﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using System;
using System.Diagnostics;
using System.Collections.Generic;
using System.IO;
using System.Linq;
using System.Threading;
using System.Windows.Forms;
using System.Text.RegularExpressions;
using App.Data;
using App.src.IO.ForDocument;
using App.Utility;
using NWCore.DataModel;
using NWCore.Viewer;

namespace App.IO
{
    #region Class for keeping the document saving information

    public class DocumentSaveInfo
    {
        #region Constructors

        /// <summary>
        /// Constructor for the information for saving documents.
        /// </summary>
        /// <param name="doc">The document to save.</param>
        /// <param name="targetPath">The target file path to save to.</param>
        public DocumentSaveInfo( IDocument doc,
                                 string targetPath )
        {
            this.Document   = doc;
            this.TargetPath = targetPath;
        }


        /// <summary>
        /// Constructor for the information for saving files.
        /// </summary>
        /// <param name="sourcePath">The source file path.</param>
        /// <param name="targetPath">The target file path to save to.</param>
        public DocumentSaveInfo( string sourcePath,
                                 string targetPath )
        {
            this.SourcePath = sourcePath;
            this.TargetPath = targetPath;
        }

        #endregion

        #region Properties

        /// <summary>Get the document to be saved.</summary>
        public IDocument Document
        {
            get { return m_doc; }
            private set
            {
                m_doc = value;

                if ( m_doc!=null )
                    this.SourcePath = m_doc.FilePath;
            }
        }


        /// <summary>Get the original file path.</summary>
        public string SourcePath
        {
            get { return m_srcPath; }
            private set
            {
                if ( string.IsNullOrEmpty(value)==true )
                {
                    m_srcPath     = string.Empty;
                    m_iSrcPathCRC = 0;
                }
                else
                {
                    m_srcPath     = string.Copy( value );
                    m_iSrcPathCRC = TheApp.CRC32Helper.ComputeCRC32Str( m_srcPath.ToLower() );
                }
            }
        }


        /// <summary>Get the CRC32 hash code of the source file path.</summary>
        public uint SourcePathCRC
        {
            get { return m_iSrcPathCRC; }
        }


        /// <summary>Get the target file path.</summary>
        public string TargetPath
        {
            get { return m_trgPath; }
            private set
            {
                if ( string.IsNullOrEmpty( value )==true )
                {
                    m_trgPath     = string.Empty;
                    m_iTrgPathCRC = 0;
                }
                else
                {
                    m_trgPath     = string.Copy( value );
                    m_iTrgPathCRC = TheApp.CRC32Helper.ComputeCRC32Str( m_trgPath.ToLower() );
                }
            }
        }


        /// <summary>Get the CRC32 hash code of the target file path.</summary>
        public uint TargetPathCRC
        {
            get { return m_iTrgPathCRC; }
        }

        #endregion

        #region Member variables

        private IDocument m_doc         = null;
        private string    m_srcPath     = string.Empty;
        private string    m_trgPath     = string.Empty;
        private uint      m_iSrcPathCRC = 0;
        private uint      m_iTrgPathCRC = 0;

        #endregion
    }

    #endregion

    /// <summary>
    /// ドキュメントセーバ。
    /// </summary>
    public class DocumentSaver
    {
        #region Constructor

        public DocumentSaver()
        {
            this.EnableMessageBox = true;
        }

        #endregion

        #region Events

        /// <summary>
        /// ドキュメント保存イベント。
        /// </summary>
        public event EventHandler DocumentSaved = null;

        /// <summary>
        /// ドキュメント保存ハンドラ。
        /// </summary>
        protected virtual void OnDocumentoSaved(EventArgs e)
        {
            if ( this.DocumentSaved!=null )
            {
                ////if ( TheApp.MainFrame.InvokeRequired==true )
                ////{
                ////    TheApp.MainFrame.Invoke( new MainFrame.EventInvoker( DocumentSaved ),
                ////                             this,
                ////                             e );
                ////}
                ////else
                {
                    this.DocumentSaved( this, e );
                }
            }
        }

        #endregion

        #region Properties

        /// <summary>
        /// Get or set the flag indicating if the warning message boxes are enabled.
        /// </summary>
        public bool EnableMessageBox { get; set; }

        #endregion

        #region Save

        /// <summary>
        /// 保存するデータの種類に応じて処理を振り分け
        /// </summary>
        /// <param name="document">The document to save.</param>
        /// <param name="filePath">The destination file path.</param>
        /// <returns>True on success.</returns>
        public bool Save( IDocument document,
                          string filePath )
        {
            DocumentPropertyChangedEventArgs changed = new DocumentPropertyChangedEventArgs();

            bool bResult = true;
            if ( document is EffectProjectDocument )
            {
                bResult = SaveWorkspace( document as EffectProjectDocument,
                                         filePath,
                                         changed );
            }
            else if ( document is GameSettingsDocument )
            {
                bResult = SavePrev( document,
                                    filePath,
                                    changed );
            }
            else
            {
                // Check if the textures were assigned to the emitters.
                if ( AreTexturesAssigned( document, MessageBoxButtons.YesNo )==DialogResult.No )
                    return false;

                bResult = Save( document,
                                filePath,
                                changed );
            }

            Document.NotifyPropertyChanged(this, changed);

            return bResult;
        }


        /// <summary>
        /// エミッタセットの保存。
        /// This method accepts a property change event argument
        /// so the user can use this method to operate upon multiple
        /// documents at the same time.
        /// </summary>
        /// <param name="document">The document to save.</param>
        /// <param name="filePath">The destination file path.</param>
        /// <param name="changed">The property change event arguments.</param>
        /// <returns>True on success.</returns>
        public bool Save( IDocument document,
                          string filePath,
                          DocumentPropertyChangedEventArgs changed )
        {
            Debug.Assert( document!=null );

            // 保存場所が決まっていない場合はSaveAsにまわす
            bool bResult = true;
            if ( Path.IsPathRooted( filePath )==false )
            {
                bResult = SaveAs( document, changed );
            }
            else
            {
                EmitterSetDocument esetDoc = document as EmitterSetDocument;
                if ( esetDoc!=null )
                {
                    // アセットのチェックをSaveInternal外で行う
                    var assetList = new List<KeyValuePair<string, string>>();

                    // リストアップ
                    var procList = new List<string> { filePath };
                    procList.AddRange(assetList.Select(x => x.Value));

                    // 保存前コマンド
                    if (EventCommandExecuter.Execute(EventCommand.PreSave, procList) != 0)
                    {
                        return false;
                    }

                    // アセットコピー
                    if (!CopyWithRetry(assetList))
                    {
                        return false;
                    }

                    // 保存実処理
                    string errMsg;
                    bResult = SaveInternal(document, filePath, changed, out errMsg);
                    if (bResult == false)
                    {
                        ShowErrorSaveFailed(filePath, errMsg);
                        return false;
                    }

                    // 保存後コマンド
                    if (EventCommandExecuter.Execute(EventCommand.PostSave, procList) != 0)
                    {
                        return false;
                    }
                }
            }

            return bResult;
        }


        /// <summary>
        /// 保存。
        /// This method accepts a property change event argument
        /// so the user can use this method to operate upon multiple
        /// documents at the same time.
        /// </summary>
        /// <param name="document">The document to save.</param>
        /// <param name="filePath">The destination file path.</param>
        /// <param name="changed">The property change event arguments.</param>
        /// <returns>True on success.</returns>
        public bool SavePrev( IDocument document,
                              string filePath,
                              DocumentPropertyChangedEventArgs changed )
        {
            Debug.Assert( document!=null );

            // 保存場所が決まっていない場合はSaveAsにまわす
            bool bResult = true;
            if ( Path.IsPathRooted( filePath )==false )
            {
                bResult = SavePrevAs( document, changed );
            }
            else
            {
                string errMsg;
                bResult = SaveInternal( document, filePath, changed, out errMsg );
                if ( bResult==false )
                    ShowErrorSaveFailed( filePath, errMsg );
            }

            return bResult;
        }


        /// <summary>
        /// ワークスペースを保存。
        /// This method accepts a property change event argument
        /// so the user can use this method to operate upon multiple
        /// documents at the same time.
        /// </summary>
        /// <param name="project">The project to save.</param>
        /// <param name="filePath">The destination file path.</param>
        /// <param name="changed">The property change event arguments.</param>
        /// <returns>True on success.</returns>
        private bool SaveWorkspace( EffectProjectDocument project,
                                    string filePath,
                                    DocumentPropertyChangedEventArgs changed )
        {
            if ( project==null )
                return false;

            // Check if all the emitter sets have had their file path set.
            foreach ( IEmitterSetDocument doc in project.EmitterSetDocuments )
            {
                // Check emitter set documents ( .eset files )
                if ( System.IO.Path.IsPathRooted( doc.FileLocation )==false )
                {
                    ////ThreadSafeMsgBox.Show( res.Strings.WARNING_MSG_SAVE_WORKFILE_NOT_ALL_FILES_SAVED,
                    ////                       res.Strings.WARNING_CAPTION,
                    ////                       MessageBoxButtons.OK,
                    ////                       MessageBoxIcon.Exclamation );
                    return false;
                }
            }

            // 保存場所が決まっていない場合はSaveAsにまわす
            bool bResult = true;
            if ( Path.IsPathRooted( filePath )==false )
            {
                bResult = SaveWorkspaceAs( project, changed );
            }
            else
            {
                string errMsg;
                bResult = SaveInternal( project,
                                        filePath,
                                        changed,
                                        out errMsg );
                if ( bResult==false )
                    ShowErrorSaveFailed( filePath, errMsg );
            }

            return bResult;
        }

        #endregion

        #region Save as...

        /// <summary>
        /// ファイル名指定保存（振り分け）。
        /// </summary>
        /// <param name="document">The document to save.</param>
        /// <returns>True on success.</returns>
        public bool SaveAs( IDocument document )
        {
            DocumentPropertyChangedEventArgs changed = new DocumentPropertyChangedEventArgs();

            bool bResult = false;
            if ( document is EffectProjectDocument )
            {
                SaveWorkspaceAs( document as EffectProjectDocument, changed );
            }
            else if ( document is GameSettingsDocument )
            {
                SavePrevAs( document, changed );
            }
            else
            {
                // Check if the textures were assigned to the emitters.
                if ( AreTexturesAssigned( document, MessageBoxButtons.YesNo )==DialogResult.No )
                    return false;

                SaveAs( document, changed );
            }

            Document.NotifyPropertyChanged(this, changed);

            return bResult;
        }

        /// <summary>
        /// 別名保存のパス確定処理のみを行います
        /// </summary>
        /// <param name="document">
        /// The document.
        /// </param>
        /// <param name="changed">
        /// The changed.
        /// </param>
        /// <param name="path">
        /// The path.
        /// </param>
        /// <returns>
        /// The <see cref="bool"/>.
        /// </returns>
        private bool PreparePathForSaveAs(
            IDocument document,
            DocumentPropertyChangedEventArgs changed,
            out string path)
        {
            path = string.Empty;

            EmitterSetDocument esetDoc = document as EmitterSetDocument;
            if (esetDoc == null)
                return false;

            // Prompt user for the location to save the document.
            SaveFileDialog dialog = new SaveFileDialog();

            dialog.DefaultExt = document.FileExt;
            dialog.FileName = document.FileName;
            dialog.Filter = document.FileFilter;
            dialog.AutoUpgradeEnabled = Config.Data.DocumentIO.AutoUpgradeEnabled;

            dialog.FileOk += new System.ComponentModel.CancelEventHandler(OnSaveFileDialogOK);

            // Initial directory for the dialog.
            if (Path.IsPathRooted(document.FileLocation) == true)
                dialog.InitialDirectory = document.FileLocation;
            else
                dialog.InitialDirectory =
                    Config.Data.DocumentIO.GetLastAccessedDir(DocumentConstants.Eset);

            // Cancelled?
            //if (dialog.ShowDialog(TheApp.MainFrame) != DialogResult.OK)
            //    return false;

            using (MCSDisableBlock disableBlock = new MCSDisableBlock())
            {
                // Remember the directory path.
                Config.Data.DocumentIO.SetLastAccessedDir(DocumentConstants.Eset,
                    dialog.FileName);

                string docName = Path.GetFileNameWithoutExtension(dialog.FileName);

                // The document name has to be the same as the file name,
                // rename the document if necessary.
                if (docName != document.Name)
                {
                    ////using (MCSEnableBlock enableBlock = new MCSEnableBlock())
                    ////{
                    ////    Viewer.Message.Close.Send(NWCore.Protocols.BinaryDataType.Ptcl,
                    ////        document.FileTitle);
                    ////}

                    ////#region Execute command for changing the document name

                    ////ICommandSet commandset = new Command.CommandSet();

                    ////// ファイル名も一緒に変更します。
                    ////commandset.Add(new Command.DocumentRenameFileTitleCommand(esetDoc.Project,
                    ////    esetDoc,
                    ////    docName));

                    ////// 名前変更コマンド
                    ////commandset.Add(new Command.DocumentRenameCommand(esetDoc.Project,
                    ////    esetDoc,
                    ////    docName));

                    ////// Execute the command set
                    ////TheApp.CommandManager.ExecuteIgnoreTargetDocument(commandset);

                    ////#endregion

                    ////using (MCSEnableBlock enableBlock = new MCSEnableBlock())
                    ////{
                    ////    Viewer.Message.SendUtility.SendEmitterSetBinary(document as IEmitterSetDocument);
                    ////}
                }
            }

            path = dialog.FileName;
            return true;
        }

        /// <summary>
        /// エミッタセットのファイル名指定保存。
        /// This method accepts a property change event argument
        /// so the user can use this method to operate upon multiple
        /// documents at the same time.
        /// </summary>
        /// <param name="document">The document to save.</param>
        /// <param name="changed">The property change event arguments.</param>
        /// <returns>True on success.</returns>
        private bool SaveAs( IDocument document,
                             DocumentPropertyChangedEventArgs changed )
        {
            string path;
            bool result = PreparePathForSaveAs(document, changed, out path);
            if (!result)
                return false;

            using ( MCSDisableBlock disableBlock = new MCSDisableBlock() )
            {
                // アセットのチェックをSaveInternal外で行う
                var assetList = new List<KeyValuePair<string, string>>();

                // リストアップ
                var procList = new List<string> { path };
                procList.AddRange(assetList.Select(x => x.Value));

                // 保存前コマンド
                if (EventCommandExecuter.Execute(EventCommand.PreSave, procList) != 0)
                {
                    return false;
                }

                // アセットコピー
                if (!CopyWithRetry(assetList))
                {
                    return false;
                }

                // 保存実処理
                string errMsg;
                result = SaveInternal(document, path, changed, out errMsg);
                if (result == false)
                {
                    ShowErrorSaveFailed(path, errMsg);
                    return false;
                }

                // 保存後コマンド
                if (EventCommandExecuter.Execute(EventCommand.PostSave, procList) != 0)
                {
                    return false;
                }
            }

            // Save the directory as the default path next time we open the file dialog.
            //Config.Data.DocumentIO.SaveAsPath = Path.GetDirectoryName(dialog.FileName);

            return true;
        }


        /// <summary>
        /// ファイル名指定保存。
        /// This method accepts a property change event argument
        /// so the user can use this method to operate upon multiple
        /// documents at the same time.
        /// </summary>
        /// <param name="document">The document to save.</param>
        /// <param name="changed">The property change event arguments.</param>
        /// <returns>True on success.</returns>
        private bool SavePrevAs( IDocument document,
                                 DocumentPropertyChangedEventArgs changed )
        {
            GameSettingsDocument doc = document as GameSettingsDocument;
            if ( doc==null )
                return false;

            // Prompt user for the location to save the document.
            SaveFileDialog dialog = new SaveFileDialog();

            dialog.DefaultExt         = document.FileExt;
            dialog.FileName           = document.FileName;
            dialog.Filter             = document.FileFilter;
            dialog.AutoUpgradeEnabled = Config.Data.DocumentIO.AutoUpgradeEnabled;

            dialog.FileOk += new System.ComponentModel.CancelEventHandler( OnSaveFileDialogOK );

            // Initial directory for the dialog.
            if ( Path.IsPathRooted( document.FileLocation )==true )
                dialog.InitialDirectory = document.FileLocation;
            else
                dialog.InitialDirectory =
                    Config.Data.DocumentIO.GetLastAccessedDir( DocumentConstants.Prev );

            // Cancelled?
            ////if ( dialog.ShowDialog(TheApp.MainFrame)!=DialogResult.OK )
            ////    return false;

            // Remember the directory path.
            Config.Data.DocumentIO.SetLastAccessedDir( DocumentConstants.Prev,
                                                       dialog.FileName );

            string docName = Path.GetFileNameWithoutExtension( dialog.FileName );

            // The document name has to be the same as the file name,
            // rename the document if necessary.
            if ( docName!=document.Name )
            {
                #region Execute command for changing the document name

                ////ICommandSet commandset = new Command.CommandSet();

                ////// 名前変更コマンド
                ////commandset.Add( new Command.DocumentRenameCommand( doc.EmitterSetDocument.Project,
                ////                                                   doc,
                ////                                                   docName ) );

                ////// Execute the command set
                ////TheApp.CommandManager.ExecuteIgnoreTargetDocument( commandset );

                doc.SetFileLocation( Path.GetDirectoryName(dialog.FileName) );

                #endregion
            }

            // Save the document.
            string errMsg;
            if ( SaveInternal(document,
                              dialog.FileName,
                              changed,
                              out errMsg)==false )
            {
                ShowErrorSaveFailed( dialog.FileName, errMsg );
                return false;
            }

            return true;
        }


        /// <summary>
        /// ワークスペースをファイル名指定保存。
        /// This method accepts a property change event argument
        /// so the user can use this method to operate upon multiple
        /// documents at the same time.
        /// </summary>
        /// <param name="project">The document to save.</param>
        /// <param name="changed">The property change event arguments.</param>
        /// <returns>True on success.</returns>
        private bool SaveWorkspaceAs( EffectProjectDocument project,
                                      DocumentPropertyChangedEventArgs changed )
        {
            // 上書き確認がファイルダイアログで行われる
            Debug.Assert( project!=null );

            // Check if all the emitter sets have had their file path set.
            foreach ( IEmitterSetDocument doc in project.EmitterSetDocuments )
            {
                // Check emitter set documents ( .eset files )
                if ( System.IO.Path.IsPathRooted( doc.FileLocation )==false )
                {
                    ////ThreadSafeMsgBox.Show( res.Strings.WARNING_MSG_SAVE_WORKFILE_NOT_ALL_FILES_SAVED,
                    ////                       res.Strings.WARNING_CAPTION,
                    ////                       MessageBoxButtons.OK,
                    ////                       MessageBoxIcon.Exclamation );
                    return false;
                }
            }

            SaveFileDialog dialog = new SaveFileDialog();
            dialog.DefaultExt = project.FileExt;
            dialog.FileName   = project.FileName;
            dialog.Filter     = project.FileFilter;
            dialog.AutoUpgradeEnabled = Config.Data.DocumentIO.AutoUpgradeEnabled;

            dialog.FileOk += new System.ComponentModel.CancelEventHandler( OnSaveFileDialogOK );

            if ( Path.IsPathRooted( project.FileLocation )==true )
                dialog.InitialDirectory = project.FileLocation;
            else
                dialog.InitialDirectory =
                    Config.Data.DocumentIO.GetLastAccessedDir( DocumentConstants.FEnv );

            ////if ( dialog.ShowDialog(TheApp.MainFrame)!=DialogResult.OK )
            ////    return false;

            // Remember the directory path.
            Config.Data.DocumentIO.SetLastAccessedDir( DocumentConstants.FEnv,
                                                       dialog.FileName );

            // Save the document.
            string errMsg;
            if ( SaveInternal( project,
                               dialog.FileName,
                               changed,
                               out errMsg )==false )
            {
                ShowErrorSaveFailed( dialog.FileName, errMsg );
                return false;
            }

            return true;
        }

        #endregion

        #region The internal save method

        /// <summary>
        /// Internal save method. This method is the method that actually
        /// saves the document.
        ///
        /// If bExport is set to true, the saving will not have any affect to
        /// the document, for example, the file path will not be set, the modify
        /// flag will not be removed.
        /// </summary>
        /// <param name="doc">The document to save.</param>
        /// <param name="filePath">The destination file path.</param>
        /// <param name="changed">The property change event arguments.</param>
        /// <param name="errMsg">The error message if any.</param>
        /// <param name="bExport">True to export the document.</param>
        /// <returns>True on success.</returns>
        private bool SaveInternal( IDocument doc,
                                   string filePath,
                                   DocumentPropertyChangedEventArgs changed,
                                   out string errMsg,
                                   bool bExport = false )
        {
            errMsg = string.Empty;

            // Verify the texture location.
            //EmitterSetDocument esetDoc = doc as EmitterSetDocument;
            //if ( esetDoc!=null && bExport==false )
            //{
            //    VerifyAssetsLocation( esetDoc, filePath );
            //}

            // Show information on the status bar.
            UpdateStatusBar(filePath);

            bool bResult = true;

            DateTime start = DateTime.Now;
            try
            {
                if ( bExport==false )
                {
                    CheckRename( doc, filePath );

                    // Update the document.
                    doc.UpdateDocument();
                }

                // Save the document.
                if ( doc.SaveDocument(filePath,
                                      !bExport,
                                      true)==false )
                {
                    MessageLogger.WriteWarning( "{0} : \n\t{1}\n",
                                                res.Strings.IO_ERR_FileDoesNotExport,
                                                filePath );
                    bResult = false;
                }

                if ( bExport==false )
                {
                    // File path changed?
                    if ( doc.FilePath!=filePath )
                        doc.SetFileLocation( DocumentConstants.GetDirectoryName( filePath ) );

                    changed.AddArgs( new DocumentHeaderChangedArgs( doc ) );
                }

                if ( bResult==true )
                {
                    // Trigger the event.
                    OnDocumentoSaved( EventArgs.Empty );

                    // Compute the time spent.
                    TimeSpan time = DateTime.Now - start;

                    // Show log.
                    DebugConsole.WriteLine( "{0} was saved successfully.( time spent {1,4:f3} seconds )",
                                            doc.FileName,
                                            time.TotalSeconds );
                }
            }
            catch ( SystemException ex )
            {
                errMsg = ex.Message;

                #if DEBUG
                    errMsg += "\nCall stack :\n" + ex.StackTrace;
                #endif

                bResult = false;
            }

            // Clear the modify flag
            if ( bExport==false &&
                 bResult==true )
            {
                // Clear the modification flags for the saved documents.
                DocumentManager.ClearDataModifyFlag( doc.DataScrPath );
            }

            var document = doc as Document;
            if (document != null && bExport == false)
                document.NotifyDocumentSaveFinished(!bResult);

            return bResult;
        }

        #endregion

        #region Utility methods

        /// <summary>
        /// 保存対象のファイルリストを取得します。
        /// </summary>
        /// <param name="project">対象とするプロジェクトです。</param>
        /// <param name="bCheckModification">
        /// True to filter out the documents those were not modified.
        /// </param>
        /// <param name="bIncludeGameSettingsDoc">True to include game settings documents.</param>
        /// <returns>保存対象のリスト</returns>
        private List<IDocument> GetDocumentSaveList( IProjectDocument project,
                                                     bool bCheckModification,
                                                     bool bIncludeGameSettingsDoc = false )
        {
            List<IDocument> documents       = new List<IDocument>();
            List<IDocument> effects         = new List<IDocument>();
            List<IDocument> gameSettingsDoc = new List<IDocument>();
            // 各ドキュメントごとに集める
            foreach (IDocument item in project.SubDocuments)
            {
                if ( String.IsNullOrEmpty(item.FileExt)==false )
                {
                    if ( item is IEmitterSetDocument)
                    {
                        // Emitter set document.
                        if ( bCheckModification==false ||
                             item.Modified==true )
                        {
                            effects.Add( item );
                        }

                        // Game settings documents.
                        if ( bIncludeGameSettingsDoc==true )
                        {
                            EmitterSetDocument eset = item as EmitterSetDocument;
                            foreach ( GameSettingsDocument childDoc in eset.GameSettingsDocuments )
                            {
                                if ( bCheckModification==false ||
                                     childDoc.Modified==true )
                                {
                                    gameSettingsDoc.Add( childDoc );
                                }
                            }
                        }
                    }
                }
            }

            if ( effects.Count>0 )
            {
                documents.AddRange(effects.ToArray());
            }

            if ( gameSettingsDoc.Count>0 )
            {
                documents.AddRange( gameSettingsDoc.ToArray() );
            }

            return documents;
        }


        /// <summary>
        /// Update status bar on the bottom of the main frame.
        /// </summary>
        /// <param name="filePath">The file path of the document just being saved.</param>
        protected virtual void UpdateStatusBar(string filePath)
        {
            ////if ( TheApp.MainFrame==null )
            ////    return;

            ////TheApp.MainFrame.StatusMessage =
            ////    StringResource.Get( "IO_SAVING_MSG",
            ////                        Path.GetFileName(filePath) );
        }


        /// <summary>
        /// Check if the file name or path of the document needs to be changed.
        /// If so, apply the change to the document.
        /// </summary>
        /// <param name="document">The document.</param>
        /// <param name="filePath">The destination file path.</param>
        protected virtual void CheckRename( IDocument document,
                                            string filePath )
        {
            string name = Path.GetFileNameWithoutExtension(filePath);

            // Was the file name changed?
            if ( document.Name==name )
                return;

            // Fire file name changed event.
            DocumentNameChangedArgs nameChangedArgs =
                new DocumentNameChangedArgs( document, document.Name );

            document.FileTitle = name;
            Document.NotifyPropertyChanged(null, nameChangedArgs);
        }


        /// <summary>
        /// セーブ失敗エラーの表示。
        /// </summary>
        /// <param name="filePath">The file path.</param>
        /// <param name="message">The error message to output.</param>
        private void ShowErrorSaveFailed( string filePath,
                                          string message )
        {
            if ( this.EnableMessageBox==false )
                return;

            ////string fullMessage = string.Format(res.Strings.IO_SAVE_FAILED_MSG, Path.GetFileName(filePath) + "\n" + message);
            ////ThreadSafeMsgBox.Show( fullMessage,
            ////                       res.Strings.ERROR_CAPTION,
            ////                       System.Windows.Forms.MessageBoxButtons.OK,
            ////                       System.Windows.Forms.MessageBoxIcon.Error );
        }

        #endregion

        #region Emitter asset related utility methods

        /// <summary>
        /// Check if the textures in the emitter set document are already assigned.
        /// </summary>
        /// <param name="doc">The emitter set document.</param>
        /// <returns>True if the textures are assigned or user choose to continue.</returns>
        private DialogResult AreTexturesAssigned( IDocument doc,
                                                  MessageBoxButtons dlgButtons )
        {
            EmitterSetDocument esetDoc = doc as EmitterSetDocument;
            if ( esetDoc==null )
                return DialogResult.None;

            return DialogResult.None;
        }

        /// <summary>
        /// Verify the assets locations, if they are not in the search paths,
        /// prompt user if overwrite or copy the asset to search path.
        /// </summary>
        /// <param name="doc">The emitter set document.</param>
        /// <param name="esetPath">The location of the emitter set file.</param>
        /// <returns>True on success.</returns>
        private List<KeyValuePair<string, string>>
            VerifyAssetsLocation(EmitterSetDocument doc, string esetPath)
        {
            return new List<KeyValuePair<string, string>>();
        }

        /// <summary>
        /// 10回リトライ付きのアセットコピーを行います。
        /// </summary>
        /// <param name="assetList">アセットコピーリスト</param>
        /// <returns>成功したらtrue,失敗したらfalse.</returns>
        private bool CopyWithRetry(List<KeyValuePair<string, string>> assetList)
        {
            foreach (var pair in assetList)
            {
                bool result = true;
                for (int i = 0; i < 10; ++i)
                {
                    try
                    {
                        File.Copy(pair.Key, pair.Value, true);
                    }
                    catch
                    {
                        result = false;
                        Thread.Sleep(100);
                    }

                    if (result)
                    {
                        break;
                    }
                }

                if (!result)
                {
                    ////ThreadSafeMsgBox.Show(
                    ////    string.Format(res.Strings.ERROR_FILE_COPY_EXCEPTION, pair.Value),
                    ////    res.Strings.ERROR_FILE_COPY_EXCEPTION_CAPTION,
                    ////    MessageBoxButtons.OK,
                    ////    MessageBoxIcon.Error);
                    TheApp.Logger.Error.AddMessage(
                        string.Format(res.Strings.ERROR_FILE_COPY_EXCEPTION, pair.Value));
                    return false;
                }
            }

            return true;
        }

        #endregion

        #region Check emitter immortal & ending properties

        /// <summary>
        /// Check the immortal & ending properties of the emitter documents.
        /// Show warning dialog if necessary.
        /// </summary>
        /// <param name="eset">The owner emitter set document.</param>
        /// <param name="dlgButtons">The dialog button options.</param>
        /// <returns>
        /// DialogResult.None to proceed on saving or one of the dialog buttons given.
        /// </returns>
        private DialogResult CheckEmitterImmortalEndingProperties( EmitterSetDocument eset,
                                                                   MessageBoxButtons dlgButtons )
        {
            if ( m_bSuppressImmortalWarningDialog==true )
                return DialogResult.None;

            string emitterNames = string.Empty;
            foreach ( IEmitterDocument emitter in eset.EmitterDocuments )
            {
                if ( emitter==null || emitter.EmitterData==null )
                    continue;

                if ( emitter.EmitterData.AnimEditData.UI_lifeInfinit==(uint)1 &&
                     emitter.EmitterData.UI_fadeOutType==NWCore.DataModel.FadeOutType.StopEmit )
                {
                    emitterNames += (emitterNames.Length>0 ? "\n" : string.Empty) +
                                    eset.Name + ":" + emitter.Name;
                }
            }

            ////if ( string.IsNullOrEmpty(emitterNames)==false )
            ////{
            ////    DialogResult result =
            ////        ThreadSafeMsgBox.Show( emitterNames + "\n" + res.Strings.WARNING_EMITTER_IMMORTAL_ENDING_PROPERTIES,
            ////                               res.Strings.WARNING_CAPTION,
            ////                               dlgButtons,
            ////                               MessageBoxIcon.Exclamation );
            ////    return result;
            ////}

            return DialogResult.None;
        }

        #endregion

        #region Event handler for save file dialog

        /// <summary>
        /// Handle FileOK event for the save file dialogs.
        /// </summary>
        /// <param name="sender">The sender of the event.</param>
        /// <param name="e">The event arguments.</param>
        private void OnSaveFileDialogOK( object sender,
                                         System.ComponentModel.CancelEventArgs e )
        {
            SaveFileDialog dialog = sender as SaveFileDialog;
            if ( dialog==null )
                return;

            string filePath = dialog.FileName;

            Regex validateFileName = new Regex( res.Strings.REGEXP_DOCUMENT_NAME_VALIDATION );

            if ( validateFileName.IsMatch(Path.GetFileNameWithoutExtension(filePath))==false )
            {
                ////ThreadSafeMsgBox.Show( TheApp.MainFrame,
                ////                       res.Strings.DEFAULT_INVALID_NAME_ERROR_MSG,
                ////                       res.Strings.MSGBOX_CAPTION_NAME_INPUT_DIALOG_INVALID_NAME,
                ////                       System.Windows.Forms.MessageBoxButtons.OK,
                ////                       System.Windows.Forms.MessageBoxIcon.Exclamation );
                e.Cancel = true;
            }

            // Filename length check
            if (Path.GetFileNameWithoutExtension(filePath).Length > DocumentIO.s_MaxFileNameLength)
            {
                DocumentIO.ErrorFileNameLengthSave(filePath);
                e.Cancel = true;
            }
        }

        #endregion

        #region Member variables

        private bool m_bSuppressImmortalWarningDialog = false;

        #endregion
    }

    #region Class for showing save progress on the bottom of main frame

    /// <summary>
    /// ドキュメントセーバステータスインジケータクラス。
    /// </summary>
    public sealed class DocumentSaverStatusIndicator : StatusIndicator
    {
        /// <summary>
        /// コンストラクタ。
        /// </summary>
        public DocumentSaverStatusIndicator( DocumentSaver saver,
                                             int stepCount ) :
            base(stepCount)
        {
            m_saver = saver;
            if ( m_saver!=null )
                m_saver.DocumentSaved += new EventHandler( OnDocumentSaved );
        }


        /// <summary>
        /// 破棄。
        /// </summary>
        public override void Dispose()
        {
            if ( m_saver!=null )
                m_saver.DocumentSaved -= new EventHandler( OnDocumentSaved );

            base.Dispose();
        }


        /// <summary>
        /// Handle DocumentSaved event sent from the saver.
        /// </summary>
        /// <param name="sender">The sender of the event.</param>
        /// <param name="e">The event arguments.</param>
        private void OnDocumentSaved(object sender, EventArgs e)
        {
            AdvanceProgress();
        }


        private DocumentSaver m_saver = null;
    }

    #endregion
}
