﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using System;
using System.IO;
using System.Xml;
using System.Collections.Generic;
using System.Threading;

using NWCore.DataModel;
using NWCore.Serializer;

using App.IO;

namespace App.Data
{
    /// <summary>
    /// Manager class for auto save features.
    /// </summary>
    public sealed class AutoSaveManager
    {
        #region Class for storing the auto saved document data

        /// <summary>
        /// Class for storing the auto saved document data.
        /// </summary>
        private class AutoSaveDocData
        {
            #region Constructor

            /// <summary>
            /// Constructor.
            /// </summary>
            /// <param name="doc">The document to keep track of.</param>
            /// <param name="autoSavePath">The auto save file path.</param>
            public AutoSaveDocData( IDocument doc,
                                    string autoSavePath )
            {
                // We initialize the LastAutoSaveTime to the minimum value
                // so that the document will be auto saved once any change
                // has made, instead of waiting until exceeding the minimum
                // auto save interval.

                this.Document         = doc;
                this.DocumentName     = doc.Name;
                this.AutoSaveFilePath = autoSavePath;
                this.OriginalFilePath = doc.FileLocation;
                this.LastAutoSaveTime = DateTime.Now.ToUniversalTime();
            }

            #endregion

            #region Properties

            /// <summary>Get or set the document.</summary>
            public IDocument Document { get; set; }


            /// <summary>Get or set the name of the document.</summary>
            public string DocumentName { get; set; }


            /// <summary>Get or set the auto saved file path.</summary>
            /// <remarks>
            /// This file path contains the full path to the file,
            /// including the path to the folder, the file name and the file extension.
            /// </remarks>
            public string AutoSaveFilePath
            {
                get { return m_autoSaveFilePath; }
                set
                {
                    if ( string.IsNullOrEmpty(value)==true )
                    {
                        m_autoSaveFilePath  = string.Empty;
                        AutoSaveFilePathCRC = 0;
                    }
                    else
                    {
                        m_autoSaveFilePath  = value;
                        AutoSaveFilePathCRC =
                            TheApp.CRC32Helper.ComputeCRC32Str( value.ToLower() );
                    }
                }
            }


            /// <summary>Get or set the original file path of the document.</summary>
            /// <remarks>
            /// The original file path contains only the path to the folder,
            /// the file name is not included.
            /// </remarks>
            public string OriginalFilePath
            {
                get { return m_originalFilePath; }
                set
                {
                    if ( string.IsNullOrEmpty(value)==true )
                    {
                        m_originalFilePath  = string.Empty;
                        AutoSaveFilePathCRC = 0;
                    }
                    else
                    {
                        m_originalFilePath  = value;
                        OriginalFilePathCRC =
                            TheApp.CRC32Helper.ComputeCRC32Str( value.ToLower() );
                    }
                }
            }


            /// <summary>Get or set the CRC32 hash code of the auto saved file path.</summary>
            public uint AutoSaveFilePathCRC { get; set; }


            /// <summary>Get or set the CRC32 hash code of the original file path.</summary>
            public uint OriginalFilePathCRC { get; set; }


            /// <summary>Get the latest time when the document being auto saved.</summary>
            public DateTime LastAutoSaveTime { get; private set; }

            #endregion

            #region Utility methods

            /// <summary>
            /// Update the data while the document has been auto saved.
            /// The auto save time will be updated to the time this method
            /// is called.
            /// </summary>
            public void SetAutoSaved()
            {
                this.LastAutoSaveTime = DateTime.Now.ToUniversalTime();
                this.DocumentName     = this.Document.Name;
                this.OriginalFilePath = this.Document.FileLocation;
            }

            #endregion

            #region Member variables

            private string m_autoSaveFilePath = string.Empty;
            private string m_originalFilePath = string.Empty;

            #endregion
        }

        #endregion

        #region Constructor

        /// <summary>
        /// Constructor.
        /// </summary>
        public AutoSaveManager()
        {
        }

        #endregion

        #region Properties

        /// <summary>Get or set the flag indicating if auto save service is enabled.</summary>
        public bool IsEnabled
        {
            get { return m_bEnabled; }
            set { m_bEnabled = value; }
        }


        /// <summary>Get or set the flag indicating if should delete expired files.</summary>
        public bool IsDeleteExpiredFilesEnabled
        {
            get { return m_bDelExpiredFiles; }
            set { m_bDelExpiredFiles = value; }
        }

        #endregion

        #region Init / deinit

        /// <summary>
        /// Initialize.
        /// </summary>
        /// <returns>True on success.</returns>
        public bool Init()
        {
            UpdateAppSettings();

            if ( m_bDelExpiredFiles==true )
                DeleteExpiredAutoSaveFiles();

            TheApp.ConfigChanged += OnAppConfigChanged;

            // Start the thread.
            m_thread = new Thread( new ThreadStart(this.ThreadLoop) );
            m_thread.Start();

            return true;
        }


        /// <summary>
        /// Stop the tasks and threads to destroy the manager.
        /// </summary>
        public void Deinit()
        {
            m_queue.Clear();

            // Is out thread still alive?
            if ( m_thread!=null && m_thread.IsAlive==true )
            {
                m_bThreadFinishing = true;

                // Wait 0.1 seconds before we force terminating the thread.
                m_thread.Join( 100 );
            }

            TheApp.ConfigChanged -= OnAppConfigChanged;
        }

        #endregion

        #region Documents to be auto saved

        /// <summary>
        /// Register document for auto save.
        /// </summary>
        /// <param name="doc">The document to be auto saved.</param>
        public void RegisterAutoSaveDoc( IDocument doc )
        {
            if ( doc==null )
                return;

            // Already registered?
            if ( FindAutoSaveData(doc)!=null )
                return;

            // Get the application auto save folder.
            string autoSaveFolder = TheApp.AutoSaveFolderPath;
            if ( string.IsNullOrEmpty(autoSaveFolder)==true )
                return;

            // Compose the auto saved file name.
            string fileName = doc.FileName;
            string filePath = Path.Combine( autoSaveFolder, fileName );

            // Create the auto save document data object.
            AutoSaveDocData data = new AutoSaveDocData( doc, filePath );

            // Push to the first element, so that it would be the first to be saved.
            m_queue.AddFirst( data );
        }


        /// <summary>
        /// Unregister the document from the auto save queue.
        /// </summary>
        /// <param name="doc">The document to unregister.</param>
        public void UnregisterAutoSaveDoc( IDocument doc )
        {
            AutoSaveDocData data = FindAutoSaveData( doc );
            if ( data==null )
                return;

            m_queue.Remove( data );
        }


        /// <summary>
        /// Find the auto save data with the given document.
        /// </summary>
        /// <param name="doc">The document.</param>
        /// <returns>The auto save data.</returns>
        private AutoSaveDocData FindAutoSaveData( IDocument doc )
        {
            if ( doc==null )
                return null;

            foreach ( AutoSaveDocData data in m_queue )
            {
                if ( data!=null && data.Document==doc )
                {
                    return data;
                }
            }

            return null;
        }

        #endregion

        #region Auto save documents

        /// <summary>
        /// Auto save the document in the given auto save data.
        /// </summary>
        /// <param name="data">The auto save data.</param>
        /// <returns>True on success.</returns>
        private bool DoAutoSave( AutoSaveDocData data )
        {
            if ( data==null || data.Document==null )
                return false;

            // Does the auto save folder exist?
            string autoSaveFolderPath = TheApp.AutoSaveFolderPath;
            if ( string.IsNullOrEmpty(autoSaveFolderPath)==true )
                return false;

            try
            {
                using ( MemoryStream stream = new MemoryStream() )
                {
                    // Now backup the document to our auto save folder.
                    bool bSuccess = data.Document.SaveDocument( data.AutoSaveFilePath,
                                                                false,
                                                                false );
                    if ( bSuccess==false )
                        return false;
                }
            }
            catch ( Exception ex )
            {
                string msg = string.Format( res.Strings.WARNING_FAILED_WRITING_AUTOSAVE_FILE,
                                            data.Document.GetType().Name,
                                            data.DocumentName,
                                            ex.Message );
                TheApp.OutputLogMsg( NWCore.LogLevels.Error, msg );
                return false;
            }

            // Output debug message.
            string debugMsg = string.Format( "AutoSaveManager : {0}({1}) has been auto saved.",
                                             data.Document.GetType().Name,
                                             data.Document.Name,
                                             data.AutoSaveFilePath );
            TheApp.OutputLogMsg( NWCore.LogLevels.Debug, debugMsg );

            // Update the auto save data.
            data.SetAutoSaved();

            return true;
        }

        #endregion

        #region Delete backup files

        /// <summary>
        /// Delete expired backup files.
        /// </summary>
        public void DeleteExpiredAutoSaveFiles()
        {
            // Get the application auto save folder.
            string autoSaveFolder = TheApp.AutoSaveFolderPath;
            if ( string.IsNullOrEmpty(autoSaveFolder)==true )
                return;

            try
            {
                string[] paths = Directory.GetFiles( autoSaveFolder );
                foreach ( string filePath in paths )
                {
                    try
                    {
                        DateTime lastAccessedTime = File.GetLastAccessTimeUtc( filePath );

                        TimeSpan elapsed = DateTime.Now.ToUniversalTime() - lastAccessedTime;

                        // The auto saved file is out-dated, delete it.
                        if ( elapsed.TotalDays>=m_fNumDaysExpire )
                            File.Delete( filePath );
                    }
                    catch
                    {
                        // Ignore file access failures.
                        continue;
                    }
                }
            }
            catch
            {
                // Ignore file access failures.
            }
        }


        /// <summary>
        /// Clear all the auto save files.
        /// </summary>
        public void ClearAutoSaveFiles()
        {
            // Get the application auto save folder.
            string autoSaveFolder = TheApp.AutoSaveFolderPath;
            if ( string.IsNullOrEmpty(autoSaveFolder)==true )
                return;

            try
            {
                string[] paths = Directory.GetFiles( autoSaveFolder );
                foreach ( string filePath in paths )
                {
                    try
                    {
                        File.Delete( filePath );
                    }
                    catch
                    {
                        // Ignore file access failures.
                        continue;
                    }
                }
            }
            catch
            {
                // Ignore file access failures.
            }
        }

        #endregion

        #region Thread loop

        /// <summary>
        /// The main loop for the thread.
        /// </summary>
        private void ThreadLoop()
        {
            // The main loop.
            while ( m_bThreadFinishing==false )
            {
                if ( m_bEnabled==false )
                {
                    Thread.Sleep( 500 );
                    continue;
                }

                LinkedListNode<AutoSaveDocData> node = m_queue.First;
                if ( node!=null )
                {
                    if ( node.Value==null )
                    {
                        // Shouldn't really come here...
                        m_queue.RemoveFirst();
                    }
                    else if ( ShouldAutoSave(node.Value)==true )
                    {
                        // Save the document then put it to the end of the queue.
                        m_queue.RemoveFirst();

                        DoAutoSave( node.Value );

                        m_queue.AddLast( node );
                    }
                }

                if ( m_queue.Count<=0 )
                    Thread.Sleep( 1000 );
                else
                    Thread.Sleep( 50 );
            }
        }


        /// <summary>
        /// Check if the given document should be backed up.
        /// </summary>
        /// <param name="data">The auto save document data.</param>
        /// <returns>True if should be backed up.</returns>
        private bool ShouldAutoSave( AutoSaveDocData data )
        {
            if ( data==null )
                return false;

            TimeSpan elapsed = DateTime.Now.ToUniversalTime() - data.LastAutoSaveTime;
            if ( elapsed.TotalSeconds>=m_fAutoSaveInterval )
                return true;

            return false;
        }

        #endregion

        #region Application settings

        /// <summary>
        /// Update settings from the application configs.
        /// </summary>
        private void UpdateAppSettings()
        {
            if ( Config.Data.AutoSave==null )
                Config.Data.AutoSave = new CfAutoSave();

            m_fAutoSaveInterval = (double)Config.Data.AutoSave.AutoSaveInterval * 60.0;
            m_fNumDaysExpire    = (double)Config.Data.AutoSave.DaysUntilFileExpires;

            this.IsEnabled                   = Config.Data.AutoSave.EnableAutoSave;
            this.IsDeleteExpiredFilesEnabled = Config.Data.AutoSave.EnableDeleteExpiredFiles;
        }

        #endregion

        #region Event handlers

        /// <summary>
        /// Handle ConfigChanged event from the application.
        /// </summary>
        /// <param name="sender">The sender of the event.</param>
        /// <param name="e">The event arguments.</param>
        private void OnAppConfigChanged( object sender,
                                         EventArgs e )
        {
            UpdateAppSettings();
        }

        #endregion

        #region Member variables

        private const string AUTO_SAVE_DATA_FILENAME = "AutoSaveData.xml";

        private bool   m_bEnabled          = true;
        private bool   m_bDelExpiredFiles  = true;

        // The minimum time span in seconds from the last auto save of a document.
        // One minute by default.
        private double m_fAutoSaveInterval = 60.0;
        private double m_fNumDaysExpire    = 3.0;

        private Thread m_thread            = null;
        private bool   m_bThreadFinishing  = false;

        private LinkedList<AutoSaveDocData> m_queue = new LinkedList<AutoSaveDocData>();

        #endregion
    }
}
