﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using System;
using System.Collections.Generic;
using System.Collections.ObjectModel;
using System.ComponentModel;
using System.Diagnostics;
using System.Dynamic;
using System.Linq;
using System.Reflection;
using System.Runtime.CompilerServices;

using EffectMaker.DataModelMaker.Core.Definitions;

using EffectMaker.Foundation.Dynamic;
using EffectMaker.Foundation.Log;
using EffectMaker.Foundation.Utility;

namespace EffectMaker.DataModelMaker.UILogic.ViewModels
{
    /// <summary>
    /// Base class for the view models.
    /// </summary>
    public abstract class ViewModelBase : DynamicObject, INotifyPropertyChanged, IDisposable
    {
        /// <summary>The data model.</summary>
        private DefinitionBase dataModel;

        /// <summary>
        /// Dictionary of data model properties.
        /// </summary>
        private Dictionary<string, PropertyInfo> dataModelPropInfoMap =
            new Dictionary<string, PropertyInfo>();

        /// <summary>
        /// Dictionary of view model properties.
        /// </summary>
        private Dictionary<string, PropertyInfo> viewModelPropInfoMap =
            new Dictionary<string, PropertyInfo>();

        /// <summary>
        /// Initializes the ViewModelBase instance.
        /// </summary>
        /// <param name="parent">The parent view model.</param>
        /// <param name="dataModel">The encapsulated data model.</param>
        public ViewModelBase(ViewModelBase parent, DefinitionBase dataModel)
        {
            this.Parent = parent;
            this.Children = new ObservableCollection<ViewModelBase>();
            this.dataModel = dataModel;
            this.UpdateDataModelPropertyDescriptorMap();
            this.UpdateMyDataModelPropertyDescriptorMap();
        }

        /// <summary>
        /// PropertyChangedEventHandler.
        /// </summary>
        public event PropertyChangedEventHandler PropertyChanged;

        /// <summary>
        /// Get the parent view model.
        /// </summary>
        public ViewModelBase Parent { get; protected set; }

        /// <summary>
        /// Get the child view models.
        /// </summary>
        public ObservableCollection<ViewModelBase> Children { get; protected set; }

        /// <summary>
        /// Gets the encapsulated data model.
        /// </summary>
        public DefinitionBase DataModel
        {
            get { return this.dataModel; }
        }

        /// <summary>
        /// Disposes the instance.
        /// </summary>
        public virtual void Dispose()
        {
            foreach (ViewModelBase child in this.Children)
            {
                if (child == null)
                {
                    continue;
                }

                child.Dispose();
            }
        }

        /// <summary>
        /// Try to get the requested property value from the data model.
        /// </summary>
        /// <param name="binder">The get member binder.</param>
        /// <param name="result">The property value.</param>
        /// <returns>True on success.</returns>
        public override bool TryGetMember(
            GetMemberBinder binder,
            out object result)
        {
            result = null;
            PropertyInfo info = null;

            // Try to get the descriptor of the requested property.
            var effectMakerBinder = binder as EffectMakerGetMemberBinder;
            if (effectMakerBinder != null &&
                effectMakerBinder.ShouldGetNonDynamicMember == true &&
                this.viewModelPropInfoMap.TryGetValue(binder.Name, out info) == true)
            {
                result = info.GetValue(this, null);
                return true;
            }
            else if (this.dataModel != null &&
                this.dataModelPropInfoMap.TryGetValue(binder.Name, out info) == true)
            {
                result = info.GetValue(this.DataModel, null);
                return true;
            }
            else
            {
                Logger.Log(LogLevels.Error, "ViewModelBase.TryGetMember : Getting non-existing member '{0}' from view model '{1}'", binder.Name, this.GetType().Name);
                return false;
            }
        }

        /// <summary>
        /// Try to set value to the property from the data model.
        /// </summary>
        /// <param name="binder">The set member binder.</param>
        /// <param name="value">The value to set.</param>
        /// <returns>True on success.</returns>
        public override bool TrySetMember(
            SetMemberBinder binder,
            object value)
        {
            // Is the binder our binder? There are some extra parameters that we can use.
            var effectMakerBinder = binder as EffectMakerSetMemberBinder;

            // Get the original value.
            var getMemberBinder = new EffectMakerGetMemberBinder(binder.Name);
            if (effectMakerBinder != null && effectMakerBinder.ShouldSetNonDynamicMember == true)
            {
                getMemberBinder.ShouldGetNonDynamicMember = true;
            }

            object origValue;
            if (this.TryGetMember(getMemberBinder, out origValue) == false)
            {
                Debug.Assert(false, "ViewModelBase.TrySetMember : Failed getting member '{0}' from view model '{1}' for modification checking.", binder.Name, this.GetType().Name);
                return false;
            }

            // Clone the original value if possible.
            if (origValue is ICloneable)
            {
                origValue = (origValue as ICloneable).Clone();
            }

            // Check if the new value is different from the current value.
            object propertyOwner = null;
            if (object.Equals(origValue, value) == false)
            {
                // Set the new value...
                PropertyInfo info = null;
                if (this.viewModelPropInfoMap.TryGetValue(binder.Name, out info) == true &&
                    effectMakerBinder != null &&
                    effectMakerBinder.ShouldSetNonDynamicMember == true)
                {
                    propertyOwner = this;
                }
                else if (this.dataModel != null &&
                    this.dataModelPropInfoMap.TryGetValue(binder.Name, out info) == true)
                {
                    propertyOwner = this.DataModel;
                }
                else
                {
                    return false;
                }

                // Try to convert the value to correct type.
                var canAssign = TypeConversionUtility.TryConvert(
                    value.GetType(),
                    info.PropertyType,
                    ref value);

                if (canAssign == false)
                {
                    Logger.Log(LogLevels.Error, "ViewModelBase.TrySetMember : Failed setting incompatible value type to member '{0}' of view model '{1}'", binder.Name, this.GetType().Name);
                    return false;
                }

                info.SetValue(propertyOwner, value, null);

                // ...and raise an event to tell the value changed.
                this.OnPropertyChanged(binder.Name);
            }

            return true;
        }

        /// <summary>
        /// Attributes a value to a field.
        /// Raises a PropertyChanged event if the field value changed.
        /// </summary>
        /// <typeparam name="T">Type of the property.</typeparam>
        /// <param name="field">The field that stores the property value.</param>
        /// <param name="value">The new value to set.</param>
        /// <param name="propertyName">The name of the property to set.</param>
        /// <returns>Returns true if the value changed, false otherwise.</returns>
        protected bool SetValue<T>(
            ref T field,
            T value,
            [CallerMemberName]string propertyName = null)
        {
            if (EqualityComparer<T>.Default.Equals(field, value))
            {
                return false;
            }

            field = value;
            this.OnPropertyChanged(propertyName);

            return true;
        }

        /// <summary>
        /// Method to be called when the value of a property changes.
        /// </summary>
        /// <param name="propertyName">The name of the property that changed.</param>
        protected virtual void OnPropertyChanged([CallerMemberName]string propertyName = null)
        {
            var handler = this.PropertyChanged;
            if (handler != null)
            {
                handler(this, new PropertyChangedEventArgs(propertyName));
            }
        }

        /// <summary>
        /// Update the property descriptor map for the data model.
        /// </summary>
        private void UpdateDataModelPropertyDescriptorMap()
        {
            this.dataModelPropInfoMap.Clear();

            if (this.dataModel == null)
            {
                return;
            }

            // Get all the PropertyInfo from the data model.
            PropertyInfo[] properties = this.dataModel.GetType().GetProperties();

            // Save the property descriptors.
            foreach (var propertyInfo in properties)
            {
                this.dataModelPropInfoMap.Add(propertyInfo.Name, propertyInfo);
            }
        }

        /// <summary>
        /// Update the property descriptor map for the my data model.
        /// </summary>
        private void UpdateMyDataModelPropertyDescriptorMap()
        {
            this.viewModelPropInfoMap.Clear();

            var properties = this.GetType().GetProperties();

            foreach (PropertyInfo propertyInfo in properties.Distinct(
                new PropertyEqualityComparer()))
            {
                this.viewModelPropInfoMap.Add(propertyInfo.Name, propertyInfo);
            }
        }
    }

    /// <summary>
    /// Compare property names.
    /// </summary>
    public class PropertyEqualityComparer : IEqualityComparer<PropertyInfo>
    {
        /// <summary>
        /// Compare.
        /// </summary>
        /// <param name="x">The property.</param>
        /// <param name="y">The other property.</param>
        /// <returns>True if both properties have the same name.</returns>
        public bool Equals(PropertyInfo x, PropertyInfo y)
        {
            if (x == null || y == null)
            {
                return false;
            }

            return x.Name == y.Name;
        }

        /// <summary>
        /// Get hash code.
        /// </summary>
        /// <param name="obj">The object to get hash code.</param>
        /// <returns>The hash code.</returns>
        public int GetHashCode(PropertyInfo obj)
        {
            if (obj == null)
            {
                return 0;
            }

            return obj.Name.GetHashCode();
        }
    }
}
