﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using System;
using System.Collections.Generic;
using System.Linq;

using EffectMaker.DataModelMaker.Core.DataTypes;
using EffectMaker.DataModelMaker.Core.Definitions;

using EffectMaker.Foundation.Input;
using EffectMaker.Foundation.Interfaces;

namespace EffectMaker.DataModelMaker.UILogic.ViewModels
{
    /// <summary>
    /// View model class for the binary data definitions.
    /// </summary>
    public class RootBinaryDataDefinitionViewModel : ViewModelBase
    {
        /// <summary>The data model with the correct type.</summary>
        private RuntimeDataModelRootDefinition myTypedDataModel;

        /// <summary>The binary data definition view models.</summary>
        private List<BinaryDataDefinitionViewModel> binaryDataViewModels =
            new List<BinaryDataDefinitionViewModel>();

        /// <summary>The selected binary data definition view model.</summary>
        private BinaryDataDefinitionViewModel selectedBinaryDataViewModel = null;

        /// <summary>The binary data definition view model to be displayed.</summary>
        private BinaryDataDefinitionViewModel displayedBinaryDataViewModel = null;

        /// <summary>The view models for the binary field group target list.</summary>
        private List<BinaryFieldGroupTargetItemViewModel> binaryFieldGroupTargetViewModels =
            new List<BinaryFieldGroupTargetItemViewModel>();

        /// <summary>
        /// The constructor.
        /// </summary>
        /// <param name="parent">The parent view model.</param>
        /// <param name="dataModel">The data model to encapsulate.</param>
        public RootBinaryDataDefinitionViewModel(
            ViewModelBase parent,
            RuntimeDataModelRootDefinition dataModel) :
            base(parent, dataModel)
        {
            this.myTypedDataModel = dataModel;

            foreach (BinaryDataDefinition def in dataModel.BinaryDatas)
            {
                // Binary data view models.
                var child = new BinaryDataDefinitionViewModel(this, def);
                this.Children.Add(child);
                this.binaryDataViewModels.Add(child);
            }

            this.UpdateTargetViewModels();

            RootRuntimeDataModelDefinitionViewModel.DataModelListChanged += (s, e) =>
                {
                    this.UpdateTargetViewModels();
                };
        }

        /// <summary>
        /// Static event triggered when any binary data is created or removed from the list.
        /// </summary>
        public static event EventHandler BinaryDataListChanged = null;

        /// <summary>
        /// Enumerate binary data definition view models.
        /// </summary>
        public IEnumerable<BinaryDataDefinitionViewModel> BinaryDataDefinitionViewModels
        {
            get { return this.binaryDataViewModels; }
        }

        /// <summary>
        /// Get or set the binary data definition view model to be displayed.
        /// </summary>
        public object DisplayedBinaryDataViewModel
        {
            get
            {
                if (this.displayedBinaryDataViewModel != null)
                {
                    return this.displayedBinaryDataViewModel;
                }
                else
                {
                    return this.selectedBinaryDataViewModel;
                }
            }

            set
            {
                if (value != null && (value is BinaryDataDefinitionViewModel) == false)
                {
                    return;
                }

                this.SetValue(
                    ref this.displayedBinaryDataViewModel,
                    (BinaryDataDefinitionViewModel)value);
            }
        }

        /// <summary>
        /// Get or set the selected binary data definition view model.
        /// </summary>
        public object SelectedBinaryDataViewModel
        {
            get
            {
                return this.selectedBinaryDataViewModel;
            }

            set
            {
                if (value != null && (value is BinaryDataDefinitionViewModel) == false)
                {
                    return;
                }

                this.SetValue(
                    ref this.selectedBinaryDataViewModel,
                    (BinaryDataDefinitionViewModel)value);
            }
        }

        /// <summary>
        /// Enumerate the view models for the binary field group target list.
        /// </summary>
        public IEnumerable<BinaryFieldGroupTargetItemViewModel> BinaryFieldGroupTargetViewModels
        {
            get { return this.binaryFieldGroupTargetViewModels.OrderBy(it => it.TargetName); }
        }

        /// <summary>
        /// Delete the specified binary data.
        /// </summary>
        /// <param name="viewModel">The view model of the data model to delete.</param>
        public void DeleteBinaryData(BinaryDataDefinitionViewModel viewModel)
        {
            if (this.myTypedDataModel == null)
            {
                return;
            }

            int index = this.binaryDataViewModels.IndexOf(viewModel);
            if (index < 0)
            {
                return;
            }

            // First deselect the binary data.
            if (this.SelectedBinaryDataViewModel == viewModel)
            {
                this.SelectedBinaryDataViewModel = null;
            }

            this.binaryDataViewModels.RemoveAt(index);

            var fieldGroupTargetVM =
                this.binaryFieldGroupTargetViewModels.Find(vm => vm.DataModel == viewModel.DataModel);
            if (fieldGroupTargetVM != null)
            {
                this.binaryFieldGroupTargetViewModels.Remove(fieldGroupTargetVM);
            }

            var def = viewModel.DataModel as BinaryDataDefinition;
            this.myTypedDataModel.BinaryDatas.Remove(def);

            viewModel.Dispose();
            def.Dispose();

            this.OnPropertyChanged("BinaryDataDefinitionViewModels");
            this.OnPropertyChanged("BinaryFieldGroupTargetViewModels");

            if (BinaryDataListChanged != null)
            {
                BinaryDataListChanged(this, EventArgs.Empty);
            }
        }

        /// <summary>
        /// Create a new binary data.
        /// </summary>
        /// <returns>The created binary data view model.</returns>
        public BinaryDataDefinitionViewModel CreateBinaryData()
        {
            if (this.myTypedDataModel == null)
            {
                return null;
            }

            // Create a unique name for the property.
            string defaultName = "BinaryData_";
            var regex = new System.Text.RegularExpressions.Regex("^" + defaultName + "([0-9]+)$");

            List<int> existingSerialList = new List<int>();

            // Collect all the numbers that is used.
            foreach (BinaryDataDefinition item in this.myTypedDataModel.BinaryDatas)
            {
                System.Text.RegularExpressions.Match match = regex.Match(item.Name);
                if (match.Groups.Count < 2)
                {
                    continue;
                }

                int currSerial;
                if (int.TryParse(match.Groups[1].Value, out currSerial) == true)
                {
                    existingSerialList.Add(currSerial);
                }
            }

            // Find an unused number for the default name.
            defaultName += this.FindUnusedSerialNumber(existingSerialList).ToString();

            // Create the new binary data and add it to the list.
            var def = new BinaryDataDefinition()
            {
                Name = defaultName
            };

            this.myTypedDataModel.BinaryDatas.Add(def);

            // Create a view model for the new binary data definition.
            var child = new BinaryDataDefinitionViewModel(this, def);
            this.Children.Add(child);
            this.binaryDataViewModels.Add(child);

            var targetItemVM = new BinaryFieldGroupTargetItemViewModel(this, def);
            this.binaryFieldGroupTargetViewModels.Add(targetItemVM);

            // Issue property changed event.
            this.OnPropertyChanged("BinaryDataDefinitionViewModels");
            this.OnPropertyChanged("BinaryFieldGroupTargetViewModels");

            // Select the created binary data view model.
            this.SelectedBinaryDataViewModel = child;

            if (BinaryDataListChanged != null)
            {
                BinaryDataListChanged(this, EventArgs.Empty);
            }

            return child;
        }

        /// <summary>
        /// Called when the name of a binary data is changed, so this view model
        /// can notify the property change event to update the UI.
        /// </summary>
        /// <param name="vm">The binary data view model that the name is changed.</param>
        public void NotifyBinaryDataNameChanged(BinaryDataDefinitionViewModel vm)
        {
            this.OnPropertyChanged("BinaryDataDefinitionViewModels");
            this.OnPropertyChanged("BinaryFieldGroupTargetViewModels");
        }

        /// <summary>
        /// Update the list of the binary field group target view models.
        /// </summary>
        private void UpdateTargetViewModels()
        {
            var existingDefinitionMap = new Dictionary<Guid, DefinitionBase>();

            // First add all the definition we want to the map.
            foreach (RuntimeDataModelDefinition def in this.myTypedDataModel.DataModels)
            {
                existingDefinitionMap.Add(def.Guid, def);
            }

            foreach (BinaryDataDefinition def in this.myTypedDataModel.BinaryDatas)
            {
                existingDefinitionMap.Add(def.Guid, def);
            }

            // Remove all the view models that their target definition no longer exists.
            int removeCount = this.binaryFieldGroupTargetViewModels.RemoveAll(
                it => existingDefinitionMap.ContainsKey(it.DataModel.Guid) == false);

            // Now find the definitions that hasn't been added to the target view models.
            var newDefinitions =
                from item in existingDefinitionMap
                where this.binaryFieldGroupTargetViewModels.Any(vm => vm.DataModel == item.Value) == false
                select item.Value;

            // Add these new definitions to the target item view models.
            int addCount = 0;
            foreach (DefinitionBase def in newDefinitions)
            {
                var child = new BinaryFieldGroupTargetItemViewModel(this, def);
                this.Children.Add(child);
                this.binaryFieldGroupTargetViewModels.Add(child);

                ++addCount;
            }

            if (removeCount > 0 || addCount > 0)
            {
                this.OnPropertyChanged("BinaryFieldGroupTargetViewModels");
            }
        }

        /// <summary>
        /// Find an unused serial number from the given used serial number list.
        /// </summary>
        /// <param name="usedSerialNumList">The serial numbers that has already being used.</param>
        /// <returns>An unused serial number.</returns>
        private int FindUnusedSerialNumber(List<int> usedSerialNumList)
        {
            int serialNumber = 1;
            bool found = true;
            while (found == true)
            {
                found = false;
                foreach (int i in usedSerialNumList)
                {
                    if (i == serialNumber)
                    {
                        ++serialNumber;
                        found = true;
                        break;
                    }
                }
            }

            return serialNumber;
        }
    }
}
