﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using System;
using System.Collections.Generic;
using System.ComponentModel;
using System.Data;
using System.Drawing;
using System.Linq;
using System.Text;
using System.Threading.Tasks;
using System.Windows.Forms;

using EffectMaker.DataModelMaker.Core.Converters;

using EffectMaker.Foundation.Log;
using EffectMaker.Foundation.Render.ObjectPicking;

namespace EffectMaker.DataModelMaker.UIControls.SelectConverter
{
    /// <summary>
    /// A windows form class for displaying the converter selection context menu.
    /// </summary>
    public partial class SelectConverterMenu : Form
    {
        /// <summary>The viewport that renders the converter list.</summary>
        private ConverterListViewport converterListViewport = null;

        /// <summary>The selected converter info.</summary>
        private ConverterInfo selectedConverterInfo = null;

        /// <summary>
        /// Constructor.
        /// </summary>
        private SelectConverterMenu()
        {
            this.InitializeComponent();

            this.converterListViewport = new ConverterListViewport(this);
        }

        /// <summary>
        /// Show the converter selection menu.
        /// </summary>
        /// <param name="parent">The parent control of the menu.</param>
        /// <param name="selectedConverter">The originally selected converter info.</param>
        /// <returns>The selected converter info.</returns>
        public static ConverterInfo ShowMenu(
            Control parent,
            ConverterInfo selectedConverter)
        {
            var instance = new SelectConverterMenu()
            {
                Location = Control.MousePosition,
                Size = new Size(300, 50),
                selectedConverterInfo = selectedConverter,
            };

            // Select the renderable that represents the selected converter.
            foreach (var rd in instance.converterListViewport.ConverterRenderables)
            {
                if (rd.ConverterInfo == selectedConverter)
                {
                    instance.converterListViewport.SelectedItem = rd;
                    break;
                }
            }

            instance.ShowDialog(parent);

            return instance.selectedConverterInfo;
        }

        /// <summary>
        /// Handle Show event.
        /// </summary>
        /// <param name="e">The event arguments.</param>
        protected override void OnShown(EventArgs e)
        {
            base.OnShown(e);
            this.Capture = true;
        }

        /// <summary>
        /// Handle MouseCaptureChanged event.
        /// </summary>
        /// <param name="e">The event arguments.</param>
        protected override void OnMouseCaptureChanged(EventArgs e)
        {
            base.OnMouseCaptureChanged(e);

            if (this.Capture == false)
            {
                if (this.RectangleToScreen(this.DisplayRectangle).Contains(Cursor.Position) == false)
                {
                    // Close the form if the first click is outside of it.
                    this.Close();
                }
                else
                {
                    this.Capture = true;
                }
            }
        }

        /// <summary>
        /// Handle MouseMove event.
        /// </summary>
        /// <param name="e">The event arguments.</param>
        protected override void OnMouseMove(MouseEventArgs e)
        {
            base.OnMouseMove(e);

            // Do mouse picking.
            ObjectPickContext context = this.converterListViewport.DoPicking(e.Location);
            if (context.PickedObjects != null &&
                context.PickedObjects.Count > 0 &&
                context.PickedObjects[0] is ConverterListItemRenderable)
            {
                this.converterListViewport.MouseOverItem =
                    context.PickedObjects[0] as ConverterListItemRenderable;
            }
        }

        /// <summary>
        /// Handle MouseDown event.
        /// </summary>
        /// <param name="e">The event arguments.</param>
        protected override void OnMouseDown(MouseEventArgs e)
        {
            base.OnMouseDown(e);

            if (e.Button == MouseButtons.Left)
            {
                // Do mouse picking.
                ObjectPickContext context = this.converterListViewport.DoPicking(e.Location);
                if (context.PickedObjects != null &&
                    context.PickedObjects.Count > 0 &&
                    context.PickedObjects[0] is ConverterListItemRenderable)
                {
                    var pickedConverterRenderable =
                        context.PickedObjects[0] as ConverterListItemRenderable;
                    if (pickedConverterRenderable != null)
                    {
                        this.selectedConverterInfo = pickedConverterRenderable.ConverterInfo;
                    }
                }
            }

            this.Close();
        }

        /// <summary>
        /// Handle SizeChanged event.
        /// </summary>
        /// <param name="e">The event arguments.</param>
        protected override void OnSizeChanged(EventArgs e)
        {
            base.OnSizeChanged(e);

            if (this.converterListViewport != null)
            {
                this.converterListViewport.Size = new SizeF(this.Width, this.Height);
            }
        }

        /// <summary>
        /// Handle Paint event.
        /// </summary>
        /// <param name="e">The event arguments.</param>
        protected override void OnPaint(PaintEventArgs e)
        {
            e.Graphics.SmoothingMode = System.Drawing.Drawing2D.SmoothingMode.HighQuality;
            e.Graphics.TextRenderingHint = System.Drawing.Text.TextRenderingHint.ClearTypeGridFit;

            // Render the viewport.
            if (this.converterListViewport != null)
            {
                this.converterListViewport.Update(e.Graphics);
                this.converterListViewport.Draw(e.Graphics);

                // Adjust the menu size.
                this.Size = new Size(
                    this.Width,
                    (int)Math.Ceiling(this.converterListViewport.Height));

                // Don't let the menu go out of the working area of the screen.
                var screen = Screen.FromControl(this);
                if (this.Bottom > screen.WorkingArea.Bottom)
                {
                    this.Location = new Point(
                        this.Location.X,
                        Math.Max(screen.WorkingArea.Top, screen.WorkingArea.Bottom - this.Height));
                }
            }
        }
    }
}
