﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using System;
using System.Collections;
using System.Collections.Generic;
using System.ComponentModel;
using System.Drawing;
using System.Linq;

using EffectMaker.DataModelMaker.Core.Core;
using EffectMaker.DataModelMaker.Core.Definitions;

using EffectMaker.DataModelMaker.UIControls.Extenders;

using EffectMaker.Foundation.Extensions;
using EffectMaker.Foundation.Interfaces;
using EffectMaker.Foundation.Log;
using EffectMaker.Foundation.Render.Renderable;
using EffectMaker.Foundation.Utility;

using EffectMaker.UIControls;
using EffectMaker.UIControls.DataBinding;
using EffectMaker.UIControls.Extenders;

namespace EffectMaker.DataModelMaker.UIControls.ConversionView
{
    /// <summary>
    /// Class for rendering source data model information.
    /// </summary>
    internal class SourceDataModelRenderable :
        RoundedRectangleShape,
        ILogicalTreeElement,
        ISourceConnectible
    {
        /// <summary>Constant for the height of the binary data information area.</summary>
        private const float DataModelInfoHeight = 45.0f;

        /// <summary>Constant for the margin of the child renderables.</summary>
        private const float ChildMargin = 5.0f;

        /// <summary>Constant for the spacing between child renderables.</summary>
        private const float ChildSpacing = 5.0f;

        /// <summary>The large font for rendering.</summary>
        private static Font largeFont = null;

        /// <summary>The small font for rendering.</summary>
        private static Font smallFont = null;

        /// <summary>The extender for easily implementing ILogicalTreeElement.</summary>
        private LogicalTreeElementExtender logicalTreeElementExtender = null;

        /// <summary>Backing field for Controls property.</summary>
        private IIndexableCollection<ILogicalTreeElement> controlsWrapper;

        /// <summary>The extender for easily implement operations for child renderables.</summary>
        private ChildRenderableOperationExtender childRenderableOperationExtender = null;

        /// <summary>The child renderables for rendering the data model properties.</summary>
        private List<SourceDataModelPropertyRenderable> propertyRenderables =
            new List<SourceDataModelPropertyRenderable>();

        /// <summary>Flag indicating whether this renderable is collapsed on UI.</summary>
        private bool isCollapsed = false;

        /// <summary>
        /// Static constructor.
        /// </summary>
        static SourceDataModelRenderable()
        {
            // Setup the fonts for rendering.
            smallFont = new Font(FontFamily.GenericSansSerif, 8.0f);
            largeFont = new Font(FontFamily.GenericSansSerif, 14.0f, FontStyle.Bold);
        }

        /// <summary>
        /// Constructor.
        /// </summary>
        /// <param name="owner">The owner of the view.</param>
        public SourceDataModelRenderable(RenderableBase owner) :
            base()
        {
            this.SeparatorLineColor = Color.Silver;

            this.Translation = new PointF(ChildMargin, DataModelInfoHeight);
            this.BorderThickness = 2.0f;
            this.FillColor = Color.Beige;
            this.SetCornerRadius(5.0f);

            this.Bindings = new BindingContainer(this);
            this.logicalTreeElementExtender = new LogicalTreeElementExtender(this);
            this.childRenderableOperationExtender = new ChildRenderableOperationExtender(this);

            this.Owner = owner;

            this.Size = new SizeF(0.0f, DataModelInfoHeight);

            this.Connector = new ConnectorRenderable(this)
            {
                ConnectorLength = 20.0f,
                ConnectorDirection = ConnectorRenderable.ConnectorDirections.Right,
                Offset = new SizeF(ChildMargin * 2.0f, -15.0f)
            };

            this.AddRenderable(this.Connector);

            this.Bindings.Add(new Binder(this, "DataModelNamespace", "DataModelNamespace"));
            this.Bindings.Add(new Binder(this, "DataModelName", "DataModelName"));
            this.Bindings.Add(new Binder(this, "IsCollapsed", "IsCollapsed"));
            this.Bindings.Add(new Binder(this, "DataModelDescription", "DataModelDescription"));
            this.Bindings.Add(new Binder(this, "PropertyItems", "PropertyInstanceViewModels"));
        }

        /// <summary>
        /// Raised when the value of a property on this control changed.
        /// </summary>
#pragma warning disable 67
        public event PropertyChangedEventHandler PropertyChanged;
#pragma warning restore 67

        /// <summary>
        /// Get the owner binary data info viewport.
        /// </summary>
        public RenderableBase Owner { get; private set; }

        /// <summary>
        /// gets the parent control.
        /// </summary>
        public new ILogicalTreeElement Parent
        {
            get { return this.Owner as ILogicalTreeElement; }
            protected set { this.Owner = value as RenderableBase; }
        }

        /// <summary>
        /// Gets a collection of children controls of the the current control instance.
        /// </summary>
        public IIndexableCollection<ILogicalTreeElement> Children
        {
            get
            {
                if (this.controlsWrapper == null)
                {
                    this.controlsWrapper = new IndexableCollection<ILogicalTreeElement>();
                }

                return this.controlsWrapper;
            }
        }

        /// <summary>
        /// Gets the control extender instance of this control.
        /// </summary>
        public LogicalTreeElementExtender LogicalTreeElementExtender
        {
            get { return this.logicalTreeElementExtender; }
        }

        /// <summary>
        /// Gets or sets the DataContext.
        /// This property may raise a 'DataContext' change notification.
        /// See LogicalTreeElementExtender for more information.
        /// <see cref="LogicalTreeElementExtender"/>
        /// </summary>
        [DesignerSerializationVisibility(DesignerSerializationVisibility.Hidden)]
        public object DataContext
        {
            get
            {
                return this.logicalTreeElementExtender.DataContext;
            }

            set
            {
                this.logicalTreeElementExtender.DataContext = value;
                this.Visible = value != null;
            }
        }

        /// <summary>
        /// Gets the binders collection.
        /// </summary>
        public BindingContainer Bindings { get; private set; }

        /// <summary>
        /// Get the flag indicating whether this source data model instance is collapsed on UI.
        /// </summary>
        public bool IsCollapsed
        {
            get
            {
                return this.isCollapsed;
            }

            set
            {
                this.isCollapsed = value;
                this.logicalTreeElementExtender.NotifyPropertyChanged();
            }
        }

        /// <summary>
        /// Get the connector.
        /// </summary>
        public ConnectorRenderable Connector { get; private set; }

        /// <summary>
        /// Get the location of the connector.
        /// </summary>
        public PointF ConnectorLocation
        {
            get { return this.Connector.ConnectorLocation; }
        }

        /// <summary>
        /// Get the transformed location of the connector.
        /// </summary>
        public PointF TransformedConnectorLocation
        {
            get { return this.Connector.TransformedConnectorLocation; }
        }

        /// <summary>
        /// Get the connection source Guid.
        /// </summary>
        public Guid SourceGuid
        {
            get
            {
                var propertyRenderable = this.Owner as SourceDataModelPropertyRenderable;
                if (propertyRenderable == null)
                {
                    return Guid.Empty;
                }

                return propertyRenderable.SourceGuid;
            }
        }

        /// <summary>
        /// Get or set the color of the separator lines.
        /// </summary>
        public Color SeparatorLineColor { get; set; }

        /// <summary>
        /// Enumerate the data context of the property instances.
        /// </summary>
        public IEnumerable PropertyItems
        {
            get { return this.propertyRenderables.Select(rd => rd.DataContext); }
            set { this.UpdateItemsFromDataSource(value); }
        }

        /// <summary>
        /// Get or set the description of the connectible.
        /// </summary>
        public string Description
        {
            get
            {
                var ownerProperty = this.Owner as SourceDataModelPropertyRenderable;
                if (ownerProperty != null)
                {
                    return ownerProperty.Description;
                }
                else
                {
                    return this.DataModelDescription;
                }
            }
        }

        /// <summary>
        /// Get the string as the type of the connectible value.
        /// </summary>
        public string ValueType
        {
            get { return this.DataModelName; }
        }

        /// <summary>
        /// Get the name of the connectible.
        /// </summary>
        public string DisplayName
        {
            get
            {
                var ownerProperty = this.Owner as SourceDataModelPropertyRenderable;
                if (ownerProperty != null)
                {
                    return ownerProperty.DisplayName;
                }
                else
                {
                    return string.Empty;
                }
            }
        }

        /// <summary>
        /// Get the flag indicating whether the connection of the connectible is set.
        /// </summary>
        public bool IsConnectionSet
        {
            get
            {
                if (ConnectionRenderable.FindConnections(this.Connector).Count() > 0)
                {
                    return true;
                }

                return false;
            }
        }

        /// <summary>
        /// Get or set the data model namespace.
        /// </summary>
        public string DataModelNamespace { get; set; }

        /// <summary>
        /// Get or set the data model name.
        /// </summary>
        public string DataModelName { get; set; }

        /// <summary>
        /// Get or set the data model description.
        /// </summary>
        public string DataModelDescription { get; set; }

        /// <summary>
        /// Enumerates all the child property renderables.
        /// </summary>
        public IEnumerable<SourceDataModelPropertyRenderable> PropertyRenderables
        {
            get { return this.propertyRenderables; }
        }

        /// <summary>
        /// Clears the DataContext.
        /// See LogicalTreeElementExtender for more details.
        /// <see cref="LogicalTreeElementExtender"/>
        /// </summary>
        public void ClearDataContext()
        {
            this.logicalTreeElementExtender.ClearDataContext();
        }

        /// <summary>
        /// Determine if the specified target can be connected to this connectible item.
        /// </summary>
        /// <param name="target">The connection target.</param>
        /// <returns>True if can connect.</returns>
        public bool CanConnect(IConnectible target)
        {
            if (target == null ||
                target is ISourceConnectible)
            {
                return false;
            }

            return true;
        }

        /// <summary>
        /// Add a child renderable.
        /// </summary>
        /// <param name="child">The child to add.</param>
        public override void AddRenderable(RenderableBase child)
        {
            this.childRenderableOperationExtender.AddRenderable(child);
            base.AddRenderable(child);

            var childRenderable = child as SourceDataModelPropertyRenderable;
            if (childRenderable != null)
            {
                this.propertyRenderables.Add(childRenderable);
            }
        }

        /// <summary>
        /// Add child renderables.
        /// </summary>
        /// <param name="children">The children to add.</param>
        public override void AddRenderableRange(IEnumerable<RenderableBase> children)
        {
            this.childRenderableOperationExtender.AddRenderableRange(children);
            base.AddRenderableRange(children);

            foreach (RenderableBase child in children)
            {
                var childRenderable = child as SourceDataModelPropertyRenderable;
                if (childRenderable != null)
                {
                    this.propertyRenderables.Add(childRenderable);
                }
            }
        }

        /// <summary>
        /// Remove a child renderable.
        /// </summary>
        /// <param name="child">The child to remove.</param>
        public override void RemoveRenderable(RenderableBase child)
        {
            this.childRenderableOperationExtender.RemoveRenderable(child);
            base.RemoveRenderable(child);

            var childRenderable = child as SourceDataModelPropertyRenderable;
            if (childRenderable != null)
            {
                this.propertyRenderables.Remove(childRenderable);
            }
            else if (child == this.Connector)
            {
                this.Connector = null;
            }
        }

        /// <summary>
        /// Remove child renderables.
        /// </summary>
        /// <param name="children">The children to remove.</param>
        public override void RemoveRenderableRange(IEnumerable<RenderableBase> children)
        {
            this.childRenderableOperationExtender.RemoveRenderableRange(children);
            base.RemoveRenderableRange(children);

            // To prevent modifying the property renderable list while looping it,
            // copy the children to another array first.
            var tmpChildren = children;
            if (tmpChildren == this.propertyRenderables)
            {
                tmpChildren = children.ToArray();
            }

            foreach (RenderableBase child in tmpChildren)
            {
                var childRenderable = child as SourceDataModelPropertyRenderable;
                if (childRenderable != null)
                {
                    this.propertyRenderables.Remove(childRenderable);
                }
                else if (child == this.Connector)
                {
                    this.Connector = null;
                }
            }
        }

        /// <summary>
        /// Clear child renderables.
        /// </summary>
        public override void ClearRenderables()
        {
            this.childRenderableOperationExtender.ClearRenderables();
            base.ClearRenderables();

            this.propertyRenderables.Clear();
            this.Connector = null;
        }

        /// <summary>
        /// Update the children for rendering.
        /// </summary>
        /// <param name="context">Data context that contains information for rendering.</param>
        protected override void UpdateChildren(RenderContext context)
        {
            float childWidth = this.Width - (ChildMargin * 2.0f);

            // Loop through field group renderables to update the content size.
            float location = 0.0f;
            foreach (RenderableBase child in this.propertyRenderables)
            {
                child.Size = new SizeF(childWidth, child.Height);
                child.Location = new PointF(0.0f, location);

                location += child.Height;
            }

            var myRC = context as ConnectorRenderContext;
            if (myRC != null &&
                myRC.ShowConnector == true &&
                this.Owner is SourceDataModelPropertyRenderable)
            {
                // Update connector length
                float length = 15.0f;
                ILogicalTreeElement element = this.Parent;
                while (element != null)
                {
                    if (element is SourceDataModelPropertyRenderable)
                    {
                        length += SourceDataModelPropertyRenderable.ChildMargin;
                    }

                    element = element.Parent;
                }

                this.Connector.Visible = true;
                this.Connector.ConnectorLength = length;
            }
            else
            {
                this.Connector.Visible = false;
            }

            base.UpdateChildren(context);

            // Loop through binary field renderables to find out the height of the children.
            float height = DataModelInfoHeight;
            foreach (RenderableBase child in this.propertyRenderables)
            {
                height += child.Height;
            }

            this.Size = new SizeF(this.Width, height);
        }

        /// <summary>
        /// Render the item's background.
        /// </summary>
        /// <param name="g">The graphics object for rendering.</param>
        protected override void DrawBackground(Graphics g)
        {
            base.DrawBackground(g);

            // Draw the separation line on the top edge of the field renderables.
            using (var pen = new Pen(this.SeparatorLineColor, 1.0f))
            {
                foreach (var renderable in this.propertyRenderables)
                {
                    var rect = renderable.TransformedBounds;
                    g.DrawLine(pen, rect.Location, new PointF(rect.Right, rect.Top));
                }
            }
        }

        /// <summary>
        /// Render the item's foreground.
        /// </summary>
        /// <param name="g">The graphics object for rendering.</param>
        protected override void DrawForeground(Graphics g)
        {
            var rect = this.TransformedBounds;

            // Compute the text size of the data model type.
            SizeF dataModelTypeTextSize = g.MeasureString(
                this.DataModelName,
                SourceDataModelRenderable.smallFont);

            string parentPropertyName = string.Empty;
            if (this.Owner is SourceDataModelPropertyRenderable)
            {
                parentPropertyName = ((SourceDataModelPropertyRenderable)this.Owner).PropertyName;
            }

            // Compute the text size of the parent property.
            SizeF parentPropertyNameTextSize = g.MeasureString(
                parentPropertyName,
                SourceDataModelRenderable.largeFont);

            // The rectangle for the data model type name.
            RectangleF dataModelTypeTextRect = rect;
            dataModelTypeTextRect.X += 3.0f;
            dataModelTypeTextRect.Y += 6.0f;
            dataModelTypeTextRect.Width = rect.Width - 3.0f;
            dataModelTypeTextRect.Height = dataModelTypeTextSize.Height;

            // The rectangle for the parent property name.
            RectangleF propertyNameTextRect = rect;
            propertyNameTextRect.X += 2.0f;
            propertyNameTextRect.Y = rect.Top + DataModelInfoHeight - parentPropertyNameTextSize.Height - 5.0f;
            propertyNameTextRect.Width = rect.Width - 2.0f;
            propertyNameTextRect.Height = parentPropertyNameTextSize.Height;

            using (var strFormat = new StringFormat())
            {
                // Top-left alignment.
                strFormat.Alignment = StringAlignment.Near;
                strFormat.LineAlignment = StringAlignment.Far;

                strFormat.Trimming = StringTrimming.EllipsisCharacter;

                g.DrawString(
                    this.DataModelName,
                    SourceDataModelRenderable.smallFont,
                    Brushes.DarkCyan,
                    dataModelTypeTextRect,
                    strFormat);

                strFormat.LineAlignment = StringAlignment.Near;

                g.DrawString(
                    parentPropertyName,
                    SourceDataModelRenderable.largeFont,
                    Brushes.DarkBlue,
                    propertyNameTextRect,
                    strFormat);
            }
        }

        /// <summary>
        /// Update items from data source.
        /// </summary>
        /// <param name="items">The items from data source.</param>
        private void UpdateItemsFromDataSource(IEnumerable items)
        {
            this.RemoveRenderableRange(this.propertyRenderables);

            // Loop through data model property definitions.
            float height = 0.0f;
            foreach (object item in items)
            {
                // Create a data model view for the data model source.
                var renderable = new SourceDataModelPropertyRenderable(this)
                {
                    DataContext = item,
                    Location = new PointF(0.0f, height)
                };

                // Set up size.
                renderable.Size =
                    new SizeF(this.Width - (ChildMargin * 2.0f), renderable.Height);

                height += renderable.Height;

                // Add the list item to the child renderable list.
                this.AddRenderable(renderable);
            }
        }
    }
}
