﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using System;
using System.Collections.Generic;
using System.Drawing;
using System.Linq;
using System.Text;
using System.Threading.Tasks;
using System.Windows.Forms;

using EffectMaker.DataModelMaker.Core.Converters;

using EffectMaker.DataModelMaker.UIControls.BinaryDataEditor;
using EffectMaker.DataModelMaker.UIControls.BinaryDataView;
using EffectMaker.DataModelMaker.UIControls.BinaryFieldEditor;
using EffectMaker.DataModelMaker.UIControls.DataModelEditor;
using EffectMaker.DataModelMaker.UIControls.DataModelPropertyEditor;
using EffectMaker.DataModelMaker.UIControls.SelectConverter;
using EffectMaker.DataModelMaker.UIControls.Selection;

using EffectMaker.DataModelMaker.UILogic.ViewModels;

using EffectMaker.Foundation.Log;
using EffectMaker.Foundation.Render.ObjectPicking;
using EffectMaker.Foundation.Render.Renderable;
using EffectMaker.Foundation.Render.ScrollBar;

using EffectMaker.UIControls.BaseControls;

namespace EffectMaker.DataModelMaker.UIControls.ConversionView
{
    /// <summary>
    /// The panel that renders the binary data list.
    /// </summary>
    internal class ConversionViewPanel : Panel
    {
        /// <summary>Constant for the factor applying to the vertical mouse scroll speed.</summary>
        private const int VerticalScrollSpeedFactor = 50;

        /// <summary>The viewport for rendering the conversion view.</summary>
        private ConversionViewViewport conversionViewport = null;

        /// <summary>The scroll bars.</summary>
        private AutoHiddenScrollBar scrollBars = null;

        /// <summary>
        /// Constructor.
        /// </summary>
        public ConversionViewPanel()
        {
            // Enable double buffer.
            this.DoubleBuffered = true;

            // Make the panel selectable.
            this.SetStyle(ControlStyles.Selectable, true);
            this.TabStop = true;

            this.conversionViewport = new ConversionViewViewport(this);
            this.conversionViewport.Bounds = new Rectangle(
                0,
                0,
                this.ClientRectangle.Width,
                this.ClientRectangle.Height);

            // Create the scroll bars.
            this.scrollBars = new AutoHiddenScrollBar(this)
            {
                ForeColor = Color.Black,
                BackColor = Color.Black,
                IncrementV = VerticalScrollSpeedFactor
            };

            this.scrollBars.RenderRequired += (s, e) => { this.Invalidate(); };
            this.scrollBars.Scroll += (s, e) =>
            {
                this.conversionViewport.Translation =
                    new PointF(-this.ScrollPosition.X, -this.ScrollPosition.Y);
                this.Invalidate();
            };

            this.scrollBars.SetContentSize(new Size(
                (int)this.conversionViewport.ContentSize.Width,
                (int)this.conversionViewport.ContentSize.Height));

            SelectionManager.SelectionChanged += (s, e) =>
            {
                this.Invalidate();
            };

            SelectionManager.MouseOverChanged += (s, e) =>
            {
                this.Invalidate();
            };
        }

        /// <summary>
        /// Get or set the data context.
        /// </summary>
        public object DataContext
        {
            get
            {
                return this.conversionViewport.DataContext;
            }

            set
            {
                // Set up the data context to the viewport for binding.
                this.conversionViewport.DataContext = value;

                // Update the content size to the scroll bars.
                this.scrollBars.SetContentSize(new Size(
                    (int)this.conversionViewport.ContentSize.Width,
                    (int)this.conversionViewport.ContentSize.Height));
            }
        }

        /// <summary>
        /// Get or set the scroll position.
        /// </summary>
        public Point ScrollPosition
        {
            get
            {
                if (this.scrollBars == null)
                {
                    return Point.Empty;
                }

                return this.scrollBars.ScrollPosition;
            }

            set
            {
                this.SetScrollPosition(value.X, value.Y);
            }
        }

        /// <summary>
        /// Set scroll position.
        /// </summary>
        /// <param name="x">The horizontal position.</param>
        /// <param name="y">The vertical position.</param>
        public void SetScrollPosition(int x, int y)
        {
            if (this.scrollBars == null)
            {
                return;
            }

            var contentSize = new Size(
                (int)this.conversionViewport.ContentSize.Width,
                (int)this.conversionViewport.ContentSize.Height);

            int maxX = contentSize.Width - this.Width;
            int maxY = contentSize.Height - this.Height;

            int posX = Math.Max(Math.Min(x, maxX), 0);
            int posY = Math.Max(Math.Min(y, maxY), 0);

            this.scrollBars.ScrollPosition = new Point(posX, posY);

            this.conversionViewport.Translation = new PointF(-posX, -posY);
        }

        /// <summary>
        /// Clean up any resources being used.
        /// </summary>
        /// <param name="disposing">True if managed resources should be disposed; otherwise, false.</param>
        protected override void Dispose(bool disposing)
        {
            if (disposing == true)
            {
                if (this.conversionViewport != null)
                {
                    this.conversionViewport.Dispose();
                    this.conversionViewport = null;
                }
            }

            base.Dispose(disposing);
        }

        /// <summary>
        /// Handle MouseWheel event.
        /// </summary>
        /// <param name="e">The event arguments.</param>
        protected override void OnMouseWheel(MouseEventArgs e)
        {
            base.OnMouseWheel(e);

            int increment = -(e.Delta / 120) * VerticalScrollSpeedFactor;

            this.SetScrollPosition(this.ScrollPosition.X, this.ScrollPosition.Y + increment);

            this.Invalidate();
        }

        /// <summary>
        /// Handle MouseDoubleClick event.
        /// </summary>
        /// <param name="e">The event arguments.</param>
        protected override void OnMouseDoubleClick(MouseEventArgs e)
        {
            base.OnMouseDoubleClick(e);

            // Do mouse picking.
            ObjectPickContext context = this.conversionViewport.DoPicking(e.Location);
            if (context.PickedObjects != null &&
                context.PickedObjects.Count > 0)
            {
                foreach (var obj in context.PickedObjects)
                {
                    if (obj is BinaryDataInfoFieldRenderable)
                    {
                        using (var dialog = new BinaryFieldEditorDialog())
                        {
                            dialog.DataSource = ((BinaryDataInfoFieldRenderable)obj).DataContext;
                            DialogResult result = dialog.ShowDialog(this);
                        }

                        break;
                    }
                }
            }
        }

        /// <summary>
        /// Handle MouseDown event.
        /// </summary>
        /// <param name="e">The event arguments.</param>
        protected override void OnMouseDown(MouseEventArgs e)
        {
            base.OnMouseDown(e);

            bool shouldBeginConnect = true;
            if (Control.ModifierKeys == Keys.Control &&
                this.conversionViewport.IsConnecting == true)
            {
                this.conversionViewport.EndConnect();
                shouldBeginConnect = false;
            }

            // Do mouse picking.
            ObjectPickContext context = this.conversionViewport.DoPicking(e.Location);
            if (context.PickedObjects != null &&
                context.PickedObjects.Count > 0 &&
                context.PickedObjects[0] != null)
            {
                if (context.PickedObjects[0] is ConnectionRenderable)
                {
                    SelectionManager.SetSelection(
                        SelectionGroups.BinaryConvertConnection,
                        context.PickedObjects[0]);
                }
                else
                {
                    ConnectorRenderable connector = null;
                    if (context.PickedObjects[0] is ConnectorRenderable)
                    {
                        connector = (ConnectorRenderable)context.PickedObjects[0];
                    }
                    else if (context.PickedObjects[0] is IConnectible)
                    {
                        connector = ((IConnectible)context.PickedObjects[0]).Connector;
                    }

                    if (connector != null && connector.Visible == true)
                    {
                        SelectionManager.SetSelection(
                            SelectionGroups.BinaryConvertConnection, connector);

                        if (shouldBeginConnect == true)
                        {
                            this.conversionViewport.BeginConnect(connector.Owner);
                        }
                    }
                    else
                    {
                        SelectionManager.SetSelection(
                            SelectionGroups.BinaryConvertConnection, null);
                    }
                }
            }
        }

        /// <summary>
        /// Handle MouseUp event.
        /// </summary>
        /// <param name="e">The event arguments.</param>
        protected override void OnMouseUp(MouseEventArgs e)
        {
            base.OnMouseUp(e);

            if (Control.ModifierKeys != Keys.Control)
            {
                this.conversionViewport.EndConnect();
            }

            // Do mouse picking.
            ObjectPickContext context = this.conversionViewport.DoPicking(e.Location);
            if (context.PickedObjects != null &&
                context.PickedObjects.Count > 0 &&
                context.PickedObjects[0] != null)
            {
                ConnectorRenderable connector = null;
                if (context.PickedObjects[0] is ConnectorRenderable)
                {
                    connector = (ConnectorRenderable)context.PickedObjects[0];
                }
                else if (context.PickedObjects[0] is IConnectible)
                {
                    connector = ((IConnectible)context.PickedObjects[0]).Connector;
                }

                if (connector != null && connector.Visible == true)
                {
                    SelectionManager.SetSelection(
                        SelectionGroups.BinaryConvertConnection, connector);

                    if (e.Button == MouseButtons.Right)
                    {
                        var destination = connector.Owner as IDestinationConnectible;
                        if (destination != null)
                        {
                            ConverterInfo selectedConverter =
                                SelectConverterMenu.ShowMenu(this, destination.Converter);

                            destination.Converter = selectedConverter;
                        }
                    }
                }
            }
        }

        /// <summary>
        /// Handle MouseMove event.
        /// </summary>
        /// <param name="e">The event arguments.</param>
        protected override void OnMouseMove(MouseEventArgs e)
        {
            if (this.Focused == false)
            {
                this.Focus();
            }

            base.OnMouseMove(e);

            this.conversionViewport.SuspendUpdateAndRendering();

            object mouseOverObj = null;
            IConnectible mouseOverConnectible = null;

            // Do mouse picking.
            ObjectPickContext context = this.conversionViewport.DoPicking(e.Location);
            if (context.PickedObjects != null &&
                context.PickedObjects.Count > 0 &&
                context.PickedObjects[0] != null)
            {
                if (context.PickedObjects[0] is ConnectorRenderable &&
                    ((ConnectorRenderable)context.PickedObjects[0]).Visible == true)
                {
                    mouseOverObj = context.PickedObjects[0];
                    mouseOverConnectible = ((ConnectorRenderable)context.PickedObjects[0]).Owner;
                }
                else if (context.PickedObjects[0] is IConnectible &&
                    ((IConnectible)context.PickedObjects[0]).Connector.Visible == true)
                {
                    mouseOverObj = ((IConnectible)context.PickedObjects[0]).Connector;
                    mouseOverConnectible = (IConnectible)context.PickedObjects[0];
                }
                else if (context.PickedObjects[0] is ConnectionRenderable &&
                    ((ConnectionRenderable)context.PickedObjects[0]).Visible == true)
                {
                    mouseOverObj = context.PickedObjects[0];
                }
            }

            if (mouseOverConnectible != null &&
                mouseOverConnectible.Connector != null)
            {
                this.conversionViewport.ShowConnectorInfo(mouseOverConnectible.Connector);
            }
            else
            {
                this.conversionViewport.ShowConnectorInfo(null);
            }

            this.conversionViewport.PerformConnectEffect(e.Location, mouseOverConnectible);

            SelectionManager.SetMouseOverObject(
                SelectionGroups.BinaryConvertConnection,
                mouseOverObj);

            if (this.conversionViewport.ResumeUpdateAndRendering() == true)
            {
                this.Invalidate();
            }
        }

        /// <summary>
        /// Handle MouseLeave event.
        /// </summary>
        /// <param name="e">The event arguments.</param>
        protected override void OnMouseLeave(EventArgs e)
        {
            base.OnMouseLeave(e);

            SelectionManager.SetMouseOverObject(SelectionGroups.BinaryConvertConnection, null);
        }

        /// <summary>
        /// Handle KeyUp event.
        /// </summary>
        /// <param name="e">The event arguments.</param>
        protected override void OnKeyUp(KeyEventArgs e)
        {
            base.OnKeyUp(e);

            if (e.KeyCode == Keys.Delete)
            {
                var selectedObj =
                    SelectionManager.GetSelectedObject(SelectionGroups.BinaryConvertConnection);
                if (selectedObj is ConnectionRenderable)
                {
                    var connection = (ConnectionRenderable)selectedObj;
                    if (connection.Source != null &&
                        connection.Destination != null)
                    {
                        connection.Destination.RemoveConnectionSources(
                            new Guid[] { connection.Source.SourceGuid });
                    }
                }
                else if (selectedObj is ConnectorRenderable)
                {
                    DialogResult result = MessageBox.Show(
                        Properties.Resources.WarningRemoveAllConnections,
                        Properties.Resources.WarningMessageBoxTitle,
                        MessageBoxButtons.OKCancel,
                        MessageBoxIcon.Exclamation);
                    if (result != DialogResult.OK)
                    {
                        return;
                    }

                    var connector = (ConnectorRenderable)selectedObj;

                    // Find the connections that are connected to this connector.
                    IEnumerable<ConnectionRenderable> connections =
                        ConnectionRenderable.FindConnections(connector);

                    if (connector.Owner is ISourceConnectible)
                    {
                        foreach (ConnectionRenderable connection in connections.ToArray())
                        {
                            // Remove these connections.
                            if (connection.Source != null &&
                                connection.Destination != null)
                            {
                                connection.Destination.RemoveConnectionSources(
                                    new Guid[] { connection.Source.SourceGuid });
                            }
                        }
                    }
                    else if (connector.Owner is IDestinationConnectible)
                    {
                        ((IDestinationConnectible)connector.Owner).RemoveConnectionSources(
                            connections.Select(it => it.Source.SourceGuid));
                    }
                }
            }
            else if (e.KeyCode == Keys.ControlKey)
            {
                if (this.conversionViewport.IsConnecting == true)
                {
                    this.conversionViewport.EndConnect();
                }
            }
        }

        /// <summary>
        /// Handle SizeChanged event.
        /// </summary>
        /// <param name="e">The event arguments.</param>
        protected override void OnSizeChanged(EventArgs e)
        {
            if (this.conversionViewport != null)
            {
                this.conversionViewport.Size = this.ClientSize;
            }

            if (this.scrollBars != null)
            {
                Point scrollPos = this.scrollBars.ScrollPosition;

                this.scrollBars.SetViewSize(this.Size);
                this.scrollBars.SetDisplayRectangle(this.DisplayRectangle);

                // Reset to the same scroll position again.
                this.scrollBars.ScrollPosition = scrollPos;
            }
        }

        /// <summary>
        /// Handle Paint event.
        /// </summary>
        /// <param name="e">The event arguments.</param>
        protected override void OnPaint(PaintEventArgs e)
        {
            e.Graphics.SmoothingMode = System.Drawing.Drawing2D.SmoothingMode.HighQuality;
            e.Graphics.TextRenderingHint = System.Drawing.Text.TextRenderingHint.ClearTypeGridFit;

            // Render the viewport.
            if (this.conversionViewport != null && this.scrollBars != null)
            {
                this.conversionViewport.Update(e.Graphics);
                this.conversionViewport.Draw(e.Graphics);

                // The data model list is updated, now set the updated content size to the scroll bars.
                this.scrollBars.SetContentSize(new Size(
                    (int)this.conversionViewport.ContentSize.Width,
                    (int)this.conversionViewport.ContentSize.Height));

                // Render the scroll bars.
                this.scrollBars.Render(e.Graphics);
            }
        }
    }
}
