﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using System;
using System.Collections;
using System.Collections.Generic;
using System.ComponentModel;
using System.Drawing;

using EffectMaker.Foundation.Interfaces;
using EffectMaker.Foundation.Render.Renderable;
using EffectMaker.Foundation.Utility;

using EffectMaker.UIControls;
using EffectMaker.UIControls.DataBinding;
using EffectMaker.UIControls.Extenders;

namespace EffectMaker.DataModelMaker.UIControls.BinaryDataView
{
    /// <summary>
    /// Class for rendering data model list item.
    /// </summary>
    internal class BinaryFieldGroupTargetListItemRenderable : RoundedRectangleShape, ILogicalTreeElement
    {
        /// <summary>The large font for rendering.</summary>
        private static Font largeFont = null;

        /// <summary>The small font for rendering.</summary>
        private static Font smallFont = null;

        /// <summary>The extender for easily implementing ILogicalTreeElement.</summary>
        private LogicalTreeElementExtender logicalTreeElementExtender = null;

        /// <summary>Backing field for Controls property.</summary>
        private IIndexableCollection<ILogicalTreeElement> controlsWrapper;

        /// <summary>The flag indicating if mouse is over this item.</summary>
        private bool isMouseOver = false;

        /// <summary>
        /// Static constructor.
        /// </summary>
        static BinaryFieldGroupTargetListItemRenderable()
        {
            // Setup the fonts for rendering.
            largeFont = new Font(FontFamily.GenericSansSerif, 12.0f, FontStyle.Bold);
            smallFont = new Font(FontFamily.GenericSansSerif, 8.0f);
        }

        /// <summary>
        /// Constructor.
        /// </summary>
        /// <param name="owner">The owner of the view.</param>
        public BinaryFieldGroupTargetListItemRenderable(BinaryFieldGroupTargetListViewport owner) :
            base()
        {
            this.Bindings = new BindingContainer(this);
            this.logicalTreeElementExtender = new LogicalTreeElementExtender(this);

            this.Owner = owner;

            this.SetCornerRadius(5.0f);
            this.Size = new SizeF(0.0f, 35.0f);

            // Set up binding.
            this.Bindings.Add(new Binder(this, "IsTargetBinaryData", "IsTargetBinaryData"));
            this.Bindings.Add(new Binder(this, "TargetName", "TargetName"));
            this.Bindings.Add(new Binder(this, "TargetGuid", "Guid"));

            this.UpdateAppearance();
        }

        /// <summary>
        /// Raised when the value of a property on this control changed.
        /// </summary>
#pragma warning disable 67
        public event PropertyChangedEventHandler PropertyChanged;
#pragma warning restore 67

        /// <summary>
        /// Get the owner binary data list renderable.
        /// </summary>
        public BinaryFieldGroupTargetListViewport Owner { get; private set; }

        /// <summary>
        /// gets the parent control.
        /// </summary>
        public new ILogicalTreeElement Parent
        {
            get { return this.Owner; }
            protected set { this.Owner = value as BinaryFieldGroupTargetListViewport; }
        }

        /// <summary>
        /// Gets a collection of children controls of the the current control instance.
        /// </summary>
        public IIndexableCollection<ILogicalTreeElement> Children
        {
            get
            {
                if (this.controlsWrapper == null)
                {
                    this.controlsWrapper = new IndexableCollection<ILogicalTreeElement>();
                }

                return this.controlsWrapper;
            }
        }

        /// <summary>
        /// Gets the control extender instance of this control.
        /// </summary>
        public LogicalTreeElementExtender LogicalTreeElementExtender
        {
            get { return this.logicalTreeElementExtender; }
        }

        /// <summary>
        /// Gets or sets the DataContext.
        /// This property may raise a 'DataContext' change notification.
        /// See LogicalTreeElementExtender for more information.
        /// <see cref="LogicalTreeElementExtender"/>
        /// </summary>
        [DesignerSerializationVisibility(DesignerSerializationVisibility.Hidden)]
        public object DataContext
        {
            get { return this.logicalTreeElementExtender.DataContext; }
            set { this.logicalTreeElementExtender.DataContext = value; }
        }

        /// <summary>
        /// Gets the binders collection.
        /// </summary>
        public BindingContainer Bindings { get; private set; }

        /// <summary>
        /// Get or set the flag indicating if mouse is over this item.
        /// </summary>
        public bool IsMouseOver
        {
            get
            {
                return this.isMouseOver;
            }

            set
            {
                if (this.isMouseOver != value)
                {
                    this.isMouseOver = value;
                    this.UpdateAppearance();
                }
            }
        }

        /// <summary>
        /// Get or set the Guid of the binary field group target.
        /// </summary>
        public Guid TargetGuid { get; set; }

        /// <summary>
        /// Get or set the name of the binary field group target.
        /// </summary>
        public string TargetName { get; set; }

        /// <summary>
        /// Get the flag indicating whether the target is a binary data or not.
        /// </summary>
        public bool IsTargetBinaryData { get; set; }

        /// <summary>
        /// Clears the DataContext.
        /// See LogicalTreeElementExtender for more details.
        /// <see cref="LogicalTreeElementExtender"/>
        /// </summary>
        public void ClearDataContext()
        {
            this.logicalTreeElementExtender.ClearDataContext();
        }

        /// <summary>
        /// Render the item's foreground.
        /// </summary>
        /// <param name="g">The graphics object for rendering.</param>
        protected override void DrawForeground(Graphics g)
        {
            var rect = this.TransformedBounds;

            Brush targetTypeBrush;
            string strTargetType;
            if (this.IsTargetBinaryData == true)
            {
                targetTypeBrush = Brushes.DarkGoldenrod;
                strTargetType = Properties.Resources.BinaryFieldGroupTargetTypeBinaryData;
            }
            else
            {
                targetTypeBrush = Brushes.DarkOliveGreen;
                strTargetType = Properties.Resources.BinaryFieldGroupTargetTypeDataModel;
            }

            // Compute the text size of the target type.
            SizeF targetTypeTextSize = g.MeasureString(
                strTargetType,
                BinaryFieldGroupTargetListItemRenderable.smallFont);

            // Compute the text size of the target name.
            SizeF targetNameTextSize = g.MeasureString(
                this.TargetName,
                BinaryFieldGroupTargetListItemRenderable.largeFont);

            // Compute the rectangle for rendering.
            RectangleF targetTypeTextRect = rect;
            targetTypeTextRect.X += 3.0f;
            targetTypeTextRect.Y += 3.0f;
            targetTypeTextRect.Width = rect.Width - 3.0f;
            targetTypeTextRect.Height = targetTypeTextSize.Height;

            // Compute the rectangle for rendering.
            RectangleF targetNameTextRect = rect;
            targetNameTextRect.X += 2.0f;
            targetNameTextRect.Y = rect.Bottom - targetNameTextSize.Height - 2.0f;
            targetNameTextRect.Width = rect.Width - 2.0f;
            targetNameTextRect.Height = targetNameTextSize.Height;

            using (var strFormat = new StringFormat())
            {
                // Top-left alignment.
                strFormat.Alignment = StringAlignment.Near;
                strFormat.LineAlignment = StringAlignment.Far;

                strFormat.Trimming = StringTrimming.EllipsisCharacter;

                g.DrawString(
                    strTargetType,
                    BinaryFieldGroupTargetListItemRenderable.smallFont,
                    targetTypeBrush,
                    targetTypeTextRect,
                    strFormat);

                g.DrawString(
                    this.TargetName,
                    BinaryFieldGroupTargetListItemRenderable.largeFont,
                    Brushes.DarkBlue,
                    targetNameTextRect,
                    strFormat);
            }
        }

        /// <summary>
        /// Update the appearance of the item.
        /// </summary>
        protected virtual void UpdateAppearance()
        {
            if (this.IsMouseOver == true)
            {
                this.BorderThickness = 0.0f;
                this.BorderColor = Color.Transparent;
                this.FillColor = Color.Khaki;
            }
            else
            {
                this.BorderThickness = 0.0f;
                this.BorderColor = Color.Transparent;
                this.FillColor = Color.Beige;
            }
        }
    }
}
