﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using System;
using System.Collections.Generic;
using System.Drawing;
using System.Linq;
using System.Text;
using System.Threading.Tasks;
using System.Windows.Forms;

using EffectMaker.DataModelMaker.UIControls.BinaryDataEditor;
using EffectMaker.DataModelMaker.UIControls.DataModelEditor;
using EffectMaker.DataModelMaker.UIControls.DataModelPropertyEditor;

using EffectMaker.DataModelMaker.UILogic.ViewModels;

using EffectMaker.Foundation.Log;
using EffectMaker.Foundation.Render.ObjectPicking;
using EffectMaker.Foundation.Render.Renderable;
using EffectMaker.Foundation.Render.ScrollBar;

using EffectMaker.UIControls.BaseControls;

namespace EffectMaker.DataModelMaker.UIControls.BinaryDataView
{
    /// <summary>
    /// The panel that renders the binary data list.
    /// </summary>
    internal class BinaryDataListPanel : Panel
    {
        /// <summary>Constant for the factor applying to the vertical mouse scroll speed.</summary>
        private const int VerticalScrollSpeedFactor = 50;

        /// <summary>The viewport for rendering the binary data list.</summary>
        private BinaryDataListViewport binaryDataListViewport = null;

        /// <summary>The scroll bars.</summary>
        private AutoHiddenScrollBar scrollBars = null;

        /// <summary>
        /// Constructor.
        /// </summary>
        public BinaryDataListPanel()
        {
            // Enable double buffer.
            this.DoubleBuffered = true;

            // Make the panel selectable.
            this.SetStyle(ControlStyles.Selectable, true);
            this.TabStop = true;

            this.binaryDataListViewport = new BinaryDataListViewport(this);
            this.binaryDataListViewport.Bounds = new Rectangle(
                0,
                0,
                this.ClientRectangle.Width,
                this.ClientRectangle.Height);

            // Create the scroll bars.
            this.scrollBars = new AutoHiddenScrollBar(this)
            {
                ForeColor = Color.Black,
                BackColor = Color.Black,
                IncrementV = VerticalScrollSpeedFactor
            };

            this.scrollBars.RenderRequired += (s, e) => { this.Invalidate(); };
            this.scrollBars.Scroll += (s, e) =>
            {
                this.binaryDataListViewport.Translation =
                    new PointF(-this.ScrollPosition.X, -this.ScrollPosition.Y);
                this.Invalidate();
            };

            this.scrollBars.SetContentSize(new Size(
                (int)this.binaryDataListViewport.ContentSize.Width,
                (int)this.binaryDataListViewport.ContentSize.Height));
        }

        /// <summary>
        /// Get or set the data context.
        /// </summary>
        public object DataContext
        {
            get
            {
                return this.binaryDataListViewport.DataContext;
            }

            set
            {
                // Set up the data context to the viewport for binding.
                this.binaryDataListViewport.DataContext = value;

                // Update the content size to the scroll bars.
                this.scrollBars.SetContentSize(new Size(
                    (int)this.binaryDataListViewport.ContentSize.Width,
                    (int)this.binaryDataListViewport.ContentSize.Height));
            }
        }

        /// <summary>
        /// Get or set the scroll position.
        /// </summary>
        public Point ScrollPosition
        {
            get
            {
                if (this.scrollBars == null)
                {
                    return Point.Empty;
                }

                return this.scrollBars.ScrollPosition;
            }

            set
            {
                this.SetScrollPosition(value.X, value.Y);
            }
        }

        /// <summary>
        /// Set scroll position.
        /// </summary>
        /// <param name="x">The horizontal position.</param>
        /// <param name="y">The vertical position.</param>
        public void SetScrollPosition(int x, int y)
        {
            if (this.scrollBars == null)
            {
                return;
            }

            var contentSize = new Size(
                (int)this.binaryDataListViewport.ContentSize.Width,
                (int)this.binaryDataListViewport.ContentSize.Height);

            int maxX = contentSize.Width - this.Width;
            int maxY = contentSize.Height - this.Height;

            int posX = Math.Max(Math.Min(x, maxX), 0);
            int posY = Math.Max(Math.Min(y, maxY), 0);

            this.scrollBars.ScrollPosition = new Point(posX, posY);

            this.binaryDataListViewport.Translation = new PointF(-posX, -posY);
        }

        /// <summary>
        /// Show binary data editor dialog for editing the selected binary data, if any.
        /// </summary>
        public void ShowSelectedBinaryDataEditorDialog()
        {
            if (this.binaryDataListViewport.SelectedItem != null)
            {
                var dialog = new BinaryDataEditorDialog()
                {
                    DataSource = this.binaryDataListViewport.SelectedItem.DataContext,
                    StartPosition = FormStartPosition.CenterParent
                };

                dialog.ShowDialog(this);
            }
        }

        /// <summary>
        /// Clean up any resources being used.
        /// </summary>
        /// <param name="disposing">true if managed resources should be disposed; otherwise, false.</param>
        protected override void Dispose(bool disposing)
        {
            if (disposing == true)
            {
                if (this.binaryDataListViewport != null)
                {
                    this.binaryDataListViewport.Dispose();
                }
            }

            base.Dispose(disposing);
        }

        /// <summary>
        /// Handle MouseWheel event.
        /// </summary>
        /// <param name="e">The event arguments.</param>
        protected override void OnMouseWheel(MouseEventArgs e)
        {
            base.OnMouseWheel(e);

            int increment = -(e.Delta / 120) * VerticalScrollSpeedFactor;

            this.SetScrollPosition(this.ScrollPosition.X, this.ScrollPosition.Y + increment);

            this.Invalidate();
        }

        /// <summary>
        /// Handle MouseDoubleClick event.
        /// </summary>
        /// <param name="e">The event arguments.</param>
        protected override void OnMouseDoubleClick(MouseEventArgs e)
        {
            base.OnMouseDoubleClick(e);

            // Do mouse picking.
            ObjectPickContext context = this.binaryDataListViewport.DoPicking(e.Location);
            if (context.PickedObjects != null &&
                context.PickedObjects.Count > 0 &&
                context.PickedObjects[0] is BinaryDataListItemRenderable)
            {
                this.binaryDataListViewport.SelectedItem =
                    context.PickedObjects[0] as BinaryDataListItemRenderable;
                this.ShowSelectedBinaryDataEditorDialog();
            }
        }

        /// <summary>
        /// Handle MouseDown event.
        /// </summary>
        /// <param name="e">The event arguments.</param>
        protected override void OnMouseDown(MouseEventArgs e)
        {
            base.OnMouseDown(e);

            // Do mouse picking.
            if (e.Button == MouseButtons.Left)
            {
                ObjectPickContext context = this.binaryDataListViewport.DoPicking(e.Location);
                if (context.PickedObjects != null &&
                    context.PickedObjects.Count > 0 &&
                    context.PickedObjects[0] is BinaryDataListItemRenderable)
                {
                    this.binaryDataListViewport.SelectedItem =
                        context.PickedObjects[0] as BinaryDataListItemRenderable;
                }
            }
            else if (e.Button == MouseButtons.Right &&
                this.binaryDataListViewport.SelectedItem != null)
            {
                this.binaryDataListViewport.SelectedItem = null;
            }
        }

        /// <summary>
        /// Handle MouseUp event.
        /// </summary>
        /// <param name="e">The event arguments.</param>
        protected override void OnMouseUp(MouseEventArgs e)
        {
            base.OnMouseUp(e);
        }

        /// <summary>
        /// Handle MouseMove event.
        /// </summary>
        /// <param name="e">The event arguments.</param>
        protected override void OnMouseMove(MouseEventArgs e)
        {
            if (this.Focused == false)
            {
                this.Focus();
            }

            base.OnMouseMove(e);

            // Do mouse picking.
            ObjectPickContext context = this.binaryDataListViewport.DoPicking(e.Location);
            if (context.PickedObjects != null &&
                context.PickedObjects.Count > 0 &&
                context.PickedObjects[0] is BinaryDataListItemRenderable)
            {
                this.binaryDataListViewport.MouseOverItem =
                    context.PickedObjects[0] as BinaryDataListItemRenderable;
            }
        }

        /// <summary>
        /// Handle MouseLeave event.
        /// </summary>
        /// <param name="e">The event arguments.</param>
        protected override void OnMouseLeave(EventArgs e)
        {
            base.OnMouseLeave(e);

            if (this.binaryDataListViewport.MouseOverItem != null)
            {
                this.binaryDataListViewport.MouseOverItem = null;
            }
        }

        /// <summary>
        /// Handle PreviewKeyDown event.
        /// </summary>
        /// <param name="e">The event arguments.</param>
        protected override void OnPreviewKeyDown(PreviewKeyDownEventArgs e)
        {
            base.OnPreviewKeyDown(e);

            switch (e.KeyCode)
            {
                case Keys.Down:
                    this.binaryDataListViewport.SelectNextItem();
                    break;

                case Keys.Up:
                    this.binaryDataListViewport.SelectPreviousItem();
                    break;

                case Keys.Enter:
                    this.ShowSelectedBinaryDataEditorDialog();
                    break;
            }
        }

        /// <summary>
        /// Handle SizeChanged event.
        /// </summary>
        /// <param name="e">The event arguments.</param>
        protected override void OnSizeChanged(EventArgs e)
        {
            if (this.binaryDataListViewport != null)
            {
                this.binaryDataListViewport.Size = this.ClientSize;
            }

            if (this.scrollBars != null)
            {
                Point scrollPos = this.scrollBars.ScrollPosition;

                this.scrollBars.SetViewSize(this.Size);
                this.scrollBars.SetDisplayRectangle(this.DisplayRectangle);

                // Reset to the same scroll position again.
                this.scrollBars.ScrollPosition = scrollPos;
            }
        }

        /// <summary>
        /// Handle Paint event.
        /// </summary>
        /// <param name="e">The event arguments.</param>
        protected override void OnPaint(PaintEventArgs e)
        {
            e.Graphics.SmoothingMode = System.Drawing.Drawing2D.SmoothingMode.HighQuality;
            e.Graphics.TextRenderingHint = System.Drawing.Text.TextRenderingHint.ClearTypeGridFit;

            // Render the viewport.
            if (this.binaryDataListViewport != null && this.scrollBars != null)
            {
                this.binaryDataListViewport.Update(e.Graphics);
                this.binaryDataListViewport.Draw(e.Graphics);

                // The data model list is updated, now set the updated content size to the scroll bars.
                this.scrollBars.SetContentSize(new Size(
                    (int)this.binaryDataListViewport.ContentSize.Width,
                    (int)this.binaryDataListViewport.ContentSize.Height));

                // Render the scroll bars.
                this.scrollBars.Render(e.Graphics);
            }
        }
    }
}
