﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using System;
using System.Collections;
using System.Collections.Generic;
using System.ComponentModel;
using System.Drawing;
using System.Linq;

using EffectMaker.DataModelMaker.Core.Core;
using EffectMaker.DataModelMaker.Core.Definitions;

using EffectMaker.DataModelMaker.UIControls.Extenders;

using EffectMaker.Foundation.Extensions;
using EffectMaker.Foundation.Interfaces;
using EffectMaker.Foundation.Log;
using EffectMaker.Foundation.Render.Renderable;
using EffectMaker.Foundation.Utility;

using EffectMaker.UIControls;
using EffectMaker.UIControls.DataBinding;
using EffectMaker.UIControls.Extenders;

namespace EffectMaker.DataModelMaker.UIControls.BinaryDataView
{
    /// <summary>
    /// Class for rendering binary data information.
    /// </summary>
    internal class BinaryDataInfoRenderable : RoundedRectangleShape, ILogicalTreeElement
    {
        /// <summary>Constant for the height of the binary data information area.</summary>
        private const float BinaryDataInfoHeight = 65.0f;

        /// <summary>Constant for the margin of the child renderables.</summary>
        private const float ChildMargin = 5.0f;

        /// <summary>Constant for the spacing between child renderables.</summary>
        private const float ChildSpacing = 5.0f;

        /// <summary>The large font for rendering.</summary>
        private static Font largeFont = null;

        /// <summary>The small font for rendering.</summary>
        private static Font smallFont = null;

        /// <summary>The extender for easily implementing ILogicalTreeElement.</summary>
        private LogicalTreeElementExtender logicalTreeElementExtender = null;

        /// <summary>Backing field for Controls property.</summary>
        private IIndexableCollection<ILogicalTreeElement> controlsWrapper;

        /// <summary>The extender for easily implement operations for child renderables.</summary>
        private ChildRenderableOperationExtender childRenderableOperationExtender = null;

        /// <summary>The child renderables for rendering the fields.</summary>
        private List<BinaryDataInfoFieldRenderable> fieldRenderables =
            new List<BinaryDataInfoFieldRenderable>();

        /// <summary>
        /// Static constructor.
        /// </summary>
        static BinaryDataInfoRenderable()
        {
            // Setup the fonts for rendering.
            smallFont = new Font(FontFamily.GenericSansSerif, 8.0f);
            largeFont = new Font(FontFamily.GenericSansSerif, 14.0f, FontStyle.Bold);
        }

        /// <summary>
        /// Constructor.
        /// </summary>
        /// <param name="owner">The owner of the view.</param>
        public BinaryDataInfoRenderable(Viewport owner) :
            base()
        {
            this.Translation = new PointF(ChildMargin, BinaryDataInfoHeight);
            this.BorderThickness = 2.0f;
            this.FillColor = Color.Khaki;
            this.SetCornerRadius(5.0f);

            this.SelectedFieldDataContext = null;

            this.Bindings = new BindingContainer(this);
            this.logicalTreeElementExtender = new LogicalTreeElementExtender(this);
            this.childRenderableOperationExtender = new ChildRenderableOperationExtender(this);

            this.Owner = owner;

            this.Size = new SizeF(0.0f, BinaryDataInfoHeight);

            // Set up binding.
            this.Bindings.Add(new Binder(this, "BinaryDataName", "Name"));
            this.Bindings.Add(new Binder(this, "FileName", "FileName"));
            this.Bindings.Add(new Binder(this, "SourceDataModelGuid", "SourceDataModelGuid"));
            this.Bindings.Add(new Binder(this, "BinaryFieldSources", "BinaryFieldViewModels"));
            this.Bindings.Add(new Binder(this, "SelectedFieldDataContext", "SelectedFieldViewModel"));

            this.Bindings.Add(new Binder(this, "OnCreateBinaryFieldExecutable", "OnCreateBinaryFieldExecutable"));
            this.Bindings.Add(new Binder(this, "OnRemoveSelectedBinaryFieldExecutable", "OnRemoveSelectedBinaryFieldExecutable"));
            this.Bindings.Add(new Binder(this, "OnMoveBinaryFieldExecutable", "OnMoveBinaryFieldExecutable"));
        }

        /// <summary>
        /// Raised when the value of a property on this control changed.
        /// </summary>
#pragma warning disable 67
        public event PropertyChangedEventHandler PropertyChanged;
#pragma warning restore 67

        /// <summary>
        /// Get the owner binary data info viewport.
        /// </summary>
        public Viewport Owner { get; private set; }

        /// <summary>
        /// gets the parent control.
        /// </summary>
        public new ILogicalTreeElement Parent
        {
            get { return this.Owner as ILogicalTreeElement; }
            protected set { this.Owner = value as Viewport; }
        }

        /// <summary>
        /// Gets a collection of children controls of the the current control instance.
        /// </summary>
        public IIndexableCollection<ILogicalTreeElement> Children
        {
            get
            {
                if (this.controlsWrapper == null)
                {
                    this.controlsWrapper = new IndexableCollection<ILogicalTreeElement>();
                }

                return this.controlsWrapper;
            }
        }

        /// <summary>
        /// Gets the control extender instance of this control.
        /// </summary>
        public LogicalTreeElementExtender LogicalTreeElementExtender
        {
            get { return this.logicalTreeElementExtender; }
        }

        /// <summary>
        /// Gets or sets the DataContext.
        /// This property may raise a 'DataContext' change notification.
        /// See LogicalTreeElementExtender for more information.
        /// <see cref="LogicalTreeElementExtender"/>
        /// </summary>
        [DesignerSerializationVisibility(DesignerSerializationVisibility.Hidden)]
        public object DataContext
        {
            get
            {
                return this.logicalTreeElementExtender.DataContext;
            }

            set
            {
                this.logicalTreeElementExtender.DataContext = value;
                this.Visible = value != null;
            }
        }

        /// <summary>
        /// Gets the binders collection.
        /// </summary>
        public BindingContainer Bindings { get; private set; }

        /// <summary>
        /// Get or set the binary field data sources.
        /// </summary>
        public IEnumerable BinaryFieldSources
        {
            get { return this.fieldRenderables.Select(pr => pr.DataContext).AsEnumerable(); }
            set { this.UpdateItemsFromDataSource(value); }
        }

        /// <summary>
        /// Get or set the executable for creating a new field.
        /// </summary>
        public IExecutable OnCreateBinaryFieldExecutable { get; set; }

        /// <summary>
        /// Get or set the executable for removing the selected field.
        /// </summary>
        public IExecutable OnRemoveSelectedBinaryFieldExecutable { get; set; }

        /// <summary>
        /// Get or set the executable for moving the selected field.
        /// </summary>
        public IExecutable OnMoveBinaryFieldExecutable { get; set; }

        /// <summary>
        /// Get or set the binary data name.
        /// </summary>
        public string BinaryDataName { get; set; }

        /// <summary>
        /// Get or set the source file name to output the binary data definition.
        /// </summary>
        public string FileName { get; set; }

        /// <summary>
        /// Get or set the Guid of the source editor data model.
        /// </summary>
        public Guid SourceDataModelGuid { get; set; }

        /// <summary>
        /// Enumerates all the child field group renderables.
        /// </summary>
        public IEnumerable<BinaryDataInfoFieldRenderable> BinaryFieldRenderables
        {
            get { return this.fieldRenderables; }
        }

        /// <summary>
        /// Get or set the data context of the selected field.
        /// </summary>
        public object SelectedFieldDataContext { get; set; }

        /// <summary>
        /// Clears the DataContext.
        /// See LogicalTreeElementExtender for more details.
        /// <see cref="LogicalTreeElementExtender"/>
        /// </summary>
        public void ClearDataContext()
        {
            this.logicalTreeElementExtender.ClearDataContext();
        }

        /// <summary>
        /// Find binary field with the specified data context.
        /// </summary>
        /// <param name="dataContext">The data context.</param>
        /// <returns>The binary field with the specified data context or null if not found.</returns>
        public BinaryDataInfoFieldRenderable FindFieldWithDataContext(object dataContext)
        {
            return this.fieldRenderables.Find(rd => rd.DataContext == dataContext);
        }

        /// <summary>
        /// Set the selected field renderable.
        /// </summary>
        /// <param name="selectedItem">The selected renderable.</param>
        public void SetSelectedField(BinaryDataInfoFieldRenderable selectedItem)
        {
            // Check if the selected item is the same.
            if (selectedItem == null &&
                this.SelectedFieldDataContext == null)
            {
                return;
            }
            else if (selectedItem != null &&
                selectedItem.DataContext == this.SelectedFieldDataContext)
            {
                return;
            }

            if (selectedItem == null)
            {
                this.SelectedFieldDataContext = null;
            }
            else
            {
                this.SelectedFieldDataContext = selectedItem.DataContext;
            }

            this.logicalTreeElementExtender.NotifyPropertyChanged(propertyName: "SelectedFieldDataContext");
        }

        /// <summary>
        /// Create a new field with the specified target definition Guid.
        /// </summary>
        /// <param name="targetGuid">The Guid of the target definition.</param>
        /// <returns>True on success.</returns>
        public bool CreateField(Guid targetGuid)
        {
            // Find the field group target definition.
            var def = WorkspaceManager.FindDefinition(targetGuid);
            if (def == null)
            {
                // The target is not found, reject it.
                return false;
            }

            if ((def is BinaryDataDefinition) == false &&
                (def is RuntimeDataModelDefinition) == false)
            {
                // The target is not what we want, reject it.
                return false;
            }

            object prevSelection = this.SelectedFieldDataContext;

            // Execute!
            if (this.OnCreateBinaryFieldExecutable != null)
            {
                this.OnCreateBinaryFieldExecutable.Execute(def);
            }

            // The selection should become the created field.
            if (prevSelection == this.SelectedFieldDataContext)
            {
                return false;
            }

            return true;
        }

        /// <summary>
        /// Move the selected field to the specified location.
        /// </summary>
        /// <param name="location">The location.</param>
        public void MoveSelectedField(PointF location)
        {
            if (this.OnMoveBinaryFieldExecutable == null)
            {
                return;
            }

            // Find the child to insert the field to.
            BinaryDataInfoFieldRenderable childAfterCursor = null;
            foreach (var child in this.fieldRenderables)
            {
                if (child.TransformedBounds.GetCenter().Y > location.Y)
                {
                    childAfterCursor = child;
                    break;
                }
            }

            object dataContext = null;
            if (childAfterCursor != null)
            {
                dataContext = childAfterCursor.DataContext;
            }

            this.OnMoveBinaryFieldExecutable.Execute(dataContext);
        }

        /// <summary>
        /// Remove the selected field.
        /// </summary>
        public void RemoveSelectedField()
        {
            if (this.OnRemoveSelectedBinaryFieldExecutable != null)
            {
                this.OnRemoveSelectedBinaryFieldExecutable.Execute(null);
            }
        }

        /// <summary>
        /// Add a child renderable.
        /// </summary>
        /// <param name="child">The child to add.</param>
        public override void AddRenderable(RenderableBase child)
        {
            this.childRenderableOperationExtender.AddRenderable(child);
            base.AddRenderable(child);

            var childRenderable = child as BinaryDataInfoFieldRenderable;
            if (childRenderable != null)
            {
                this.fieldRenderables.Add(childRenderable);
            }
        }

        /// <summary>
        /// Add child renderables.
        /// </summary>
        /// <param name="children">The children to add.</param>
        public override void AddRenderableRange(IEnumerable<RenderableBase> children)
        {
            this.childRenderableOperationExtender.AddRenderableRange(children);
            base.AddRenderableRange(children);

            foreach (RenderableBase child in children)
            {
                var childRenderable = child as BinaryDataInfoFieldRenderable;
                if (childRenderable != null)
                {
                    this.fieldRenderables.Add(childRenderable);
                }
            }
        }

        /// <summary>
        /// Remove a child renderable.
        /// </summary>
        /// <param name="child">The child to remove.</param>
        public override void RemoveRenderable(RenderableBase child)
        {
            this.childRenderableOperationExtender.RemoveRenderable(child);
            base.RemoveRenderable(child);

            var childRenderable = child as BinaryDataInfoFieldRenderable;
            if (childRenderable != null)
            {
                this.fieldRenderables.Remove(childRenderable);
            }
        }

        /// <summary>
        /// Remove child renderables.
        /// </summary>
        /// <param name="children">The children to remove.</param>
        public override void RemoveRenderableRange(IEnumerable<RenderableBase> children)
        {
            this.childRenderableOperationExtender.RemoveRenderableRange(children);
            base.RemoveRenderableRange(children);

            // To prevent modifying the property renderable list while looping it,
            // copy the children to another array first.
            var tmpChildren = children;
            if (tmpChildren == this.fieldRenderables)
            {
                tmpChildren = children.ToArray();
            }

            foreach (RenderableBase child in tmpChildren)
            {
                var childRenderable = child as BinaryDataInfoFieldRenderable;
                if (childRenderable != null)
                {
                    this.fieldRenderables.Remove(childRenderable);
                }
            }
        }

        /// <summary>
        /// Clear child renderables.
        /// </summary>
        public override void ClearRenderables()
        {
            this.childRenderableOperationExtender.ClearRenderables();
            base.ClearRenderables();

            this.fieldRenderables.Clear();
        }

        /// <summary>
        /// Update all the connections.
        /// </summary>
        public void UpdateConnections()
        {
            foreach (BinaryDataInfoFieldRenderable rd in this.fieldRenderables)
            {
                rd.UpdateConnections();
            }
        }

        /// <summary>
        /// Update the children for rendering.
        /// </summary>
        /// <param name="context">Data context that contains information for rendering.</param>
        protected override void UpdateChildren(RenderContext context)
        {
            float childWidth = this.Width - (ChildMargin * 2.0f);

            // Loop through field group renderables to update the content size.
            float location = 0.0f;
            foreach (RenderableBase child in this.fieldRenderables)
            {
                child.Size = new SizeF(childWidth, child.Height);
                child.Location = new PointF(0.0f, location);

                location += child.Height;
            }

            base.UpdateChildren(context);

            // Loop through binary field renderables to find out the height of the children.
            float height = BinaryDataInfoHeight;
            foreach (RenderableBase child in this.fieldRenderables)
            {
                height += child.Height;
            }

            this.Size = new SizeF(this.Width, height);
        }

        /// <summary>
        /// Render the item's background.
        /// </summary>
        /// <param name="g">The graphics object for rendering.</param>
        protected override void DrawBackground(Graphics g)
        {
            base.DrawBackground(g);

            // Draw the separation line on the top edge of the field renderables.
            using (var pen = new Pen(Color.Silver, 1.0f))
            {
                foreach (var renderable in this.fieldRenderables)
                {
                    var rect = renderable.TransformedBounds;
                    g.DrawLine(pen, rect.Location, new PointF(rect.Right, rect.Top));
                }
            }
        }

        /// <summary>
        /// Render the item's foreground.
        /// </summary>
        /// <param name="g">The graphics object for rendering.</param>
        protected override void DrawForeground(Graphics g)
        {
            if (Guid.Empty.Equals(this.SourceDataModelGuid) == true)
            {
                return;
            }

            var rect = this.TransformedBounds;

            string sourceDataModelName = string.Empty;
            var def = WorkspaceManager.FindDefinition(this.SourceDataModelGuid) as EditorDataModelDefinition;
            if (def != null)
            {
                sourceDataModelName = def.Name;
            }

            // Compute the text size of the file name.
            SizeF fileNameTextSize = g.MeasureString(
                this.FileName,
                BinaryDataInfoRenderable.smallFont);

            // Compute the text size of the source data model name.
            SizeF srcDataModelNameTextSize = g.MeasureString(
                sourceDataModelName,
                BinaryDataInfoRenderable.smallFont);

            // Compute the text size of the binary data name.
            SizeF binaryDataNameTextSize = g.MeasureString(
                this.BinaryDataName,
                BinaryDataInfoRenderable.largeFont);

            // Compute the rectangle for rendering.
            RectangleF fileNameTextRect = rect;
            fileNameTextRect.X += 3.0f;
            fileNameTextRect.Y += 6.0f;
            fileNameTextRect.Width = rect.Width - 3.0f;
            fileNameTextRect.Height = fileNameTextSize.Height;

            // Compute the rectangle for rendering.
            RectangleF binaryDataNameTextRect = rect;
            binaryDataNameTextRect.X += 2.0f;
            binaryDataNameTextRect.Y = rect.Top + BinaryDataInfoHeight - binaryDataNameTextSize.Height - 5.0f;
            binaryDataNameTextRect.Width = rect.Width - 2.0f;
            binaryDataNameTextRect.Height = binaryDataNameTextSize.Height;

            // Compute the rectangle for rendering.
            RectangleF srcDataModelNameTextRect = rect;
            srcDataModelNameTextRect.X += 3.0f;
            srcDataModelNameTextRect.Y = binaryDataNameTextRect.Top - srcDataModelNameTextSize.Height;
            srcDataModelNameTextRect.Width = rect.Width - 3.0f;
            srcDataModelNameTextRect.Height = srcDataModelNameTextSize.Height;

            using (var strFormat = new StringFormat())
            {
                // Top-left alignment.
                strFormat.Alignment = StringAlignment.Near;
                strFormat.LineAlignment = StringAlignment.Far;

                strFormat.Trimming = StringTrimming.EllipsisPath;

                g.DrawString(
                    this.FileName,
                    BinaryDataInfoRenderable.smallFont,
                    Brushes.DarkCyan,
                    fileNameTextRect,
                    strFormat);

                strFormat.LineAlignment = StringAlignment.Center;
                strFormat.Trimming = StringTrimming.EllipsisCharacter;

                g.DrawString(
                    sourceDataModelName,
                    BinaryDataInfoRenderable.smallFont,
                    Brushes.Maroon,
                    srcDataModelNameTextRect,
                    strFormat);

                strFormat.LineAlignment = StringAlignment.Near;

                g.DrawString(
                    this.BinaryDataName,
                    BinaryDataInfoRenderable.largeFont,
                    Brushes.DarkBlue,
                    binaryDataNameTextRect,
                    strFormat);
            }
        }

        /// <summary>
        /// Update items from data source.
        /// </summary>
        /// <param name="items">The items from data source.</param>
        private void UpdateItemsFromDataSource(IEnumerable items)
        {
            var tmpList = this.fieldRenderables.ToList();

            float height = 0.0f;

            // Loop through binary field group source items.
            foreach (object item in items)
            {
                // Create a renderable for the item source.
                var renderable = tmpList.Find(it => it.DataContext == item);
                if (renderable == null)
                {
                    renderable = new BinaryDataInfoFieldRenderable(this);
                }
                else
                {
                    // This renderable still exists, remove from the list.
                    // Whatever left in the list in the end will be removed from the children list.
                    tmpList.Remove(renderable);

                    this.Children.Remove(renderable);
                    base.RemoveRenderable(renderable);
                    this.fieldRenderables.Remove(renderable);
                }

                // Set up the data context.
                renderable.DataContext = item;

                // Setup location and size.
                renderable.Location = new PointF(0.0f, height);

                height += renderable.Height + ChildSpacing;

                // Add the list item to the child renderable list.
                this.AddRenderable(renderable);
            }

            // Remove the renderables that no longer exists.
            this.RemoveRenderableRange(tmpList);

            this.Size = new SizeF(this.Width, height + BinaryDataInfoHeight);
        }
    }
}
