﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using System;
using System.Collections.Generic;
using System.Drawing;
using System.Linq;
using System.Text;
using System.Threading.Tasks;
using System.Windows.Forms;

using EffectMaker.DataModelMaker.Core.Core;

using EffectMaker.DataModelMaker.UILogic.ViewModels;

using EffectMaker.Foundation.Log;
using EffectMaker.Foundation.Render.ObjectPicking;
using EffectMaker.Foundation.Render.Renderable;
using EffectMaker.Foundation.Render.ScrollBar;

using EffectMaker.UIControls;
using EffectMaker.UIControls.BaseControls;

namespace EffectMaker.DataModelMaker.UIControls.BinaryDataView
{
    /// <summary>
    /// Panel to show the information of a binary data.
    /// </summary>
    internal class BinaryDataInfoPanel : Panel
    {
        /// <summary>Constant for the factor applying to the vertical mouse scroll speed.</summary>
        private const int VerticalScrollSpeedFactor = 50;

        /// <summary>The viewport for rendering the binary data information.</summary>
        private BinaryDataInfoViewport binaryDataInfoViewport = null;

        /// <summary>The scroll bars.</summary>
        private AutoHiddenScrollBar scrollBars = null;

        /// <summary>The binary data property that the cursor is hovering over.</summary>
        private BinaryDataInfoFieldRenderable mouseOverItem = null;

        /// <summary>The binary data property that is currently selected.</summary>
        private BinaryDataInfoFieldRenderable selectedItem = null;

        /// <summary>Flag indicating whether the user is dragging on the panel.</summary>
        private bool isDragging = false;

        /// <summary>
        /// Constructor.
        /// </summary>
        public BinaryDataInfoPanel()
        {
            // Enable double buffer.
            this.DoubleBuffered = true;
            this.AllowDrop = true;

            // Make the panel selectable.
            this.SetStyle(ControlStyles.Selectable, true);
            this.TabStop = true;

            // Create the viewport for rendering the binary data information.
            this.binaryDataInfoViewport = new BinaryDataInfoViewport(this)
            {
                Bounds = new Rectangle(
                    0,
                    0,
                    this.ClientRectangle.Width,
                    this.ClientRectangle.Height)
            };

            // Create and set up the scroll bars.
            this.scrollBars = new AutoHiddenScrollBar(this)
            {
                ForeColor = Color.Black,
                BackColor = Color.Black,
                IncrementV = VerticalScrollSpeedFactor
            };

            this.scrollBars.RenderRequired += (s, e) => { this.Invalidate(); };
            this.scrollBars.Scroll += (s, e) =>
            {
                this.binaryDataInfoViewport.Translation =
                    new PointF(-this.ScrollPosition.X, -this.ScrollPosition.Y);
                this.Invalidate();
            };

            this.scrollBars.SetContentSize(new Size(
                (int)this.binaryDataInfoViewport.ContentSize.Width,
                (int)this.binaryDataInfoViewport.ContentSize.Height));
        }

        /// <summary>
        /// Get or set the data context.
        /// </summary>
        public object DataContext
        {
            get
            {
                return this.binaryDataInfoViewport.DataContext;
            }

            set
            {
                this.binaryDataInfoViewport.DataContext = value;

                if (value == null)
                {
                    this.binaryDataInfoViewport.Visible = false;
                    return;
                }
                else
                {
                    this.binaryDataInfoViewport.Visible = true;
                }

                // Restore the selection.
                if (this.SelectedItem != null)
                {
                    var selectedViewModel = this.selectedItem.DataContext as ViewModelBase;
                    if (selectedViewModel != null)
                    {
                        // Get the binary data renderable.
                        var binaryDataRenderable =
                            this.binaryDataInfoViewport.BinaryDataInfoRenderable;
                        if (binaryDataRenderable != null)
                        {
                            // Find the property renderable that references
                            // the same property definition as the selected one.
                            foreach (var renderable in binaryDataRenderable.BinaryFieldRenderables)
                            {
                                var viewModel = renderable.DataContext as ViewModelBase;
                                if (viewModel != null &&
                                    viewModel.DataModel == selectedViewModel.DataModel)
                                {
                                    this.SelectedItem = renderable;
                                    break;
                                }
                            }
                        }
                    }
                    else
                    {
                        this.SelectedItem = null;
                    }
                }
            }
        }

        /// <summary>
        /// Get or set the scroll position.
        /// </summary>
        public Point ScrollPosition
        {
            get { return this.scrollBars.ScrollPosition; }
            set { this.SetScrollPosition(value.X, value.Y); }
        }

        /// <summary>
        /// Get or set the binary field item that the cursor is currently hovering over.
        /// </summary>
        public BinaryDataInfoFieldRenderable MouseOverItem
        {
            get
            {
                return this.mouseOverItem;
            }

            set
            {
                if (this.mouseOverItem != value)
                {
                    if (this.mouseOverItem != null)
                    {
                        this.mouseOverItem.IsMouseOver = false;
                    }

                    this.mouseOverItem = value;

                    if (this.mouseOverItem != null)
                    {
                        this.mouseOverItem.IsMouseOver = true;
                    }
                }
            }
        }

        /// <summary>
        /// Get or set the selected binary field item.
        /// </summary>
        public BinaryDataInfoFieldRenderable SelectedItem
        {
            get
            {
                return this.selectedItem;
            }

            set
            {
                if (object.ReferenceEquals(this.selectedItem, value) == false)
                {
                    this.binaryDataInfoViewport.SuspendUpdateAndRendering();

                    if (this.selectedItem != null)
                    {
                        this.selectedItem.IsSelected = false;
                    }

                    this.selectedItem = value;

                    if (this.selectedItem != null)
                    {
                        this.selectedItem.IsSelected = true;
                    }

                    if (this.binaryDataInfoViewport != null &&
                        this.binaryDataInfoViewport.BinaryDataInfoRenderable != null)
                    {
                        this.binaryDataInfoViewport.BinaryDataInfoRenderable.SetSelectedField(this.selectedItem);
                    }

                    this.binaryDataInfoViewport.ResumeUpdateAndRendering();
                    this.Invalidate();
                    this.Update();
                }
            }
        }

        /// <summary>
        /// Set scroll position.
        /// </summary>
        /// <param name="x">The horizontal position.</param>
        /// <param name="y">The vertical position.</param>
        public void SetScrollPosition(int x, int y)
        {
            var contentSize = new Size(
                (int)this.binaryDataInfoViewport.ContentSize.Width,
                (int)this.binaryDataInfoViewport.ContentSize.Height);

            int maxX = contentSize.Width - this.Width;
            int maxY = contentSize.Height - this.Height;

            int posX = Math.Max(Math.Min(x, maxX), 0);
            int posY = Math.Max(Math.Min(y, maxY), 0);

            this.scrollBars.ScrollPosition = new Point(posX, posY);

            this.binaryDataInfoViewport.Translation = new PointF(-posX, -posY);
        }

        /// <summary>
        /// Clean up any resources being used.
        /// </summary>
        /// <param name="disposing">true if managed resources should be disposed; otherwise, false.</param>
        protected override void Dispose(bool disposing)
        {
            if (disposing == true)
            {
                if (this.binaryDataInfoViewport != null)
                {
                    this.binaryDataInfoViewport.Dispose();
                }
            }

            base.Dispose(disposing);
        }

        /// <summary>
        /// Handle MouseWheel event.
        /// </summary>
        /// <param name="e">The event arguments.</param>
        protected override void OnMouseWheel(MouseEventArgs e)
        {
            base.OnMouseWheel(e);

            int increment = -(e.Delta / 120) * VerticalScrollSpeedFactor;

            this.SetScrollPosition(this.ScrollPosition.X, this.ScrollPosition.Y + increment);

            this.Invalidate();
        }

        /// <summary>
        /// Handle MouseDoubleClick event.
        /// </summary>
        /// <param name="e">The event arguments.</param>
        protected override void OnMouseDoubleClick(MouseEventArgs e)
        {
            base.OnMouseDoubleClick(e);
        }

        /// <summary>
        /// Handle MouseDown event.
        /// </summary>
        /// <param name="e">The event arguments.</param>
        protected override void OnMouseDown(MouseEventArgs e)
        {
            base.OnMouseDown(e);

            // Do mouse picking.
            ObjectPickContext context = this.binaryDataInfoViewport.DoPicking(e.Location);
            if (context.PickedObjects != null &&
                context.PickedObjects.Count > 0)
            {
                var renderable =
                    this.FindTopLevelFieldRenderable(context.PickedObjects[0] as ILogicalTreeElement);

                this.SelectedItem = renderable;
                if (this.SelectedItem != null)
                {
                    this.DoDragDrop(this.SelectedItem.TargetGuid, DragDropEffects.Move);
                }
            }
            else if (this.SelectedItem != null)
            {
                this.SelectedItem = null;
            }
        }

        /// <summary>
        /// Handle MouseUp event.
        /// </summary>
        /// <param name="e">The event arguments.</param>
        protected override void OnMouseUp(MouseEventArgs e)
        {
            base.OnMouseUp(e);
        }

        /// <summary>
        /// Handle MouseMove event.
        /// </summary>
        /// <param name="e">The event arguments.</param>
        protected override void OnMouseMove(MouseEventArgs e)
        {
            if (this.Focused == false)
            {
                this.Focus();
            }

            base.OnMouseMove(e);

            if (this.isDragging == false)
            {
                // Do mouse picking.
                ObjectPickContext context = this.binaryDataInfoViewport.DoPicking(e.Location);
                if (context.PickedObjects != null &&
                    context.PickedObjects.Count > 0)
                {
                    this.MouseOverItem =
                        this.FindTopLevelFieldRenderable(context.PickedObjects[0] as ILogicalTreeElement);
                }
                else if (this.MouseOverItem != null)
                {
                    this.MouseOverItem = null;
                }
            }
        }

        /// <summary>
        /// Handle MouseLeave event.
        /// </summary>
        /// <param name="e">The event arguments.</param>
        protected override void OnMouseLeave(EventArgs e)
        {
            base.OnMouseLeave(e);

            if (this.MouseOverItem != null)
            {
                this.MouseOverItem = null;
            }
        }

        /// <summary>
        /// Handle DragEnter event.
        /// </summary>
        /// <param name="e">The event arguments.</param>
        protected override void OnDragEnter(DragEventArgs e)
        {
            base.OnDragEnter(e);

            object dropData = e.Data.GetData(DataFormats.Text);
            Guid guid;
            if ((dropData is string) == false ||
                Guid.TryParse((string)dropData, out guid) == false)
            {
                // Not the data we want, reject it.
                e.Effect = DragDropEffects.None;
                return;
            }

            if (this.binaryDataInfoViewport == null ||
                this.binaryDataInfoViewport.BinaryDataInfoRenderable == null ||
                this.binaryDataInfoViewport.BinaryDataInfoRenderable.Visible == false)
            {
                // There is no binary data renderable on the screen, reject it.
                e.Effect = DragDropEffects.None;
                return;
            }

            var renderable = this.binaryDataInfoViewport.BinaryDataInfoRenderable;
            if (renderable.CreateField(guid) == false)
            {
                // Rejected by the binary data renderable.
                e.Effect = DragDropEffects.None;
                return;
            }

            Point clientLocation = this.PointToClient(new Point(e.X, e.Y));
            renderable.MoveSelectedField(clientLocation);

            e.Effect = DragDropEffects.Move;

            this.isDragging = true;
        }

        /// <summary>
        /// Handle DragOver event.
        /// </summary>
        /// <param name="e">The event arguments.</param>
        protected override void OnDragOver(DragEventArgs e)
        {
            base.OnDragOver(e);

            if (this.binaryDataInfoViewport == null ||
                this.binaryDataInfoViewport.BinaryDataInfoRenderable == null ||
                this.binaryDataInfoViewport.BinaryDataInfoRenderable.Visible == false)
            {
                // There is no binary data renderable on the screen, reject it.
                e.Effect = DragDropEffects.None;
                return;
            }

            var renderable = this.binaryDataInfoViewport.BinaryDataInfoRenderable;

            this.SelectedItem =
                renderable.FindFieldWithDataContext(renderable.SelectedFieldDataContext);

            Point clientLocation = this.PointToClient(new Point(e.X, e.Y));
            renderable.MoveSelectedField(clientLocation);

            e.Effect = DragDropEffects.Move;

            this.isDragging = true;
        }

        /// <summary>
        /// Handle DragLeave event.
        /// </summary>
        /// <param name="e">The event arguments.</param>
        protected override void OnDragLeave(EventArgs e)
        {
            base.OnDragLeave(e);

            if (this.binaryDataInfoViewport == null ||
                this.binaryDataInfoViewport.BinaryDataInfoRenderable == null)
            {
                return;
            }

            // Cancel the drag and drop.
            var renderable = this.binaryDataInfoViewport.BinaryDataInfoRenderable;

            Point clientLocation = this.PointToClient(Cursor.Position);
            if (renderable.TransformedBounds.Contains(clientLocation) == false)
            {
                renderable.RemoveSelectedField();
            }

            this.isDragging = false;
        }

        /// <summary>
        /// Handle DragDrop event.
        /// </summary>
        /// <param name="e">The event arguments.</param>
        protected override void OnDragDrop(DragEventArgs e)
        {
            base.OnDragDrop(e);

            this.isDragging = false;
        }

        /// <summary>
        /// Handle SizeChanged event.
        /// </summary>
        /// <param name="e">The event arguments.</param>
        protected override void OnSizeChanged(EventArgs e)
        {
            if (this.binaryDataInfoViewport != null)
            {
                this.binaryDataInfoViewport.Size = this.ClientSize;
            }

            Point scrollPos = this.scrollBars.ScrollPosition;

            this.scrollBars.SetViewSize(this.Size);
            this.scrollBars.SetDisplayRectangle(this.DisplayRectangle);

            // Reset to the same scroll position again.
            this.scrollBars.ScrollPosition = scrollPos;
        }

        /// <summary>
        /// Handle Paint event.
        /// </summary>
        /// <param name="e">The event arguments.</param>
        protected override void OnPaint(PaintEventArgs e)
        {
            e.Graphics.SmoothingMode = System.Drawing.Drawing2D.SmoothingMode.HighQuality;
            e.Graphics.TextRenderingHint = System.Drawing.Text.TextRenderingHint.ClearTypeGridFit;

            // Render the viewport.
            if (this.binaryDataInfoViewport != null && this.scrollBars != null)
            {
                this.binaryDataInfoViewport.Update(e.Graphics);
                this.binaryDataInfoViewport.Draw(e.Graphics);

                // The binary data renderable is updated, now set the
                // updated content size to the scroll bars.
                this.scrollBars.SetContentSize(new Size(
                    (int)this.binaryDataInfoViewport.ContentSize.Width,
                    (int)this.binaryDataInfoViewport.ContentSize.Height));

                // Render the scroll bars.
                this.scrollBars.Render(e.Graphics);
            }
        }

        /// <summary>
        /// Find the top level binary field renderable beginning from the specified element.
        /// </summary>
        /// <param name="element">The logical tree element to start searching.</param>
        /// <returns>The field group renderable.</returns>
        private BinaryDataInfoFieldRenderable FindTopLevelFieldRenderable(ILogicalTreeElement element)
        {
            while (element != null)
            {
                var renderable = element as BinaryDataInfoFieldRenderable;
                if (renderable != null && renderable.Parent is BinaryDataInfoRenderable)
                {
                    return renderable;
                }

                element = element.Parent;
            }

            return null;
        }
    }
}
