﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using System;
using System.Collections.Generic;
using System.Diagnostics;
using System.IO;
using System.Linq;
using System.Reflection;
using System.Text;
using System.Xml;
using System.Xml.Serialization;

using EffectMaker.DataModelMaker.Core;
using EffectMaker.DataModelMaker.Core.DataTypes;
using EffectMaker.DataModelMaker.Core.Definitions;
using EffectMaker.DataModelMaker.Core.Properties;
using EffectMaker.DataModelMaker.Core.Template;
using EffectMaker.DataModelMaker.Core.Writers;

namespace EffectMaker.DataModelMaker.Generators
{
    /// <summary>
    /// データモデルメーカーです.
    /// </summary>
    public class DataModelGenerator : IGenerator
    {
        /// <summary>
        /// 定義データです.
        /// </summary>
        private EditorDataModelRootDefinition root;

        /// <summary>
        /// コンストラクタです.
        /// </summary>
        public DataModelGenerator()
        {
            this.root = new EditorDataModelRootDefinition();
        }

        /// <summary>
        /// コンストラクタです.
        /// </summary>
        /// <param name="definition">The definition.</param>
        public DataModelGenerator(EditorDataModelRootDefinition definition)
        {
            this.root = definition;
        }

        /// <summary>
        /// Event triggered when the generation progress is advanced.
        /// </summary>
        public event EventHandler ProgressAdvanced;

        /// <summary>
        /// Evaluate total step count for Generate().
        /// </summary>
        /// <returns>The total stap count.</returns>
        public int EvaluateTotalProgressSteps()
        {
            // Count the non-versioned data models.
            int count = this.root.DataModels.Count(
                dm => dm.Export == true && dm.IsVersionEnabled == false);

            // Count the versioned data models.
            for (int i = 0; i < this.root.Versions.Count; ++i)
            {
                count += this.root.DataModels.Count(
                    dm => dm.Export == true && dm.IsVersionEnabled == true && dm.DeletedVersionIndex < i);

                // Plus one for the data model serializer.
                ++count;
            }

            return count;
        }

        /// <summary>
        /// 定義ファイルをロードします.
        /// </summary>
        /// <param name="path">入力ファイルパス.</param>
        /// <returns>ロードに成功したらtrueを返却します.</returns>
        public bool LoadDefinitionFile(string path)
        {
            // ファイルパス/ファイルの存在をチェック.
            if (Path.IsPathRooted(path) == false ||
                File.Exists(path) == false)
            {
                return false;
            }

            // 破棄処理.
            if (this.root != null)
            {
                this.root.Dispose();
            }

            // デシリアライズ.
            using (XmlReader reader = new XmlTextReader(new FileStream(path, FileMode.Open)))
            {
                XmlSerializer serializer = new XmlSerializer(typeof(EditorDataModelRootDefinition));
                this.root = (EditorDataModelRootDefinition)serializer.Deserialize(reader);
            }

            return true;
        }

        /// <summary>
        /// 定義ファイルをセーブします.
        /// </summary>
        /// <param name="path">出力ファイルパス.</param>
        /// <returns>セーブに成功したらtrueを返却します.</returns>
        public bool SaveDefinitionFile(string path)
        {
            // シリアライズ.
            using (TextWriter writer = new StreamWriter(path))
            {
                XmlSerializer serializer = new XmlSerializer(typeof(EditorDataModelRootDefinition));
                serializer.Serialize(writer, this.root);
            }

            return true;
        }

        /// <summary>
        /// C# ファイルを出力します.
        /// </summary>
        /// <param name="path">出力フォルダパス.</param>
        /// <returns>出力に成功したらtrueを返却します.</returns>
        public bool Generate(string path)
        {
            // 準備しようね.
            this.Setup();

            if (this.root.IsVersionEnabled == true)
            {
                var nonVersionedFolderPath = Path.Combine(path, "NonVersioned");
                var versionedFolderPath = Path.Combine(path, "Versioned");

                // Find non-versioned data models.
                var nonVersionedDataModels =
                    from dm in this.root.DataModels
                    where dm.Export == true &&
                          dm.IsVersionEnabled == false
                    select dm;

                // Generate non-versioned data models.
                this.GenerateNonVersionedDataModels(
                    nonVersionedFolderPath,
                    nonVersionedDataModels);

                Version previousVersion = null;
                for (int i = 0; i < this.root.Versions.Count; ++i)
                {
                    EditorVersionDefinition versionDef = this.root.Versions[i];

                    // Find versioned data models.
                    var versionedDataModels =
                        from dm in this.root.DataModels
                        where dm.Export == true &&
                              dm.IsVersionEnabled == true &&
                              dm.CreatedVersionIndex <= i &&
                              (dm.DeletedVersionIndex > i || dm.DeletedVersionIndex < 0)
                        select dm;

                    this.GenerateVersionedDataModels(
                        versionedFolderPath,
                        versionedDataModels,
                        versionDef.Version);

                    this.GenerateSerializer(
                        versionedFolderPath,
                        previousVersion,
                        versionDef.Version);

                    previousVersion = versionDef.Version;
                }
            }
            else
            {
                // Find all data models to export.
                var nonVersionedDataModels =
                    from dm in this.root.DataModels
                    where dm.Export == true
                    select dm;

                // Generate data models.
                this.GenerateNonVersionedDataModels(path, nonVersionedDataModels);
            }

            return true;
        }

        /// <summary>
        /// 定義データを設定します.
        /// </summary>
        /// <param name="definition">設定する定義データです.</param>
        /// <returns>定義データの設定に成功したらtrueを返却します.</returns>
        public bool SetDefinitionData(DefinitionBase definition)
        {
            // キャスト.
            var rootDef = definition as EditorDataModelRootDefinition;
            if (rootDef == null)
            {
                // 異常終了.
                return false;
            }

            // 破棄処理.
            if (this.root != null)
            {
                this.root.Dispose();
            }

            // 定義データを設定.
            this.root = rootDef;

            // 正常終了.
            return true;
        }

        /// <summary>
        /// 定義データを取得します.
        /// </summary>
        /// <returns>定義データを返却します.</returns>
        public DefinitionBase GetDefinitionData()
        {
            return this.root;
        }

        /// <summary>
        /// デバッグ用セットアップ.
        /// </summary>
        private void Setup()
        {
            // 名前空間の重複を削除し，ソートする.
            this.root.Setup();
        }

        /// <summary>
        /// Generate source files for non-versioned data models.
        /// </summary>
        /// <param name="baseOutputPath">The base output folder path.</param>
        /// <param name="dataModels">The data model definitions to generate.</param>
        /// <returns>True on success.</returns>
        private bool GenerateNonVersionedDataModels(
            string baseOutputPath,
            IEnumerable<EditorDataModelDefinition> dataModels)
        {
            string path = baseOutputPath;

            // Set up editing version.
            this.root.EditingVersionIndex = this.root.LatestVersionIndex;

            // Generate data model source files.
            foreach (EditorDataModelDefinition dataModelDef in dataModels)
            {
                Debug.Assert(
                    string.IsNullOrEmpty(dataModelDef.Name) == false,
                    "One or more data model has no name defined.");

                // Compose output folder path.
                string outputFolderPath =
                    Path.Combine(path, dataModelDef.ExportFolder);

                // Check if the output folder path exists.
                if (Directory.Exists(outputFolderPath) == false)
                {
                    // Create the folder if it doesn't exist.
                    Directory.CreateDirectory(outputFolderPath);
                }

                // Compose the output source file path.
                string outputFilePath =
                    Path.Combine(outputFolderPath, dataModelDef.Name + ".cs");

                // Generate data model source code with templates.
                var code = (new EditorDataModelWriter()).Write(dataModelDef);

                // Write generated source code to file.
                File.WriteAllText(outputFilePath, code.Replace(" " + Environment.NewLine, Environment.NewLine), Encoding.UTF8);

                // Advance progress.
                if (this.ProgressAdvanced != null)
                {
                    this.ProgressAdvanced(this, EventArgs.Empty);
                }
            }

            return true;
        }

        /// <summary>
        /// Generate source files for versioned data models.
        /// </summary>
        /// <param name="baseOutputPath">The base output folder path.</param>
        /// <param name="dataModels">The data model definitions to generate.</param>
        /// <param name="version">The version to generate the data models with.</param>
        /// <returns>True on success.</returns>
        private bool GenerateVersionedDataModels(
            string baseOutputPath,
            IEnumerable<EditorDataModelDefinition> dataModels,
            Version version)
        {
            // Find the version index.
            int versionIndex = this.root.FindVersionIndex(version);
            if (versionIndex < 0)
            {
                return false;
            }

            // Set up editing version.
            this.root.EditingVersionIndex = versionIndex;

            // Compose the output folder path for the version.
            string versionFolderPath = string.Format(
                "{0}_{1}_{2}_{3}",
                version.Major,
                version.Minor,
                version.Build,
                version.Revision);

            var path = Path.Combine(baseOutputPath, versionFolderPath);

            // Generate data model source files.
            foreach (EditorDataModelDefinition dataModelDef in dataModels)
            {
                Debug.Assert(
                    string.IsNullOrEmpty(dataModelDef.Name) == false,
                    "One or more data model has no name defined.");

                // Compose output folder path.
                string outputFolderPath =
                    Path.Combine(path, dataModelDef.ExportFolder);

                // Check if the output folder path exists.
                if (Directory.Exists(outputFolderPath) == false)
                {
                    // Create the folder if it doesn't exist.
                    Directory.CreateDirectory(outputFolderPath);
                }

                // Compose the output source file path.
                string outputFilePath =
                    Path.Combine(outputFolderPath, dataModelDef.Name + ".cs");

                // Generate data model source code with templates.
                var code = (new EditorDataModelWriter()).Write(dataModelDef);

                // Write generated source code to file.
                File.WriteAllText(outputFilePath, code.Replace(" " + Environment.NewLine, Environment.NewLine), Encoding.UTF8);

                // Advance progress.
                if (this.ProgressAdvanced != null)
                {
                    this.ProgressAdvanced(this, EventArgs.Empty);
                }
            }

            return true;
        }

        /// <summary>
        /// Generate data model serializer source file.
        /// </summary>
        /// <param name="baseOutputPath">The base output folder path.</param>
        /// <param name="previousVesion">The previous version.</param>
        /// <param name="currentVersion">The current version.</param>
        /// <returns>True on success.</returns>
        private bool GenerateSerializer(
            string baseOutputPath,
            Version previousVesion,
            Version currentVersion)
        {
            // Find the version index.
            int versionIndex = this.root.FindVersionIndex(currentVersion);
            if (versionIndex < 0)
            {
                return false;
            }

            // Set up editing version.
            this.root.EditingVersionIndex = versionIndex;

            // Compose the name space for the current version.
            string currVersionNamespace = string.Format(
                "{0}_{1}_{2}_{3}",
                currentVersion.Major,
                currentVersion.Minor,
                currentVersion.Build,
                currentVersion.Revision);

            // Compose the name space for the previous version.
            TemplateHelper helper = new TemplateHelper();
            if (previousVesion != null)
            {
                string prevVersionNamespace = string.Format(
                    "{0}_{1}_{2}_{3}",
                    previousVesion.Major,
                    previousVesion.Minor,
                    previousVesion.Build,
                    previousVesion.Revision);

                // Find versioned data models.
                var dataModelNames =
                    from dm in this.root.DataModels
                    where dm.Export == true &&
                          dm.IsVersionEnabled == true &&
                          (dm.CreatedVersionIndex < versionIndex) &&
                          (dm.DeletedVersionIndex > versionIndex || dm.DeletedVersionIndex < 0)
                    select dm.Name;

                var versionMappingsBuilder = new StringBuilder();
                var inverseVersionMappingsBuilder = new StringBuilder();
                foreach (string dataModelName in dataModelNames)
                {
                    versionMappingsBuilder.Append(string.Format(
                        Resources.EditorDataModelTemplate_VersionMappingItem,
                        dataModelName));

                    inverseVersionMappingsBuilder.Append(string.Format(
                        Resources.EditorDataModelTemplate_InverseVersionMappingItem,
                        dataModelName));
                }

                if (currentVersion == this.root.LatestVersion.Version)
                {
                    helper.ParseSimpleTemplate(Resources.DataModelSerializer_LatestVersion);
                }
                else
                {
                    helper.ParseSimpleTemplate(Resources.DataModelSerializer);
                }

                helper.SetTemplateTagValue("PrevVersionNamespace", prevVersionNamespace);
                helper.SetTemplateTagValue("CurrVersionNamespace", currVersionNamespace);
                helper.SetTemplateTagValue("CurrVersion", currentVersion.ToString());
                helper.SetTemplateTagValue("VersionMappings", versionMappingsBuilder.ToString());
                helper.SetTemplateTagValue("InverseVersionMappings", inverseVersionMappingsBuilder.ToString());
            }
            else
            {
                string currVersionNamespaceString = currVersionNamespace;
                if (currentVersion == this.root.LatestVersion.Version)
                {
                    currVersionNamespaceString = string.Empty;
                    helper.ParseSimpleTemplate(Resources.DataModelSerializer_FirstVersionOnly);
                }
                else
                {
                    helper.ParseSimpleTemplate(Resources.DataModelSerializer_FirstVersion);
                }

                helper.SetTemplateTagValue("CurrVersionNamespace", currVersionNamespaceString);
                helper.SetTemplateTagValue("CurrVersion", currentVersion.ToString());
            }

            // Generate the source code.
            var code = helper.Compose();

            // Compose the output file path.
            var path = Path.Combine(baseOutputPath, currVersionNamespace);
            path = Path.Combine(path, "DataModelSerializer.cs");

            // Write generated source code to file.
            File.WriteAllText(path, code.Replace(" " + Environment.NewLine, Environment.NewLine), Encoding.UTF8);

            // Advance progress.
            if (this.ProgressAdvanced != null)
            {
                this.ProgressAdvanced(this, EventArgs.Empty);
            }

            return true;
        }
    }
}
