﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using System;
using System.Collections.Generic;
using System.Threading;
using EffectMaker.Foundation.Log;
using System.Diagnostics;

namespace EffectMaker.DataModelMaker.Application
{
    /// <summary>
    /// Logger class to output message to console.
    /// </summary>
    public class ConsoleLogger : ILogHandler
    {
        /// <summary>The default background color for messages.</summary>
        private static ConsoleColor defaultBackColor = ConsoleColor.Black;

        /// <summary>The text color for debug messages.</summary>
        private static ConsoleColor debugColor = ConsoleColor.Gray;

        /// <summary>The text color for information messages.</summary>
        private static ConsoleColor informationColor = ConsoleColor.White;

        /// <summary>The text color for warning messages.</summary>
        private static ConsoleColor warningColor = ConsoleColor.Yellow;

        /// <summary>The text color for error messages.</summary>
        private static ConsoleColor errorColor = ConsoleColor.Red;

        /// <summary>The text color for fatal messages.</summary>
        private static ConsoleColor fatalForeColor = ConsoleColor.Black;

        /// <summary>The background color for fatal messages.</summary>
        private static ConsoleColor fatalBackColor = ConsoleColor.Red;

        /// <summary>
        /// Constructor.
        /// </summary>
        public ConsoleLogger()
        {
        }

        /// <summary>
        /// ログハンドラの名前を取得します。
        /// </summary>
        public string LogHandlerName
        {
            get { return "ConsoleLogger"; }
        }

        /// <summary>
        /// ログハンドラの同期コンテキストを取得します。
        /// </summary>
        public SynchronizationContext SynchronizationContext
        {
            get { return null; }
        }

        /// <summary>
        /// ロガーからメッセージが送られたときの処理を行います。
        /// </summary>
        /// <param name="destinations">ログ出力先</param>
        /// <param name="level">ログレベル</param>
        /// <param name="message">ログメッセージ</param>
        /// <param name="callStack">コールスタック</param>
        public void Log(IEnumerable<string> destinations, LogLevels level, string message, StackFrame callStack)
        {
            ConsoleColor backColor = defaultBackColor;
            ConsoleColor foreColor;

            switch (level)
            {
                case LogLevels.Debug:
                    foreColor = debugColor;
                    break;

                case LogLevels.Information:
                    foreColor = informationColor;
                    break;

                case LogLevels.Warning:
                    foreColor = warningColor;
                    break;

                case LogLevels.Error:
                    foreColor = errorColor;
                    break;

                case LogLevels.Fatal:
                    foreColor = fatalForeColor;
                    backColor = fatalBackColor;
                    break;

                default:
                    // We don't handle any other types of messages.
                    return;
            }

            ConsoleColor origBackColor = Console.BackgroundColor;
            ConsoleColor origForeColor = Console.ForegroundColor;

            if (backColor != origBackColor)
            {
                Console.BackgroundColor = backColor;
            }

            if (foreColor != origForeColor)
            {
                Console.ForegroundColor = foreColor;
            }

            Console.WriteLine(message);

            if (backColor != origBackColor)
            {
                Console.BackgroundColor = origBackColor;
            }

            if (foreColor != origForeColor)
            {
                Console.ForegroundColor = origForeColor;
            }
        }
    }
}
