﻿// ========================================================================
// <copyright file="DoubleUtility.cs" company="Nintendo">
//      Copyright 2009 Nintendo.  All rights reserved.
// </copyright>
//
// These coded instructions, statements, and computer programs contain
// proprietary information of Nintendo of America Inc. and/or Nintendo
// Company Ltd., and are protected by Federal copyright law.  They may
// not be disclosed to third parties or copied or duplicated in any form,
// in whole or in part, without the prior written consent of Nintendo.
// ========================================================================

namespace NintendoWare.ToolDevelopmentKit
{
    using System;

    /// <summary>
    /// double 関連のユーティリティです。
    /// </summary>
    public static class DoubleUtility
    {
        /// <summary>
        /// 許容誤差です。
        /// <remarks>
        /// 機械イプシロンの値です。
        /// </remarks>
        /// </summary>
        public const double Epsilon = 2.2204460492503131E-16;

        /// <summary>
        /// 2 つの double の値がほぼ等しいかどうかを示します。
        /// </summary>
        /// <remarks>
        /// Epsilon を差の許容値として利用します。
        /// </remarks>
        /// <param name="lhs">比較する左辺値です。</param>
        /// <param name="rhs">比較する右辺値です。</param>
        /// <returns>2 つの double の差が Epsilon 以下の場合は true を返します。
        /// それ以外の場合は false を返します。</returns>
        public static bool NearlyEqual(double lhs, double rhs)
        {
            return System.Math.Abs(lhs - rhs) <= Epsilon;
        }

        /// <summary>
        /// 2 つの double の値がほぼ等しいかどうかを相対許誤差で判定します。
        /// </summary>
        /// <remarks>
        /// 書籍：ゲームプログラミングのためのリアルタイム衝突判定(ISBN4-939007-91-X)
        /// を参考にしました。
        /// </remarks>
        /// <param name="lhs">比較する左辺値です。</param>
        /// <param name="rhs">比較する右辺値です。</param>
        /// <returns>2 つの double の比差が Epsilon 以下の場合は true を返します。
        /// それ以外の場合は false を返します。</returns>
        public static bool NearlyEqualInRelativeManner(double lhs, double rhs)
        {
            // Abs(x) < Abs(y) と仮定して、2値の比がどれだけ１に近いかという考え方から
            //  Math.Abs(x/y) - 1.0f) <= Epsilon
            // v
            //  Math.Abs(x/y) / y  <= Epsilon // 変形
            // v
            //  Math.Abs(x/y) / y  <= Epsilon // 分母にAbs(y)を乗算
            // v
            //  Math.Abs(x/y) <= Epsilon * Math.Abs(y) // 分母にAbs(y)を乗算
            // v
            //  Abs(x) < Abs(y) の仮定を外し、xやyが1.0より小さい場合も考慮すると下式
            // v
            return System.Math.Abs(lhs - rhs) <= Epsilon * Math.Max(Math.Max(lhs, rhs), 1.0f);
        }

        /// <summary>
        /// 2 つの double の値がほぼ等しいかどうかを示します。
        /// </summary>
        /// <param name="lhs">比較する左辺値です。</param>
        /// <param name="rhs">比較する右辺値です。</param>
        /// <param name="tolerance">差の許容値です。</param>
        /// <returns>2 つの double の差が tolerance 以下の場合は true を返します。
        /// それ以外の場合は false を返します。</returns>
        public static bool NearlyEqual(double lhs, double rhs, double tolerance)
        {
            return System.Math.Abs(lhs - rhs) <= tolerance;
        }

        /// <summary>
        /// 2 つの double の値が小数点以下の指定の桁数（切捨て）内で等しいかどうかを示します。
        /// </summary>
        /// <param name="lhs">比較する左辺値です。</param>
        /// <param name="rhs">比較する右辺値です。</param>
        /// <param name="precision">小数点以下の桁数です。</param>
        /// <returns>2 つの double の値が小数点以下の指定の桁数（切捨て）内で
        /// 等しい場合は true を返します。それ以外の場合は false を返します。</returns>
        public static bool NearlyEqualNumberOfSignificantFigures(
            double lhs, double rhs, int precision)
        {
            // TODO: 実装
            throw new NotImplementedException();
        }

        /// <summary>
        /// double 型の値１が値２より大きいかどうかを調べます。
        /// </summary>
        /// <param name="value1">１つ目の値を指定します。</param>
        /// <param name="value2">２つ目の値を指定します。</param>
        /// <returns>値１が値２より大きい場合は true、そうでない場合は false を返します。</returns>
        public static bool GreaterThan(double value1, double value2)
        {
            return value1 > value2 && !NearlyEqual(value1, value2);
        }

        /// <summary>
        /// double 型の値１が値２より小さいかどうかを調べます。
        /// </summary>
        /// <param name="value1">１つ目の値を指定します。</param>
        /// <param name="value2">２つ目の値を指定します。</param>
        /// <returns>値１が値２より小さい場合は true、そうでない場合は false を返します。</returns>
        public static bool LessThan(double value1, double value2)
        {
            return value1 < value2 && !NearlyEqual(value1, value2);
        }

        /// <summary>
        /// double の値を指定の小数点以下の桁数で切り上げを行います。
        /// </summary>
        /// <param name="v">double の値です。</param>
        /// <param name="presicion">小数点以下の桁数です。</param>
        /// <returns>切り上げの結果です。</returns>
        public static double RoundUp(double v, int presicion)
        {
            // TODO: 実装
            throw new NotImplementedException();
        }

        /// <summary>
        /// double の値を指定の小数点以下の桁数で切り捨てを行います。
        /// </summary>
        /// <param name="v">double の値です。</param>
        /// <param name="precision">小数点以下の桁数です。</param>
        /// <returns>切り上げの結果です。</returns>
        public static double RoundDown(double v, int precision)
        {
            // TODO: 実装
            throw new NotImplementedException();
        }

        /// <summary>
        /// 2 つの double の値を交換します。
        /// </summary>
        /// <param name="rhs">交換する左辺値です。</param>
        /// <param name="lhs">交換する右辺値です。</param>
        public static void Swap(ref double rhs, ref double lhs)
        {
            double temp = rhs;
            rhs = lhs;
            lhs = temp;
        }

        /// <summary>
        /// 指定した double の値以下の最大の整数を取得します。
        /// </summary>
        /// <param name="v">double の値です。</param>
        /// <returns>整数値です。</returns>
        public static int Floor(double v)
        {
            return (int)Math.Floor(v);
        }

        /// <summary>
        /// double 値を最大値と最小値の間にクランプします。
        /// </summary>
        /// <param name="value">クランプする値です。</param>
        /// <param name="minValue">最小値です。</param>
        /// <param name="maxValue">最大値です。</param>
        /// <returns>クランプした値です。</returns>
        public static double Clamp(double value, double minValue, double maxValue)
        {
            if (value < minValue)
            {
                return minValue;
            }

            if (value > maxValue)
            {
                return maxValue;
            }

            return value;
        }

        /// <summary>
        /// 値が指定された範囲内にあるか調べます。
        /// </summary>
        /// <param name="value">範囲チェックする値です。</param>
        /// <param name="min">最小値です。</param>
        /// <param name="max">最大値です。</param>
        /// <returns>範囲内であれば true を返します。</returns>
        public static bool Contains(double value, double min, double max)
        {
            Ensure.Operation.True(min <= max);

            if (value < min)
            {
                return false;
            }

            if (max < value)
            {
                return false;
            }

            return true;
        }

        /// <summary>
        /// 値を元の範囲から目的の範囲に範囲における位置の比率を保ちつつコンバートします。
        /// </summary>
        /// <param name="value">コンバート対象です。</param>
        /// <param name="srcMin">元の範囲の最小値です。</param>
        /// <param name="srcMmx">元の範囲の最大値です。</param>
        /// <param name="dstMin">目的の範囲の最小値です。</param>
        /// <param name="dstMax">目的の範囲の最大値です。</param>
        /// <returns>コンバートした結果です。</returns>
        public static double Convert(
            double value, double srcMin, double srcMmx, double dstMin, double dstMax)
        {
            return ((value - srcMin) * (dstMax - dstMin) / (srcMmx - srcMin)) + dstMin;
        }

        /// <summary>
        /// 値が0に近い場合0に丸めます。
        /// </summary>
        /// <param name="value">元の値です。</param>
        /// <returns>結果です。</returns>
        public static double RoundToZero(double value)
        {
            if (Contains(value, -Epsilon, Epsilon))
            {
                return 0.0F;
            }

            return value;
        }
    }
}
