﻿// ========================================================================
// <copyright file="Matrix22Function.cs" company="Nintendo">
//      Copyright 2009 Nintendo.  All rights reserved.
// </copyright>
//
// These coded instructions, statements, and computer programs contain
// proprietary information of Nintendo of America Inc. and/or Nintendo
// Company Ltd., and are protected by Federal copyright law.  They may
// not be disclosed to third parties or copied or duplicated in any form,
// in whole or in part, without the prior written consent of Nintendo.
// ========================================================================

namespace NintendoWare.ToolDevelopmentKit
{
    using System.Diagnostics;

    /// <summary>
    /// 2 × 2 行列の演算関数です。
    /// </summary>
    /// <typeparam name="TMatrix">テンプレート行列型です。</typeparam>
    internal static class Matrix22Function<TMatrix> where TMatrix : IMatrix, new()
    {
        /// <summary>
        /// 2 × 2 の部分行列の逆行列を取得します。
        /// </summary>
        /// <param name="mtx">対象の行列です。</param>
        /// <returns>逆行列です。</returns>
        public static TMatrix Invert(TMatrix mtx)
        {
            DebugMatrixRowColumnCount(mtx);

            float det = GetDeterminant(mtx);
            if (det == 0f)
            {
                return default(TMatrix);
            } // null を返す

            det = 1.0f / det;

            TMatrix inv = new TMatrix();

            inv[0, 0] = mtx[1, 1] * det;
            inv[0, 1] = -mtx[0, 1] * det;

            inv[1, 0] = -mtx[1, 0] * det;
            inv[1, 1] = mtx[0, 0] * det;

            return inv;
        }

        /// <summary>
        /// 3 × 3 の部分行列の行列式を取得します。
        /// </summary>
        /// <param name="mtx">対象の行列です。</param>
        /// <returns>行列式です。</returns>
        public static float GetDeterminant(TMatrix mtx)
        {
            DebugMatrixRowColumnCount(mtx);

            return (mtx[0, 0] * mtx[1, 1]) - (mtx[0, 1] * mtx[1, 0]);
        }

        /// <summary>
        /// 行列の行数と列数が演算に対して適切であることを確認します。
        /// </summary>
        /// <param name="mtx">対象の行列です。</param>
        [Conditional("DEBUG")]
        private static void DebugMatrixRowColumnCount(TMatrix mtx)
        {
            Debug.Assert(mtx.RowCount >= 2, "Unexpected case!");
            Debug.Assert(mtx.ColumnCount >= 2, "Unexpected case!");
        }
    }
}
