﻿// ========================================================================
// <copyright file="ColorUtilityRgba5551.cs" company="Nintendo">
//      Copyright 2009 Nintendo.  All rights reserved.
// </copyright>
//
// These coded instructions, statements, and computer programs contain
// proprietary information of Nintendo of America Inc. and/or Nintendo
// Company Ltd., and are protected by Federal copyright law.  They may
// not be disclosed to third parties or copied or duplicated in any form,
// in whole or in part, without the prior written consent of Nintendo.
// ========================================================================

namespace NintendoWare.ToolDevelopmentKit
{
    /// <summary>
    /// Rgba5551 フォーマットのユーティリティです。
    /// </summary>
    public static class ColorUtilityRgba5551
    {
        /// <summary>
        /// R 成分の最大値です。
        /// </summary>
        public const byte MaxR = 31;

        /// <summary>
        /// G 成分の最大値です。
        /// </summary>
        public const byte MaxG = 31;

        /// <summary>
        /// B 成分の最大値です。
        /// </summary>
        public const byte MaxB = 31;

        /// <summary>
        /// A 成分の最大値です。
        /// </summary>
        public const byte MaxA = 1;

        /// <summary>
        /// ビットフィールド内の R 成分の位置です。
        /// </summary>
        public const int ShiftR = 0;

        /// <summary>
        /// ビットフィールド内の G 成分の位置です。
        /// </summary>
        public const int ShiftG = 5;

        /// <summary>
        /// ビットフィールド内の B 成分の位置です。
        /// </summary>
        public const int ShiftB = 10;

        /// <summary>
        /// ビットフィールド内の A 成分の位置です。
        /// </summary>
        public const int ShiftA = 15;

        /// <summary>
        /// 指定した IColor の R 成分を byte で取得します。
        /// </summary>
        /// <param name="color">IColor です。</param>
        /// <returns>byte です。</returns>
        public static byte GetR(IColor color)
        {
            Ensure.Argument.NotNull(color);

            return (byte)FloatUtility.Clamp((color.R * MaxR), 0f, MaxR);
        }

        /// <summary>
        /// 指定した IColor の G 成分を byte で取得します。
        /// </summary>
        /// <param name="color">IColor です。</param>
        /// <returns>byte です。</returns>
        public static byte GetG(IColor color)
        {
            Ensure.Argument.NotNull(color);

            return (byte)FloatUtility.Clamp((color.G * MaxG), 0f, MaxG);
        }

        /// <summary>
        /// 指定した IColor の B 成分を byte で取得します。
        /// </summary>
        /// <param name="color">IColor です。</param>
        /// <returns>byte です。</returns>
        public static byte GetB(IColor color)
        {
            Ensure.Argument.NotNull(color);

            return (byte)FloatUtility.Clamp((color.B * MaxB), 0f, MaxB);
        }

        /// <summary>
        /// 指定した IColor の A 成分を byte で取得します。
        /// </summary>
        /// <param name="color">IColor です。</param>
        /// <returns>byte です。</returns>
        public static byte GetA(IColor color)
        {
            Ensure.Argument.NotNull(color);

            return (byte)FloatUtility.Clamp((color.A * MaxA), 0f, MaxA);
        }

        /// <summary>
        /// 指定した RgbaColor の R 成分を byte で設定します。
        /// </summary>
        /// <param name="color">RgbaColor です。</param>
        /// <param name="value">Rgba5551 です。</param>
        public static void SetR(RgbaColor color, byte value)
        {
            Ensure.Argument.NotNull(color);
            color.R = value / (float)MaxR;
        }

        /// <summary>
        /// 指定した RgbaColor の G 成分を byte で設定します。
        /// </summary>
        /// <param name="color">RgbaColor です。</param>
        /// <param name="value">Rgba5551 です。</param>
        public static void SetG(RgbaColor color, byte value)
        {
            Ensure.Argument.NotNull(color);
            color.G = value / (float)MaxG;
        }

        /// <summary>
        /// 指定した RgbaColor の B 成分を byte で設定します。
        /// </summary>
        /// <param name="color">RgbaColor です。</param>
        /// <param name="value">Rgba5551 です。</param>
        public static void SetB(RgbaColor color, byte value)
        {
            Ensure.Argument.NotNull(color);
            color.B = value / (float)MaxB;
        }

        /// <summary>
        /// 指定した RgbaColor の A 成分を byte で設定します。
        /// </summary>
        /// <param name="color">RgbaColor です。</param>
        /// <param name="value">Rgba5551 です。</param>
        public static void SetA(RgbaColor color, byte value)
        {
            Ensure.Argument.NotNull(color);
            color.A = value / (float)MaxA;
        }

        /// <summary>
        /// 指定した RgbaColor の R 成分を 指定した Rgba5551 の R 成分で設定します。
        /// </summary>
        /// <param name="color">RgbaColor です。</param>
        /// <param name="value">Rgba5551 です。</param>
        public static void SetMaskR(RgbaColor color, int value)
        {
            SetR(color, (byte)((value >> ShiftR) & MaxR));
        }

        /// <summary>
        /// 指定した RgbaColor の G 成分を 指定した Rgba5551 の G 成分で設定します。
        /// </summary>
        /// <param name="color">RgbaColor です。</param>
        /// <param name="value">Rgba5551 です。</param>
        public static void SetMaskG(RgbaColor color, int value)
        {
            SetG(color, (byte)((value >> ShiftG) & MaxG));
        }

        /// <summary>
        /// 指定した RgbaColor の B 成分を 指定した Rgba5551 の B 成分で設定します。
        /// </summary>
        /// <param name="color">RgbaColor です。</param>
        /// <param name="value">Rgba5551 です。</param>
        public static void SetMaskB(RgbaColor color, int value)
        {
            SetB(color, (byte)((value >> ShiftB) & MaxB));
        }

        /// <summary>
        /// 指定した RgbaColor の A 成分を 指定した Rgba5551 の A 成分で設定します。
        /// </summary>
        /// <param name="color">RgbaColor です。</param>
        /// <param name="value">Rgba5551 です。</param>
        public static void SetMaskA(RgbaColor color, int value)
        {
            SetA(color, (byte)((value >> ShiftA) & MaxA));
        }

        /// <summary>
        /// 指定した IColor を Rgba5551 に変換します。
        /// </summary>
        /// <param name="color">IColor です。</param>
        /// <returns>Rgba5551 です。</returns>
        public static int ToRgba5551(IColor color)
        {
            Ensure.Argument.NotNull(color);
            return
                (GetR(color) << ShiftR) +
                (GetG(color) << ShiftG) +
                (GetB(color) << ShiftB) +
                (GetA(color) << ShiftA);
        }

        /// <summary>
        /// 指定した Rgba5551 を RgbaColor に変換します。
        /// </summary>
        /// <param name="value">Rgba5551 です。</param>
        /// <returns>RgbaColor です。</returns>
        public static RgbaColor ToRgbaColor(int value)
        {
            RgbaColor rgbaColor = new RgbaColor();
            SetMaskR(rgbaColor, value);
            SetMaskG(rgbaColor, value);
            SetMaskB(rgbaColor, value);
            SetMaskA(rgbaColor, value);
            return rgbaColor;
        }
    }
}
