﻿// ========================================================================
// <copyright file="AxisAlignedBoundingBox.cs" company="Nintendo">
//      Copyright 2009 Nintendo.  All rights reserved.
// </copyright>
//
// These coded instructions, statements, and computer programs contain
// proprietary information of Nintendo of America Inc. and/or Nintendo
// Company Ltd., and are protected by Federal copyright law.  They may
// not be disclosed to third parties or copied or duplicated in any form,
// in whole or in part, without the prior written consent of Nintendo.
// ========================================================================

namespace NintendoWare.ToolDevelopmentKit
{
    using System;
    using System.Collections.Generic;
    using System.Linq;
    using System.Text;

    /// <summary>
    /// AABB(境界矩形)クラスです。
    /// </summary>
    public class AxisAlignedBoundingBox : BoundingVolume
    {
        /// <summary>
        /// キューブのデフォルトサイズです。
        /// </summary>
        public const float DefaultSize = 1.0f;

        /// <summary>
        /// 頂点の数です。
        /// </summary>
        public static readonly int VertexCount = 8;

        private Vector3 centerPosition = new Vector3();
        private Vector3 size = new Vector3(DefaultSize, DefaultSize, DefaultSize);

        /// <summary>
        /// コンストラクタです。
        /// </summary>
        public AxisAlignedBoundingBox()
        {
        }

        /// <summary>
        /// コピーコンストラクタです。
        /// </summary>
        /// <param name="source">コピー元のデータです。</param>
        public AxisAlignedBoundingBox(AxisAlignedBoundingBox source)
            : this()
        {
            this.Set(source);
        }

        /// <summary>
        /// 中心位置を取得します。
        /// </summary>
        public Vector3 CenterPosition
        {
            get
            {
                return this.centerPosition;
            }

            set
            {
                this.centerPosition = value;
            }
        }

        /// <summary>
        /// ローカル座標空間におけるサイズを取得します。
        /// </summary>
        public Vector3 Size
        {
            get { return this.size; }
            set { this.size = value; }
        }

        /// <summary>
        /// 各頂点位置を取得します。
        /// </summary>
        public Vector3[] VertexPositions
        {
            get
            {
                float half_x = this.Size.X * 0.5f;
                float half_y = this.Size.Y * 0.5f;
                float half_z = this.Size.Z * 0.5f;

                Vector3[] posArray = new[]
                {
                    new Vector3(-half_x, -half_y, -half_z),
                    new Vector3(-half_x, -half_y, +half_z),
                    new Vector3(-half_x, +half_y, -half_z),
                    new Vector3(-half_x, +half_y, +half_z),
                    new Vector3(+half_x, -half_y, -half_z),
                    new Vector3(+half_x, -half_y, +half_z),
                    new Vector3(+half_x, +half_y, -half_z),
                    new Vector3(+half_x, +half_y, +half_z),
                };

                for (int i = 0; i != VertexCount; ++i)
                {
                    posArray[i] = posArray[i] + this.CenterPosition;
                }

                return posArray;
            }
        }

        /// <summary>
        /// 現在のインスタンスのコピーである新しいオブジェクトを作成します。
        /// </summary>
        /// <returns>このインスタンスのコピーである新しいオブジェクトです。</returns>
        public override object Clone()
        {
            return new AxisAlignedBoundingBox(this);
        }

        /// <summary>
        /// オブジェクトを設定します。
        /// </summary>
        /// <param name="source">設定するオブジェクトです。</param>
        public override void Set(object source)
        {
            this.Set(source as AxisAlignedBoundingBox);
        }

        /// <summary>
        /// 点の集合から AABB を設定します。
        /// </summary>
        /// <param name="positions">内包させたい点座標です。</param>
        public void Set(IList<Vector3> positions)
        {
            if (positions.Count > 0)
            {
                Vector3 min_pos = new Vector3(float.MaxValue, float.MaxValue, float.MaxValue);
                Vector3 max_pos = new Vector3(float.MinValue, float.MinValue, float.MinValue);

                foreach (Vector3 pos in positions)
                {
                    min_pos.X = Math.Min(min_pos.X, pos.X);
                    min_pos.Y = Math.Min(min_pos.Y, pos.Y);
                    min_pos.Z = Math.Min(min_pos.Z, pos.Z);

                    max_pos.X = Math.Max(max_pos.X, pos.X);
                    max_pos.Y = Math.Max(max_pos.Y, pos.Y);
                    max_pos.Z = Math.Max(max_pos.Z, pos.Z);
                }

                Assertion.Operation.True(min_pos.X <= max_pos.X);
                Assertion.Operation.True(min_pos.Y <= max_pos.Y);
                Assertion.Operation.True(min_pos.Z <= max_pos.Z);

                this.CenterPosition.Set((max_pos + min_pos) * 0.5f);
                this.Size.Set(max_pos - min_pos);
            }
        }

        /// <summary>
        /// オブジェクトを設定します。
        /// </summary>
        /// <param name="source">設定するオブジェクトです。</param>
        protected void Set(AxisAlignedBoundingBox source)
        {
            Ensure.Argument.NotNull(source);

            this.CenterPosition.Set(source.CenterPosition);
            this.Size.Set(source.Size);
        }
    }
}
