﻿// ========================================================================
// <copyright file="XmlAttributeOverridesGenerator.cs" company="Nintendo">
//      Copyright 2009 Nintendo.  All rights reserved.
// </copyright>
//
// These coded instructions, statements, and computer programs contain
// proprietary information of Nintendo of America Inc. and/or Nintendo
// Company Ltd., and are protected by Federal copyright law.  They may
// not be disclosed to third parties or copied or duplicated in any form,
// in whole or in part, without the prior written consent of Nintendo.
// ========================================================================

namespace NintendoWare.ToolDevelopmentKit.Xml
{
    using System;
    using System.Collections.Generic;
    using System.Reflection;
    using System.Xml.Serialization;
    using NintendoWare.ToolDevelopmentKit.Conversion;

    /// <summary>
    /// XmlAttributeOverrideを作成するユーティリティクラスです。
    /// </summary>
    public static class XmlAttributeOverridesGenerator
    {
        /// <summary>
        /// 指定した型のリストに対して、XmlAttributeOverridesを作成します。
        /// <remarks>
        /// targetTypesにobject型が含まれていないことと、
        /// 検索対象が型名の末尾が"Xml"で終わっているという前提で動作します。
        /// </remarks>
        /// </summary>
        /// <param name="targetTypes">調査対象の型のリストです。</param>
        /// <returns>調査により作成されたXmlAttributeOverridesです。</returns>
        public static XmlAttributeOverrides GenerateXmlAttributeOverrides(IEnumerable<Type> targetTypes)
        {
            XmlAttributeOverrides attributeOverrides = new XmlAttributeOverrides();

            foreach (Type targetType in targetTypes)
            {
                Ensure.Operation.True(targetType != typeof(object));

                RegisterXmlAttributes(
                    attributeOverrides,
                    targetType,
                    XmlUtility.EnumerateXmlTypes());
            }

            return attributeOverrides;
        }

        /// <summary>
        /// 指定した型で、マークされたプロパティのXmlAttributesを作成し、
        /// XmlAttributeOverridesに登録します。
        /// <remarks>
        /// GenerateXmlAttributeOverridesAttributeが
        /// ついていることが対象のプロパティとなる条件です。
        /// </remarks>
        /// </summary>
        /// <param name="attributeOverrides">
        /// 登録対象のXmlAttributeOverridesです。
        /// </param>
        /// <param name="type">対象の型です。</param>
        /// <param name="searchTargetTypes">
        /// アプリケーションドメインに存在するすべての検索対象の型のリストです。
        /// </param>
        public static void RegisterXmlAttributes(
            XmlAttributeOverrides attributeOverrides,
            Type type,
            IEnumerable<Type> searchTargetTypes)
        {
            PropertyInfo[] propertyInfos = type.GetProperties();

            foreach (PropertyInfo propertyInfo in propertyInfos)
            {
                GenerateXmlAttributeOverridesAttribute targetAttribute =
                    ConverterUtil.FindPropertyAttribute<GenerateXmlAttributeOverridesAttribute>(
                        propertyInfo);

                if (targetAttribute == null)
                {
                    continue;
                }

                Type targetType = propertyInfo.PropertyType;
                XmlAttributes xmlAttributes = null;
                if (ConverterUtil.CheckTypeImplementsList(targetType))
                {
                    Type listElementType = ConverterUtil.GetIListElementType(targetType);
                    if (listElementType != null)
                    {
                        xmlAttributes =
                            GenerateXmlArrayItemAttributes(listElementType, searchTargetTypes);
                    }
                }
                else
                {
                    xmlAttributes =
                        GenerateXmlElementAttributes(targetType, searchTargetTypes);
                }

                if (xmlAttributes != null)
                {
                    attributeOverrides.Add(type, propertyInfo.Name, xmlAttributes);
                }
            }
        }

        /// <summary>
        /// 指定した型を派生している型を元に、XmlArrayItemアトリビュート
        /// を要素とするXmlAttributesを作成します。
        /// </summary>
        /// <param name="type">対象の型です。</param>
        /// <param name="searchTargetTypes">
        /// アプリケーションドメインに存在するすべての検索対象の型のリストです。
        /// </param>
        /// <returns>
        /// 指定した型を派生している型で作られる
        /// XmlArrayItemアトリビュートを要素とするXmlAttributesです。
        /// </returns>
        private static XmlAttributes GenerateXmlArrayItemAttributes(
            Type type,
            IEnumerable<Type> searchTargetTypes)
        {
            XmlAttributes attributes = new XmlAttributes();

            foreach (Type searchTargetType in searchTargetTypes)
            {
                if (IsImplement(searchTargetType, type))
                {
                    XmlArrayItemAttribute arrayItemAttribute =
                        new XmlArrayItemAttribute(searchTargetType);
                    attributes.XmlArrayItems.Add(arrayItemAttribute);
                }
            }

            return attributes;
        }

        /// <summary>
        /// 指定した型を派生している型を元に、XmlElementアトリビュート
        /// を要素とするXmlAttributesを作成します。
        /// </summary>
        /// <param name="type">対象の型です。</param>
        /// <param name="searchTargetTypes">
        /// アプリケーションドメインに存在する
        /// すべての検索対象の型のリストです。
        /// </param>
        /// <returns>
        /// 指定した型を派生している型で作られるXmlElement
        /// アトリビュートを要素とするXmlAttributesです。
        /// </returns>
        private static XmlAttributes GenerateXmlElementAttributes(
            Type type,
            IEnumerable<Type> searchTargetTypes)
        {
            XmlAttributes attributes = new XmlAttributes();

            foreach (Type searchTargetType in searchTargetTypes)
            {
                if (IsImplement(searchTargetType, type))
                {
                    XmlElementAttribute elementAttribute =
                        new XmlElementAttribute(searchTargetType);
                    attributes.XmlElements.Add(elementAttribute);
                }
            }

            return attributes;
        }

        /// <summary>
        /// 指定した型がcheckTypeの派生型であるかを調べます。
        /// </summary>
        /// <param name="type">調査対象の型です。</param>
        /// <param name="checkType">派生型です。</param>
        /// <returns>checkTypeを派生していればtrueを返します。</returns>
        private static bool IsImplement(Type type, Type checkType)
        {
            if (type == null)
            {
                return false;
            }

            if (type == checkType)
            {
                return true;
            }
            else
            {
                return IsImplement(type.BaseType, checkType);
            }
        }
    }
}
