﻿// ========================================================================
// <copyright file="XmlContent.cs" company="Nintendo">
//      Copyright 2009 Nintendo.  All rights reserved.
// </copyright>
//
// These coded instructions, statements, and computer programs contain
// proprietary information of Nintendo of America Inc. and/or Nintendo
// Company Ltd., and are protected by Federal copyright law.  They may
// not be disclosed to third parties or copied or duplicated in any form,
// in whole or in part, without the prior written consent of Nintendo.
// ========================================================================
namespace NintendoWare.ToolDevelopmentKit.Xml.Complex
{
    using System.Xml;
    using NintendoWare.ToolDevelopmentKit.ComponentModel;

    /// <summary>
    /// XML コンテンツです。
    /// </summary>
    public class XmlContent : ObservableObject, IXmlContent
    {
        private string xml = string.Empty;

        /// <summary>
        /// デフォルトコンストラクタです。
        /// </summary>
        public XmlContent()
        {
        }

        /// <summary>
        /// コピーコンストラクタです。
        /// </summary>
        /// <param name="source">コピー元の XML コンテンツです。</param>
        public XmlContent(IXmlContent source)
        {
            this.Set(source);
        }

        /// <summary>
        /// XML文章 を設定または取得します。
        /// </summary>
        public virtual string Xml
        {
            get
            {
                return this.xml;
            }

            set
            {
                Ensure.Argument.StringNotEmpty(value);
                this.xml = value;
                OnPropertyChanged("Xml");
            }
        }

        /// <summary>
        /// オブジェクトを複製します。
        /// </summary>
        /// <returns>複製したオブジェクトです。</returns>
        public virtual object Clone()
        {
            return new XmlContent(this);
        }

        /// <summary>
        /// オブジェクトを設定します。
        /// </summary>
        /// <param name="source">設定するオブジェクトです。</param>
        public virtual void Set(object source)
        {
            this.Set(source as IXmlContent);
        }

        /// <summary>
        /// XmlWriterに内容を書き出します。
        /// </summary>
        /// <param name="writer">XmlWriterです。</param>
        public virtual void Export(XmlWriter writer)
        {
            Ensure.Argument.NotNull(writer);

            // 文字列として書き出します。
            writer.WriteRaw(this.Xml);
        }

        /// <summary>
        /// 現在のオブジェクトを表す文字列を返します。
        /// </summary>
        /// <returns>現在の object を表す stringです。</returns>
        public override string ToString()
        {
            return this.Xml;
        }

        /// <summary>
        /// 等値であるかどうか比較します。
        /// </summary>
        /// <param name="other">比較対象です。</param>
        /// <returns>等値であれば true を返します。</returns>
        public override bool Equals(object other)
        {
            return CheckClassEqualityUtil<IXmlContent>.Equals(this, other, this.Equals);
        }

        /// <summary>
        /// ハッシュ値を取得します。
        /// </summary>
        /// <returns>ハッシュ値です。</returns>
        public override int GetHashCode()
        {
            return this.Xml.GetHashCode();
        }

        /// <summary>
        /// XML コンテンツを設定します。
        /// </summary>
        /// <param name="source">設定する XML コンテンツです。</param>
        protected void Set(IXmlContent source)
        {
            Ensure.Argument.NotNull(source);

            this.xml = source.Xml;
        }

        /// <summary>
        /// 等値であるかどうか比較します。
        /// <remarks>
        /// Xmlフィールドについては、文字列が完全に一致しなければ同一とみなしません。
        /// XML処理モジュールで、整形された場合には等価な内容のXML文章が不一致と判定されます。
        /// </remarks>
        /// </summary>
        /// <param name="other">比較対象です。</param>
        /// <returns>等値であれば true を返します。</returns>
        protected bool Equals(IXmlContent other)
        {
            Ensure.Argument.NotNull(other);

            return other.Xml == this.Xml;
        }
    }
}
