﻿// ========================================================================
// <copyright file="ObservableFixedSizeList.cs" company="Nintendo">
//      Copyright 2009 Nintendo.  All rights reserved.
// </copyright>
//
// These coded instructions, statements, and computer programs contain
// proprietary information of Nintendo of America Inc. and/or Nintendo
// Company Ltd., and are protected by Federal copyright law.  They may
// not be disclosed to third parties or copied or duplicated in any form,
// in whole or in part, without the prior written consent of Nintendo.
// ========================================================================
namespace NintendoWare.ToolDevelopmentKit.Collections
{
    using System.Collections.Generic;
    using System.Collections.Specialized;

    /// <summary>
    /// 通知機能付きの固定サイズリストです。
    /// </summary>
    /// <remarks>
    /// Remove、RemoveAt、Clear は null で置き換えます。
    /// IndexOf 等の同一オブジェクトの評価は、同値ではなく参照で評価します。
    /// リストへの追加や削除を監視する場合、以下のメソッドをオーバーライドします。
    /// Insert、RemoveAt、Clear、インデクサ。
    /// </remarks>
    /// <typeparam name="TItem">リスト内の要素のテンプレート型です。</typeparam>
    public class ObservableFixedSizeList<TItem> : IList<TItem>, INotifyCollectionChanged
        where TItem : class
    {
        //-----------------------------------------------------------------
        private readonly TItem[] array;

        //-----------------------------------------------------------------
        // オブジェクトの生成
        //-----------------------------------------------------------------

        /// <summary>
        /// コンストラクタです。
        /// </summary>
        /// <param name="size">リストのサイズを指定します。</param>
        public ObservableFixedSizeList(int size)
        {
            this.array = new TItem[size];
        }

        /// <summary>
        /// コンストラクタです。
        /// </summary>
        /// <param name="source">コピー元の列挙です。</param>
        public ObservableFixedSizeList(IEnumerable<TItem> source)
        {
            int size = 0;
            foreach (TItem item in source)
            {
                size++;
            }

            this.array = new TItem[size];

            int i = 0;
            foreach (TItem item in source)
            {
                this[i] = item;
                i++;
            }
        }

        //-----------------------------------------------------------------
        // 公開しているイベント
        //-----------------------------------------------------------------

        /// <summary>
        /// コレクション変更イベントです。
        /// </summary>
        public event NotifyCollectionChangedEventHandler CollectionChanged;

        /// <summary>
        /// 格納されている要素の数を取得します。
        /// </summary>
        /// <returns>格納されている要素の数です。</returns>
        public virtual int Count
        {
            get
            {
                return this.array.Length;
            }
        }

        /// <summary>
        /// 読み取り専用かどうかを示す値を取得します。
        /// </summary>
        /// <returns>読み取り専用の場合は trueです。それ以外の場合は false です。</returns>
        public virtual bool IsReadOnly
        {
            get
            {
                return false;
            }
        }

        //-----------------------------------------------------------------
        // 要素の取得または設定
        //-----------------------------------------------------------------

        /// <summary>
        /// 指定したインデックス位置へのアクセスを行うインデクサです。
        /// </summary>
        /// <param name="index">アクセスする項目の 0 から始まるインデックスです。</param>
        /// <returns>インデックス位置の項目です。</returns>
        public virtual TItem this[int index]
        {
            get { return this.array[index]; }
            set { this.Replace(index, value); }
        }

        //-----------------------------------------------------------------
        // 要素の追加または削除
        //-----------------------------------------------------------------

        /// <summary>
        /// 指定したインデックス位置に項目を挿入します。
        /// </summary>
        /// <param name="index">項目を挿入する位置の、0 から始まるインデックス番号です。</param>
        /// <param name="item">挿入する項目です。</param>
        public virtual void Insert(int index, TItem item)
        {
            int emptyIndex = -1;
            int count = this.Count;
            for (int i = index; i < count; i++)
            {
                if (this.array[i] == null)
                {
                    emptyIndex = i;
                    break;
                }
            }

            Ensure.Operation.True(emptyIndex != -1);

            for (int i = emptyIndex; i > index; i--)
            {
                this.Replace(i, this.array[i - 1]);
            }

            this.Replace(index, item);
        }

        /// <summary>
        /// 項目を追加します。
        /// </summary>
        /// <param name="item">追加する項目です。</param>
        public virtual void Add(TItem item)
        {
            int emptyIndex = this.IndexOf(null);
            Ensure.Operation.True(emptyIndex != -1);

            this.Insert(emptyIndex, item);
        }

        /// <summary>
        /// 指定したインデックス位置の項目を削除します。
        /// </summary>
        /// <param name="index">削除する項目の 0 から始まるインデックスです。</param>
        public virtual void RemoveAt(int index)
        {
            this.Replace(index, null);
        }

        /// <summary>
        /// 最初に見つかった特定のオブジェクトを削除します。
        /// </summary>
        /// <param name="item">削除する項目です。</param>
        /// <returns>
        /// 項目が正常に削除された場合は true を返します。それ以外の場合は false を返します。
        /// このメソッドは、項目が見つからない場合にも false を返します。
        /// </returns>
        public virtual bool Remove(TItem item)
        {
            int index = this.IndexOf(item);
            if (index == -1)
            {
                return false;
            }

            this.RemoveAt(index);
            return true;
        }

        /// <summary>
        /// すべての項目を削除します。
        /// </summary>
        public virtual void Clear()
        {
            int count = this.Count;
            for (int i = 0; i < count; i++)
            {
                this.Replace(i, null);
            }
        }

        /// <summary>
        /// 指定した項目のインデックスを調べます。
        /// </summary>
        /// <param name="item">検索する項目です。</param>
        /// <returns>
        /// リストに存在する場合は項目のインデックスです。それ以外の場合は -1 を返します。
        /// </returns>
        public virtual int IndexOf(TItem item)
        {
            // 同値でなく参照によりチェックを行う
            int count = this.Count;
            for (int i = 0; i < count; i++)
            {
                if (object.ReferenceEquals(this[i], item))
                {
                    return i;
                }
            }

            return -1;
        }

        /// <summary>
        /// 特定の値が格納されているかどうかを判断します。
        /// </summary>
        /// <param name="item">検索する項目です。</param>
        /// <returns>項目が存在する場合は true を返します。それ以外の場合は false を返します。</returns>
        public virtual bool Contains(TItem item)
        {
            return this.IndexOf(item) != -1;
        }

        /// <summary>
        /// 要素を配列にコピーします。
        /// </summary>
        /// <param name="array">要素がコピーされる 1 次元の配列です。</param>
        /// <param name="arrayIndex">
        /// コピーの開始位置となる、配列の 0 から始まるインデックス番号です。
        /// </param>
        public virtual void CopyTo(TItem[] array, int arrayIndex)
        {
            this.array.CopyTo(array, arrayIndex);
        }

        //-----------------------------------------------------------------
        // 要素の列挙
        //-----------------------------------------------------------------

        /// <summary>
        /// コレクションを反復処理する列挙子を返します。
        /// </summary>
        /// <returns>コレクションを反復処理するために使用できる列挙子です。</returns>
        public virtual IEnumerator<TItem> GetEnumerator()
        {
            foreach (TItem item in this.array)
            {
                yield return item;
            }
        }

        /// <summary>
        /// コレクションを反復処理する列挙子を返します。
        /// </summary>
        /// <returns>コレクションを反復処理するために使用できる列挙子です。</returns>
        System.Collections.IEnumerator System.Collections.IEnumerable.GetEnumerator()
        {
            return this.GetEnumerator();
        }

        //-----------------------------------------------------------------
        // 同値比較
        //-----------------------------------------------------------------

        /// <summary>
        /// 等値であるかどうか比較します。
        /// </summary>
        /// <param name="other">比較対象です。</param>
        /// <returns>等値であれば true を返します。</returns>
        public override bool Equals(object other)
        {
            if (other == this)
            {
                return true;
            }

            if ((other == null) || (other.GetType() != GetType()))
            {
                return false;
            }

            return this.Equals(other as ObservableFixedSizeList<TItem>);
        }

        /// <summary>
        /// ハッシュ値を取得します。
        /// </summary>
        /// <returns>ハッシュ値です。</returns>
        public override int GetHashCode()
        {
            int hashCode = GetType().GetHashCode();
            foreach (TItem item in this)
            {
                if (item != null)
                {
                    hashCode ^= item.GetHashCode();
                }
            }

            return hashCode;
        }

        //-----------------------------------------------------------------

        /// <summary>
        /// 要素を置き換えます。
        /// </summary>
        /// <param name="index">置き換える要素のインデックスです。</param>
        /// <param name="newItem">置き換える要素です。</param>
        /// <returns>置き換えられた要素です。</returns>
        protected virtual TItem Replace(int index, TItem newItem)
        {
            TItem oldItem = this.array[index];
            this.array[index] = newItem;
            if (this.CollectionChanged != null)
            {
                NotifyCollectionChangedEventArgs args =
                    new NotifyCollectionChangedEventArgs(
                        NotifyCollectionChangedAction.Replace, newItem, oldItem, index);
                this.CollectionChanged(this, args);
            }

            return oldItem;
        }

        /// <summary>
        /// 等値であるかどうか比較します。
        /// </summary>
        /// <param name="other">比較対象です。</param>
        /// <returns>等値であれば true を返します。</returns>
        protected bool Equals(ObservableFixedSizeList<TItem> other)
        {
            Ensure.Argument.NotNull(other);

            int count = this.Count;
            if (count != other.Count)
            {
                return false;
            }

            for (int i = 0; i < count; i++)
            {
                if (!object.Equals(this[i], other[i]))
                {
                    return false;
                }
            }

            return true;
        }
    }
}
