﻿// ========================================================================
// <copyright file="MultiValueDictionary.cs" company="Nintendo">
//      Copyright 2009 Nintendo.  All rights reserved.
// </copyright>
//
// These coded instructions, statements, and computer programs contain
// proprietary information of Nintendo of America Inc. and/or Nintendo
// Company Ltd., and are protected by Federal copyright law.  They may
// not be disclosed to third parties or copied or duplicated in any form,
// in whole or in part, without the prior written consent of Nintendo.
// ========================================================================

namespace NintendoWare.ToolDevelopmentKit.Collections
{
    using System.Collections.Generic;

    /// <summary>
    /// 複数値の辞書クラスです。
    /// </summary>
    /// <typeparam name="TKey">キーの型です。</typeparam>
    /// <typeparam name="TValue">キーに対応する値の型です。</typeparam>
    public class MultiValueDictionary<TKey, TValue> : Dictionary<TKey, HashSet<TValue>>
    {
        /// <summary>
        /// コンストラクタです。
        /// </summary>
        public MultiValueDictionary()
            : base()
        {
        }

        /// <summary>
        /// 追加します。
        /// </summary>
        /// <param name="key">追加するキーです。</param>
        /// <param name="value">キーに対応する値です。</param>
        public void Add(TKey key, TValue value)
        {
            Ensure.Argument.NotNull(key);

            HashSet<TValue> valueSet = null;
            if (!this.TryGetValue(key, out valueSet))
            {
                valueSet = new HashSet<TValue>();
                base.Add(key, valueSet);
            }

            valueSet.Add(value);
        }

        /// <summary>
        /// 値追加を試行します。
        /// </summary>
        /// <param name="key">追加する値のキーです。</param>
        /// <param name="value">追加する値です。</param>
        /// <returns>追加成功ならば、true を返します。</returns>
        public bool TryAddValue(TKey key, TValue value)
        {
            Ensure.Argument.NotNull(key);

            if (this.ContainsValue(key, value))
            {
                return false;
            }

            this.Add(key, value);

            return true;
        }

        /// <summary>
        /// 値が辞書に含まれているか判定します。
        /// </summary>
        /// <param name="key">対象のキーです。</param>
        /// <param name="value">キーに対応する値です。</param>
        /// <returns>含まれていれば、true を返します。</returns>
        public bool ContainsValue(TKey key, TValue value)
        {
            Ensure.Argument.NotNull(key);

            bool result = false;
            HashSet<TValue> values = null;
            if (this.TryGetValue(key, out values))
            {
                result = values.Contains(value);
            }

            return result;
        }

        /// <summary>
        /// 削除します。
        /// 複数値を持つので、キーに他の値が存在する場合は、キーを削除しません。
        /// </summary>
        /// <param name="key">対象キーです。</param>
        /// <param name="value">キーに対応する値です。</param>
        public void Remove(TKey key, TValue value)
        {
            Ensure.Argument.NotNull(key);

            HashSet<TValue> valueSet = null;
            if (this.TryGetValue(key, out valueSet))
            {
                valueSet.Remove(value);
                if (valueSet.Count <= 0)
                {
                    this.Remove(key);
                }
            }
        }

        /// <summary>
        /// マージします。
        /// </summary>
        /// <param name="mergeSource">対象マージ元です。</param>
        public void Merge(MultiValueDictionary<TKey, TValue> mergeSource)
        {
            if (mergeSource == null)
            {
                return;
            }

            foreach (KeyValuePair<TKey, HashSet<TValue>> pair in mergeSource)
            {
                foreach (TValue value in pair.Value)
                {
                    this.Add(pair.Key, value);
                }
            }
        }

        /// <summary>
        /// 複数値を取得します。
        /// </summary>
        /// <param name="key">取得対象のキーです。</param>
        /// <param name="returnEmptySet">値が存在しない場合に空のセットを返すフラグです。</param>
        /// <returns>値のセットを返します。</returns>
        public HashSet<TValue> GetValues(TKey key, bool returnEmptySet)
        {
            HashSet<TValue> result = null;
            if (!this.TryGetValue(key, out result) && returnEmptySet)
            {
                result = new HashSet<TValue>();
            }

            return result;
        }
    }
}
