﻿// ========================================================================
// <copyright file="TcpViewerConnecter.cs" company="Nintendo">
//      Copyright 2009 Nintendo.  All rights reserved.
// </copyright>
//
// These coded instructions, statements, and computer programs contain
// proprietary information of Nintendo of America Inc. and/or Nintendo
// Company Ltd., and are protected by Federal copyright law.  They may
// not be disclosed to third parties or copied or duplicated in any form,
// in whole or in part, without the prior written consent of Nintendo.
// ========================================================================

using System;
using System.Collections;
using System.Collections.Generic;
using System.Runtime.InteropServices;
using System.IO;
using System.Net.Sockets;
using System.Text;

using NWCore.Protocols;

namespace NWCore.Viewer
{
    public class TcpViewerConnecter : IConnecter
    {
        #region Constructor

        /// <summary>
        /// Constructor.
        /// </summary>
        /// <param name="manager">The message manager.</param>
        public TcpViewerConnecter( MessageManager manager )
        {
            m_msgManager = manager;
        }

        #endregion

        #region Initialize

        /// <summary>
        /// Initialize.
        /// </summary>
        /// <returns>true on success.</returns>
        public bool Initialize()
        {
            Close();

            // Create an empty connection context
            m_context = new ConnectionContext();
            if ( m_context==null )
                return false;

            return true;
        }

        #endregion

        #region Dispose

        /// <summary>
        /// Dispose.
        /// </summary>
        public void Dispose()
        {
            Close();
        }

        #endregion

        #region Properties

        /// <summary>
        /// Get the identifier of the connector.
        /// </summary>
        public string Identifier
        {
            get { return "TCP"; }
        }


        /// <summary>
        /// 接続しているかどうか？
        /// </summary>
        public bool IsConnected
        {
            get
            {
                if ( m_context!=null &&
                     m_context.IsReady==true &&
                     m_context.UseWinMessage==true )
                {
                    return CheckTargetExistence( m_context );
                }
                else if ( m_socket!=null && m_socket.Connected==true )
                {
                    return true;
                }

                return false;
            }
        }


        /// <summary>
        /// Get the size of the buffered incoming data.
        /// </summary>
        public bool IsIncomingBufferFull
        {
            get
            {
                if ( IsConnected==false ||
                     m_socket==null )
                {
                    return false;
                }

                return ( m_socket.Available>=SOCKET_BUFFER_SIZE );
            }
        }

        #endregion

        #region Connection

        /// <summary>
        /// Check the existence of the target of the given connection information.
        /// </summary>
        /// <param name="context">The connection context.</param>
        /// <returns>True if the target exists.</returns>
        public bool CheckTargetExistence( ConnectionContext context )
        {
            if ( context.UseWinMessage==true )
                return WinMsgCheckTargetExistence( context );

            bool bExist = false;
            using ( TcpClient client = new TcpClient() )
            {
                try
                {
                    // Which port number to use?
                    int iPort;
                    if ( context.ShouldAcquireChannelInfo==true )
                        iPort = context.ChannelInfoPort;
                    else
                        iPort = context.Port;

                    if ( iPort<=0 )
                        return false;

                    // Try to connect to the target.
                    client.Connect( context.IPAddress, iPort );

                    // Connected?
                    bExist = client.Connected;
                }
                catch
                {
                    // Connection failed, the target is not reachable.
                    bExist = false;
                }

                // Close the connection anyway.
                client.Close();
            }

            return bExist;
        }


        /// <summary>
        /// Connect to the given target.
        /// </summary>
        /// <param name="context">The information about the target to connect to.</param>
        public bool Connect( ConnectionContext context )
        {
            if ( context==null || context.IsReady==false )
                return false;

            // Already connected?
            if ( this.IsConnected==true )
            {
                // If the target of the given connection context is different
                // from the current target, close the current then connect to
                // the new one.
                if ( context.IsSameTarget( m_context )==true )
                    return true;
                else
                    Close();
            }

            if ( context.UseWinMessage==true )
                return WinMsgConnect( context );

            lock ( m_lock )
            {
                try
                {
                    // Save the connection context
                    m_context.Copy( context );

                    // Should we acquire the channel information first?
                    if ( m_context.ShouldAcquireChannelInfo==true )
                    {
                        // Acquire the port number
                        if ( AcquireTargetPort( ref m_context )==false )
                            return false;
                    }

                    if ( m_context.Port<=0 )
                        return false;

                    // Connect to the server
                    m_socket = new TcpClient();
                    m_socket.ReceiveBufferSize = SOCKET_BUFFER_SIZE;
                    m_socket.Connect( m_context.IPAddress, m_context.Port );
                }
                catch ( SocketException ex )
                {
                    // SocketError.ConnectionRefused means the server does not exist.
                    // Do not log the message if this is the case.
                    if ( ex.SocketErrorCode!=SocketError.ConnectionRefused )
                    {
                        // Failed, log the error message.
                        DebugConsole.DebugConsole.Instance.WriteLine( ex.Message );
                    }
                }
                catch ( Exception ex )
                {
                    // Failed, log the error message.
                    DebugConsole.DebugConsole.Instance.WriteLine(ex.Message);
                }
            }

            if ( m_socket==null || m_socket.Connected==false )
            {
                m_socket = null;
                m_context.Reset();
                return false;
            }

            return true;
        }


        /// <summary>
        /// Close connection.
        /// </summary>
        public void Close()
        {
            if ( m_context!=null &&
                 m_context.UseWinMessage==true )
            {
                WinMsgClose();
                m_socket = null;
                return;
            }

            if ( m_socket==null ||
                 m_socket.Connected==false )
            {
                m_socket = null;
                return;
            }

            lock ( m_lock )
            {
                m_socket.Close();
                m_socket = null;

                if ( m_context!=null )
                    m_context.Reset();
            }
        }


        /// <summary>
        /// Get the information of the current connection.
        /// Null is returned if there is no connection.
        /// </summary>
        /// <returns>The information of the current connection or null if no connection.</returns>
        public ConnectionContext GetConnectionContext()
        {
            return m_context;
        }

        #endregion

        #region Send / receive

        /// <summary>
        /// Send raw data through current connection.
        /// </summary>
        /// <param name="buffer">The data buffer to send.</param>
        /// <param name="iBufferSize">Size of the data buffer.</param>
        /// <returns>True on success.</returns>
        public bool Send( byte[] buffer,
                          int iBufferSize )
        {
            if ( this.IsConnected==false )
                return false;

            if ( m_context.UseWinMessage==true )
                return WinMsgSend( buffer, iBufferSize );

            if ( m_socket==null || m_socket.Connected==false )
                return false;

            try
            {
                lock ( m_lock )
                {
                    NetworkStream stream = m_socket.GetStream();

                    stream.Write( buffer, 0, iBufferSize );
                    stream.Flush();
                }
            }
            catch ( Exception ex )
            {
                // Failed sending data
                DebugConsole.DebugConsole.Instance.WriteLine(ex.Message);
                Close();
                return false;
            }

            return true;
        }


        /// <summary>
        /// Receive data from the connection.
        /// </summary>
        /// <param name="buffer">The buffer to store the read data.</param>
        /// <returns>Size of the buffer read from the connection.</returns>
        public int Receive( out byte[] buffer )
        {
            buffer = null;

            if ( this.IsConnected==false )
                return 0;

            if ( m_context.UseWinMessage==true )
                return WinMsgReceive( out buffer );

            if ( m_socket==null || m_socket.Connected==false )
                return 0;

            int iSize = 0;
            try
            {
                if ( m_socket.Available>0 )
                {
                    lock ( m_lock )
                    {
                        NetworkStream stream = m_socket.GetStream();

                        buffer = new byte[ m_socket.Available ];
                        iSize  = stream.Read( buffer, 0, m_socket.Available );
                    }
                }
            }
            catch
            {
                // Failed reading data from the connection.
            }

            if ( buffer==null )
                return 0;

            // Check if this is a disconnect message
            if ( string.IsNullOrEmpty( m_context.ChannelHeaderCode )==false &&
                 buffer[0]=='-' )
            {
                bool   bMatch     = true;
                byte[] binaryData = Encoding.ASCII.GetBytes( "-" + m_context.ChannelHeaderCode );
                for ( int i=0;i<binaryData.Length;++i )
                {
                    if ( i>=buffer.Length ||
                         buffer[i]!=binaryData[i] )
                    {
                        bMatch = false;
                        break;
                    }
                }

                // This is a disconnect message, close the connection.
                if ( bMatch==true )
                {
                    Close();
                    return 0;
                }
            }

            return System.Math.Min( iSize, buffer.Length );
        }

        #endregion

        #region Create stream reader / writer from socket

        /// <summary>
        /// Create stream writer according to the connection settings.
        /// ( encoding, platform, etc... )
        /// </summary>
        /// <param name="stream">The stream to write.</param>
        /// <returns>The created writer.</returns>
        public Protocol4FWriter CreateWriter( Stream stream )
        {
            if ( m_context.Platform==PlatformType.Windows )
            {
                return new LittleEndianProtocolWriter( stream, m_context.Encoding );
            }
            else if ( m_context.Platform==PlatformType.Cafe )
            {
                return new BigEndianProtocolWriter( stream, m_context.Encoding );
            }

            return null;
        }


        /// <summary>
        /// Create stream reader according to the connection settings.
        /// ( encoding, platform, etc... )
        /// </summary>
        /// <param name="stream">The stream to read.</param>
        /// <returns>The created reader.</returns>
        public Protocol4FReader CreateReader( Stream stream )
        {
            if ( m_context.Platform==PlatformType.Windows )
            {
                return new LittleEndianProtocolReader( stream, m_context.Encoding );
            }
            else if ( m_context.Platform==PlatformType.Cafe )
            {
                return new BigEndianProtocolReader( stream, m_context.Encoding );
            }

            return null;
        }

        #endregion

        #region Acquire port number

        /// <summary>
        /// Parse channel information.
        /// </summary>
        /// <param name="message">全チャンネル情報のはいった文字列です。</param>
        /// <returns>チャンネル情報単位に分割して配列を返します。</returns>
        private IList<string> SplitChannelInfos( string message )
        {
            List<string> stringList = new List<string>();

            for (int index = 0; index < message.Length; )
            {
                if (message[index] == '+' || message[index] == '-')
                {
                    int i = index + 1;
                    for (; i < message.Length; ++i)
                    {
                        if (message[i] == '+' || message[i] == '-')
                        {
                            break;
                        }
                    }

                    stringList.Add(message.Substring(index, i - index));
                    index = i;
                }
                else
                {
                    ++index;
                }
            }

            return stringList;
        }


        /// <summary>
        /// Acquire port number.
        /// </summary>
        /// <param name="socket">The socket to connect to the server.</param>
        /// <param name="context">The connection context.</param>
        /// <returns>True on success.</returns>
        private bool AcquireTargetPort( ref ConnectionContext context )
        {
            TcpClient socket = new TcpClient();

            try
            {
                if ( context.ChannelInfoPort<=0 )
                    return false;

                // Connect to server for channel information.
                socket.Connect( context.IPAddress,
                                context.ChannelInfoPort );

                // Unable to connect for channel information.
                if ( socket.Connected==false )
                    return false;

                // No data being received
                int iNumTimeWaited = 0;
                while ( socket.Available<=0 )
                {
                    ++iNumTimeWaited;
                    if ( iNumTimeWaited>=100 )
                    {
                        socket.Close();
                        return false;
                    }

                    System.Threading.Thread.Sleep( 100 );
                }

                // Load the raw data from server.
                MemoryStream  memory = new MemoryStream();
                NetworkStream stream = socket.GetStream();

                int    iNumBytesRead = 0;
                byte[] buffer        = new byte[256];

                do
                {
                    iNumBytesRead = stream.Read( buffer, 0, buffer.Length );
                    if ( iNumBytesRead>0 )
                        memory.Write( buffer, 0, iNumBytesRead );
                } while ( socket.Available>0 );

                // Close the connection.
                socket.Close();

                // Encode the loaded raw data to string
                string infoString = Encoding.ASCII.GetString(memory.ToArray());

                // Split each channel information to string array
                IList<string> channels = this.SplitChannelInfos(infoString);

                // Parse each channel information
                int iMCSPortNo = -1;
                foreach ( string channel in channels )
                {
                    string[] tokens = channel.Split( new char[] { ':' }, 2 );
                    if ( tokens.Length!=2 )
                        continue;

                    string channelName = tokens[0];
                    int    iPortNum;
                    if ( int.TryParse( tokens[1], out iPortNum )==false )
                        continue;

                    // Log the channel name and port number
                    DebugConsole.DebugConsole.Instance.WriteLine( "- Channel: " +
                                                                  channelName.Substring(1) +
                                                                  ", port: " +
                                                                  iPortNum );

                    // Is the channel we want?
                    if ( channelName.Substring(1).CompareTo( context.ChannelHeaderCode )!=0 )
                        continue;

                    // Is the port number what we want?
                    if ( channelName[0]=='+' || channelName[0]=='-' )
                    {
                        context.Port = iPortNum;
                        return true;
                    }

                    iMCSPortNo = iPortNum;
                }

                context.Port = iMCSPortNo;

                return true;
            }
            catch
            {
                if ( socket!=null && socket.Connected==true )
                    socket.Close();

                // Failed loading channel information from server.
                return false;
            }
        }

        #endregion

        #region Backward support for windows message connection

        /// <summary>
        /// Enumerator for message types
        /// </summary>
        private enum WinMessageTypes
        {
            RELOAD_MESSAGE = 100,
            PACKET = 101
        }

        /// <summary>
        /// Check the existence of the target of the given connection information.
        /// </summary>
        /// <param name="context">The connection context.</param>
        /// <returns>True if the target exists.</returns>
        private bool WinMsgCheckTargetExistence( ConnectionContext context )
        {
            System.Diagnostics.Process[] processes =
                System.Diagnostics.Process.GetProcessesByName( context.ViewerProcessName );

            return ( processes.Length>0 );
        }


        /// <summary>
        /// Connect to the given target.
        /// </summary>
        /// <param name="context">The information about the target to connect to.</param>
        private bool WinMsgConnect( ConnectionContext context )
        {
            if ( context==null || context.IsReady==false )
                return false;

            if ( WinMsgCheckTargetExistence( context )==false )
                return false;

            lock ( m_lock )
            {
                m_context.Copy( context );
            }

            return true;
        }


        /// <summary>
        /// Close connection.
        /// </summary>
        private void WinMsgClose()
        {
            lock ( m_lock )
            {
                // Clear message queue.
                m_winMsgQueue.Clear();

                // Clear the connection context.
                if ( m_context!=null )
                    m_context.Reset();
            }
        }


        /// <summary>
        /// Send raw data through current connection.
        /// </summary>
        /// <param name="buffer">The data buffer to send.</param>
        /// <param name="iBufferSize">Size of the data buffer.</param>
        /// <returns>True on success.</returns>
        private bool WinMsgSend( byte[] buffer,
                                 int iBufferSize )
        {
            // Find the processes to send message to.
            System.Diagnostics.Process[] processes =
                System.Diagnostics.Process.GetProcessesByName( m_context.ViewerProcessName );
            if ( processes.Length<=0 )
                return false;

            IntPtr bufPtr  = IntPtr.Zero;
            IntPtr dataPtr = IntPtr.Zero;
            bool   bResult = true;

            try
            {
                // Allocate memory for the packet and copy it to the allocated location.
                bufPtr = Marshal.AllocHGlobal( buffer.Length );
                Marshal.Copy( buffer, 0, bufPtr, buffer.Length );

                // Setup the information for the buffer size, type and location.
                Win32.COPYDATASTRUCT copyData = new Win32.COPYDATASTRUCT();
                copyData.dwData = (IntPtr)WinMessageTypes.PACKET;
                copyData.cbData = buffer.Length;
                copyData.lpData = bufPtr;

                // Allocate memory for the buffer info and copy it to the allocated location.
                dataPtr = Marshal.AllocHGlobal( Marshal.SizeOf( copyData ) );
                Marshal.StructureToPtr( copyData, dataPtr, false );

                foreach ( System.Diagnostics.Process process in processes )
                {
                    IntPtr result = SendMessage( process.MainWindowHandle,
                                    Win32.WM.WM_COPYDATA,
                                    (IntPtr)0,
                                    dataPtr );
                    if ( result!=IntPtr.Zero )
                    {
                        DebugConsole.DebugConsole.Instance.WriteLine( "Failed sending message to viewer process, return code {0}.", (int)result );
                    }
                }
            }
            catch(Exception ex)
            {
                bResult = false;
                DebugConsole.DebugConsole.Instance.WriteLine( ex.Message );
                System.Diagnostics.Trace.WriteLine( ex.Message );
            }
            finally
            {
                if ( dataPtr!=IntPtr.Zero )
                    Marshal.FreeHGlobal( dataPtr );

                if ( bufPtr!=IntPtr.Zero )
                    Marshal.FreeHGlobal( bufPtr );
            }

            return bResult;
        }


        /// <summary>
        /// Read data from the stream of the current connection.
        /// </summary>
        /// <param name="buffer">The buffer to store the read data.</param>
        /// <returns>Size of the buffer read from the connection.</returns>
        private int WinMsgReceive( out byte[] buffer )
        {
            buffer = null;

            // Confirm that the target process exists.
            if ( WinMsgCheckTargetExistence( m_context )==false )
            {
                m_msgManager.RequestDisconnect();
                return 0;
            }

            // We have any data queued?
            if ( m_winMsgQueue.Count<=0 )
                return 0;

            // Pop out data from our received data queue
            lock ( m_winMsgQueue )
            {
                // Count the total size of all the queued buffers first.
                int iSize = 0;
                foreach ( byte[] queuedData in m_winMsgQueue )
                {
                    iSize += queuedData.Length;
                }

                // Allocate the buffer for all the messages.
                buffer = new byte[ iSize ];

                // Copy all the data into the buffer.
                int iIndex = 0;
                foreach ( byte[] queuedData in m_winMsgQueue )
                {
                    // Copy the buffer.
                    for ( int i=0;i<queuedData.Length;++i )
                    {
                        buffer[iIndex+i] = queuedData[i];
                    }
                }

                // Clear the queue.
                m_winMsgQueue.Clear();
            }

            if ( buffer==null )
                return 0;

            return buffer.Length;
        }


        /// <summary>
        /// Enqueue data from received windows message for later use.
        /// </summary>
        /// <param name="buffer">The data buffer.</param>
        public void WinMsgPushReceivedData( byte[] buffer )
        {
            lock ( m_winMsgQueue )
            {
                m_winMsgQueue.Add( buffer );
            }
        }


        /// <summary>
        /// SendMessage
        /// </summary>
        /// <param name="hwnd">Pointer to the handler</param>
        /// <param name="Msg">Message to be sent</param>
        /// <param name="wParam">Parameters to be sent along with</param>
        /// <param name="lParam">Copy of data structure</param>
        /// <returns></returns>
        [DllImport( "user32.dll", CharSet = CharSet.Auto, SetLastError = true )]
        private static extern IntPtr SendMessage( IntPtr hWnd,
                                                  uint Msg,
                                                  IntPtr wParam,
                                                  IntPtr lParam );

        #endregion

        #region Member variables

        private const int         SOCKET_BUFFER_SIZE = 262144; // 256K Bytes

        private MessageManager    m_msgManager  = null;

        private ConnectionContext m_context     = null;
        private TcpClient         m_socket      = null;

        private volatile object   m_lock        = new object();

        //---------------------------------------------------------------------
        // Variables for the backward support for windows message connection
        //---------------------------------------------------------------------
        private List<byte[]>      m_winMsgQueue = new List<byte[]>();

        #endregion
    }
}
