﻿// ========================================================================
// <copyright file="EnvConfigDataXml.cs" company="Nintendo">
//      Copyright 2011 Nintendo.  All rights reserved.
// </copyright>
//
// These coded instructions, statements, and computer programs contain
// proprietary information of Nintendo of America Inc. and/or Nintendo
// Company Ltd., and are protected by Federal copyright law.  They may
// not be disclosed to third parties or copied or duplicated in any form,
// in whole or in part, without the prior written consent of Nintendo.
// ========================================================================

using System;
using System.Collections.Generic;
using System.Xml.Serialization;
using NintendoWare.ToolDevelopmentKit;
using NintendoWare.ToolDevelopmentKit.Conversion;
using NintendoWare.ToolDevelopmentKit.Xml;
using NWCore.DataModel;

namespace NWCore.Serializer
{
    /// <summary>
    /// 全エミッタ共通データのシリアライズ用クラスです
    /// </summary>
    [XmlRoot("root")]
    public class EnvConfigDataXml
    {
        #region Classes

        /// <summary>
        /// エミッタセット用コンフィグ設定用
        /// </summary>
        [XmlType("EmitterSetData")]
        public class EmitterSetConfigXml
        {
            /// <summary>
            /// コンストラクタ
            /// </summary>
            public EmitterSetConfigXml()
            {
                this.EmitterGameDatas = new List<EmitterGameData>();
                this.PreviewGameDatas = new List<PreviewGameData>();
            }

            /// <summary>
            /// コンストラクタ
            /// </summary>
            public EmitterSetConfigXml(IEmitterSetDocument emitterSet)
                : this()
            {
                SetForSerialize(emitterSet);
            }

            /// <summary>
            /// シリアライズのための設定
            /// </summary>
            public void SetForSerialize(IEmitterSetDocument emitterSet)
            {
                this.Name = emitterSet.FileName;
                this.IsShow = emitterSet.EmitterSetData.EditData.DebugData.DispEmitterSet;

                this.EmitterGameDatas.Clear();
                int index = 0;
                foreach (var item in emitterSet.EmitterDocuments)
                {
                    EmitterGameData config = new EmitterGameData();
                    config.SetForSerialize(item.EmitterData, index++);
                    this.EmitterGameDatas.Add(config);
                }

                index = 0;
                foreach (var item in emitterSet.GameSettingsDocuments)
                {
                    PreviewGameData config = new PreviewGameData();
                    config.SetForSerialize(item, index++);
                    this.PreviewGameDatas.Add(config);
                }
            }

            /// <summary>
            /// 復元：EmitterSet用コンフィグ
            /// </summary>
            public void SetForDeserialize(IEmitterSetDocument emitterSet)
            {
                emitterSet.EmitterSetData.EditData.DebugData.DispEmitterSet = this.IsShow;

                int index = 0;
                foreach (var emitter in emitterSet.EmitterDocuments)
                {
                    var foundItem = FindEmitterGameDatas(emitter.Name, index++);
                    if (foundItem != null)
                    {
                        foundItem.SetForDeserialize(emitter.EmitterData);
                    }
                }

                index = 0;
                foreach (var preview in emitterSet.GameSettingsDocuments)
                {
                    var foundItem = FindPreviewGameDatas(preview.Name, index++);
                    if (foundItem != null)
                    {
                        foundItem.SetForDeserialize(preview);
                    }
                }
            }

            private EmitterGameData FindEmitterGameDatas(string emitterName, int index)
            {
                foreach (var item in EmitterGameDatas)
                {
                    if (item.Name != null
                        && item.Name.Equals(emitterName, StringComparison.InvariantCultureIgnoreCase)
                        && item.Index == index)
                    {
                        return item;
                    }
                }
                return null;
            }

            private PreviewGameData FindPreviewGameDatas(string preivewName, int index)
            {
                foreach (var item in PreviewGameDatas)
                {
                    if (item.Name != null
                        && item.Name.Equals(preivewName, StringComparison.InvariantCultureIgnoreCase)
                        && item.Index == index)
                    {
                        return item;
                    }
                }
                return null;
            }

            /// <summary>
            /// エミッタセットのファイルパス
            /// </summary>
            [XmlAttribute]
            public string Name { get; set; }

            /// <summary>
            /// エミッタセットの表示、非表示
            /// </summary>
            [XmlAttribute]
            public bool IsShow { get; set; }

            /// <summary>
            /// エミッタセットの表示、非表示
            /// </summary>
            [XmlArrayItem("EmitterConfig")]
            public List<EmitterGameData> EmitterGameDatas
            {
                get;
                set;
            }

            /// <summary>
            /// プレビューの表示、非表示
            /// </summary>
            [XmlArrayItem("PreviewConfig")]
            public List<PreviewGameData> PreviewGameDatas
            {
                get;
                set;
            }
        }

        #endregion

        #region Construtors

        /// <summary>
        /// コンストラクタ
        /// </summary>
        public EnvConfigDataXml()
        {
            this.ConfigData        = new EnvConfigData();
            this.PrevDataList      = new List<GameConfigDataXml>();
            this.EmitterSetConfigs = new List<EmitterSetConfigXml>();

            this.ConfigData.Initialize();
        }

        /// <summary>
        /// コンストラクタ
        /// </summary>
        public EnvConfigDataXml( EnvConfigData srcData )
            : this()
        {
            this.ConfigData.Set( srcData );
        }

        /// <summary>
        /// 復元：EmitterSet用コンフィグ
        /// </summary>
        public void DeserializeToEmitterSet(IEmitterSetDocument emitterSet)
        {
            EmitterSetConfigXml item = FindFromEmitterSets(emitterSet.FileName);
            if (item != null)
            {
                item.SetForDeserialize(emitterSet);
            }
        }

        /// <summary>
        /// EmitterSetConfigs から対応するEmitterSetのコンフィグ情報を取得する
        /// </summary>
        public EmitterSetConfigXml FindFromEmitterSets(string emitterSetName)
        {
            foreach (var item in EmitterSetConfigs)
            {
                if (item.Name != null
                    && item.Name.Equals(emitterSetName, StringComparison.InvariantCultureIgnoreCase))
                {
                    return item;
                }
            }
            return null;
        }

        #endregion

        #region Properties

        /// <summary>
        /// Preview file paths
        /// </summary>
        [XmlArrayItem("PrevDataList")]
        public List<GameConfigDataXml> PrevDataList
        {
            get;
            set;
        }

        /// <summary>
        /// Emitter set config data
        /// </summary>
        [XmlArrayItem( "EmitterSetConfigs" )]
        public List<EmitterSetConfigXml> EmitterSetConfigs
        {
            get;
            set;
        }

        /// <summary>
        /// コンフィグ設定データ
        /// </summary>
        public EnvConfigData ConfigData
        {
            get;
            set;
        }

        #endregion

    }
}
