﻿// ========================================================================
// <copyright file="ProtocolEM4F.cs" company="Nintendo">
//      Copyright 2009 Nintendo.  All rights reserved.
// </copyright>
//
// These coded instructions, statements, and computer programs contain
// proprietary information of Nintendo of America Inc. and/or Nintendo
// Company Ltd., and are protected by Federal copyright law.  They may
// not be disclosed to third parties or copied or duplicated in any form,
// in whole or in part, without the prior written consent of Nintendo.
// ========================================================================

using System;
using System.IO;
using System.Text;
using System.Globalization;
using NintendoWare.ToolDevelopmentKit;
using NWCore.DataModel;

namespace NWCore.Protocols
{
    /// <summary>Constants</summary>
    public class Constants
    {
        /// <summary>プロトコルバージョン</summary>
        public const uint VERSION_EM4F = 0x03000000;

        /// <summary>プロトコルのメジャーバージョンマスク</summary>
        public const uint VERSION_MAJOR_EM4C_MASK = 0xFF000000;

        /// <summary>プロトコルのマイナーバージョンマスク</summary>
        public const uint VERSION_MINOR_EM4C_MASK = 0x00FF0000;

        /// <summary>プロトコルのリビジョンマスク</summary>
        public const uint VERSION_REVISION_EM4C_MASK = 0x0000FF00;

        /// <summary>メッセージのカテゴリーマスク</summary>
        public const uint MESSAGE_CATEGORY_EM4C_MASK = 0x0000FF00;

        /// <summary>EffectMaker用チャンネルＩＤです。送信用</summary>
        public const uint CHANNEL_ID_EFFECTMAKER_ACTIVE = 0x02;

        /// <summary>EffectMaker用チャンネルＩＤです。受信用</summary>
        public const uint CHANNEL_ID_EFFECTMAKER_PASSIVE = 0x03;

    };

    /// <summary>編集ターゲットのオブジェクトタイプ</summary>
    public enum EditTargetType : int
    {
        /// <summary>未選択</summary>
        None = 0x00,

        /// <summary>エミッタセット・ターゲット</summary>
        EmitterSet = 0x01,

        /// <summary>エミッタ・ターゲット</summary>
        Emitter = 0x02,

        /// <summary>ゲーム設定（現プレビューノード）</summary>
        GameSetting = 0x10,

        /// <summary>プレビュー設定（現ビューアノード）</summary>
        PreviewSetting = 0x20,

        /// <summary>モデル設定（現ビューアモデルノード）</summary>
        PreviewModel = 0x30,

        /// <summary>マスク</summary>
        Mask = 0xFF,

    }

    /// <summary>バイナリーのタイプ</summary>
    public enum BinaryDataType : int
    {
        /// <summary>未設定</summary>
        None = 0x00,

        /// <summary>EmitterSet 用</summary>
        Ptcl = 0x01,

        /// <summary>Model 用</summary>
        Model = 0x02,

    }

    /// <summary>Flag用ビット定義（=IsEnabled|IsResourceCopyEnabledのように使用します）</summary>
    public enum FlagBitType : int
    {
        /// <summary>未設定</summary>
        None = 0x00,

        /// <summary>Initializer/Updater用フラグ：許可／不許可フラグ</summary>
        IsEnabled = 0x01,

    }

    /// <summary>プラットフォームのタイプ</summary>
    public enum PlatformType : int
    {
        /// <summary>Platform:Windows Little Endian(=0x0000(Win Viewer, Little Endian) + 0x0000)</summary>
        Windows = 0x0000,

        /// <summary>Platform:Cafe Big Endian(=0x1100(実機、Big Endian) + 0x0001)</summary>
        Cafe = 0x1101,

        /// <summary>Platform:CTR Little Endian(=0x1000(実機、Little Endian) + 0x0002)</summary>
        CTR = 0x1002,

        /// <summary>エンディアン用マスク</summary>
        EndianMask = 0x0F00,

        /// <summary>実機(Use Native Format)用マスク</summary>
        NativeFormatMask = 0xF000,

    }

    /// <summary>ビューア側で利用する文字コードのエンコーディングタイプ</summary>
    public enum ViewEncodingType : int
    {
        /// <summary>ビューアで使用する文字をUTF8とする</summary>
        UTF8 = 0x00,

        /// <summary>ビューアで使用する文字をUnicodeとする</summary>
        Unicode = 0x01,

        /// <summary>ビューアで使用する文字をShiftJISとする</summary>
        ShiftJIS = 0x02,

    }

    /// <summary>文字列のタイプ</summary>
    public enum StringType : int
    {
        /// <summary>文字列を名前として使用</summary>
        Name = 0x00,

        /// <summary>文字列をファイルとして使用</summary>
        Path = 0x01,

    }

    /// <summary>メッセージカテゴリ</summary>
    public enum MessageCategory : int
    {
        /// <summary>ファイル</summary>
        File = 0x0000,

        /// <summary>オブジェクト</summary>
        Object = 0x0100,

        /// <summary>エミッタ</summary>
        Emitter = 0x0200,

        /// <summary>プレビュー設定</summary>
        Preview = 0x0300,

        /// <summary>ビューア</summary>
        Viewer = 0x0400,

    }

    /// <summary>メッセージタイプ</summary>
    public enum MessageType : int
    {
        /// <summary>最後のパケットであることを示すパケット</summary>
        [MessageTypeAttribute(typeof(MESSAGEPACKET_File_EndOfPacket))]
        File_EndOfPacket = 0x00 | MessageCategory.File,

        /// <summary>接続時に送信するパケット</summary>
        [MessageTypeAttribute(typeof(MESSAGEPACKET_File_Connect))]
        File_Connect = 0x01 | MessageCategory.File,

        /// <summary>すべてのファイルを閉じます。</summary>
        [MessageTypeAttribute(typeof(MESSAGEPACKET_File_CloseAll))]
        File_CloseAll = 0x02 | MessageCategory.File,

        /// <summary>選択されたモデルを閉じます。</summary>
        [MessageTypeAttribute(typeof(MESSAGEPACKET_File_Close))]
        File_Close = 0x03 | MessageCategory.File,

        /// <summary>Viewer側からファイルを閉じます。</summary>
        [MessageTypeAttribute(typeof(MESSAGEPACKET_File_CloseV2E))]
        File_CloseV2E = 0x04 | MessageCategory.File,

        /// <summary>バイナリデータを読み込みます。</summary>
        [MessageTypeAttribute(typeof(MESSAGEPACKET_File_LoadBinaryData))]
        File_LoadBinaryData = 0x05 | MessageCategory.File,

        /// <summary>指定ファイルの読み込みをビューア→EffectMakerへリクエストします。</summary>
        [MessageTypeAttribute(typeof(MESSAGEPACKET_File_LoadFileV2E))]
        File_LoadFileV2E = 0x06 | MessageCategory.File,

        /// <summary>読み込んだファイルを Byte[] としてEffectMaker→ビューアへ送ります。</summary>
        [MessageTypeAttribute(typeof(MESSAGEPACKET_File_LoadFileE2V))]
        File_LoadFileE2V = 0x07 | MessageCategory.File,

        /// <summary>Viewer側で利用しているEmitterSetをEffectMaker側へ通知</summary>
        [MessageTypeAttribute(typeof(MESSAGEPACKET_File_RequireEmitterSetV2E))]
        File_RequireEmitterSetV2E = 0x08 | MessageCategory.File,

        /// <summary>接続時にビューア情報をEffectMakerへ通知するパケット</summary>
        [MessageTypeAttribute(typeof(MESSAGEPACKET_File_ConnectInfoV2E))]
        File_ConnectInfoV2E = 0x09 | MessageCategory.File,

        /// <summary>Viewer側からエミッタセットのリンク開始を通知するパケット</summary>
        [MessageTypeAttribute(typeof(MESSAGEPACKET_File_LinkEmitterSetV2E))]
        File_LinkEmitterSetV2E = 0x0A | MessageCategory.File,

        /// <summary>Viewer側からエミッタセットのリンク解除を通知するパケット</summary>
        [MessageTypeAttribute(typeof(MESSAGEPACKET_File_UnlinkEmitterSetV2E))]
        File_UnlinkEmitterSetV2E = 0x0B | MessageCategory.File,

        /// <summary>選択しているエミッタセットのデータを送信します。</summary>
        [MessageTypeAttribute(typeof(MESSAGEPACKET_File_SendEmitterSet))]
        File_SendEmitterSet = 0x0C | MessageCategory.File,

        /// <summary>編集ターゲットを選択します。</summary>
        [MessageTypeAttribute(typeof(MESSAGEPACKET_Object_SelectEM))]
        Object_SelectEM = 0x00 | MessageCategory.Object,

        /// <summary>選択しているエミッタのデータを送信します。</summary>
        [MessageTypeAttribute(typeof(MESSAGEPACKET_Emitter_SendEmitter))]
        Emitter_SendEmitter = 0x00 | MessageCategory.Emitter,

        /// <summary>編集しているエミッタのアニメバイナリを送信します。</summary>
        [MessageTypeAttribute(typeof(MESSAGEPACKET_Emitter_SendAnimTable))]
        Emitter_SendAnimTable = 0x01 | MessageCategory.Emitter,

        /// <summary>編集しているエミッタのカスタムシェーダパラメータを送信します。</summary>
        [MessageTypeAttribute(typeof(MESSAGEPACKET_Emitter_SendShaderParam))]
        Emitter_SendShaderParam = 0x02 | MessageCategory.Emitter,

        /// <summary>エミッタセットごとのプレビュー設定</summary>
        [MessageTypeAttribute(typeof(MESSAGEPACKET_Preview_SetPreviewEmitterSet))]
        Preview_SetPreviewEmitterSet = 0x00 | MessageCategory.Preview,

        /// <summary>Function送信</summary>
        [MessageTypeAttribute(typeof(MESSAGEPACKET_Preview_SendFunctionInfo))]
        Preview_SendFunctionInfo = 0x01 | MessageCategory.Preview,

        /// <summary>エフェクトのインスタンスを生成します。</summary>
        [MessageTypeAttribute(typeof(MESSAGEPACKET_Preview_AddEmitterSetInst))]
        Preview_AddEmitterSetInst = 0x02 | MessageCategory.Preview,

        /// <summary>エフェクトのインスタンスを削除します。</summary>
        [MessageTypeAttribute(typeof(MESSAGEPACKET_Preview_RemoveEmitterSetInst))]
        Preview_RemoveEmitterSetInst = 0x03 | MessageCategory.Preview,

        /// <summary>開始フレームまでフレーム移動します。</summary>
        [MessageTypeAttribute(typeof(MESSAGEPACKET_Preview_RewindToFirst))]
        Preview_RewindToFirst = 0x04 | MessageCategory.Preview,

        /// <summary>全てのアニメーションの更新を開始/一時停止します。</summary>
        [MessageTypeAttribute(typeof(MESSAGEPACKET_Preview_Play))]
        Preview_Play = 0x05 | MessageCategory.Preview,

        /// <summary>全てのアニメーションを停止し、フレームを最初に戻します。</summary>
        [MessageTypeAttribute(typeof(MESSAGEPACKET_Preview_Stop))]
        Preview_Stop = 0x06 | MessageCategory.Preview,

        /// <summary>全てのアニメーションの１フレーム更新（前に進める）します。</summary>
        [MessageTypeAttribute(typeof(MESSAGEPACKET_Preview_StepFrame))]
        Preview_StepFrame = 0x07 | MessageCategory.Preview,

        /// <summary>再生するフレーム範囲を指定します。</summary>
        [MessageTypeAttribute(typeof(MESSAGEPACKET_Preview_SetFrameRange))]
        Preview_SetFrameRange = 0x08 | MessageCategory.Preview,

        /// <summary>エミッタセット/エミッタ/プレビューノードの表示/非表示を切り替えます</summary>
        [MessageTypeAttribute(typeof(MESSAGEPACKET_Preview_SetVisibility))]
        Preview_SetVisibility = 0x09 | MessageCategory.Preview,

        /// <summary>初期設定です。</summary>
        [MessageTypeAttribute(typeof(MESSAGEPACKET_Viewer_InitSettings))]
        Viewer_InitSettings = 0x00 | MessageCategory.Viewer,

        /// <summary>背景画像の設定です。</summary>
        [MessageTypeAttribute(typeof(MESSAGEPACKET_Viewer_SendBackgroundImage))]
        Viewer_SendBackgroundImage = 0x01 | MessageCategory.Viewer,

        /// <summary>プレビュー設定のモデル情報を送信します</summary>
        [MessageTypeAttribute(typeof(MESSAGEPACKET_Viewer_SendPreviewModel))]
        Viewer_SendPreviewModel = 0x02 | MessageCategory.Viewer,

        /// <summary>モデル情報を削除します。</summary>
        [MessageTypeAttribute(typeof(MESSAGEPACKET_Viewer_ClosePreviewModel))]
        Viewer_ClosePreviewModel = 0x03 | MessageCategory.Viewer,

        /// <summary>モデル情報を送信します（Viewer -> EM4C）</summary>
        [MessageTypeAttribute(typeof(MESSAGEPACKET_Viewer_SendModelInfoV2E))]
        Viewer_SendModelInfoV2E = 0x04 | MessageCategory.Viewer,

        /// <summary>モデル情報を初期化します。</summary>
        [MessageTypeAttribute(typeof(MESSAGEPACKET_Viewer_ClearPreviewModelV2E))]
        Viewer_ClearPreviewModelV2E = 0x05 | MessageCategory.Viewer,

        /// <summary>モデル情報の転送を開始します。</summary>
        [MessageTypeAttribute(typeof(MESSAGEPACKET_Viewer_BeginSendModelInfoV2E))]
        Viewer_BeginSendModelInfoV2E = 0x06 | MessageCategory.Viewer,

        /// <summary>モデル情報の転送を終了します。</summary>
        [MessageTypeAttribute(typeof(MESSAGEPACKET_Viewer_EndSendModelInfoV2E))]
        Viewer_EndSendModelInfoV2E = 0x07 | MessageCategory.Viewer,

        /// <summary>カスタムシェーダフォルダパスを指定します。</summary>
        [MessageTypeAttribute(typeof(MESSAGEPACKET_Viewer_SetUserShaderPathV2E))]
        Viewer_SetUserShaderPathV2E = 0x08 | MessageCategory.Viewer,

        /// <summary>カスタムアクションフォルダパスを指定します。</summary>
        [MessageTypeAttribute(typeof(MESSAGEPACKET_Viewer_SetUserDataPathV2E))]
        Viewer_SetUserDataPathV2E = 0x09 | MessageCategory.Viewer,

        /// <summary>ユーザー設定ファイルパスを指定します。</summary>
        [MessageTypeAttribute(typeof(MESSAGEPACKET_Viewer_SetUserConfigPathV2E))]
        Viewer_SetUserConfigPathV2E = 0x0A | MessageCategory.Viewer,

        /// <summary>ビューアからエミッタセットを新規作成します。</summary>
        [MessageTypeAttribute(typeof(MESSAGEPACKET_Viewer_CreateEmitterSetV2E))]
        Viewer_CreateEmitterSetV2E = 0x0B | MessageCategory.Viewer,

        /// <summary>指定したエミッタセットの複製します。</summary>
        [MessageTypeAttribute(typeof(MESSAGEPACKET_Viewer_CloneEmitterSetV2E))]
        Viewer_CloneEmitterSetV2E = 0x0C | MessageCategory.Viewer,

        /// <summary>コネクションを切断します。</summary>
        [MessageTypeAttribute(typeof(MESSAGEPACKET_Viewer_CloseConnection))]
        Viewer_CloseConnection = 0x0D | MessageCategory.Viewer,

        /// <summary>カメラの設定を送信します。</summary>
        [MessageTypeAttribute(typeof(MESSAGEPACKET_Viewer_CameraSettings))]
        Viewer_CameraSettings = 0x0E | MessageCategory.Viewer,

        /// <summary>プレビュー設定のモデルの再生パラメータを送信します</summary>
        [MessageTypeAttribute(typeof(MESSAGEPACKET_Viewer_SendModelPlaybackParam))]
        Viewer_SendModelPlaybackParam = 0x0F | MessageCategory.Viewer,

    }

    /// <summary>Header</summary>
    public class Header
    {
        /// <summary>構造体にした場合のサイズ</summary>
        public const int StructSize = (4 * 3);

        /// <summary>コンストラクタ</summary>
        public Header()
        {
        }

        /// <summary>コンストラクタ</summary>
        public Header(MessageType type, int size, int arraySize)
        {
            Type = type;
            Size = size;
            ArraySize = arraySize;
        }

        /// <summary>文字列化</summary>
        public override string ToString()
        {
            return "Type:" + Type.ToString() + " Size:" + Size.ToString() + " ArraySize:" + ArraySize.ToString();
        }

        /// <summary>Type</summary>
        public MessageType Type{ get; set; }

        /// <summary>Size</summary>
        public int Size{ get; set; }

        /// <summary>ArraySize</summary>
        public int ArraySize{ get; set; }

        /// <summary>Read</summary>
        public void Read(Protocol4FReader reader)
        {
            Type = (MessageType)reader.ReadInt32();
            if (Type == MessageType.File_EndOfPacket)
            {
                Size = 0;
                ArraySize = 0;
            }
            else
            {
                Size = reader.ReadInt32();
                ArraySize = reader.ReadInt32();
            }
        }

        /// <summary>Write</summary>
        public void Write(Protocol4FWriter writer)
        {
            writer.Write((int)Type);
            writer.Write(Size);
            writer.Write(ArraySize);
        }
    };

    /// <summary>NByte</summary>
    public class NByte
    {
        /// <summary>Value</summary>
        public byte Value{ get; set; }

        /// <summary>Size</summary>
        public int Size{ get { return 1; } }

        /// <summary>コンストラクタ</summary>
        public NByte()
        {
            Value = (byte)0;
        }

        /// <summary>コンストラクタ</summary>
        public NByte(byte src)
        {
            Value = src;
        }

        /// <summary>文字列化</summary>
        public override string ToString()
        {
            return Value.ToString( CultureInfo.InvariantCulture );
        }

        /// <summary>Read</summary>
        public void Read(Protocol4FReader reader)
        {
            Value = reader.ReadByte();
        }

        /// <summary>Write</summary>
        public void Write(Protocol4FWriter writer)
        {
            writer.Write(Value);
        }
    };

    /// <summary>NInt</summary>
    public class NInt
    {
        /// <summary>Value</summary>
        public int Value{ get; set; }

        /// <summary>Size</summary>
        public int Size{ get { return 4; } }

        /// <summary>コンストラクタ</summary>
        public NInt()
        {
        }

        /// <summary>コンストラクタ</summary>
        public NInt(int src)
        {
            Value = src;
        }

        /// <summary>文字列化</summary>
        public override string ToString()
        {
            return Value.ToString( CultureInfo.InvariantCulture );
        }

        /// <summary>Read</summary>
        public void Read(Protocol4FReader reader)
        {
            Value = reader.ReadInt32();
        }

        /// <summary>Write</summary>
        public void Write(Protocol4FWriter writer)
        {
            writer.Write(Value);
        }
    };

    /// <summary>NUint</summary>
    public class NUint
    {
        /// <summary>Value</summary>
        public uint Value{ get; set; }

        /// <summary>Size</summary>
        public int Size{ get { return 4; } }

        /// <summary>コンストラクタ</summary>
        public NUint()
        {
        }

        /// <summary>コンストラクタ</summary>
        public NUint(uint src)
        {
            Value = src;
        }

        /// <summary>文字列化</summary>
        public override string ToString()
        {
            return Value.ToString( CultureInfo.InvariantCulture );
        }

        /// <summary>Read</summary>
        public void Read(Protocol4FReader reader)
        {
            Value = reader.ReadUInt32();
        }

        /// <summary>Write</summary>
        public void Write(Protocol4FWriter writer)
        {
            writer.Write(Value);
        }
    };

    /// <summary>NFloat</summary>
    public class NFloat
    {
        /// <summary>Value</summary>
        public float Value{ get; set; }

        /// <summary>Size</summary>
        public int Size{ get { return 4; } }

        /// <summary>コンストラクタ</summary>
        public NFloat()
        {
        }

        /// <summary>コンストラクタ</summary>
        public NFloat(float src)
        {
            Value = src;
        }

        /// <summary>文字列化</summary>
        public override string ToString()
        {
            return Value.ToString( CultureInfo.InvariantCulture );
        }

        /// <summary>Read</summary>
        public void Read(Protocol4FReader reader)
        {
            Value = reader.ReadSingle();
        }

        /// <summary>Write</summary>
        public void Write(Protocol4FWriter writer)
        {
            writer.Write(Value);
        }
    };

    /// <summary>NString</summary>
    public class NTString
    {
        /// <summary>Value</summary>
        public byte[] Value{ get; set; }

        /// <summary>Size</summary>
        public int Size{ get { return Value.Length; } }

        /// <summary>コンストラクタ</summary>
        public NTString(int size)
        {
            Value = new byte[size * 2];
        }

        /// <summary>コンストラクタ</summary>
        public NTString(int size, string src) : this(size)
        {
            FromString(src);
        }

        /// <summary>ToString</summary>
        public override string ToString()
        {
            int size = 0;
            for (; size < Value.Length; ++size)
            {
                if (Value[size] == '\0')
                {
                    break;
                }
            }

            byte[] temp = new byte[size];
            for (int i = 0; i != size; ++i)
            {
                temp[i] = Value[i];
            }
            byte[] dstData = Encoding.Convert(NWCore.Viewer.MCSManager.EncodingType, Encoding.UTF8, temp);
            return Encoding.UTF8.GetString(dstData, 0, dstData.Length);
        }

        /// <summary>FromString</summary>
        public void FromString(string src)
        {
            if (NWCore.Viewer.MCSManager.EncodingType.GetByteCount(src) > Value.Length)
            {
                /*ShowError();*/
                return;
            }
            byte[] srcData = Encoding.UTF8.GetBytes(src);
            byte[] dstData = Encoding.Convert(Encoding.UTF8, NWCore.Viewer.MCSManager.EncodingType, srcData);
            for (int i = 0; i < Value.Length; ++i)
            {
                Value[i] = (i < dstData.Length) ? dstData[i] : (byte)0;
            }
        }

        /// <summary>Read</summary>
        public void Read(Protocol4FReader reader)
        {
            for(int i = 0;i < Size;++ i)
            {
                Value[i] = reader.ReadByte();
            }
        }

        /// <summary>Write</summary>
        public void Write(Protocol4FWriter writer)
        {
            for(int i = 0;i < Size;++ i)
            {
                writer.Write(Value[i]);
            }
        }
    };

    /// <summary>NString</summary>
    public class NString : NTString
    {
        /// <summary>コンストラクタ</summary>
        public NString() : base(260) { }

        /// <summary>コンストラクタ</summary>
        public NString(string src) : base(260, src) { }
    }

    /// <summary>NString</summary>
    public class NString16 : NTString
    {
        /// <summary>コンストラクタ</summary>
        public NString16() : base(16) { }

        /// <summary>コンストラクタ</summary>
        public NString16(string src) : base(16, src) { }
    }

    /// <summary>NString</summary>
    public class NString32 : NTString
    {
        /// <summary>コンストラクタ</summary>
        public NString32() : base(32) { }

        /// <summary>コンストラクタ</summary>
        public NString32(string src) : base(32, src) { }
    }

    /// <summary>NRgb</summary>
    public class NRgb
    {
        /// <summary>R</summary>
        public float R{ get; set; }

        /// <summary>G</summary>
        public float G{ get; set; }

        /// <summary>B</summary>
        public float B{ get; set; }

        /// <summary>Size</summary>
        public int Size{ get { return (4 * 3); } }

        /// <summary>コンストラクタ</summary>
        public NRgb()
        {
        }

        /// <summary>コンストラクタ</summary>
        public NRgb(RgbaColor src)
        {
            R = src.R;
            G = src.G;
            B = src.B;
        }

        /// <summary>文字列化</summary>
        public override string ToString()
        {
            return "R:" + R.ToString( CultureInfo.InvariantCulture ) +
                  " G:" + G.ToString( CultureInfo.InvariantCulture ) +
                  " B:" + B.ToString( CultureInfo.InvariantCulture );
        }

        /// <summary>Read</summary>
        public void Read(Protocol4FReader reader)
        {
            R = reader.ReadSingle();
            G = reader.ReadSingle();
            B = reader.ReadSingle();
        }

        /// <summary>Write</summary>
        public void Write(Protocol4FWriter writer)
        {
            writer.Write(R);
            writer.Write(G);
            writer.Write(B);
        }
    };

    /// <summary>NRgba</summary>
    public class NRgba
    {
        /// <summary>R</summary>
        public float R{ get; set; }

        /// <summary>G</summary>
        public float G{ get; set; }

        /// <summary>B</summary>
        public float B{ get; set; }

        /// <summary>A</summary>
        public float A{ get; set; }

        /// <summary>Size</summary>
        public int Size{ get { return (4 * 4); } }

        /// <summary>コンストラクタ</summary>
        public NRgba()
        {
        }

        /// <summary>コンストラクタ</summary>
        public NRgba(RgbaColor src)
        {
            R = src.R;
            G = src.G;
            B = src.B;
            A = src.A;
        }

        /// <summary>文字列化</summary>
        public override string ToString()
        {
            return "R:" + R.ToString( CultureInfo.InvariantCulture ) +
                  " G:" + G.ToString( CultureInfo.InvariantCulture ) +
                  " B:" + B.ToString( CultureInfo.InvariantCulture ) +
                  " A:" + A.ToString( CultureInfo.InvariantCulture );
        }

        /// <summary>Read</summary>
        public void Read(Protocol4FReader reader)
        {
            R = reader.ReadSingle();
            G = reader.ReadSingle();
            B = reader.ReadSingle();
            A = reader.ReadSingle();
        }

        /// <summary>Write</summary>
        public void Write(Protocol4FWriter writer)
        {
            writer.Write(R);
            writer.Write(G);
            writer.Write(B);
            writer.Write(A);
        }
    };

    /// <summary>NVector2</summary>
    public class NVector2
    {
        /// <summary>X</summary>
        public float X{ get; set; }

        /// <summary>Y</summary>
        public float Y{ get; set; }

        /// <summary>Size</summary>
        public int Size{ get { return (4 * 2); } }

        /// <summary>コンストラクタ</summary>
        public NVector2()
        {
        }

        /// <summary>コンストラクタ</summary>
        public NVector2(NVector2 src)
        {
            X = src.X;
            Y = src.Y;
        }

        /// <summary>文字列化</summary>
        public override string ToString()
        {
            return "X:" + X.ToString( CultureInfo.InvariantCulture ) +
                  " Y:" + Y.ToString( CultureInfo.InvariantCulture );
        }

        /// <summary>Read</summary>
        public void Read(Protocol4FReader reader)
        {
            X = reader.ReadSingle();
            Y = reader.ReadSingle();
        }

        /// <summary>Write</summary>
        public void Write(Protocol4FWriter writer)
        {
            writer.Write(X);
            writer.Write(Y);
        }
    };

    /// <summary>NVector3</summary>
    public class NVector3
    {
        /// <summary>X</summary>
        public float X{ get; set; }

        /// <summary>Y</summary>
        public float Y{ get; set; }

        /// <summary>Z</summary>
        public float Z{ get; set; }

        /// <summary>Size</summary>
        public int Size{ get { return (4 * 3); } }

        /// <summary>コンストラクタ</summary>
        public NVector3()
        {
        }

        /// <summary>コンストラクタ</summary>
        public NVector3(NVector3 src)
        {
            X = src.X;
            Y = src.Y;
            Z = src.Z;
        }

        /// <summary>文字列化</summary>
        public override string ToString()
        {
            return "X:" + X.ToString( CultureInfo.InvariantCulture ) +
                  " Y:" + Y.ToString( CultureInfo.InvariantCulture ) +
                  " Z:" + Z.ToString( CultureInfo.InvariantCulture );
        }

        /// <summary>Read</summary>
        public void Read(Protocol4FReader reader)
        {
            X = reader.ReadSingle();
            Y = reader.ReadSingle();
            Z = reader.ReadSingle();
        }

        /// <summary>Write</summary>
        public void Write(Protocol4FWriter writer)
        {
            writer.Write(X);
            writer.Write(Y);
            writer.Write(Z);
        }
    };

    /// <summary>NVector4</summary>
    public class NVector4
    {
        /// <summary>X</summary>
        public float X{ get; set; }

        /// <summary>Y</summary>
        public float Y{ get; set; }

        /// <summary>Z</summary>
        public float Z{ get; set; }

        /// <summary>W</summary>
        public float W{ get; set; }

        /// <summary>Size</summary>
        public int Size{ get { return (4 * 4); } }

        /// <summary>コンストラクタ</summary>
        public NVector4()
        {
        }

        /// <summary>コンストラクタ</summary>
        public NVector4(NVector4 src)
        {
            X = src.X;
            Y = src.Y;
            Z = src.Z;
            W = src.W;
        }

        /// <summary>文字列化</summary>
        public override string ToString()
        {
            return "X:" + X.ToString( CultureInfo.InvariantCulture ) +
                  " Y:" + Y.ToString( CultureInfo.InvariantCulture ) +
                  " Z:" + Z.ToString( CultureInfo.InvariantCulture ) +
                  " W:" + W.ToString( CultureInfo.InvariantCulture );
        }

        /// <summary>Read</summary>
        public void Read(Protocol4FReader reader)
        {
            X = reader.ReadSingle();
            Y = reader.ReadSingle();
            Z = reader.ReadSingle();
            W = reader.ReadSingle();
        }

        /// <summary>Write</summary>
        public void Write(Protocol4FWriter writer)
        {
            writer.Write(X);
            writer.Write(Y);
            writer.Write(Z);
            writer.Write(W);
        }
    };

    /// <summary>ByteArray</summary>
    public class ByteArray
    {
        /// <summary>Length</summary>
        public int    Length { get; set; }

        /// <summary>Value</summary>
        public byte[] Value { get; set; }

        /// <summary>Size</summary>
        public int Size{ get { return (4 + Value.Length); } }

        /// <summary>コンストラクタ</summary>
        public ByteArray()
        {
            Value = new byte[1];
        }

        /// <summary>コンストラクタ</summary>
        public ByteArray(ByteArray src) : this()
        {
            SetValue(Value.Clone() as byte[]);
        }

        /// <summary>設定</summary>
        public void SetValue(byte[] data)
        {
            Length = data.Length;
            Value = data;
        }

        /// <summary>文字列化</summary>
        public override string ToString()
        {
            return "Length:" + Length.ToString();
        }

        /// <summary>Read</summary>
        public void Read(Protocol4FReader reader)
        {
            Length = reader.ReadInt32();
            Value = reader.ReadBytes(Length);
        }

        /// <summary>Write</summary>
        public void Write(Protocol4FWriter writer)
        {
            writer.Write(Length);
            writer.Write(Value);
        }
    };

    /// <summary>最後のパケットであることを示すパケット</summary>
    public class MESSAGEPACKET_File_EndOfPacket : BaseMessagePacket
    {
        /// <summary>メッセージタイプ</summary>
        public const MessageType MessagePacketType = MessageType.File_EndOfPacket;

        /// <summary>構造体にした場合のサイズ</summary>
        public const int StructSize = 0 + 4 + 4;   // + 4 は ModifiedProperties と ModifiedPropertyElements

        /// <summary>後ろに続く配列のサイズ(byte)</summary>
        public virtual int ArraySize{ get{ return 0; } }

        /// <summary>編集されたプロパティ</summary>
        public NUint    ModifiedProperties{ get; set; }

        /// <summary>編集されたプロパティ要素</summary>
        public NUint    ModifiedPropertyElements{ get; set; }


        /// <summary>コンストラクタ</summary>
        public MESSAGEPACKET_File_EndOfPacket()
        {
            ModifiedProperties = new NUint(0xFFFFFFFF);
            ModifiedPropertyElements = new NUint(0xFFFFFFFF);

        }

        /// <summary>文字列化</summary>
        public override string ToString()
        {
            string temp = string.Empty;
            return temp;
        }

        /// <summary>パケットサイズ（可変サイズに対応）</summary>
        public int PacketSize
        {
            get {
                int temp = (4 + 4);   // + 4 は ModifiedProperties と ModifiedPropertyElements
                temp += (4 - (temp & 0x03)) & 0x03;
                return temp;
            }
        }

        /// <summary>後に続く配列の長さ</summary>
        public override int ArrayLength
        {
            get{
                return 0;
            }
        }

        /// <summary>Read</summary>
        public override void Read(Protocol4FReader reader)
        {
             ModifiedProperties.Read(reader);
             ModifiedPropertyElements.Read(reader);
        }

        /// <summary>Write</summary>
        public override void Write(Protocol4FWriter writer)
        {
             ModifiedProperties.Write(writer);
             ModifiedPropertyElements.Write(writer);
        }
    };

    /// <summary>接続時に送信するパケット</summary>
    public class MESSAGEPACKET_File_Connect : BaseMessagePacket
    {
        /// <summary>メッセージタイプ</summary>
        public const MessageType MessagePacketType = MessageType.File_Connect;

        /// <summary>構造体にした場合のサイズ</summary>
        public const int StructSize = 64 + 4 + 4;   // + 4 は ModifiedProperties と ModifiedPropertyElements

        /// <summary>後ろに続く配列のサイズ(byte)</summary>
        public virtual int ArraySize{ get{ return 0; } }

        /// <summary>プロパティ</summary>
        [Flags]
        public enum PropertyFlag : uint
        {
            /// <summary>EffectMaker名（Viewer->EM4Fへの通信時に利用）</summary>
            EffectMakerName = 1<<0,

        }

        /// <summary>編集されたプロパティ</summary>
        public NUint    ModifiedProperties{ get; set; }

        /// <summary>編集されたプロパティ要素</summary>
        public NUint    ModifiedPropertyElements{ get; set; }

        /// <summary>EffectMaker名（Viewer->EM4Fへの通信時に利用）</summary>
        public NString32    EffectMakerName{ get; set; }


        /// <summary>コンストラクタ</summary>
        public MESSAGEPACKET_File_Connect()
        {
            ModifiedProperties = new NUint(0xFFFFFFFF);
            ModifiedPropertyElements = new NUint(0xFFFFFFFF);

            EffectMakerName = new NString32();
        }

        /// <summary>コンストラクタ</summary>
        public MESSAGEPACKET_File_Connect(NString32 _EffectMakerName)
        {
            ModifiedProperties = new NUint(0xFFFFFFFF);
            ModifiedPropertyElements = new NUint(0xFFFFFFFF);

            EffectMakerName = _EffectMakerName;
        }
        /// <summary>文字列化</summary>
        public override string ToString()
        {
            string temp = string.Empty;
            temp += "EffectMakerName:(" + EffectMakerName.ToString() + ") ";
            return temp;
        }

        /// <summary>パケットサイズ（可変サイズに対応）</summary>
        public int PacketSize
        {
            get {
                int temp = (4 + 4);   // + 4 は ModifiedProperties と ModifiedPropertyElements
                temp += EffectMakerName.Size;
                temp += (4 - (temp & 0x03)) & 0x03;
                return temp;
            }
        }

        /// <summary>後に続く配列の長さ</summary>
        public override int ArrayLength
        {
            get{
                return 0;
            }
        }

        /// <summary>Read</summary>
        public override void Read(Protocol4FReader reader)
        {
             ModifiedProperties.Read(reader);
             ModifiedPropertyElements.Read(reader);
             EffectMakerName.Read(reader);
        }

        /// <summary>Write</summary>
        public override void Write(Protocol4FWriter writer)
        {
             ModifiedProperties.Write(writer);
             ModifiedPropertyElements.Write(writer);
             EffectMakerName.Write(writer);
        }
    };

    /// <summary>すべてのファイルを閉じます。</summary>
    public class MESSAGEPACKET_File_CloseAll : BaseMessagePacket
    {
        /// <summary>メッセージタイプ</summary>
        public const MessageType MessagePacketType = MessageType.File_CloseAll;

        /// <summary>構造体にした場合のサイズ</summary>
        public const int StructSize = 0 + 4 + 4;   // + 4 は ModifiedProperties と ModifiedPropertyElements

        /// <summary>後ろに続く配列のサイズ(byte)</summary>
        public virtual int ArraySize{ get{ return 0; } }

        /// <summary>編集されたプロパティ</summary>
        public NUint    ModifiedProperties{ get; set; }

        /// <summary>編集されたプロパティ要素</summary>
        public NUint    ModifiedPropertyElements{ get; set; }


        /// <summary>コンストラクタ</summary>
        public MESSAGEPACKET_File_CloseAll()
        {
            ModifiedProperties = new NUint(0xFFFFFFFF);
            ModifiedPropertyElements = new NUint(0xFFFFFFFF);

        }

        /// <summary>文字列化</summary>
        public override string ToString()
        {
            string temp = string.Empty;
            return temp;
        }

        /// <summary>パケットサイズ（可変サイズに対応）</summary>
        public int PacketSize
        {
            get {
                int temp = (4 + 4);   // + 4 は ModifiedProperties と ModifiedPropertyElements
                temp += (4 - (temp & 0x03)) & 0x03;
                return temp;
            }
        }

        /// <summary>後に続く配列の長さ</summary>
        public override int ArrayLength
        {
            get{
                return 0;
            }
        }

        /// <summary>Read</summary>
        public override void Read(Protocol4FReader reader)
        {
             ModifiedProperties.Read(reader);
             ModifiedPropertyElements.Read(reader);
        }

        /// <summary>Write</summary>
        public override void Write(Protocol4FWriter writer)
        {
             ModifiedProperties.Write(writer);
             ModifiedPropertyElements.Write(writer);
        }
    };

    /// <summary>選択されたモデルを閉じます。</summary>
    public class MESSAGEPACKET_File_Close : BaseMessagePacket
    {
        /// <summary>メッセージタイプ</summary>
        public const MessageType MessagePacketType = MessageType.File_Close;

        /// <summary>構造体にした場合のサイズ</summary>
        public const int StructSize = 524 + 4 + 4;   // + 4 は ModifiedProperties と ModifiedPropertyElements

        /// <summary>後ろに続く配列のサイズ(byte)</summary>
        public virtual int ArraySize{ get{ return 0; } }

        /// <summary>プロパティ</summary>
        [Flags]
        public enum PropertyFlag : uint
        {
            /// <summary>タイプです。</summary>
            Type = 1<<0,

            /// <summary>対象のエミッタセット名</summary>
            EmitterSetName = 1<<1,

        }

        /// <summary>編集されたプロパティ</summary>
        public NUint    ModifiedProperties{ get; set; }

        /// <summary>編集されたプロパティ要素</summary>
        public NUint    ModifiedPropertyElements{ get; set; }

        /// <summary>タイプです。</summary>
        public BinaryDataType    Type{ get; set; }

        /// <summary>対象のエミッタセット名</summary>
        public NString    EmitterSetName{ get; set; }


        /// <summary>コンストラクタ</summary>
        public MESSAGEPACKET_File_Close()
        {
            ModifiedProperties = new NUint(0xFFFFFFFF);
            ModifiedPropertyElements = new NUint(0xFFFFFFFF);

            Type = new BinaryDataType();
            EmitterSetName = new NString();
        }

        /// <summary>コンストラクタ</summary>
        public MESSAGEPACKET_File_Close(BinaryDataType _Type, NString _EmitterSetName)
        {
            ModifiedProperties = new NUint(0xFFFFFFFF);
            ModifiedPropertyElements = new NUint(0xFFFFFFFF);

            Type = _Type;
            EmitterSetName = _EmitterSetName;
        }
        /// <summary>文字列化</summary>
        public override string ToString()
        {
            string temp = string.Empty;
            temp += "Type:(" + Type.ToString() + ") ";
            temp += "EmitterSetName:(" + EmitterSetName.ToString() + ") ";
            return temp;
        }

        /// <summary>パケットサイズ（可変サイズに対応）</summary>
        public int PacketSize
        {
            get {
                int temp = (4 + 4);   // + 4 は ModifiedProperties と ModifiedPropertyElements
                temp += 4;  //Type
                temp += EmitterSetName.Size;
                temp += (4 - (temp & 0x03)) & 0x03;
                return temp;
            }
        }

        /// <summary>後に続く配列の長さ</summary>
        public override int ArrayLength
        {
            get{
                return 0;
            }
        }

        /// <summary>Read</summary>
        public override void Read(Protocol4FReader reader)
        {
             ModifiedProperties.Read(reader);
             ModifiedPropertyElements.Read(reader);
             {
                 NInt temp = new NInt();
                 temp.Read(reader);
                 Type = (BinaryDataType)temp.Value;
             }
             EmitterSetName.Read(reader);
        }

        /// <summary>Write</summary>
        public override void Write(Protocol4FWriter writer)
        {
             ModifiedProperties.Write(writer);
             ModifiedPropertyElements.Write(writer);
             (new NInt((int)Type)).Write(writer);
             EmitterSetName.Write(writer);
        }
    };

    /// <summary>Viewer側からファイルを閉じます。</summary>
    public class MESSAGEPACKET_File_CloseV2E : BaseMessagePacket
    {
        /// <summary>メッセージタイプ</summary>
        public const MessageType MessagePacketType = MessageType.File_CloseV2E;

        /// <summary>構造体にした場合のサイズ</summary>
        public const int StructSize = 524 + 4 + 4;   // + 4 は ModifiedProperties と ModifiedPropertyElements

        /// <summary>後ろに続く配列のサイズ(byte)</summary>
        public virtual int ArraySize{ get{ return 0; } }

        /// <summary>プロパティ</summary>
        [Flags]
        public enum PropertyFlag : uint
        {
            /// <summary>タイプです。</summary>
            Type = 1<<0,

            /// <summary>対象のドキュメント名</summary>
            DocumentName = 1<<1,

        }

        /// <summary>編集されたプロパティ</summary>
        public NUint    ModifiedProperties{ get; set; }

        /// <summary>編集されたプロパティ要素</summary>
        public NUint    ModifiedPropertyElements{ get; set; }

        /// <summary>タイプです。</summary>
        public BinaryDataType    Type{ get; set; }

        /// <summary>対象のドキュメント名</summary>
        public NString    DocumentName{ get; set; }


        /// <summary>コンストラクタ</summary>
        public MESSAGEPACKET_File_CloseV2E()
        {
            ModifiedProperties = new NUint(0xFFFFFFFF);
            ModifiedPropertyElements = new NUint(0xFFFFFFFF);

            Type = new BinaryDataType();
            DocumentName = new NString();
        }

        /// <summary>コンストラクタ</summary>
        public MESSAGEPACKET_File_CloseV2E(BinaryDataType _Type, NString _DocumentName)
        {
            ModifiedProperties = new NUint(0xFFFFFFFF);
            ModifiedPropertyElements = new NUint(0xFFFFFFFF);

            Type = _Type;
            DocumentName = _DocumentName;
        }
        /// <summary>文字列化</summary>
        public override string ToString()
        {
            string temp = string.Empty;
            temp += "Type:(" + Type.ToString() + ") ";
            temp += "DocumentName:(" + DocumentName.ToString() + ") ";
            return temp;
        }

        /// <summary>パケットサイズ（可変サイズに対応）</summary>
        public int PacketSize
        {
            get {
                int temp = (4 + 4);   // + 4 は ModifiedProperties と ModifiedPropertyElements
                temp += 4;  //Type
                temp += DocumentName.Size;
                temp += (4 - (temp & 0x03)) & 0x03;
                return temp;
            }
        }

        /// <summary>後に続く配列の長さ</summary>
        public override int ArrayLength
        {
            get{
                return 0;
            }
        }

        /// <summary>Read</summary>
        public override void Read(Protocol4FReader reader)
        {
             ModifiedProperties.Read(reader);
             ModifiedPropertyElements.Read(reader);
             {
                 NInt temp = new NInt();
                 temp.Read(reader);
                 Type = (BinaryDataType)temp.Value;
             }
             DocumentName.Read(reader);
        }

        /// <summary>Write</summary>
        public override void Write(Protocol4FWriter writer)
        {
             ModifiedProperties.Write(writer);
             ModifiedPropertyElements.Write(writer);
             (new NInt((int)Type)).Write(writer);
             DocumentName.Write(writer);
        }
    };

    /// <summary>バイナリデータを読み込みます。</summary>
    public class MESSAGEPACKET_File_LoadBinaryData : BaseMessagePacket
    {
        /// <summary>メッセージタイプ</summary>
        public const MessageType MessagePacketType = MessageType.File_LoadBinaryData;

        /// <summary>構造体にした場合のサイズ</summary>
        public const int StructSize = 1056 + 4 + 4;   // + 4 は ModifiedProperties と ModifiedPropertyElements

        /// <summary>後ろに続く配列のサイズ(byte)</summary>
        public virtual int ArraySize{ get{ return 0; } }

        /// <summary>プロパティ</summary>
        [Flags]
        public enum PropertyFlag : uint
        {
            /// <summary>タイプです。</summary>
            Type = 1<<0,

            /// <summary>IDです。</summary>
            Id = 1<<1,

            /// <summary>ファイルサイズです。</summary>
            FileSize = 1<<2,

            /// <summary>ファイルパスです。</summary>
            FilePath = 1<<3,

            /// <summary>エミッタセット名</summary>
            EmitterSetName = 1<<4,

            /// <summary>エミッタセット名を識別するＩＤ</summary>
            EmitterSetID = 1<<5,

        }

        /// <summary>編集されたプロパティ</summary>
        public NUint    ModifiedProperties{ get; set; }

        /// <summary>編集されたプロパティ要素</summary>
        public NUint    ModifiedPropertyElements{ get; set; }

        /// <summary>タイプです。</summary>
        public BinaryDataType    Type{ get; set; }

        /// <summary>IDです。</summary>
        public NUint    Id{ get; set; }

        /// <summary>ファイルサイズです。</summary>
        public NUint    FileSize{ get; set; }

        /// <summary>ファイルパスです。</summary>
        public NString    FilePath{ get; set; }

        /// <summary>エミッタセット名</summary>
        public NString    EmitterSetName{ get; set; }

        /// <summary>エミッタセット名を識別するＩＤ</summary>
        public NInt    EmitterSetID{ get; set; }


        /// <summary>コンストラクタ</summary>
        public MESSAGEPACKET_File_LoadBinaryData()
        {
            ModifiedProperties = new NUint(0xFFFFFFFF);
            ModifiedPropertyElements = new NUint(0xFFFFFFFF);

            Type = new BinaryDataType();
            Id = new NUint();
            FileSize = new NUint();
            FilePath = new NString();
            EmitterSetName = new NString();
            EmitterSetID = new NInt();
        }

        /// <summary>コンストラクタ</summary>
        public MESSAGEPACKET_File_LoadBinaryData(BinaryDataType _Type, NUint _Id, NUint _FileSize, NString _FilePath, NString _EmitterSetName, NInt _EmitterSetID)
        {
            ModifiedProperties = new NUint(0xFFFFFFFF);
            ModifiedPropertyElements = new NUint(0xFFFFFFFF);

            Type = _Type;
            Id = _Id;
            FileSize = _FileSize;
            FilePath = _FilePath;
            EmitterSetName = _EmitterSetName;
            EmitterSetID = _EmitterSetID;
        }
        /// <summary>文字列化</summary>
        public override string ToString()
        {
            string temp = string.Empty;
            temp += "Type:(" + Type.ToString() + ") ";
            temp += "Id:(" + Id.ToString() + ") ";
            temp += "FileSize:(" + FileSize.ToString() + ") ";
            temp += "FilePath:(" + FilePath.ToString() + ") ";
            temp += "EmitterSetName:(" + EmitterSetName.ToString() + ") ";
            temp += "EmitterSetID:(" + EmitterSetID.ToString() + ") ";
            return temp;
        }

        /// <summary>パケットサイズ（可変サイズに対応）</summary>
        public int PacketSize
        {
            get {
                int temp = (4 + 4);   // + 4 は ModifiedProperties と ModifiedPropertyElements
                temp += 4;  //Type
                temp += Id.Size;
                temp += FileSize.Size;
                temp += FilePath.Size;
                temp += EmitterSetName.Size;
                temp += EmitterSetID.Size;
                temp += (4 - (temp & 0x03)) & 0x03;
                return temp;
            }
        }

        /// <summary>後に続く配列の長さ</summary>
        public override int ArrayLength
        {
            get{
                return 0;
            }
        }

        /// <summary>Read</summary>
        public override void Read(Protocol4FReader reader)
        {
             ModifiedProperties.Read(reader);
             ModifiedPropertyElements.Read(reader);
             {
                 NInt temp = new NInt();
                 temp.Read(reader);
                 Type = (BinaryDataType)temp.Value;
             }
             Id.Read(reader);
             FileSize.Read(reader);
             FilePath.Read(reader);
             EmitterSetName.Read(reader);
             EmitterSetID.Read(reader);
        }

        /// <summary>Write</summary>
        public override void Write(Protocol4FWriter writer)
        {
             ModifiedProperties.Write(writer);
             ModifiedPropertyElements.Write(writer);
             (new NInt((int)Type)).Write(writer);
             Id.Write(writer);
             FileSize.Write(writer);
             FilePath.Write(writer);
             EmitterSetName.Write(writer);
             EmitterSetID.Write(writer);
        }
    };

    /// <summary>指定ファイルの読み込みをビューア→EffectMakerへリクエストします。</summary>
    public class MESSAGEPACKET_File_LoadFileV2E : BaseMessagePacket
    {
        /// <summary>メッセージタイプ</summary>
        public const MessageType MessagePacketType = MessageType.File_LoadFileV2E;

        /// <summary>構造体にした場合のサイズ</summary>
        public const int StructSize = 520 + 4 + 4;   // + 4 は ModifiedProperties と ModifiedPropertyElements

        /// <summary>後ろに続く配列のサイズ(byte)</summary>
        public virtual int ArraySize{ get{ return 0; } }

        /// <summary>プロパティ</summary>
        [Flags]
        public enum PropertyFlag : uint
        {
            /// <summary>ファイルパスです。</summary>
            FilePath = 1<<0,

        }

        /// <summary>編集されたプロパティ</summary>
        public NUint    ModifiedProperties{ get; set; }

        /// <summary>編集されたプロパティ要素</summary>
        public NUint    ModifiedPropertyElements{ get; set; }

        /// <summary>ファイルパスです。</summary>
        public NString    FilePath{ get; set; }


        /// <summary>コンストラクタ</summary>
        public MESSAGEPACKET_File_LoadFileV2E()
        {
            ModifiedProperties = new NUint(0xFFFFFFFF);
            ModifiedPropertyElements = new NUint(0xFFFFFFFF);

            FilePath = new NString();
        }

        /// <summary>コンストラクタ</summary>
        public MESSAGEPACKET_File_LoadFileV2E(NString _FilePath)
        {
            ModifiedProperties = new NUint(0xFFFFFFFF);
            ModifiedPropertyElements = new NUint(0xFFFFFFFF);

            FilePath = _FilePath;
        }
        /// <summary>文字列化</summary>
        public override string ToString()
        {
            string temp = string.Empty;
            temp += "FilePath:(" + FilePath.ToString() + ") ";
            return temp;
        }

        /// <summary>パケットサイズ（可変サイズに対応）</summary>
        public int PacketSize
        {
            get {
                int temp = (4 + 4);   // + 4 は ModifiedProperties と ModifiedPropertyElements
                temp += FilePath.Size;
                temp += (4 - (temp & 0x03)) & 0x03;
                return temp;
            }
        }

        /// <summary>後に続く配列の長さ</summary>
        public override int ArrayLength
        {
            get{
                return 0;
            }
        }

        /// <summary>Read</summary>
        public override void Read(Protocol4FReader reader)
        {
             ModifiedProperties.Read(reader);
             ModifiedPropertyElements.Read(reader);
             FilePath.Read(reader);
        }

        /// <summary>Write</summary>
        public override void Write(Protocol4FWriter writer)
        {
             ModifiedProperties.Write(writer);
             ModifiedPropertyElements.Write(writer);
             FilePath.Write(writer);
        }
    };

    /// <summary>読み込んだファイルを Byte[] としてEffectMaker→ビューアへ送ります。</summary>
    public class MESSAGEPACKET_File_LoadFileE2V : BaseMessagePacket
    {
        /// <summary>メッセージタイプ</summary>
        public const MessageType MessagePacketType = MessageType.File_LoadFileE2V;

        /// <summary>構造体にした場合のサイズ</summary>
        public const int StructSize = 528 + 4 + 4;   // + 4 は ModifiedProperties と ModifiedPropertyElements

        /// <summary>後ろに続く配列のサイズ(byte)</summary>
        public virtual int ArraySize{ get{ return 0; } }

        /// <summary>プロパティ</summary>
        [Flags]
        public enum PropertyFlag : uint
        {
            /// <summary>読み込んだファイルパスです。</summary>
            FilePath = 1<<0,

            /// <summary>データ（&Data からLength分データが続く）</summary>
            Data = 1<<1,

        }

        /// <summary>編集されたプロパティ</summary>
        public NUint    ModifiedProperties{ get; set; }

        /// <summary>編集されたプロパティ要素</summary>
        public NUint    ModifiedPropertyElements{ get; set; }

        /// <summary>読み込んだファイルパスです。</summary>
        public NString    FilePath{ get; set; }

        /// <summary>データ（&Data からLength分データが続く）</summary>
        public ByteArray    Data{ get; set; }


        /// <summary>コンストラクタ</summary>
        public MESSAGEPACKET_File_LoadFileE2V()
        {
            ModifiedProperties = new NUint(0xFFFFFFFF);
            ModifiedPropertyElements = new NUint(0xFFFFFFFF);

            FilePath = new NString();
            Data = new ByteArray();
        }

        /// <summary>コンストラクタ</summary>
        public MESSAGEPACKET_File_LoadFileE2V(NString _FilePath, ByteArray _Data)
        {
            ModifiedProperties = new NUint(0xFFFFFFFF);
            ModifiedPropertyElements = new NUint(0xFFFFFFFF);

            FilePath = _FilePath;
            Data = _Data;
        }
        /// <summary>文字列化</summary>
        public override string ToString()
        {
            string temp = string.Empty;
            temp += "FilePath:(" + FilePath.ToString() + ") ";
            temp += "Data:(" + Data.ToString() + ") ";
            return temp;
        }

        /// <summary>パケットサイズ（可変サイズに対応）</summary>
        public int PacketSize
        {
            get {
                int temp = (4 + 4);   // + 4 は ModifiedProperties と ModifiedPropertyElements
                temp += FilePath.Size;
                temp += Data.Size;
                temp += (4 - (temp & 0x03)) & 0x03;
                return temp;
            }
        }

        /// <summary>後に続く配列の長さ</summary>
        public override int ArrayLength
        {
            get{
                return 0;
            }
        }

        /// <summary>Read</summary>
        public override void Read(Protocol4FReader reader)
        {
             ModifiedProperties.Read(reader);
             ModifiedPropertyElements.Read(reader);
             FilePath.Read(reader);
             Data.Read(reader);
        }

        /// <summary>Write</summary>
        public override void Write(Protocol4FWriter writer)
        {
             ModifiedProperties.Write(writer);
             ModifiedPropertyElements.Write(writer);
             FilePath.Write(writer);
             Data.Write(writer);
        }
    };

    /// <summary>Viewer側で利用しているEmitterSetをEffectMaker側へ通知</summary>
    public class MESSAGEPACKET_File_RequireEmitterSetV2E : BaseMessagePacket
    {
        /// <summary>メッセージタイプ</summary>
        public const MessageType MessagePacketType = MessageType.File_RequireEmitterSetV2E;

        /// <summary>構造体にした場合のサイズ</summary>
        public const int StructSize = 528 + 4 + 4;   // + 4 は ModifiedProperties と ModifiedPropertyElements

        /// <summary>後ろに続く配列のサイズ(byte)</summary>
        public virtual int ArraySize{ get{ return 0; } }

        /// <summary>プロパティ</summary>
        [Flags]
        public enum PropertyFlag : uint
        {
            /// <summary>EffectMaker名／ファイルのフルパス</summary>
            EmitterSetName = 1<<0,

            /// <summary>EmitterSetNameの文字列タイプ（Name/Path）</summary>
            StrType = 1<<1,

            /// <summary>要求するタイプ</summary>
            BinType = 1<<2,

        }

        /// <summary>編集されたプロパティ</summary>
        public NUint    ModifiedProperties{ get; set; }

        /// <summary>編集されたプロパティ要素</summary>
        public NUint    ModifiedPropertyElements{ get; set; }

        /// <summary>EffectMaker名／ファイルのフルパス</summary>
        public NString    EmitterSetName{ get; set; }

        /// <summary>EmitterSetNameの文字列タイプ（Name/Path）</summary>
        public StringType    StrType{ get; set; }

        /// <summary>要求するタイプ</summary>
        public BinaryDataType    BinType{ get; set; }


        /// <summary>コンストラクタ</summary>
        public MESSAGEPACKET_File_RequireEmitterSetV2E()
        {
            ModifiedProperties = new NUint(0xFFFFFFFF);
            ModifiedPropertyElements = new NUint(0xFFFFFFFF);

            EmitterSetName = new NString();
            StrType = new StringType();
            BinType = new BinaryDataType();
        }

        /// <summary>コンストラクタ</summary>
        public MESSAGEPACKET_File_RequireEmitterSetV2E(NString _EmitterSetName, StringType _StrType, BinaryDataType _BinType)
        {
            ModifiedProperties = new NUint(0xFFFFFFFF);
            ModifiedPropertyElements = new NUint(0xFFFFFFFF);

            EmitterSetName = _EmitterSetName;
            StrType = _StrType;
            BinType = _BinType;
        }
        /// <summary>文字列化</summary>
        public override string ToString()
        {
            string temp = string.Empty;
            temp += "EmitterSetName:(" + EmitterSetName.ToString() + ") ";
            temp += "StrType:(" + StrType.ToString() + ") ";
            temp += "BinType:(" + BinType.ToString() + ") ";
            return temp;
        }

        /// <summary>パケットサイズ（可変サイズに対応）</summary>
        public int PacketSize
        {
            get {
                int temp = (4 + 4);   // + 4 は ModifiedProperties と ModifiedPropertyElements
                temp += EmitterSetName.Size;
                temp += 4;  //StrType
                temp += 4;  //BinType
                temp += (4 - (temp & 0x03)) & 0x03;
                return temp;
            }
        }

        /// <summary>後に続く配列の長さ</summary>
        public override int ArrayLength
        {
            get{
                return 0;
            }
        }

        /// <summary>Read</summary>
        public override void Read(Protocol4FReader reader)
        {
             ModifiedProperties.Read(reader);
             ModifiedPropertyElements.Read(reader);
             EmitterSetName.Read(reader);
             {
                 NInt temp = new NInt();
                 temp.Read(reader);
                 StrType = (StringType)temp.Value;
             }
             {
                 NInt temp = new NInt();
                 temp.Read(reader);
                 BinType = (BinaryDataType)temp.Value;
             }
        }

        /// <summary>Write</summary>
        public override void Write(Protocol4FWriter writer)
        {
             ModifiedProperties.Write(writer);
             ModifiedPropertyElements.Write(writer);
             EmitterSetName.Write(writer);
             (new NInt((int)StrType)).Write(writer);
             (new NInt((int)BinType)).Write(writer);
        }
    };

    /// <summary>接続時にビューア情報をEffectMakerへ通知するパケット</summary>
    public class MESSAGEPACKET_File_ConnectInfoV2E : BaseMessagePacket
    {
        /// <summary>メッセージタイプ</summary>
        public const MessageType MessagePacketType = MessageType.File_ConnectInfoV2E;

        /// <summary>構造体にした場合のサイズ</summary>
        public const int StructSize = 4 + 4 + 4;   // + 4 は ModifiedProperties と ModifiedPropertyElements

        /// <summary>後ろに続く配列のサイズ(byte)</summary>
        public virtual int ArraySize{ get{ return 0; } }

        /// <summary>プロパティ</summary>
        [Flags]
        public enum PropertyFlag : uint
        {
            /// <summary>文字コードのエンコーディングタイプ</summary>
            EncodingType = 1<<0,

        }

        /// <summary>編集されたプロパティ</summary>
        public NUint    ModifiedProperties{ get; set; }

        /// <summary>編集されたプロパティ要素</summary>
        public NUint    ModifiedPropertyElements{ get; set; }

        /// <summary>文字コードのエンコーディングタイプ</summary>
        public ViewEncodingType    EncodingType{ get; set; }


        /// <summary>コンストラクタ</summary>
        public MESSAGEPACKET_File_ConnectInfoV2E()
        {
            ModifiedProperties = new NUint(0xFFFFFFFF);
            ModifiedPropertyElements = new NUint(0xFFFFFFFF);

            EncodingType = new ViewEncodingType();
        }

        /// <summary>コンストラクタ</summary>
        public MESSAGEPACKET_File_ConnectInfoV2E(ViewEncodingType _EncodingType)
        {
            ModifiedProperties = new NUint(0xFFFFFFFF);
            ModifiedPropertyElements = new NUint(0xFFFFFFFF);

            EncodingType = _EncodingType;
        }
        /// <summary>文字列化</summary>
        public override string ToString()
        {
            string temp = string.Empty;
            temp += "EncodingType:(" + EncodingType.ToString() + ") ";
            return temp;
        }

        /// <summary>パケットサイズ（可変サイズに対応）</summary>
        public int PacketSize
        {
            get {
                int temp = (4 + 4);   // + 4 は ModifiedProperties と ModifiedPropertyElements
                temp += 4;  //EncodingType
                temp += (4 - (temp & 0x03)) & 0x03;
                return temp;
            }
        }

        /// <summary>後に続く配列の長さ</summary>
        public override int ArrayLength
        {
            get{
                return 0;
            }
        }

        /// <summary>Read</summary>
        public override void Read(Protocol4FReader reader)
        {
             ModifiedProperties.Read(reader);
             ModifiedPropertyElements.Read(reader);
             {
                 NInt temp = new NInt();
                 temp.Read(reader);
                 EncodingType = (ViewEncodingType)temp.Value;
             }
        }

        /// <summary>Write</summary>
        public override void Write(Protocol4FWriter writer)
        {
             ModifiedProperties.Write(writer);
             ModifiedPropertyElements.Write(writer);
             (new NInt((int)EncodingType)).Write(writer);
        }
    };

    /// <summary>Viewer側からエミッタセットのリンク開始を通知するパケット</summary>
    public class MESSAGEPACKET_File_LinkEmitterSetV2E : BaseMessagePacket
    {
        /// <summary>メッセージタイプ</summary>
        public const MessageType MessagePacketType = MessageType.File_LinkEmitterSetV2E;

        /// <summary>構造体にした場合のサイズ</summary>
        public const int StructSize = 524 + 4 + 4;   // + 4 は ModifiedProperties と ModifiedPropertyElements

        /// <summary>後ろに続く配列のサイズ(byte)</summary>
        public virtual int ArraySize{ get{ return 0; } }

        /// <summary>プロパティ</summary>
        [Flags]
        public enum PropertyFlag : uint
        {
            /// <summary>エミッタセット名</summary>
            EmitterSetName = 1<<0,

            /// <summary>エミッタセット名を識別するＩＤ</summary>
            EmitterSetID = 1<<1,

        }

        /// <summary>編集されたプロパティ</summary>
        public NUint    ModifiedProperties{ get; set; }

        /// <summary>編集されたプロパティ要素</summary>
        public NUint    ModifiedPropertyElements{ get; set; }

        /// <summary>エミッタセット名</summary>
        public NString    EmitterSetName{ get; set; }

        /// <summary>エミッタセット名を識別するＩＤ</summary>
        public NInt    EmitterSetID{ get; set; }


        /// <summary>コンストラクタ</summary>
        public MESSAGEPACKET_File_LinkEmitterSetV2E()
        {
            ModifiedProperties = new NUint(0xFFFFFFFF);
            ModifiedPropertyElements = new NUint(0xFFFFFFFF);

            EmitterSetName = new NString();
            EmitterSetID = new NInt();
        }

        /// <summary>コンストラクタ</summary>
        public MESSAGEPACKET_File_LinkEmitterSetV2E(NString _EmitterSetName, NInt _EmitterSetID)
        {
            ModifiedProperties = new NUint(0xFFFFFFFF);
            ModifiedPropertyElements = new NUint(0xFFFFFFFF);

            EmitterSetName = _EmitterSetName;
            EmitterSetID = _EmitterSetID;
        }
        /// <summary>文字列化</summary>
        public override string ToString()
        {
            string temp = string.Empty;
            temp += "EmitterSetName:(" + EmitterSetName.ToString() + ") ";
            temp += "EmitterSetID:(" + EmitterSetID.ToString() + ") ";
            return temp;
        }

        /// <summary>パケットサイズ（可変サイズに対応）</summary>
        public int PacketSize
        {
            get {
                int temp = (4 + 4);   // + 4 は ModifiedProperties と ModifiedPropertyElements
                temp += EmitterSetName.Size;
                temp += EmitterSetID.Size;
                temp += (4 - (temp & 0x03)) & 0x03;
                return temp;
            }
        }

        /// <summary>後に続く配列の長さ</summary>
        public override int ArrayLength
        {
            get{
                return 0;
            }
        }

        /// <summary>Read</summary>
        public override void Read(Protocol4FReader reader)
        {
             ModifiedProperties.Read(reader);
             ModifiedPropertyElements.Read(reader);
             EmitterSetName.Read(reader);
             EmitterSetID.Read(reader);
        }

        /// <summary>Write</summary>
        public override void Write(Protocol4FWriter writer)
        {
             ModifiedProperties.Write(writer);
             ModifiedPropertyElements.Write(writer);
             EmitterSetName.Write(writer);
             EmitterSetID.Write(writer);
        }
    };

    /// <summary>Viewer側からエミッタセットのリンク解除を通知するパケット</summary>
    public class MESSAGEPACKET_File_UnlinkEmitterSetV2E : BaseMessagePacket
    {
        /// <summary>メッセージタイプ</summary>
        public const MessageType MessagePacketType = MessageType.File_UnlinkEmitterSetV2E;

        /// <summary>構造体にした場合のサイズ</summary>
        public const int StructSize = 524 + 4 + 4;   // + 4 は ModifiedProperties と ModifiedPropertyElements

        /// <summary>後ろに続く配列のサイズ(byte)</summary>
        public virtual int ArraySize{ get{ return 0; } }

        /// <summary>プロパティ</summary>
        [Flags]
        public enum PropertyFlag : uint
        {
            /// <summary>エミッタセット名</summary>
            EmitterSetName = 1<<0,

            /// <summary>エミッタセット名を識別するＩＤ</summary>
            EmitterSetID = 1<<1,

        }

        /// <summary>編集されたプロパティ</summary>
        public NUint    ModifiedProperties{ get; set; }

        /// <summary>編集されたプロパティ要素</summary>
        public NUint    ModifiedPropertyElements{ get; set; }

        /// <summary>エミッタセット名</summary>
        public NString    EmitterSetName{ get; set; }

        /// <summary>エミッタセット名を識別するＩＤ</summary>
        public NInt    EmitterSetID{ get; set; }


        /// <summary>コンストラクタ</summary>
        public MESSAGEPACKET_File_UnlinkEmitterSetV2E()
        {
            ModifiedProperties = new NUint(0xFFFFFFFF);
            ModifiedPropertyElements = new NUint(0xFFFFFFFF);

            EmitterSetName = new NString();
            EmitterSetID = new NInt();
        }

        /// <summary>コンストラクタ</summary>
        public MESSAGEPACKET_File_UnlinkEmitterSetV2E(NString _EmitterSetName, NInt _EmitterSetID)
        {
            ModifiedProperties = new NUint(0xFFFFFFFF);
            ModifiedPropertyElements = new NUint(0xFFFFFFFF);

            EmitterSetName = _EmitterSetName;
            EmitterSetID = _EmitterSetID;
        }
        /// <summary>文字列化</summary>
        public override string ToString()
        {
            string temp = string.Empty;
            temp += "EmitterSetName:(" + EmitterSetName.ToString() + ") ";
            temp += "EmitterSetID:(" + EmitterSetID.ToString() + ") ";
            return temp;
        }

        /// <summary>パケットサイズ（可変サイズに対応）</summary>
        public int PacketSize
        {
            get {
                int temp = (4 + 4);   // + 4 は ModifiedProperties と ModifiedPropertyElements
                temp += EmitterSetName.Size;
                temp += EmitterSetID.Size;
                temp += (4 - (temp & 0x03)) & 0x03;
                return temp;
            }
        }

        /// <summary>後に続く配列の長さ</summary>
        public override int ArrayLength
        {
            get{
                return 0;
            }
        }

        /// <summary>Read</summary>
        public override void Read(Protocol4FReader reader)
        {
             ModifiedProperties.Read(reader);
             ModifiedPropertyElements.Read(reader);
             EmitterSetName.Read(reader);
             EmitterSetID.Read(reader);
        }

        /// <summary>Write</summary>
        public override void Write(Protocol4FWriter writer)
        {
             ModifiedProperties.Write(writer);
             ModifiedPropertyElements.Write(writer);
             EmitterSetName.Write(writer);
             EmitterSetID.Write(writer);
        }
    };

    /// <summary>選択しているエミッタセットのデータを送信します。</summary>
    public class MESSAGEPACKET_File_SendEmitterSet : BaseMessagePacket
    {
        /// <summary>メッセージタイプ</summary>
        public const MessageType MessagePacketType = MessageType.File_SendEmitterSet;

        /// <summary>構造体にした場合のサイズ</summary>
        public const int StructSize = 540 + 4 + 4;   // + 4 は ModifiedProperties と ModifiedPropertyElements

        /// <summary>後ろに続く配列のサイズ(byte)</summary>
        public virtual int ArraySize{ get{ return 0; } }

        /// <summary>プロパティ</summary>
        [Flags]
        public enum PropertyFlag : uint
        {
            /// <summary>IDです。</summary>
            Id = 1<<0,

            /// <summary>エミッタセット名を識別するＩＤ</summary>
            EmitterSetID = 1<<1,

            /// <summary>エミッタセット名</summary>
            EmitterSetName = 1<<2,

            /// <summary>バイナリデータのサイズです。</summary>
            DataSize = 1<<3,

            /// <summary>エミッタセットのバイナリデータです。</summary>
            Data = 1<<4,

        }

        /// <summary>編集されたプロパティ</summary>
        public NUint    ModifiedProperties{ get; set; }

        /// <summary>編集されたプロパティ要素</summary>
        public NUint    ModifiedPropertyElements{ get; set; }

        /// <summary>IDです。</summary>
        public NUint    Id{ get; set; }

        /// <summary>エミッタセット名を識別するＩＤ</summary>
        public NInt    EmitterSetID{ get; set; }

        /// <summary>エミッタセット名</summary>
        public NString    EmitterSetName{ get; set; }

        /// <summary>バイナリデータのサイズです。</summary>
        public NUint    DataSize{ get; set; }

        /// <summary>エミッタセットのバイナリデータです。</summary>
        public ByteArray    Data{ get; set; }


        /// <summary>コンストラクタ</summary>
        public MESSAGEPACKET_File_SendEmitterSet()
        {
            ModifiedProperties = new NUint(0xFFFFFFFF);
            ModifiedPropertyElements = new NUint(0xFFFFFFFF);

            Id = new NUint();
            EmitterSetID = new NInt();
            EmitterSetName = new NString();
            DataSize = new NUint();
            Data = new ByteArray();
        }

        /// <summary>コンストラクタ</summary>
        public MESSAGEPACKET_File_SendEmitterSet(NUint _Id, NInt _EmitterSetID, NString _EmitterSetName, NUint _DataSize, ByteArray _Data)
        {
            ModifiedProperties = new NUint(0xFFFFFFFF);
            ModifiedPropertyElements = new NUint(0xFFFFFFFF);

            Id = _Id;
            EmitterSetID = _EmitterSetID;
            EmitterSetName = _EmitterSetName;
            DataSize = _DataSize;
            Data = _Data;
        }
        /// <summary>文字列化</summary>
        public override string ToString()
        {
            string temp = string.Empty;
            temp += "Id:(" + Id.ToString() + ") ";
            temp += "EmitterSetID:(" + EmitterSetID.ToString() + ") ";
            temp += "EmitterSetName:(" + EmitterSetName.ToString() + ") ";
            temp += "DataSize:(" + DataSize.ToString() + ") ";
            temp += "Data:(" + Data.ToString() + ") ";
            return temp;
        }

        /// <summary>パケットサイズ（可変サイズに対応）</summary>
        public int PacketSize
        {
            get {
                int temp = (4 + 4);   // + 4 は ModifiedProperties と ModifiedPropertyElements
                temp += Id.Size;
                temp += EmitterSetID.Size;
                temp += EmitterSetName.Size;
                temp += DataSize.Size;
                temp += Data.Size;
                temp += (4 - (temp & 0x03)) & 0x03;
                return temp;
            }
        }

        /// <summary>後に続く配列の長さ</summary>
        public override int ArrayLength
        {
            get{
                return 0;
            }
        }

        /// <summary>Read</summary>
        public override void Read(Protocol4FReader reader)
        {
             ModifiedProperties.Read(reader);
             ModifiedPropertyElements.Read(reader);
             Id.Read(reader);
             EmitterSetID.Read(reader);
             EmitterSetName.Read(reader);
             DataSize.Read(reader);
             Data.Read(reader);
        }

        /// <summary>Write</summary>
        public override void Write(Protocol4FWriter writer)
        {
             ModifiedProperties.Write(writer);
             ModifiedPropertyElements.Write(writer);
             Id.Write(writer);
             EmitterSetID.Write(writer);
             EmitterSetName.Write(writer);
             DataSize.Write(writer);
             Data.Write(writer);
        }
    };

    /// <summary>編集ターゲットを選択します。</summary>
    public class MESSAGEPACKET_Object_SelectEM : BaseMessagePacket
    {
        /// <summary>メッセージタイプ</summary>
        public const MessageType MessagePacketType = MessageType.Object_SelectEM;

        /// <summary>構造体にした場合のサイズ</summary>
        public const int StructSize = 532 + 4 + 4;   // + 4 は ModifiedProperties と ModifiedPropertyElements

        /// <summary>後ろに続く配列のサイズ(byte)</summary>
        public virtual int ArraySize{ get{ return 0; } }

        /// <summary>プロパティ</summary>
        [Flags]
        public enum PropertyFlag : uint
        {
            /// <summary>オブジェクトタイプです。</summary>
            ObjectType = 1<<0,

            /// <summary>編集ターゲットタイプです。</summary>
            TargetType = 1<<1,

            /// <summary>バイナリーのハッシュです。</summary>
            Id = 1<<2,

            /// <summary>編集ターゲット名です。</summary>
            TargetName = 1<<3,

        }

        /// <summary>編集されたプロパティ</summary>
        public NUint    ModifiedProperties{ get; set; }

        /// <summary>編集されたプロパティ要素</summary>
        public NUint    ModifiedPropertyElements{ get; set; }

        /// <summary>オブジェクトタイプです。</summary>
        public BinaryDataType    ObjectType{ get; set; }

        /// <summary>編集ターゲットタイプです。</summary>
        public EditTargetType    TargetType{ get; set; }

        /// <summary>バイナリーのハッシュです。</summary>
        public NUint    Id{ get; set; }

        /// <summary>編集ターゲット名です。</summary>
        public NString    TargetName{ get; set; }


        /// <summary>コンストラクタ</summary>
        public MESSAGEPACKET_Object_SelectEM()
        {
            ModifiedProperties = new NUint(0xFFFFFFFF);
            ModifiedPropertyElements = new NUint(0xFFFFFFFF);

            ObjectType = new BinaryDataType();
            TargetType = new EditTargetType();
            Id = new NUint();
            TargetName = new NString();
        }

        /// <summary>コンストラクタ</summary>
        public MESSAGEPACKET_Object_SelectEM(BinaryDataType _ObjectType, EditTargetType _TargetType, NUint _Id, NString _TargetName)
        {
            ModifiedProperties = new NUint(0xFFFFFFFF);
            ModifiedPropertyElements = new NUint(0xFFFFFFFF);

            ObjectType = _ObjectType;
            TargetType = _TargetType;
            Id = _Id;
            TargetName = _TargetName;
        }
        /// <summary>文字列化</summary>
        public override string ToString()
        {
            string temp = string.Empty;
            temp += "ObjectType:(" + ObjectType.ToString() + ") ";
            temp += "TargetType:(" + TargetType.ToString() + ") ";
            temp += "Id:(" + Id.ToString() + ") ";
            temp += "TargetName:(" + TargetName.ToString() + ") ";
            return temp;
        }

        /// <summary>パケットサイズ（可変サイズに対応）</summary>
        public int PacketSize
        {
            get {
                int temp = (4 + 4);   // + 4 は ModifiedProperties と ModifiedPropertyElements
                temp += 4;  //ObjectType
                temp += 4;  //TargetType
                temp += Id.Size;
                temp += TargetName.Size;
                temp += (4 - (temp & 0x03)) & 0x03;
                return temp;
            }
        }

        /// <summary>後に続く配列の長さ</summary>
        public override int ArrayLength
        {
            get{
                return 0;
            }
        }

        /// <summary>Read</summary>
        public override void Read(Protocol4FReader reader)
        {
             ModifiedProperties.Read(reader);
             ModifiedPropertyElements.Read(reader);
             {
                 NInt temp = new NInt();
                 temp.Read(reader);
                 ObjectType = (BinaryDataType)temp.Value;
             }
             {
                 NInt temp = new NInt();
                 temp.Read(reader);
                 TargetType = (EditTargetType)temp.Value;
             }
             Id.Read(reader);
             TargetName.Read(reader);
        }

        /// <summary>Write</summary>
        public override void Write(Protocol4FWriter writer)
        {
             ModifiedProperties.Write(writer);
             ModifiedPropertyElements.Write(writer);
             (new NInt((int)ObjectType)).Write(writer);
             (new NInt((int)TargetType)).Write(writer);
             Id.Write(writer);
             TargetName.Write(writer);
        }
    };

    /// <summary>選択しているエミッタのデータを送信します。</summary>
    public class MESSAGEPACKET_Emitter_SendEmitter : BaseMessagePacket
    {
        /// <summary>メッセージタイプ</summary>
        public const MessageType MessagePacketType = MessageType.Emitter_SendEmitter;

        /// <summary>構造体にした場合のサイズ</summary>
        public const int StructSize = 1052 + 4 + 4;   // + 4 は ModifiedProperties と ModifiedPropertyElements

        /// <summary>後ろに続く配列のサイズ(byte)</summary>
        public virtual int ArraySize{ get{ return 0; } }

        /// <summary>プロパティ</summary>
        [Flags]
        public enum PropertyFlag : uint
        {
            /// <summary>対象のエミッタセット名</summary>
            EmitterSetName = 1<<0,

            /// <summary>対象のエミッタ名</summary>
            EmitterName = 1<<1,

            /// <summary>対象のエミッタインデックス</summary>
            EmitterIndex = 1<<2,

            /// <summary>データ（&Data からLength分データが続く）</summary>
            Data = 1<<3,

        }

        /// <summary>編集されたプロパティ</summary>
        public NUint    ModifiedProperties{ get; set; }

        /// <summary>編集されたプロパティ要素</summary>
        public NUint    ModifiedPropertyElements{ get; set; }

        /// <summary>対象のエミッタセット名</summary>
        public NString    EmitterSetName{ get; set; }

        /// <summary>対象のエミッタ名</summary>
        public NString    EmitterName{ get; set; }

        /// <summary>対象のエミッタインデックス</summary>
        public NUint    EmitterIndex{ get; set; }

        /// <summary>データ（&Data からLength分データが続く）</summary>
        public ByteArray    Data{ get; set; }


        /// <summary>コンストラクタ</summary>
        public MESSAGEPACKET_Emitter_SendEmitter()
        {
            ModifiedProperties = new NUint(0xFFFFFFFF);
            ModifiedPropertyElements = new NUint(0xFFFFFFFF);

            EmitterSetName = new NString();
            EmitterName = new NString();
            EmitterIndex = new NUint();
            Data = new ByteArray();
        }

        /// <summary>コンストラクタ</summary>
        public MESSAGEPACKET_Emitter_SendEmitter(NString _EmitterSetName, NString _EmitterName, NUint _EmitterIndex, ByteArray _Data)
        {
            ModifiedProperties = new NUint(0xFFFFFFFF);
            ModifiedPropertyElements = new NUint(0xFFFFFFFF);

            EmitterSetName = _EmitterSetName;
            EmitterName = _EmitterName;
            EmitterIndex = _EmitterIndex;
            Data = _Data;
        }
        /// <summary>文字列化</summary>
        public override string ToString()
        {
            string temp = string.Empty;
            temp += "EmitterSetName:(" + EmitterSetName.ToString() + ") ";
            temp += "EmitterName:(" + EmitterName.ToString() + ") ";
            temp += "EmitterIndex:(" + EmitterIndex.ToString() + ") ";
            temp += "Data:(" + Data.ToString() + ") ";
            return temp;
        }

        /// <summary>パケットサイズ（可変サイズに対応）</summary>
        public int PacketSize
        {
            get {
                int temp = (4 + 4);   // + 4 は ModifiedProperties と ModifiedPropertyElements
                temp += EmitterSetName.Size;
                temp += EmitterName.Size;
                temp += EmitterIndex.Size;
                temp += Data.Size;
                temp += (4 - (temp & 0x03)) & 0x03;
                return temp;
            }
        }

        /// <summary>後に続く配列の長さ</summary>
        public override int ArrayLength
        {
            get{
                return 0;
            }
        }

        /// <summary>Read</summary>
        public override void Read(Protocol4FReader reader)
        {
             ModifiedProperties.Read(reader);
             ModifiedPropertyElements.Read(reader);
             EmitterSetName.Read(reader);
             EmitterName.Read(reader);
             EmitterIndex.Read(reader);
             Data.Read(reader);
        }

        /// <summary>Write</summary>
        public override void Write(Protocol4FWriter writer)
        {
             ModifiedProperties.Write(writer);
             ModifiedPropertyElements.Write(writer);
             EmitterSetName.Write(writer);
             EmitterName.Write(writer);
             EmitterIndex.Write(writer);
             Data.Write(writer);
        }
    };

    /// <summary>編集しているエミッタのアニメバイナリを送信します。</summary>
    public class MESSAGEPACKET_Emitter_SendAnimTable : BaseMessagePacket
    {
        /// <summary>メッセージタイプ</summary>
        public const MessageType MessagePacketType = MessageType.Emitter_SendAnimTable;

        /// <summary>構造体にした場合のサイズ</summary>
        public const int StructSize = 1056 + 4 + 4;   // + 4 は ModifiedProperties と ModifiedPropertyElements

        /// <summary>後ろに続く配列のサイズ(byte)</summary>
        public virtual int ArraySize{ get{ return 0; } }

        /// <summary>プロパティ</summary>
        [Flags]
        public enum PropertyFlag : uint
        {
            /// <summary>対象のエミッタセット名</summary>
            EmitterSetName = 1<<0,

            /// <summary>対象のエミッタインデックス</summary>
            EmitterIndex = 1<<1,

            /// <summary>対象のエミッタ名</summary>
            EmitterName = 1<<2,

            /// <summary>アニメバイナリのサイズです。</summary>
            DataSize = 1<<3,

            /// <summary>アニメバイナリデータ</summary>
            Data = 1<<4,

        }

        /// <summary>編集されたプロパティ</summary>
        public NUint    ModifiedProperties{ get; set; }

        /// <summary>編集されたプロパティ要素</summary>
        public NUint    ModifiedPropertyElements{ get; set; }

        /// <summary>対象のエミッタセット名</summary>
        public NString    EmitterSetName{ get; set; }

        /// <summary>対象のエミッタインデックス</summary>
        public NUint    EmitterIndex{ get; set; }

        /// <summary>対象のエミッタ名</summary>
        public NString    EmitterName{ get; set; }

        /// <summary>アニメバイナリのサイズです。</summary>
        public NUint    DataSize{ get; set; }

        /// <summary>アニメバイナリデータ</summary>
        public ByteArray    Data{ get; set; }


        /// <summary>コンストラクタ</summary>
        public MESSAGEPACKET_Emitter_SendAnimTable()
        {
            ModifiedProperties = new NUint(0xFFFFFFFF);
            ModifiedPropertyElements = new NUint(0xFFFFFFFF);

            EmitterSetName = new NString();
            EmitterIndex = new NUint();
            EmitterName = new NString();
            DataSize = new NUint();
            Data = new ByteArray();
        }

        /// <summary>コンストラクタ</summary>
        public MESSAGEPACKET_Emitter_SendAnimTable(NString _EmitterSetName, NUint _EmitterIndex, NString _EmitterName, NUint _DataSize, ByteArray _Data)
        {
            ModifiedProperties = new NUint(0xFFFFFFFF);
            ModifiedPropertyElements = new NUint(0xFFFFFFFF);

            EmitterSetName = _EmitterSetName;
            EmitterIndex = _EmitterIndex;
            EmitterName = _EmitterName;
            DataSize = _DataSize;
            Data = _Data;
        }
        /// <summary>文字列化</summary>
        public override string ToString()
        {
            string temp = string.Empty;
            temp += "EmitterSetName:(" + EmitterSetName.ToString() + ") ";
            temp += "EmitterIndex:(" + EmitterIndex.ToString() + ") ";
            temp += "EmitterName:(" + EmitterName.ToString() + ") ";
            temp += "DataSize:(" + DataSize.ToString() + ") ";
            temp += "Data:(" + Data.ToString() + ") ";
            return temp;
        }

        /// <summary>パケットサイズ（可変サイズに対応）</summary>
        public int PacketSize
        {
            get {
                int temp = (4 + 4);   // + 4 は ModifiedProperties と ModifiedPropertyElements
                temp += EmitterSetName.Size;
                temp += EmitterIndex.Size;
                temp += EmitterName.Size;
                temp += DataSize.Size;
                temp += Data.Size;
                temp += (4 - (temp & 0x03)) & 0x03;
                return temp;
            }
        }

        /// <summary>後に続く配列の長さ</summary>
        public override int ArrayLength
        {
            get{
                return 0;
            }
        }

        /// <summary>Read</summary>
        public override void Read(Protocol4FReader reader)
        {
             ModifiedProperties.Read(reader);
             ModifiedPropertyElements.Read(reader);
             EmitterSetName.Read(reader);
             EmitterIndex.Read(reader);
             EmitterName.Read(reader);
             DataSize.Read(reader);
             Data.Read(reader);
        }

        /// <summary>Write</summary>
        public override void Write(Protocol4FWriter writer)
        {
             ModifiedProperties.Write(writer);
             ModifiedPropertyElements.Write(writer);
             EmitterSetName.Write(writer);
             EmitterIndex.Write(writer);
             EmitterName.Write(writer);
             DataSize.Write(writer);
             Data.Write(writer);
        }
    };

    /// <summary>編集しているエミッタのカスタムシェーダパラメータを送信します。</summary>
    public class MESSAGEPACKET_Emitter_SendShaderParam : BaseMessagePacket
    {
        /// <summary>メッセージタイプ</summary>
        public const MessageType MessagePacketType = MessageType.Emitter_SendShaderParam;

        /// <summary>構造体にした場合のサイズ</summary>
        public const int StructSize = 1048 + 4 + 4;   // + 4 は ModifiedProperties と ModifiedPropertyElements

        /// <summary>後ろに続く配列のサイズ(byte)</summary>
        public virtual int ArraySize{ get{ return ByteData.StructSize * DataLength.Value; } }

        /// <summary>プロパティ</summary>
        [Flags]
        public enum PropertyFlag : uint
        {
            /// <summary>対象のエミッタセット名</summary>
            EmitterSetName = 1<<0,

            /// <summary>対象のエミッタインデックス</summary>
            EmitterIndex = 1<<1,

            /// <summary>対象のエミッタ名</summary>
            EmitterName = 1<<2,

            /// <summary>バイナリ配列の長さです。</summary>
            DataLength = 1<<3,

        }

        /// <summary>編集されたプロパティ</summary>
        public NUint    ModifiedProperties{ get; set; }

        /// <summary>編集されたプロパティ要素</summary>
        public NUint    ModifiedPropertyElements{ get; set; }

        /// <summary>対象のエミッタセット名</summary>
        public NString    EmitterSetName{ get; set; }

        /// <summary>対象のエミッタインデックス</summary>
        public NUint    EmitterIndex{ get; set; }

        /// <summary>対象のエミッタ名</summary>
        public NString    EmitterName{ get; set; }

        /// <summary>バイナリ配列の長さです。</summary>
        public NInt    DataLength{ get; set; }


        /// <summary>コンストラクタ</summary>
        public MESSAGEPACKET_Emitter_SendShaderParam()
        {
            ModifiedProperties = new NUint(0xFFFFFFFF);
            ModifiedPropertyElements = new NUint(0xFFFFFFFF);

            EmitterSetName = new NString();
            EmitterIndex = new NUint();
            EmitterName = new NString();
            DataLength = new NInt();
        }

        /// <summary>コンストラクタ</summary>
        public MESSAGEPACKET_Emitter_SendShaderParam(NString _EmitterSetName, NUint _EmitterIndex, NString _EmitterName, NInt _DataLength)
        {
            ModifiedProperties = new NUint(0xFFFFFFFF);
            ModifiedPropertyElements = new NUint(0xFFFFFFFF);

            EmitterSetName = _EmitterSetName;
            EmitterIndex = _EmitterIndex;
            EmitterName = _EmitterName;
            DataLength = _DataLength;
        }
        /// <summary>文字列化</summary>
        public override string ToString()
        {
            string temp = string.Empty;
            temp += "EmitterSetName:(" + EmitterSetName.ToString() + ") ";
            temp += "EmitterIndex:(" + EmitterIndex.ToString() + ") ";
            temp += "EmitterName:(" + EmitterName.ToString() + ") ";
            temp += "DataLength:(" + DataLength.ToString() + ") ";
            return temp;
        }

        /// <summary>パケットサイズ（可変サイズに対応）</summary>
        public int PacketSize
        {
            get {
                int temp = (4 + 4);   // + 4 は ModifiedProperties と ModifiedPropertyElements
                temp += EmitterSetName.Size;
                temp += EmitterIndex.Size;
                temp += EmitterName.Size;
                temp += DataLength.Size;
                temp += (4 - (temp & 0x03)) & 0x03;
                return temp;
            }
        }

        /// <summary>後に続く配列の長さ</summary>
        public override int ArrayLength
        {
            get{
                return DataLength.Value;
            }
        }

        /// <summary>CreateStructPacket</summary>
        public override BaseStructPacket CreateStructPacket()
        {
            return new ByteData();
        }

        /// <summary>Read</summary>
        public override void Read(Protocol4FReader reader)
        {
             ModifiedProperties.Read(reader);
             ModifiedPropertyElements.Read(reader);
             EmitterSetName.Read(reader);
             EmitterIndex.Read(reader);
             EmitterName.Read(reader);
             DataLength.Read(reader);
        }

        /// <summary>Write</summary>
        public override void Write(Protocol4FWriter writer)
        {
             ModifiedProperties.Write(writer);
             ModifiedPropertyElements.Write(writer);
             EmitterSetName.Write(writer);
             EmitterIndex.Write(writer);
             EmitterName.Write(writer);
             DataLength.Write(writer);
        }
    };

    /// <summary>エミッタセットごとのプレビュー設定</summary>
    public class MESSAGEPACKET_Preview_SetPreviewEmitterSet : BaseMessagePacket
    {
        /// <summary>メッセージタイプ</summary>
        public const MessageType MessagePacketType = MessageType.Preview_SetPreviewEmitterSet;

        /// <summary>構造体にした場合のサイズ</summary>
        public const int StructSize = 1564 + 4 + 4;   // + 4 は ModifiedProperties と ModifiedPropertyElements

        /// <summary>後ろに続く配列のサイズ(byte)</summary>
        public virtual int ArraySize{ get{ return 0; } }

        /// <summary>プロパティ</summary>
        [Flags]
        public enum PropertyFlag : uint
        {
            /// <summary>対象のエミッタセット名</summary>
            EmitterSetName = 1<<0,

            /// <summary>コンストレイン先のモデル名</summary>
            ModelName = 1<<1,

            /// <summary>コンストレイン先のボーン名</summary>
            BoneName = 1<<2,

            /// <summary>コンストレイン先のボーン番号</summary>
            BoneNum = 1<<3,

        }

        /// <summary>編集されたプロパティ</summary>
        public NUint    ModifiedProperties{ get; set; }

        /// <summary>編集されたプロパティ要素</summary>
        public NUint    ModifiedPropertyElements{ get; set; }

        /// <summary>対象のエミッタセット名</summary>
        public NString    EmitterSetName{ get; set; }

        /// <summary>コンストレイン先のモデル名</summary>
        public NString    ModelName{ get; set; }

        /// <summary>コンストレイン先のボーン名</summary>
        public NString    BoneName{ get; set; }

        /// <summary>コンストレイン先のボーン番号</summary>
        public NInt    BoneNum{ get; set; }


        /// <summary>コンストラクタ</summary>
        public MESSAGEPACKET_Preview_SetPreviewEmitterSet()
        {
            ModifiedProperties = new NUint(0xFFFFFFFF);
            ModifiedPropertyElements = new NUint(0xFFFFFFFF);

            EmitterSetName = new NString();
            ModelName = new NString();
            BoneName = new NString();
            BoneNum = new NInt();
        }

        /// <summary>コンストラクタ</summary>
        public MESSAGEPACKET_Preview_SetPreviewEmitterSet(NString _EmitterSetName, NString _ModelName, NString _BoneName, NInt _BoneNum)
        {
            ModifiedProperties = new NUint(0xFFFFFFFF);
            ModifiedPropertyElements = new NUint(0xFFFFFFFF);

            EmitterSetName = _EmitterSetName;
            ModelName = _ModelName;
            BoneName = _BoneName;
            BoneNum = _BoneNum;
        }
        /// <summary>文字列化</summary>
        public override string ToString()
        {
            string temp = string.Empty;
            temp += "EmitterSetName:(" + EmitterSetName.ToString() + ") ";
            temp += "ModelName:(" + ModelName.ToString() + ") ";
            temp += "BoneName:(" + BoneName.ToString() + ") ";
            temp += "BoneNum:(" + BoneNum.ToString() + ") ";
            return temp;
        }

        /// <summary>パケットサイズ（可変サイズに対応）</summary>
        public int PacketSize
        {
            get {
                int temp = (4 + 4);   // + 4 は ModifiedProperties と ModifiedPropertyElements
                temp += EmitterSetName.Size;
                temp += ModelName.Size;
                temp += BoneName.Size;
                temp += BoneNum.Size;
                temp += (4 - (temp & 0x03)) & 0x03;
                return temp;
            }
        }

        /// <summary>後に続く配列の長さ</summary>
        public override int ArrayLength
        {
            get{
                return 0;
            }
        }

        /// <summary>Read</summary>
        public override void Read(Protocol4FReader reader)
        {
             ModifiedProperties.Read(reader);
             ModifiedPropertyElements.Read(reader);
             EmitterSetName.Read(reader);
             ModelName.Read(reader);
             BoneName.Read(reader);
             BoneNum.Read(reader);
        }

        /// <summary>Write</summary>
        public override void Write(Protocol4FWriter writer)
        {
             ModifiedProperties.Write(writer);
             ModifiedPropertyElements.Write(writer);
             EmitterSetName.Write(writer);
             ModelName.Write(writer);
             BoneName.Write(writer);
             BoneNum.Write(writer);
        }
    };

    /// <summary>Function送信</summary>
    public class MESSAGEPACKET_Preview_SendFunctionInfo : BaseMessagePacket
    {
        /// <summary>メッセージタイプ</summary>
        public const MessageType MessagePacketType = MessageType.Preview_SendFunctionInfo;

        /// <summary>構造体にした場合のサイズ</summary>
        public const int StructSize = 536 + 4 + 4;   // + 4 は ModifiedProperties と ModifiedPropertyElements

        /// <summary>後ろに続く配列のサイズ(byte)</summary>
        public virtual int ArraySize{ get{ return 0; } }

        /// <summary>プロパティ</summary>
        [Flags]
        public enum PropertyFlag : uint
        {
            /// <summary>対象のタイプ</summary>
            TargetType = 1<<0,

            /// <summary>対象の名前</summary>
            TargetName = 1<<1,

            /// <summary>対象のＩＤ</summary>
            TargetID = 1<<2,

            /// <summary>データ（&Data からLength分データが続く）</summary>
            Data = 1<<3,

        }

        /// <summary>編集されたプロパティ</summary>
        public NUint    ModifiedProperties{ get; set; }

        /// <summary>編集されたプロパティ要素</summary>
        public NUint    ModifiedPropertyElements{ get; set; }

        /// <summary>対象のタイプ</summary>
        public EditTargetType    TargetType{ get; set; }

        /// <summary>対象の名前</summary>
        public NString    TargetName{ get; set; }

        /// <summary>対象のＩＤ</summary>
        public NUint    TargetID{ get; set; }

        /// <summary>データ（&Data からLength分データが続く）</summary>
        public ByteArray    Data{ get; set; }


        /// <summary>コンストラクタ</summary>
        public MESSAGEPACKET_Preview_SendFunctionInfo()
        {
            ModifiedProperties = new NUint(0xFFFFFFFF);
            ModifiedPropertyElements = new NUint(0xFFFFFFFF);

            TargetType = new EditTargetType();
            TargetName = new NString();
            TargetID = new NUint();
            Data = new ByteArray();
        }

        /// <summary>コンストラクタ</summary>
        public MESSAGEPACKET_Preview_SendFunctionInfo(EditTargetType _TargetType, NString _TargetName, NUint _TargetID, ByteArray _Data)
        {
            ModifiedProperties = new NUint(0xFFFFFFFF);
            ModifiedPropertyElements = new NUint(0xFFFFFFFF);

            TargetType = _TargetType;
            TargetName = _TargetName;
            TargetID = _TargetID;
            Data = _Data;
        }
        /// <summary>文字列化</summary>
        public override string ToString()
        {
            string temp = string.Empty;
            temp += "TargetType:(" + TargetType.ToString() + ") ";
            temp += "TargetName:(" + TargetName.ToString() + ") ";
            temp += "TargetID:(" + TargetID.ToString() + ") ";
            temp += "Data:(" + Data.ToString() + ") ";
            return temp;
        }

        /// <summary>パケットサイズ（可変サイズに対応）</summary>
        public int PacketSize
        {
            get {
                int temp = (4 + 4);   // + 4 は ModifiedProperties と ModifiedPropertyElements
                temp += 4;  //TargetType
                temp += TargetName.Size;
                temp += TargetID.Size;
                temp += Data.Size;
                temp += (4 - (temp & 0x03)) & 0x03;
                return temp;
            }
        }

        /// <summary>後に続く配列の長さ</summary>
        public override int ArrayLength
        {
            get{
                return 0;
            }
        }

        /// <summary>Read</summary>
        public override void Read(Protocol4FReader reader)
        {
             ModifiedProperties.Read(reader);
             ModifiedPropertyElements.Read(reader);
             {
                 NInt temp = new NInt();
                 temp.Read(reader);
                 TargetType = (EditTargetType)temp.Value;
             }
             TargetName.Read(reader);
             TargetID.Read(reader);
             Data.Read(reader);
        }

        /// <summary>Write</summary>
        public override void Write(Protocol4FWriter writer)
        {
             ModifiedProperties.Write(writer);
             ModifiedPropertyElements.Write(writer);
             (new NInt((int)TargetType)).Write(writer);
             TargetName.Write(writer);
             TargetID.Write(writer);
             Data.Write(writer);
        }
    };

    /// <summary>エフェクトのインスタンスを生成します。</summary>
    public class MESSAGEPACKET_Preview_AddEmitterSetInst : BaseMessagePacket
    {
        /// <summary>メッセージタイプ</summary>
        public const MessageType MessagePacketType = MessageType.Preview_AddEmitterSetInst;

        /// <summary>構造体にした場合のサイズ</summary>
        public const int StructSize = 536 + 4 + 4;   // + 4 は ModifiedProperties と ModifiedPropertyElements

        /// <summary>後ろに続く配列のサイズ(byte)</summary>
        public virtual int ArraySize{ get{ return 0; } }

        /// <summary>プロパティ</summary>
        [Flags]
        public enum PropertyFlag : uint
        {
            /// <summary>対象のエミッタセット名</summary>
            EmitterSetName = 1<<0,

            /// <summary>プレビューのIDです。</summary>
            PreviewId = 1<<1,

            /// <summary>表示/非表示のフラグです。</summary>
            IsVisible = 1<<2,

            /// <summary>再生開始フレーム</summary>
            PlaybackStartFrame = 1<<3,

            /// <summary>強制開始 フレーム</summary>
            ForceStartFrame = 1<<4,

        }

        /// <summary>編集されたプロパティ</summary>
        public NUint    ModifiedProperties{ get; set; }

        /// <summary>編集されたプロパティ要素</summary>
        public NUint    ModifiedPropertyElements{ get; set; }

        /// <summary>対象のエミッタセット名</summary>
        public NString    EmitterSetName{ get; set; }

        /// <summary>プレビューのIDです。</summary>
        public NUint    PreviewId{ get; set; }

        /// <summary>表示/非表示のフラグです。</summary>
        public NUint    IsVisible{ get; set; }

        /// <summary>再生開始フレーム</summary>
        public NUint    PlaybackStartFrame{ get; set; }

        /// <summary>強制開始 フレーム</summary>
        public NUint    ForceStartFrame{ get; set; }


        /// <summary>コンストラクタ</summary>
        public MESSAGEPACKET_Preview_AddEmitterSetInst()
        {
            ModifiedProperties = new NUint(0xFFFFFFFF);
            ModifiedPropertyElements = new NUint(0xFFFFFFFF);

            EmitterSetName = new NString();
            PreviewId = new NUint();
            IsVisible = new NUint();
            PlaybackStartFrame = new NUint();
            ForceStartFrame = new NUint();
        }

        /// <summary>コンストラクタ</summary>
        public MESSAGEPACKET_Preview_AddEmitterSetInst(NString _EmitterSetName, NUint _PreviewId, NUint _IsVisible, NUint _PlaybackStartFrame, NUint _ForceStartFrame)
        {
            ModifiedProperties = new NUint(0xFFFFFFFF);
            ModifiedPropertyElements = new NUint(0xFFFFFFFF);

            EmitterSetName = _EmitterSetName;
            PreviewId = _PreviewId;
            IsVisible = _IsVisible;
            PlaybackStartFrame = _PlaybackStartFrame;
            ForceStartFrame = _ForceStartFrame;
        }
        /// <summary>文字列化</summary>
        public override string ToString()
        {
            string temp = string.Empty;
            temp += "EmitterSetName:(" + EmitterSetName.ToString() + ") ";
            temp += "PreviewId:(" + PreviewId.ToString() + ") ";
            temp += "IsVisible:(" + IsVisible.ToString() + ") ";
            temp += "PlaybackStartFrame:(" + PlaybackStartFrame.ToString() + ") ";
            temp += "ForceStartFrame:(" + ForceStartFrame.ToString() + ") ";
            return temp;
        }

        /// <summary>パケットサイズ（可変サイズに対応）</summary>
        public int PacketSize
        {
            get {
                int temp = (4 + 4);   // + 4 は ModifiedProperties と ModifiedPropertyElements
                temp += EmitterSetName.Size;
                temp += PreviewId.Size;
                temp += IsVisible.Size;
                temp += PlaybackStartFrame.Size;
                temp += ForceStartFrame.Size;
                temp += (4 - (temp & 0x03)) & 0x03;
                return temp;
            }
        }

        /// <summary>後に続く配列の長さ</summary>
        public override int ArrayLength
        {
            get{
                return 0;
            }
        }

        /// <summary>Read</summary>
        public override void Read(Protocol4FReader reader)
        {
             ModifiedProperties.Read(reader);
             ModifiedPropertyElements.Read(reader);
             EmitterSetName.Read(reader);
             PreviewId.Read(reader);
             IsVisible.Read(reader);
             PlaybackStartFrame.Read(reader);
             ForceStartFrame.Read(reader);
        }

        /// <summary>Write</summary>
        public override void Write(Protocol4FWriter writer)
        {
             ModifiedProperties.Write(writer);
             ModifiedPropertyElements.Write(writer);
             EmitterSetName.Write(writer);
             PreviewId.Write(writer);
             IsVisible.Write(writer);
             PlaybackStartFrame.Write(writer);
             ForceStartFrame.Write(writer);
        }
    };

    /// <summary>エフェクトのインスタンスを削除します。</summary>
    public class MESSAGEPACKET_Preview_RemoveEmitterSetInst : BaseMessagePacket
    {
        /// <summary>メッセージタイプ</summary>
        public const MessageType MessagePacketType = MessageType.Preview_RemoveEmitterSetInst;

        /// <summary>構造体にした場合のサイズ</summary>
        public const int StructSize = 524 + 4 + 4;   // + 4 は ModifiedProperties と ModifiedPropertyElements

        /// <summary>後ろに続く配列のサイズ(byte)</summary>
        public virtual int ArraySize{ get{ return 0; } }

        /// <summary>プロパティ</summary>
        [Flags]
        public enum PropertyFlag : uint
        {
            /// <summary>対象のエミッタセット名</summary>
            EmitterSetName = 1<<0,

            /// <summary>プレビューのIDです。</summary>
            PreviewId = 1<<1,

        }

        /// <summary>編集されたプロパティ</summary>
        public NUint    ModifiedProperties{ get; set; }

        /// <summary>編集されたプロパティ要素</summary>
        public NUint    ModifiedPropertyElements{ get; set; }

        /// <summary>対象のエミッタセット名</summary>
        public NString    EmitterSetName{ get; set; }

        /// <summary>プレビューのIDです。</summary>
        public NUint    PreviewId{ get; set; }


        /// <summary>コンストラクタ</summary>
        public MESSAGEPACKET_Preview_RemoveEmitterSetInst()
        {
            ModifiedProperties = new NUint(0xFFFFFFFF);
            ModifiedPropertyElements = new NUint(0xFFFFFFFF);

            EmitterSetName = new NString();
            PreviewId = new NUint();
        }

        /// <summary>コンストラクタ</summary>
        public MESSAGEPACKET_Preview_RemoveEmitterSetInst(NString _EmitterSetName, NUint _PreviewId)
        {
            ModifiedProperties = new NUint(0xFFFFFFFF);
            ModifiedPropertyElements = new NUint(0xFFFFFFFF);

            EmitterSetName = _EmitterSetName;
            PreviewId = _PreviewId;
        }
        /// <summary>文字列化</summary>
        public override string ToString()
        {
            string temp = string.Empty;
            temp += "EmitterSetName:(" + EmitterSetName.ToString() + ") ";
            temp += "PreviewId:(" + PreviewId.ToString() + ") ";
            return temp;
        }

        /// <summary>パケットサイズ（可変サイズに対応）</summary>
        public int PacketSize
        {
            get {
                int temp = (4 + 4);   // + 4 は ModifiedProperties と ModifiedPropertyElements
                temp += EmitterSetName.Size;
                temp += PreviewId.Size;
                temp += (4 - (temp & 0x03)) & 0x03;
                return temp;
            }
        }

        /// <summary>後に続く配列の長さ</summary>
        public override int ArrayLength
        {
            get{
                return 0;
            }
        }

        /// <summary>Read</summary>
        public override void Read(Protocol4FReader reader)
        {
             ModifiedProperties.Read(reader);
             ModifiedPropertyElements.Read(reader);
             EmitterSetName.Read(reader);
             PreviewId.Read(reader);
        }

        /// <summary>Write</summary>
        public override void Write(Protocol4FWriter writer)
        {
             ModifiedProperties.Write(writer);
             ModifiedPropertyElements.Write(writer);
             EmitterSetName.Write(writer);
             PreviewId.Write(writer);
        }
    };

    /// <summary>開始フレームまでフレーム移動します。</summary>
    public class MESSAGEPACKET_Preview_RewindToFirst : BaseMessagePacket
    {
        /// <summary>メッセージタイプ</summary>
        public const MessageType MessagePacketType = MessageType.Preview_RewindToFirst;

        /// <summary>構造体にした場合のサイズ</summary>
        public const int StructSize = 0 + 4 + 4;   // + 4 は ModifiedProperties と ModifiedPropertyElements

        /// <summary>後ろに続く配列のサイズ(byte)</summary>
        public virtual int ArraySize{ get{ return 0; } }

        /// <summary>編集されたプロパティ</summary>
        public NUint    ModifiedProperties{ get; set; }

        /// <summary>編集されたプロパティ要素</summary>
        public NUint    ModifiedPropertyElements{ get; set; }


        /// <summary>コンストラクタ</summary>
        public MESSAGEPACKET_Preview_RewindToFirst()
        {
            ModifiedProperties = new NUint(0xFFFFFFFF);
            ModifiedPropertyElements = new NUint(0xFFFFFFFF);

        }

        /// <summary>文字列化</summary>
        public override string ToString()
        {
            string temp = string.Empty;
            return temp;
        }

        /// <summary>パケットサイズ（可変サイズに対応）</summary>
        public int PacketSize
        {
            get {
                int temp = (4 + 4);   // + 4 は ModifiedProperties と ModifiedPropertyElements
                temp += (4 - (temp & 0x03)) & 0x03;
                return temp;
            }
        }

        /// <summary>後に続く配列の長さ</summary>
        public override int ArrayLength
        {
            get{
                return 0;
            }
        }

        /// <summary>Read</summary>
        public override void Read(Protocol4FReader reader)
        {
             ModifiedProperties.Read(reader);
             ModifiedPropertyElements.Read(reader);
        }

        /// <summary>Write</summary>
        public override void Write(Protocol4FWriter writer)
        {
             ModifiedProperties.Write(writer);
             ModifiedPropertyElements.Write(writer);
        }
    };

    /// <summary>全てのアニメーションの更新を開始/一時停止します。</summary>
    public class MESSAGEPACKET_Preview_Play : BaseMessagePacket
    {
        /// <summary>メッセージタイプ</summary>
        public const MessageType MessagePacketType = MessageType.Preview_Play;

        /// <summary>構造体にした場合のサイズ</summary>
        public const int StructSize = 4 + 4 + 4;   // + 4 は ModifiedProperties と ModifiedPropertyElements

        /// <summary>後ろに続く配列のサイズ(byte)</summary>
        public virtual int ArraySize{ get{ return 0; } }

        /// <summary>プロパティ</summary>
        [Flags]
        public enum PropertyFlag : uint
        {
            /// <summary>強制的に再生させるフラグです。</summary>
            ForcePlay = 1<<0,

        }

        /// <summary>編集されたプロパティ</summary>
        public NUint    ModifiedProperties{ get; set; }

        /// <summary>編集されたプロパティ要素</summary>
        public NUint    ModifiedPropertyElements{ get; set; }

        /// <summary>強制的に再生させるフラグです。</summary>
        public NUint    ForcePlay{ get; set; }


        /// <summary>コンストラクタ</summary>
        public MESSAGEPACKET_Preview_Play()
        {
            ModifiedProperties = new NUint(0xFFFFFFFF);
            ModifiedPropertyElements = new NUint(0xFFFFFFFF);

            ForcePlay = new NUint();
        }

        /// <summary>コンストラクタ</summary>
        public MESSAGEPACKET_Preview_Play(NUint _ForcePlay)
        {
            ModifiedProperties = new NUint(0xFFFFFFFF);
            ModifiedPropertyElements = new NUint(0xFFFFFFFF);

            ForcePlay = _ForcePlay;
        }
        /// <summary>文字列化</summary>
        public override string ToString()
        {
            string temp = string.Empty;
            temp += "ForcePlay:(" + ForcePlay.ToString() + ") ";
            return temp;
        }

        /// <summary>パケットサイズ（可変サイズに対応）</summary>
        public int PacketSize
        {
            get {
                int temp = (4 + 4);   // + 4 は ModifiedProperties と ModifiedPropertyElements
                temp += ForcePlay.Size;
                temp += (4 - (temp & 0x03)) & 0x03;
                return temp;
            }
        }

        /// <summary>後に続く配列の長さ</summary>
        public override int ArrayLength
        {
            get{
                return 0;
            }
        }

        /// <summary>Read</summary>
        public override void Read(Protocol4FReader reader)
        {
             ModifiedProperties.Read(reader);
             ModifiedPropertyElements.Read(reader);
             ForcePlay.Read(reader);
        }

        /// <summary>Write</summary>
        public override void Write(Protocol4FWriter writer)
        {
             ModifiedProperties.Write(writer);
             ModifiedPropertyElements.Write(writer);
             ForcePlay.Write(writer);
        }
    };

    /// <summary>全てのアニメーションを停止し、フレームを最初に戻します。</summary>
    public class MESSAGEPACKET_Preview_Stop : BaseMessagePacket
    {
        /// <summary>メッセージタイプ</summary>
        public const MessageType MessagePacketType = MessageType.Preview_Stop;

        /// <summary>構造体にした場合のサイズ</summary>
        public const int StructSize = 0 + 4 + 4;   // + 4 は ModifiedProperties と ModifiedPropertyElements

        /// <summary>後ろに続く配列のサイズ(byte)</summary>
        public virtual int ArraySize{ get{ return 0; } }

        /// <summary>編集されたプロパティ</summary>
        public NUint    ModifiedProperties{ get; set; }

        /// <summary>編集されたプロパティ要素</summary>
        public NUint    ModifiedPropertyElements{ get; set; }


        /// <summary>コンストラクタ</summary>
        public MESSAGEPACKET_Preview_Stop()
        {
            ModifiedProperties = new NUint(0xFFFFFFFF);
            ModifiedPropertyElements = new NUint(0xFFFFFFFF);

        }

        /// <summary>文字列化</summary>
        public override string ToString()
        {
            string temp = string.Empty;
            return temp;
        }

        /// <summary>パケットサイズ（可変サイズに対応）</summary>
        public int PacketSize
        {
            get {
                int temp = (4 + 4);   // + 4 は ModifiedProperties と ModifiedPropertyElements
                temp += (4 - (temp & 0x03)) & 0x03;
                return temp;
            }
        }

        /// <summary>後に続く配列の長さ</summary>
        public override int ArrayLength
        {
            get{
                return 0;
            }
        }

        /// <summary>Read</summary>
        public override void Read(Protocol4FReader reader)
        {
             ModifiedProperties.Read(reader);
             ModifiedPropertyElements.Read(reader);
        }

        /// <summary>Write</summary>
        public override void Write(Protocol4FWriter writer)
        {
             ModifiedProperties.Write(writer);
             ModifiedPropertyElements.Write(writer);
        }
    };

    /// <summary>全てのアニメーションの１フレーム更新（前に進める）します。</summary>
    public class MESSAGEPACKET_Preview_StepFrame : BaseMessagePacket
    {
        /// <summary>メッセージタイプ</summary>
        public const MessageType MessagePacketType = MessageType.Preview_StepFrame;

        /// <summary>構造体にした場合のサイズ</summary>
        public const int StructSize = 0 + 4 + 4;   // + 4 は ModifiedProperties と ModifiedPropertyElements

        /// <summary>後ろに続く配列のサイズ(byte)</summary>
        public virtual int ArraySize{ get{ return 0; } }

        /// <summary>編集されたプロパティ</summary>
        public NUint    ModifiedProperties{ get; set; }

        /// <summary>編集されたプロパティ要素</summary>
        public NUint    ModifiedPropertyElements{ get; set; }


        /// <summary>コンストラクタ</summary>
        public MESSAGEPACKET_Preview_StepFrame()
        {
            ModifiedProperties = new NUint(0xFFFFFFFF);
            ModifiedPropertyElements = new NUint(0xFFFFFFFF);

        }

        /// <summary>文字列化</summary>
        public override string ToString()
        {
            string temp = string.Empty;
            return temp;
        }

        /// <summary>パケットサイズ（可変サイズに対応）</summary>
        public int PacketSize
        {
            get {
                int temp = (4 + 4);   // + 4 は ModifiedProperties と ModifiedPropertyElements
                temp += (4 - (temp & 0x03)) & 0x03;
                return temp;
            }
        }

        /// <summary>後に続く配列の長さ</summary>
        public override int ArrayLength
        {
            get{
                return 0;
            }
        }

        /// <summary>Read</summary>
        public override void Read(Protocol4FReader reader)
        {
             ModifiedProperties.Read(reader);
             ModifiedPropertyElements.Read(reader);
        }

        /// <summary>Write</summary>
        public override void Write(Protocol4FWriter writer)
        {
             ModifiedProperties.Write(writer);
             ModifiedPropertyElements.Write(writer);
        }
    };

    /// <summary>再生するフレーム範囲を指定します。</summary>
    public class MESSAGEPACKET_Preview_SetFrameRange : BaseMessagePacket
    {
        /// <summary>メッセージタイプ</summary>
        public const MessageType MessagePacketType = MessageType.Preview_SetFrameRange;

        /// <summary>構造体にした場合のサイズ</summary>
        public const int StructSize = 12 + 4 + 4;   // + 4 は ModifiedProperties と ModifiedPropertyElements

        /// <summary>後ろに続く配列のサイズ(byte)</summary>
        public virtual int ArraySize{ get{ return 0; } }

        /// <summary>プロパティ</summary>
        [Flags]
        public enum PropertyFlag : uint
        {
            /// <summary>開始フレームです。</summary>
            StartFrame = 1<<0,

            /// <summary>終了フレームです。</summary>
            EndFrame = 1<<1,

            /// <summary>フラグ設定（=1..ループ /=0..ループしない）</summary>
            DoLoop = 1<<2,

        }

        /// <summary>編集されたプロパティ</summary>
        public NUint    ModifiedProperties{ get; set; }

        /// <summary>編集されたプロパティ要素</summary>
        public NUint    ModifiedPropertyElements{ get; set; }

        /// <summary>開始フレームです。</summary>
        public NInt    StartFrame{ get; set; }

        /// <summary>終了フレームです。</summary>
        public NInt    EndFrame{ get; set; }

        /// <summary>フラグ設定（=1..ループ /=0..ループしない）</summary>
        public NInt    DoLoop{ get; set; }


        /// <summary>コンストラクタ</summary>
        public MESSAGEPACKET_Preview_SetFrameRange()
        {
            ModifiedProperties = new NUint(0xFFFFFFFF);
            ModifiedPropertyElements = new NUint(0xFFFFFFFF);

            StartFrame = new NInt();
            EndFrame = new NInt();
            DoLoop = new NInt();
        }

        /// <summary>コンストラクタ</summary>
        public MESSAGEPACKET_Preview_SetFrameRange(NInt _StartFrame, NInt _EndFrame, NInt _DoLoop)
        {
            ModifiedProperties = new NUint(0xFFFFFFFF);
            ModifiedPropertyElements = new NUint(0xFFFFFFFF);

            StartFrame = _StartFrame;
            EndFrame = _EndFrame;
            DoLoop = _DoLoop;
        }
        /// <summary>文字列化</summary>
        public override string ToString()
        {
            string temp = string.Empty;
            temp += "StartFrame:(" + StartFrame.ToString() + ") ";
            temp += "EndFrame:(" + EndFrame.ToString() + ") ";
            temp += "DoLoop:(" + DoLoop.ToString() + ") ";
            return temp;
        }

        /// <summary>パケットサイズ（可変サイズに対応）</summary>
        public int PacketSize
        {
            get {
                int temp = (4 + 4);   // + 4 は ModifiedProperties と ModifiedPropertyElements
                temp += StartFrame.Size;
                temp += EndFrame.Size;
                temp += DoLoop.Size;
                temp += (4 - (temp & 0x03)) & 0x03;
                return temp;
            }
        }

        /// <summary>後に続く配列の長さ</summary>
        public override int ArrayLength
        {
            get{
                return 0;
            }
        }

        /// <summary>Read</summary>
        public override void Read(Protocol4FReader reader)
        {
             ModifiedProperties.Read(reader);
             ModifiedPropertyElements.Read(reader);
             StartFrame.Read(reader);
             EndFrame.Read(reader);
             DoLoop.Read(reader);
        }

        /// <summary>Write</summary>
        public override void Write(Protocol4FWriter writer)
        {
             ModifiedProperties.Write(writer);
             ModifiedPropertyElements.Write(writer);
             StartFrame.Write(writer);
             EndFrame.Write(writer);
             DoLoop.Write(writer);
        }
    };

    /// <summary>エミッタセット/エミッタ/プレビューノードの表示/非表示を切り替えます</summary>
    public class MESSAGEPACKET_Preview_SetVisibility : BaseMessagePacket
    {
        /// <summary>メッセージタイプ</summary>
        public const MessageType MessagePacketType = MessageType.Preview_SetVisibility;

        /// <summary>構造体にした場合のサイズ</summary>
        public const int StructSize = 532 + 4 + 4;   // + 4 は ModifiedProperties と ModifiedPropertyElements

        /// <summary>後ろに続く配列のサイズ(byte)</summary>
        public virtual int ArraySize{ get{ return 0; } }

        /// <summary>プロパティ</summary>
        [Flags]
        public enum PropertyFlag : uint
        {
            /// <summary>対象のエミッタセット名</summary>
            TargetType = 1<<0,

            /// <summary>対象のエミッタ名</summary>
            TargetName = 1<<1,

            /// <summary>対象のエミッタインデックス</summary>
            TargetID = 1<<2,

            /// <summary>Visibilityです。</summary>
            Visibility = 1<<3,

        }

        /// <summary>編集されたプロパティ</summary>
        public NUint    ModifiedProperties{ get; set; }

        /// <summary>編集されたプロパティ要素</summary>
        public NUint    ModifiedPropertyElements{ get; set; }

        /// <summary>対象のエミッタセット名</summary>
        public EditTargetType    TargetType{ get; set; }

        /// <summary>対象のエミッタ名</summary>
        public NString    TargetName{ get; set; }

        /// <summary>対象のエミッタインデックス</summary>
        public NUint    TargetID{ get; set; }

        /// <summary>Visibilityです。</summary>
        public NUint    Visibility{ get; set; }


        /// <summary>コンストラクタ</summary>
        public MESSAGEPACKET_Preview_SetVisibility()
        {
            ModifiedProperties = new NUint(0xFFFFFFFF);
            ModifiedPropertyElements = new NUint(0xFFFFFFFF);

            TargetType = new EditTargetType();
            TargetName = new NString();
            TargetID = new NUint();
            Visibility = new NUint();
        }

        /// <summary>コンストラクタ</summary>
        public MESSAGEPACKET_Preview_SetVisibility(EditTargetType _TargetType, NString _TargetName, NUint _TargetID, NUint _Visibility)
        {
            ModifiedProperties = new NUint(0xFFFFFFFF);
            ModifiedPropertyElements = new NUint(0xFFFFFFFF);

            TargetType = _TargetType;
            TargetName = _TargetName;
            TargetID = _TargetID;
            Visibility = _Visibility;
        }
        /// <summary>文字列化</summary>
        public override string ToString()
        {
            string temp = string.Empty;
            temp += "TargetType:(" + TargetType.ToString() + ") ";
            temp += "TargetName:(" + TargetName.ToString() + ") ";
            temp += "TargetID:(" + TargetID.ToString() + ") ";
            temp += "Visibility:(" + Visibility.ToString() + ") ";
            return temp;
        }

        /// <summary>パケットサイズ（可変サイズに対応）</summary>
        public int PacketSize
        {
            get {
                int temp = (4 + 4);   // + 4 は ModifiedProperties と ModifiedPropertyElements
                temp += 4;  //TargetType
                temp += TargetName.Size;
                temp += TargetID.Size;
                temp += Visibility.Size;
                temp += (4 - (temp & 0x03)) & 0x03;
                return temp;
            }
        }

        /// <summary>後に続く配列の長さ</summary>
        public override int ArrayLength
        {
            get{
                return 0;
            }
        }

        /// <summary>Read</summary>
        public override void Read(Protocol4FReader reader)
        {
             ModifiedProperties.Read(reader);
             ModifiedPropertyElements.Read(reader);
             {
                 NInt temp = new NInt();
                 temp.Read(reader);
                 TargetType = (EditTargetType)temp.Value;
             }
             TargetName.Read(reader);
             TargetID.Read(reader);
             Visibility.Read(reader);
        }

        /// <summary>Write</summary>
        public override void Write(Protocol4FWriter writer)
        {
             ModifiedProperties.Write(writer);
             ModifiedPropertyElements.Write(writer);
             (new NInt((int)TargetType)).Write(writer);
             TargetName.Write(writer);
             TargetID.Write(writer);
             Visibility.Write(writer);
        }
    };

    /// <summary>初期設定です。</summary>
    public class MESSAGEPACKET_Viewer_InitSettings : BaseMessagePacket
    {
        /// <summary>メッセージタイプ</summary>
        public const MessageType MessagePacketType = MessageType.Viewer_InitSettings;

        /// <summary>構造体にした場合のサイズ</summary>
        public const int StructSize = 40 + 4 + 4;   // + 4 は ModifiedProperties と ModifiedPropertyElements

        /// <summary>後ろに続く配列のサイズ(byte)</summary>
        public virtual int ArraySize{ get{ return 0; } }

        /// <summary>プロパティ</summary>
        [Flags]
        public enum PropertyFlag : uint
        {
            /// <summary>フレームレートです。</summary>
            FrameRate = 1<<0,

            /// <summary>解像度です。</summary>
            Resolution = 1<<1,

            /// <summary>クリアカラーです。</summary>
            ClearColor = 1<<2,

            /// <summary>ワールドスケール 位置・速度・サイズレンジ</summary>
            WorldScaleRange = 1<<3,

            /// <summary>ワールドスケール 時間</summary>
            WorldScaleTime = 1<<4,

            /// <summary>ワールドスケール グリッドと床の表示スケール</summary>
            GridScale = 1<<5,

            /// <summary>リニア編集モードの設定状態です(0: Disabled, 1: Enabled)。</summary>
            LinearEditMode = 1<<6,

        }

        /// <summary>編集されたプロパティ</summary>
        public NUint    ModifiedProperties{ get; set; }

        /// <summary>編集されたプロパティ要素</summary>
        public NUint    ModifiedPropertyElements{ get; set; }

        /// <summary>フレームレートです。</summary>
        public NInt    FrameRate{ get; set; }

        /// <summary>解像度です。</summary>
        public NInt    Resolution{ get; set; }

        /// <summary>クリアカラーです。</summary>
        public NRgba    ClearColor{ get; set; }

        /// <summary>ワールドスケール 位置・速度・サイズレンジ</summary>
        public NFloat    WorldScaleRange{ get; set; }

        /// <summary>ワールドスケール 時間</summary>
        public NInt    WorldScaleTime{ get; set; }

        /// <summary>ワールドスケール グリッドと床の表示スケール</summary>
        public NFloat    GridScale{ get; set; }

        /// <summary>リニア編集モードの設定状態です(0: Disabled, 1: Enabled)。</summary>
        public NInt    LinearEditMode{ get; set; }


        /// <summary>コンストラクタ</summary>
        public MESSAGEPACKET_Viewer_InitSettings()
        {
            ModifiedProperties = new NUint(0xFFFFFFFF);
            ModifiedPropertyElements = new NUint(0xFFFFFFFF);

            FrameRate = new NInt();
            Resolution = new NInt();
            ClearColor = new NRgba();
            WorldScaleRange = new NFloat();
            WorldScaleTime = new NInt();
            GridScale = new NFloat();
            LinearEditMode = new NInt();
        }

        /// <summary>コンストラクタ</summary>
        public MESSAGEPACKET_Viewer_InitSettings(NInt _FrameRate, NInt _Resolution, NRgba _ClearColor, NFloat _WorldScaleRange, NInt _WorldScaleTime, NFloat _GridScale, NInt _LinearEditMode)
        {
            ModifiedProperties = new NUint(0xFFFFFFFF);
            ModifiedPropertyElements = new NUint(0xFFFFFFFF);

            FrameRate = _FrameRate;
            Resolution = _Resolution;
            ClearColor = _ClearColor;
            WorldScaleRange = _WorldScaleRange;
            WorldScaleTime = _WorldScaleTime;
            GridScale = _GridScale;
            LinearEditMode = _LinearEditMode;
        }
        /// <summary>文字列化</summary>
        public override string ToString()
        {
            string temp = string.Empty;
            temp += "FrameRate:(" + FrameRate.ToString() + ") ";
            temp += "Resolution:(" + Resolution.ToString() + ") ";
            temp += "ClearColor:(" + ClearColor.ToString() + ") ";
            temp += "WorldScaleRange:(" + WorldScaleRange.ToString() + ") ";
            temp += "WorldScaleTime:(" + WorldScaleTime.ToString() + ") ";
            temp += "GridScale:(" + GridScale.ToString() + ") ";
            temp += "LinearEditMode:(" + LinearEditMode.ToString() + ") ";
            return temp;
        }

        /// <summary>パケットサイズ（可変サイズに対応）</summary>
        public int PacketSize
        {
            get {
                int temp = (4 + 4);   // + 4 は ModifiedProperties と ModifiedPropertyElements
                temp += FrameRate.Size;
                temp += Resolution.Size;
                temp += ClearColor.Size;
                temp += WorldScaleRange.Size;
                temp += WorldScaleTime.Size;
                temp += GridScale.Size;
                temp += LinearEditMode.Size;
                temp += (4 - (temp & 0x03)) & 0x03;
                return temp;
            }
        }

        /// <summary>後に続く配列の長さ</summary>
        public override int ArrayLength
        {
            get{
                return 0;
            }
        }

        /// <summary>Read</summary>
        public override void Read(Protocol4FReader reader)
        {
             ModifiedProperties.Read(reader);
             ModifiedPropertyElements.Read(reader);
             FrameRate.Read(reader);
             Resolution.Read(reader);
             ClearColor.Read(reader);
             WorldScaleRange.Read(reader);
             WorldScaleTime.Read(reader);
             GridScale.Read(reader);
             LinearEditMode.Read(reader);
        }

        /// <summary>Write</summary>
        public override void Write(Protocol4FWriter writer)
        {
             ModifiedProperties.Write(writer);
             ModifiedPropertyElements.Write(writer);
             FrameRate.Write(writer);
             Resolution.Write(writer);
             ClearColor.Write(writer);
             WorldScaleRange.Write(writer);
             WorldScaleTime.Write(writer);
             GridScale.Write(writer);
             LinearEditMode.Write(writer);
        }
    };

    /// <summary>背景画像の設定です。</summary>
    public class MESSAGEPACKET_Viewer_SendBackgroundImage : BaseMessagePacket
    {
        /// <summary>メッセージタイプ</summary>
        public const MessageType MessagePacketType = MessageType.Viewer_SendBackgroundImage;

        /// <summary>構造体にした場合のサイズ</summary>
        public const int StructSize = 12 + 4 + 4;   // + 4 は ModifiedProperties と ModifiedPropertyElements

        /// <summary>後ろに続く配列のサイズ(byte)</summary>
        public virtual int ArraySize{ get{ return ByteData.StructSize * DataLength.Value; } }

        /// <summary>プロパティ</summary>
        [Flags]
        public enum PropertyFlag : uint
        {
            /// <summary>画像の幅です。</summary>
            Width = 1<<0,

            /// <summary>画像の高さです。</summary>
            Height = 1<<1,

            /// <summary>画像のRGBA配列の長さです。</summary>
            DataLength = 1<<2,

        }

        /// <summary>編集されたプロパティ</summary>
        public NUint    ModifiedProperties{ get; set; }

        /// <summary>編集されたプロパティ要素</summary>
        public NUint    ModifiedPropertyElements{ get; set; }

        /// <summary>画像の幅です。</summary>
        public NInt    Width{ get; set; }

        /// <summary>画像の高さです。</summary>
        public NInt    Height{ get; set; }

        /// <summary>画像のRGBA配列の長さです。</summary>
        public NInt    DataLength{ get; set; }


        /// <summary>コンストラクタ</summary>
        public MESSAGEPACKET_Viewer_SendBackgroundImage()
        {
            ModifiedProperties = new NUint(0xFFFFFFFF);
            ModifiedPropertyElements = new NUint(0xFFFFFFFF);

            Width = new NInt();
            Height = new NInt();
            DataLength = new NInt();
        }

        /// <summary>コンストラクタ</summary>
        public MESSAGEPACKET_Viewer_SendBackgroundImage(NInt _Width, NInt _Height, NInt _DataLength)
        {
            ModifiedProperties = new NUint(0xFFFFFFFF);
            ModifiedPropertyElements = new NUint(0xFFFFFFFF);

            Width = _Width;
            Height = _Height;
            DataLength = _DataLength;
        }
        /// <summary>文字列化</summary>
        public override string ToString()
        {
            string temp = string.Empty;
            temp += "Width:(" + Width.ToString() + ") ";
            temp += "Height:(" + Height.ToString() + ") ";
            temp += "DataLength:(" + DataLength.ToString() + ") ";
            return temp;
        }

        /// <summary>パケットサイズ（可変サイズに対応）</summary>
        public int PacketSize
        {
            get {
                int temp = (4 + 4);   // + 4 は ModifiedProperties と ModifiedPropertyElements
                temp += Width.Size;
                temp += Height.Size;
                temp += DataLength.Size;
                temp += (4 - (temp & 0x03)) & 0x03;
                return temp;
            }
        }

        /// <summary>後に続く配列の長さ</summary>
        public override int ArrayLength
        {
            get{
                return DataLength.Value;
            }
        }

        /// <summary>CreateStructPacket</summary>
        public override BaseStructPacket CreateStructPacket()
        {
            return new ByteData();
        }

        /// <summary>Read</summary>
        public override void Read(Protocol4FReader reader)
        {
             ModifiedProperties.Read(reader);
             ModifiedPropertyElements.Read(reader);
             Width.Read(reader);
             Height.Read(reader);
             DataLength.Read(reader);
        }

        /// <summary>Write</summary>
        public override void Write(Protocol4FWriter writer)
        {
             ModifiedProperties.Write(writer);
             ModifiedPropertyElements.Write(writer);
             Width.Write(writer);
             Height.Write(writer);
             DataLength.Write(writer);
        }
    };

    /// <summary>プレビュー設定のモデル情報を送信します</summary>
    public class MESSAGEPACKET_Viewer_SendPreviewModel : BaseMessagePacket
    {
        /// <summary>メッセージタイプ</summary>
        public const MessageType MessagePacketType = MessageType.Viewer_SendPreviewModel;

        /// <summary>構造体にした場合のサイズ</summary>
        public const int StructSize = 524 + 4 + 4;   // + 4 は ModifiedProperties と ModifiedPropertyElements

        /// <summary>後ろに続く配列のサイズ(byte)</summary>
        public virtual int ArraySize{ get{ return 0; } }

        /// <summary>プロパティ</summary>
        [Flags]
        public enum PropertyFlag : uint
        {
            /// <summary>モデルノードＩＤ</summary>
            ModelID = 1<<0,

            /// <summary>モデルファイルのパス</summary>
            ModelPath = 1<<1,

        }

        /// <summary>編集されたプロパティ</summary>
        public NUint    ModifiedProperties{ get; set; }

        /// <summary>編集されたプロパティ要素</summary>
        public NUint    ModifiedPropertyElements{ get; set; }

        /// <summary>モデルノードＩＤ</summary>
        public NInt    ModelID{ get; set; }

        /// <summary>モデルファイルのパス</summary>
        public NString    ModelPath{ get; set; }


        /// <summary>コンストラクタ</summary>
        public MESSAGEPACKET_Viewer_SendPreviewModel()
        {
            ModifiedProperties = new NUint(0xFFFFFFFF);
            ModifiedPropertyElements = new NUint(0xFFFFFFFF);

            ModelID = new NInt();
            ModelPath = new NString();
        }

        /// <summary>コンストラクタ</summary>
        public MESSAGEPACKET_Viewer_SendPreviewModel(NInt _ModelID, NString _ModelPath)
        {
            ModifiedProperties = new NUint(0xFFFFFFFF);
            ModifiedPropertyElements = new NUint(0xFFFFFFFF);

            ModelID = _ModelID;
            ModelPath = _ModelPath;
        }
        /// <summary>文字列化</summary>
        public override string ToString()
        {
            string temp = string.Empty;
            temp += "ModelID:(" + ModelID.ToString() + ") ";
            temp += "ModelPath:(" + ModelPath.ToString() + ") ";
            return temp;
        }

        /// <summary>パケットサイズ（可変サイズに対応）</summary>
        public int PacketSize
        {
            get {
                int temp = (4 + 4);   // + 4 は ModifiedProperties と ModifiedPropertyElements
                temp += ModelID.Size;
                temp += ModelPath.Size;
                temp += (4 - (temp & 0x03)) & 0x03;
                return temp;
            }
        }

        /// <summary>後に続く配列の長さ</summary>
        public override int ArrayLength
        {
            get{
                return 0;
            }
        }

        /// <summary>Read</summary>
        public override void Read(Protocol4FReader reader)
        {
             ModifiedProperties.Read(reader);
             ModifiedPropertyElements.Read(reader);
             ModelID.Read(reader);
             ModelPath.Read(reader);
        }

        /// <summary>Write</summary>
        public override void Write(Protocol4FWriter writer)
        {
             ModifiedProperties.Write(writer);
             ModifiedPropertyElements.Write(writer);
             ModelID.Write(writer);
             ModelPath.Write(writer);
        }
    };

    /// <summary>モデル情報を削除します。</summary>
    public class MESSAGEPACKET_Viewer_ClosePreviewModel : BaseMessagePacket
    {
        /// <summary>メッセージタイプ</summary>
        public const MessageType MessagePacketType = MessageType.Viewer_ClosePreviewModel;

        /// <summary>構造体にした場合のサイズ</summary>
        public const int StructSize = 524 + 4 + 4;   // + 4 は ModifiedProperties と ModifiedPropertyElements

        /// <summary>後ろに続く配列のサイズ(byte)</summary>
        public virtual int ArraySize{ get{ return 0; } }

        /// <summary>プロパティ</summary>
        [Flags]
        public enum PropertyFlag : uint
        {
            /// <summary>モデルノードＩＤ</summary>
            ModelID = 1<<0,

            /// <summary>モデル名（SendModelInfoV2E で取得したモデル名）</summary>
            ModelName = 1<<1,

        }

        /// <summary>編集されたプロパティ</summary>
        public NUint    ModifiedProperties{ get; set; }

        /// <summary>編集されたプロパティ要素</summary>
        public NUint    ModifiedPropertyElements{ get; set; }

        /// <summary>モデルノードＩＤ</summary>
        public NInt    ModelID{ get; set; }

        /// <summary>モデル名（SendModelInfoV2E で取得したモデル名）</summary>
        public NString    ModelName{ get; set; }


        /// <summary>コンストラクタ</summary>
        public MESSAGEPACKET_Viewer_ClosePreviewModel()
        {
            ModifiedProperties = new NUint(0xFFFFFFFF);
            ModifiedPropertyElements = new NUint(0xFFFFFFFF);

            ModelID = new NInt();
            ModelName = new NString();
        }

        /// <summary>コンストラクタ</summary>
        public MESSAGEPACKET_Viewer_ClosePreviewModel(NInt _ModelID, NString _ModelName)
        {
            ModifiedProperties = new NUint(0xFFFFFFFF);
            ModifiedPropertyElements = new NUint(0xFFFFFFFF);

            ModelID = _ModelID;
            ModelName = _ModelName;
        }
        /// <summary>文字列化</summary>
        public override string ToString()
        {
            string temp = string.Empty;
            temp += "ModelID:(" + ModelID.ToString() + ") ";
            temp += "ModelName:(" + ModelName.ToString() + ") ";
            return temp;
        }

        /// <summary>パケットサイズ（可変サイズに対応）</summary>
        public int PacketSize
        {
            get {
                int temp = (4 + 4);   // + 4 は ModifiedProperties と ModifiedPropertyElements
                temp += ModelID.Size;
                temp += ModelName.Size;
                temp += (4 - (temp & 0x03)) & 0x03;
                return temp;
            }
        }

        /// <summary>後に続く配列の長さ</summary>
        public override int ArrayLength
        {
            get{
                return 0;
            }
        }

        /// <summary>Read</summary>
        public override void Read(Protocol4FReader reader)
        {
             ModifiedProperties.Read(reader);
             ModifiedPropertyElements.Read(reader);
             ModelID.Read(reader);
             ModelName.Read(reader);
        }

        /// <summary>Write</summary>
        public override void Write(Protocol4FWriter writer)
        {
             ModifiedProperties.Write(writer);
             ModifiedPropertyElements.Write(writer);
             ModelID.Write(writer);
             ModelName.Write(writer);
        }
    };

    /// <summary>モデル情報を送信します（Viewer -> EM4C）</summary>
    public class MESSAGEPACKET_Viewer_SendModelInfoV2E : BaseMessagePacket
    {
        /// <summary>メッセージタイプ</summary>
        public const MessageType MessagePacketType = MessageType.Viewer_SendModelInfoV2E;

        /// <summary>構造体にした場合のサイズ</summary>
        public const int StructSize = 536 + 4 + 4;   // + 4 は ModifiedProperties と ModifiedPropertyElements

        /// <summary>後ろに続く配列のサイズ(byte)</summary>
        public virtual int ArraySize{ get{ return 0; } }

        /// <summary>プロパティ</summary>
        [Flags]
        public enum PropertyFlag : uint
        {
            /// <summary>モデルノードＩＤ</summary>
            ModelID = 1<<0,

            /// <summary>モデル名</summary>
            ModelName = 1<<1,

            /// <summary>ボーン数</summary>
            BoneCount = 1<<2,

            /// <summary>データ（&Data からLength分データが続く）</summary>
            Data = 1<<3,

        }

        /// <summary>編集されたプロパティ</summary>
        public NUint    ModifiedProperties{ get; set; }

        /// <summary>編集されたプロパティ要素</summary>
        public NUint    ModifiedPropertyElements{ get; set; }

        /// <summary>モデルノードＩＤ</summary>
        public NInt    ModelID{ get; set; }

        /// <summary>モデル名</summary>
        public NString    ModelName{ get; set; }

        /// <summary>ボーン数</summary>
        public NInt    BoneCount{ get; set; }

        /// <summary>データ（&Data からLength分データが続く）</summary>
        public ByteArray    Data{ get; set; }


        /// <summary>コンストラクタ</summary>
        public MESSAGEPACKET_Viewer_SendModelInfoV2E()
        {
            ModifiedProperties = new NUint(0xFFFFFFFF);
            ModifiedPropertyElements = new NUint(0xFFFFFFFF);

            ModelID = new NInt();
            ModelName = new NString();
            BoneCount = new NInt();
            Data = new ByteArray();
        }

        /// <summary>コンストラクタ</summary>
        public MESSAGEPACKET_Viewer_SendModelInfoV2E(NInt _ModelID, NString _ModelName, NInt _BoneCount, ByteArray _Data)
        {
            ModifiedProperties = new NUint(0xFFFFFFFF);
            ModifiedPropertyElements = new NUint(0xFFFFFFFF);

            ModelID = _ModelID;
            ModelName = _ModelName;
            BoneCount = _BoneCount;
            Data = _Data;
        }
        /// <summary>文字列化</summary>
        public override string ToString()
        {
            string temp = string.Empty;
            temp += "ModelID:(" + ModelID.ToString() + ") ";
            temp += "ModelName:(" + ModelName.ToString() + ") ";
            temp += "BoneCount:(" + BoneCount.ToString() + ") ";
            temp += "Data:(" + Data.ToString() + ") ";
            return temp;
        }

        /// <summary>パケットサイズ（可変サイズに対応）</summary>
        public int PacketSize
        {
            get {
                int temp = (4 + 4);   // + 4 は ModifiedProperties と ModifiedPropertyElements
                temp += ModelID.Size;
                temp += ModelName.Size;
                temp += BoneCount.Size;
                temp += Data.Size;
                temp += (4 - (temp & 0x03)) & 0x03;
                return temp;
            }
        }

        /// <summary>後に続く配列の長さ</summary>
        public override int ArrayLength
        {
            get{
                return 0;
            }
        }

        /// <summary>Read</summary>
        public override void Read(Protocol4FReader reader)
        {
             ModifiedProperties.Read(reader);
             ModifiedPropertyElements.Read(reader);
             ModelID.Read(reader);
             ModelName.Read(reader);
             BoneCount.Read(reader);
             Data.Read(reader);
        }

        /// <summary>Write</summary>
        public override void Write(Protocol4FWriter writer)
        {
             ModifiedProperties.Write(writer);
             ModifiedPropertyElements.Write(writer);
             ModelID.Write(writer);
             ModelName.Write(writer);
             BoneCount.Write(writer);
             Data.Write(writer);
        }
    };

    /// <summary>モデル情報を初期化します。</summary>
    public class MESSAGEPACKET_Viewer_ClearPreviewModelV2E : BaseMessagePacket
    {
        /// <summary>メッセージタイプ</summary>
        public const MessageType MessagePacketType = MessageType.Viewer_ClearPreviewModelV2E;

        /// <summary>構造体にした場合のサイズ</summary>
        public const int StructSize = 0 + 4 + 4;   // + 4 は ModifiedProperties と ModifiedPropertyElements

        /// <summary>後ろに続く配列のサイズ(byte)</summary>
        public virtual int ArraySize{ get{ return 0; } }

        /// <summary>編集されたプロパティ</summary>
        public NUint    ModifiedProperties{ get; set; }

        /// <summary>編集されたプロパティ要素</summary>
        public NUint    ModifiedPropertyElements{ get; set; }


        /// <summary>コンストラクタ</summary>
        public MESSAGEPACKET_Viewer_ClearPreviewModelV2E()
        {
            ModifiedProperties = new NUint(0xFFFFFFFF);
            ModifiedPropertyElements = new NUint(0xFFFFFFFF);

        }

        /// <summary>文字列化</summary>
        public override string ToString()
        {
            string temp = string.Empty;
            return temp;
        }

        /// <summary>パケットサイズ（可変サイズに対応）</summary>
        public int PacketSize
        {
            get {
                int temp = (4 + 4);   // + 4 は ModifiedProperties と ModifiedPropertyElements
                temp += (4 - (temp & 0x03)) & 0x03;
                return temp;
            }
        }

        /// <summary>後に続く配列の長さ</summary>
        public override int ArrayLength
        {
            get{
                return 0;
            }
        }

        /// <summary>Read</summary>
        public override void Read(Protocol4FReader reader)
        {
             ModifiedProperties.Read(reader);
             ModifiedPropertyElements.Read(reader);
        }

        /// <summary>Write</summary>
        public override void Write(Protocol4FWriter writer)
        {
             ModifiedProperties.Write(writer);
             ModifiedPropertyElements.Write(writer);
        }
    };

    /// <summary>モデル情報の転送を開始します。</summary>
    public class MESSAGEPACKET_Viewer_BeginSendModelInfoV2E : BaseMessagePacket
    {
        /// <summary>メッセージタイプ</summary>
        public const MessageType MessagePacketType = MessageType.Viewer_BeginSendModelInfoV2E;

        /// <summary>構造体にした場合のサイズ</summary>
        public const int StructSize = 0 + 4 + 4;   // + 4 は ModifiedProperties と ModifiedPropertyElements

        /// <summary>後ろに続く配列のサイズ(byte)</summary>
        public virtual int ArraySize{ get{ return 0; } }

        /// <summary>編集されたプロパティ</summary>
        public NUint    ModifiedProperties{ get; set; }

        /// <summary>編集されたプロパティ要素</summary>
        public NUint    ModifiedPropertyElements{ get; set; }


        /// <summary>コンストラクタ</summary>
        public MESSAGEPACKET_Viewer_BeginSendModelInfoV2E()
        {
            ModifiedProperties = new NUint(0xFFFFFFFF);
            ModifiedPropertyElements = new NUint(0xFFFFFFFF);

        }

        /// <summary>文字列化</summary>
        public override string ToString()
        {
            string temp = string.Empty;
            return temp;
        }

        /// <summary>パケットサイズ（可変サイズに対応）</summary>
        public int PacketSize
        {
            get {
                int temp = (4 + 4);   // + 4 は ModifiedProperties と ModifiedPropertyElements
                temp += (4 - (temp & 0x03)) & 0x03;
                return temp;
            }
        }

        /// <summary>後に続く配列の長さ</summary>
        public override int ArrayLength
        {
            get{
                return 0;
            }
        }

        /// <summary>Read</summary>
        public override void Read(Protocol4FReader reader)
        {
             ModifiedProperties.Read(reader);
             ModifiedPropertyElements.Read(reader);
        }

        /// <summary>Write</summary>
        public override void Write(Protocol4FWriter writer)
        {
             ModifiedProperties.Write(writer);
             ModifiedPropertyElements.Write(writer);
        }
    };

    /// <summary>モデル情報の転送を終了します。</summary>
    public class MESSAGEPACKET_Viewer_EndSendModelInfoV2E : BaseMessagePacket
    {
        /// <summary>メッセージタイプ</summary>
        public const MessageType MessagePacketType = MessageType.Viewer_EndSendModelInfoV2E;

        /// <summary>構造体にした場合のサイズ</summary>
        public const int StructSize = 0 + 4 + 4;   // + 4 は ModifiedProperties と ModifiedPropertyElements

        /// <summary>後ろに続く配列のサイズ(byte)</summary>
        public virtual int ArraySize{ get{ return 0; } }

        /// <summary>編集されたプロパティ</summary>
        public NUint    ModifiedProperties{ get; set; }

        /// <summary>編集されたプロパティ要素</summary>
        public NUint    ModifiedPropertyElements{ get; set; }


        /// <summary>コンストラクタ</summary>
        public MESSAGEPACKET_Viewer_EndSendModelInfoV2E()
        {
            ModifiedProperties = new NUint(0xFFFFFFFF);
            ModifiedPropertyElements = new NUint(0xFFFFFFFF);

        }

        /// <summary>文字列化</summary>
        public override string ToString()
        {
            string temp = string.Empty;
            return temp;
        }

        /// <summary>パケットサイズ（可変サイズに対応）</summary>
        public int PacketSize
        {
            get {
                int temp = (4 + 4);   // + 4 は ModifiedProperties と ModifiedPropertyElements
                temp += (4 - (temp & 0x03)) & 0x03;
                return temp;
            }
        }

        /// <summary>後に続く配列の長さ</summary>
        public override int ArrayLength
        {
            get{
                return 0;
            }
        }

        /// <summary>Read</summary>
        public override void Read(Protocol4FReader reader)
        {
             ModifiedProperties.Read(reader);
             ModifiedPropertyElements.Read(reader);
        }

        /// <summary>Write</summary>
        public override void Write(Protocol4FWriter writer)
        {
             ModifiedProperties.Write(writer);
             ModifiedPropertyElements.Write(writer);
        }
    };

    /// <summary>カスタムシェーダフォルダパスを指定します。</summary>
    public class MESSAGEPACKET_Viewer_SetUserShaderPathV2E : BaseMessagePacket
    {
        /// <summary>メッセージタイプ</summary>
        public const MessageType MessagePacketType = MessageType.Viewer_SetUserShaderPathV2E;

        /// <summary>構造体にした場合のサイズ</summary>
        public const int StructSize = 524 + 4 + 4;   // + 4 は ModifiedProperties と ModifiedPropertyElements

        /// <summary>後ろに続く配列のサイズ(byte)</summary>
        public virtual int ArraySize{ get{ return 0; } }

        /// <summary>プロパティ</summary>
        [Flags]
        public enum PropertyFlag : uint
        {
            /// <summary>ユーザシェーダパス</summary>
            UserShaderPath = 1<<0,

            /// <summary>強制的にカスタムシェーダフォルダパスを指定します。</summary>
            ForceAssign = 1<<1,

        }

        /// <summary>編集されたプロパティ</summary>
        public NUint    ModifiedProperties{ get; set; }

        /// <summary>編集されたプロパティ要素</summary>
        public NUint    ModifiedPropertyElements{ get; set; }

        /// <summary>ユーザシェーダパス</summary>
        public NString    UserShaderPath{ get; set; }

        /// <summary>強制的にカスタムシェーダフォルダパスを指定します。</summary>
        public NInt    ForceAssign{ get; set; }


        /// <summary>コンストラクタ</summary>
        public MESSAGEPACKET_Viewer_SetUserShaderPathV2E()
        {
            ModifiedProperties = new NUint(0xFFFFFFFF);
            ModifiedPropertyElements = new NUint(0xFFFFFFFF);

            UserShaderPath = new NString();
            ForceAssign = new NInt();
        }

        /// <summary>コンストラクタ</summary>
        public MESSAGEPACKET_Viewer_SetUserShaderPathV2E(NString _UserShaderPath, NInt _ForceAssign)
        {
            ModifiedProperties = new NUint(0xFFFFFFFF);
            ModifiedPropertyElements = new NUint(0xFFFFFFFF);

            UserShaderPath = _UserShaderPath;
            ForceAssign = _ForceAssign;
        }
        /// <summary>文字列化</summary>
        public override string ToString()
        {
            string temp = string.Empty;
            temp += "UserShaderPath:(" + UserShaderPath.ToString() + ") ";
            temp += "ForceAssign:(" + ForceAssign.ToString() + ") ";
            return temp;
        }

        /// <summary>パケットサイズ（可変サイズに対応）</summary>
        public int PacketSize
        {
            get {
                int temp = (4 + 4);   // + 4 は ModifiedProperties と ModifiedPropertyElements
                temp += UserShaderPath.Size;
                temp += ForceAssign.Size;
                temp += (4 - (temp & 0x03)) & 0x03;
                return temp;
            }
        }

        /// <summary>後に続く配列の長さ</summary>
        public override int ArrayLength
        {
            get{
                return 0;
            }
        }

        /// <summary>Read</summary>
        public override void Read(Protocol4FReader reader)
        {
             ModifiedProperties.Read(reader);
             ModifiedPropertyElements.Read(reader);
             UserShaderPath.Read(reader);
             ForceAssign.Read(reader);
        }

        /// <summary>Write</summary>
        public override void Write(Protocol4FWriter writer)
        {
             ModifiedProperties.Write(writer);
             ModifiedPropertyElements.Write(writer);
             UserShaderPath.Write(writer);
             ForceAssign.Write(writer);
        }
    };

    /// <summary>カスタムアクションフォルダパスを指定します。</summary>
    public class MESSAGEPACKET_Viewer_SetUserDataPathV2E : BaseMessagePacket
    {
        /// <summary>メッセージタイプ</summary>
        public const MessageType MessagePacketType = MessageType.Viewer_SetUserDataPathV2E;

        /// <summary>構造体にした場合のサイズ</summary>
        public const int StructSize = 524 + 4 + 4;   // + 4 は ModifiedProperties と ModifiedPropertyElements

        /// <summary>後ろに続く配列のサイズ(byte)</summary>
        public virtual int ArraySize{ get{ return 0; } }

        /// <summary>プロパティ</summary>
        [Flags]
        public enum PropertyFlag : uint
        {
            /// <summary>ユーザデータパス</summary>
            UserDataPath = 1<<0,

            /// <summary>強制的にカスタムアクションフォルダパスを指定します。</summary>
            ForceAssign = 1<<1,

        }

        /// <summary>編集されたプロパティ</summary>
        public NUint    ModifiedProperties{ get; set; }

        /// <summary>編集されたプロパティ要素</summary>
        public NUint    ModifiedPropertyElements{ get; set; }

        /// <summary>ユーザデータパス</summary>
        public NString    UserDataPath{ get; set; }

        /// <summary>強制的にカスタムアクションフォルダパスを指定します。</summary>
        public NInt    ForceAssign{ get; set; }


        /// <summary>コンストラクタ</summary>
        public MESSAGEPACKET_Viewer_SetUserDataPathV2E()
        {
            ModifiedProperties = new NUint(0xFFFFFFFF);
            ModifiedPropertyElements = new NUint(0xFFFFFFFF);

            UserDataPath = new NString();
            ForceAssign = new NInt();
        }

        /// <summary>コンストラクタ</summary>
        public MESSAGEPACKET_Viewer_SetUserDataPathV2E(NString _UserDataPath, NInt _ForceAssign)
        {
            ModifiedProperties = new NUint(0xFFFFFFFF);
            ModifiedPropertyElements = new NUint(0xFFFFFFFF);

            UserDataPath = _UserDataPath;
            ForceAssign = _ForceAssign;
        }
        /// <summary>文字列化</summary>
        public override string ToString()
        {
            string temp = string.Empty;
            temp += "UserDataPath:(" + UserDataPath.ToString() + ") ";
            temp += "ForceAssign:(" + ForceAssign.ToString() + ") ";
            return temp;
        }

        /// <summary>パケットサイズ（可変サイズに対応）</summary>
        public int PacketSize
        {
            get {
                int temp = (4 + 4);   // + 4 は ModifiedProperties と ModifiedPropertyElements
                temp += UserDataPath.Size;
                temp += ForceAssign.Size;
                temp += (4 - (temp & 0x03)) & 0x03;
                return temp;
            }
        }

        /// <summary>後に続く配列の長さ</summary>
        public override int ArrayLength
        {
            get{
                return 0;
            }
        }

        /// <summary>Read</summary>
        public override void Read(Protocol4FReader reader)
        {
             ModifiedProperties.Read(reader);
             ModifiedPropertyElements.Read(reader);
             UserDataPath.Read(reader);
             ForceAssign.Read(reader);
        }

        /// <summary>Write</summary>
        public override void Write(Protocol4FWriter writer)
        {
             ModifiedProperties.Write(writer);
             ModifiedPropertyElements.Write(writer);
             UserDataPath.Write(writer);
             ForceAssign.Write(writer);
        }
    };

    /// <summary>ユーザー設定ファイルパスを指定します。</summary>
    public class MESSAGEPACKET_Viewer_SetUserConfigPathV2E : BaseMessagePacket
    {
        /// <summary>メッセージタイプ</summary>
        public const MessageType MessagePacketType = MessageType.Viewer_SetUserConfigPathV2E;

        /// <summary>構造体にした場合のサイズ</summary>
        public const int StructSize = 524 + 4 + 4;   // + 4 は ModifiedProperties と ModifiedPropertyElements

        /// <summary>後ろに続く配列のサイズ(byte)</summary>
        public virtual int ArraySize{ get{ return 0; } }

        /// <summary>プロパティ</summary>
        [Flags]
        public enum PropertyFlag : uint
        {
            /// <summary>ユーザ設定ファイルパス</summary>
            FilePath = 1<<0,

            /// <summary>強制的にユーザー設定ファイルパスを指定します。</summary>
            ForceAssign = 1<<1,

        }

        /// <summary>編集されたプロパティ</summary>
        public NUint    ModifiedProperties{ get; set; }

        /// <summary>編集されたプロパティ要素</summary>
        public NUint    ModifiedPropertyElements{ get; set; }

        /// <summary>ユーザ設定ファイルパス</summary>
        public NString    FilePath{ get; set; }

        /// <summary>強制的にユーザー設定ファイルパスを指定します。</summary>
        public NInt    ForceAssign{ get; set; }


        /// <summary>コンストラクタ</summary>
        public MESSAGEPACKET_Viewer_SetUserConfigPathV2E()
        {
            ModifiedProperties = new NUint(0xFFFFFFFF);
            ModifiedPropertyElements = new NUint(0xFFFFFFFF);

            FilePath = new NString();
            ForceAssign = new NInt();
        }

        /// <summary>コンストラクタ</summary>
        public MESSAGEPACKET_Viewer_SetUserConfigPathV2E(NString _FilePath, NInt _ForceAssign)
        {
            ModifiedProperties = new NUint(0xFFFFFFFF);
            ModifiedPropertyElements = new NUint(0xFFFFFFFF);

            FilePath = _FilePath;
            ForceAssign = _ForceAssign;
        }
        /// <summary>文字列化</summary>
        public override string ToString()
        {
            string temp = string.Empty;
            temp += "FilePath:(" + FilePath.ToString() + ") ";
            temp += "ForceAssign:(" + ForceAssign.ToString() + ") ";
            return temp;
        }

        /// <summary>パケットサイズ（可変サイズに対応）</summary>
        public int PacketSize
        {
            get {
                int temp = (4 + 4);   // + 4 は ModifiedProperties と ModifiedPropertyElements
                temp += FilePath.Size;
                temp += ForceAssign.Size;
                temp += (4 - (temp & 0x03)) & 0x03;
                return temp;
            }
        }

        /// <summary>後に続く配列の長さ</summary>
        public override int ArrayLength
        {
            get{
                return 0;
            }
        }

        /// <summary>Read</summary>
        public override void Read(Protocol4FReader reader)
        {
             ModifiedProperties.Read(reader);
             ModifiedPropertyElements.Read(reader);
             FilePath.Read(reader);
             ForceAssign.Read(reader);
        }

        /// <summary>Write</summary>
        public override void Write(Protocol4FWriter writer)
        {
             ModifiedProperties.Write(writer);
             ModifiedPropertyElements.Write(writer);
             FilePath.Write(writer);
             ForceAssign.Write(writer);
        }
    };

    /// <summary>ビューアからエミッタセットを新規作成します。</summary>
    public class MESSAGEPACKET_Viewer_CreateEmitterSetV2E : BaseMessagePacket
    {
        /// <summary>メッセージタイプ</summary>
        public const MessageType MessagePacketType = MessageType.Viewer_CreateEmitterSetV2E;

        /// <summary>構造体にした場合のサイズ</summary>
        public const int StructSize = 520 + 4 + 4;   // + 4 は ModifiedProperties と ModifiedPropertyElements

        /// <summary>後ろに続く配列のサイズ(byte)</summary>
        public virtual int ArraySize{ get{ return 0; } }

        /// <summary>プロパティ</summary>
        [Flags]
        public enum PropertyFlag : uint
        {
            /// <summary>作成したエミッタセット名</summary>
            EmitterSetName = 1<<0,

        }

        /// <summary>編集されたプロパティ</summary>
        public NUint    ModifiedProperties{ get; set; }

        /// <summary>編集されたプロパティ要素</summary>
        public NUint    ModifiedPropertyElements{ get; set; }

        /// <summary>作成したエミッタセット名</summary>
        public NString    EmitterSetName{ get; set; }


        /// <summary>コンストラクタ</summary>
        public MESSAGEPACKET_Viewer_CreateEmitterSetV2E()
        {
            ModifiedProperties = new NUint(0xFFFFFFFF);
            ModifiedPropertyElements = new NUint(0xFFFFFFFF);

            EmitterSetName = new NString();
        }

        /// <summary>コンストラクタ</summary>
        public MESSAGEPACKET_Viewer_CreateEmitterSetV2E(NString _EmitterSetName)
        {
            ModifiedProperties = new NUint(0xFFFFFFFF);
            ModifiedPropertyElements = new NUint(0xFFFFFFFF);

            EmitterSetName = _EmitterSetName;
        }
        /// <summary>文字列化</summary>
        public override string ToString()
        {
            string temp = string.Empty;
            temp += "EmitterSetName:(" + EmitterSetName.ToString() + ") ";
            return temp;
        }

        /// <summary>パケットサイズ（可変サイズに対応）</summary>
        public int PacketSize
        {
            get {
                int temp = (4 + 4);   // + 4 は ModifiedProperties と ModifiedPropertyElements
                temp += EmitterSetName.Size;
                temp += (4 - (temp & 0x03)) & 0x03;
                return temp;
            }
        }

        /// <summary>後に続く配列の長さ</summary>
        public override int ArrayLength
        {
            get{
                return 0;
            }
        }

        /// <summary>Read</summary>
        public override void Read(Protocol4FReader reader)
        {
             ModifiedProperties.Read(reader);
             ModifiedPropertyElements.Read(reader);
             EmitterSetName.Read(reader);
        }

        /// <summary>Write</summary>
        public override void Write(Protocol4FWriter writer)
        {
             ModifiedProperties.Write(writer);
             ModifiedPropertyElements.Write(writer);
             EmitterSetName.Write(writer);
        }
    };

    /// <summary>指定したエミッタセットの複製します。</summary>
    public class MESSAGEPACKET_Viewer_CloneEmitterSetV2E : BaseMessagePacket
    {
        /// <summary>メッセージタイプ</summary>
        public const MessageType MessagePacketType = MessageType.Viewer_CloneEmitterSetV2E;

        /// <summary>構造体にした場合のサイズ</summary>
        public const int StructSize = 1040 + 4 + 4;   // + 4 は ModifiedProperties と ModifiedPropertyElements

        /// <summary>後ろに続く配列のサイズ(byte)</summary>
        public virtual int ArraySize{ get{ return 0; } }

        /// <summary>プロパティ</summary>
        [Flags]
        public enum PropertyFlag : uint
        {
            /// <summary>複製元のエミッタセット名</summary>
            SrcEmitterSetName = 1<<0,

            /// <summary>複製先のエミッタセット名</summary>
            DstEmitterSetName = 1<<1,

        }

        /// <summary>編集されたプロパティ</summary>
        public NUint    ModifiedProperties{ get; set; }

        /// <summary>編集されたプロパティ要素</summary>
        public NUint    ModifiedPropertyElements{ get; set; }

        /// <summary>複製元のエミッタセット名</summary>
        public NString    SrcEmitterSetName{ get; set; }

        /// <summary>複製先のエミッタセット名</summary>
        public NString    DstEmitterSetName{ get; set; }


        /// <summary>コンストラクタ</summary>
        public MESSAGEPACKET_Viewer_CloneEmitterSetV2E()
        {
            ModifiedProperties = new NUint(0xFFFFFFFF);
            ModifiedPropertyElements = new NUint(0xFFFFFFFF);

            SrcEmitterSetName = new NString();
            DstEmitterSetName = new NString();
        }

        /// <summary>コンストラクタ</summary>
        public MESSAGEPACKET_Viewer_CloneEmitterSetV2E(NString _SrcEmitterSetName, NString _DstEmitterSetName)
        {
            ModifiedProperties = new NUint(0xFFFFFFFF);
            ModifiedPropertyElements = new NUint(0xFFFFFFFF);

            SrcEmitterSetName = _SrcEmitterSetName;
            DstEmitterSetName = _DstEmitterSetName;
        }
        /// <summary>文字列化</summary>
        public override string ToString()
        {
            string temp = string.Empty;
            temp += "SrcEmitterSetName:(" + SrcEmitterSetName.ToString() + ") ";
            temp += "DstEmitterSetName:(" + DstEmitterSetName.ToString() + ") ";
            return temp;
        }

        /// <summary>パケットサイズ（可変サイズに対応）</summary>
        public int PacketSize
        {
            get {
                int temp = (4 + 4);   // + 4 は ModifiedProperties と ModifiedPropertyElements
                temp += SrcEmitterSetName.Size;
                temp += DstEmitterSetName.Size;
                temp += (4 - (temp & 0x03)) & 0x03;
                return temp;
            }
        }

        /// <summary>後に続く配列の長さ</summary>
        public override int ArrayLength
        {
            get{
                return 0;
            }
        }

        /// <summary>Read</summary>
        public override void Read(Protocol4FReader reader)
        {
             ModifiedProperties.Read(reader);
             ModifiedPropertyElements.Read(reader);
             SrcEmitterSetName.Read(reader);
             DstEmitterSetName.Read(reader);
        }

        /// <summary>Write</summary>
        public override void Write(Protocol4FWriter writer)
        {
             ModifiedProperties.Write(writer);
             ModifiedPropertyElements.Write(writer);
             SrcEmitterSetName.Write(writer);
             DstEmitterSetName.Write(writer);
        }
    };

    /// <summary>コネクションを切断します。</summary>
    public class MESSAGEPACKET_Viewer_CloseConnection : BaseMessagePacket
    {
        /// <summary>メッセージタイプ</summary>
        public const MessageType MessagePacketType = MessageType.Viewer_CloseConnection;

        /// <summary>構造体にした場合のサイズ</summary>
        public const int StructSize = 0 + 4 + 4;   // + 4 は ModifiedProperties と ModifiedPropertyElements

        /// <summary>後ろに続く配列のサイズ(byte)</summary>
        public virtual int ArraySize{ get{ return 0; } }

        /// <summary>編集されたプロパティ</summary>
        public NUint    ModifiedProperties{ get; set; }

        /// <summary>編集されたプロパティ要素</summary>
        public NUint    ModifiedPropertyElements{ get; set; }


        /// <summary>コンストラクタ</summary>
        public MESSAGEPACKET_Viewer_CloseConnection()
        {
            ModifiedProperties = new NUint(0xFFFFFFFF);
            ModifiedPropertyElements = new NUint(0xFFFFFFFF);

        }

        /// <summary>文字列化</summary>
        public override string ToString()
        {
            string temp = string.Empty;
            return temp;
        }

        /// <summary>パケットサイズ（可変サイズに対応）</summary>
        public int PacketSize
        {
            get {
                int temp = (4 + 4);   // + 4 は ModifiedProperties と ModifiedPropertyElements
                temp += (4 - (temp & 0x03)) & 0x03;
                return temp;
            }
        }

        /// <summary>後に続く配列の長さ</summary>
        public override int ArrayLength
        {
            get{
                return 0;
            }
        }

        /// <summary>Read</summary>
        public override void Read(Protocol4FReader reader)
        {
             ModifiedProperties.Read(reader);
             ModifiedPropertyElements.Read(reader);
        }

        /// <summary>Write</summary>
        public override void Write(Protocol4FWriter writer)
        {
             ModifiedProperties.Write(writer);
             ModifiedPropertyElements.Write(writer);
        }
    };

    /// <summary>カメラの設定を送信します。</summary>
    public class MESSAGEPACKET_Viewer_CameraSettings : BaseMessagePacket
    {
        /// <summary>メッセージタイプ</summary>
        public const MessageType MessagePacketType = MessageType.Viewer_CameraSettings;

        /// <summary>構造体にした場合のサイズ</summary>
        public const int StructSize = 48 + 4 + 4;   // + 4 は ModifiedProperties と ModifiedPropertyElements

        /// <summary>後ろに続く配列のサイズ(byte)</summary>
        public virtual int ArraySize{ get{ return 0; } }

        /// <summary>プロパティ</summary>
        [Flags]
        public enum PropertyFlag : uint
        {
            /// <summary>カメラの位置。</summary>
            CameraPos = 1<<0,

            /// <summary>カメラの注視点。</summary>
            LookAtPos = 1<<1,

            /// <summary>カメラの回転。</summary>
            CameraRotation = 1<<2,

            /// <summary>カメラタイプ。</summary>
            CameraProjType = 1<<3,

            /// <summary>アスペクト比。</summary>
            AspectRatio = 1<<4,

            /// <summary>画角(パースペクティブ) / 画像の縦サイズ(オルソ)</summary>
            FOV = 1<<5,

            /// <summary>ニア</summary>
            FrustumNear = 1<<6,

            /// <summary>ファー</summary>
            FrustumFar = 1<<7,

        }

        /// <summary>編集されたプロパティ</summary>
        public NUint    ModifiedProperties{ get; set; }

        /// <summary>編集されたプロパティ要素</summary>
        public NUint    ModifiedPropertyElements{ get; set; }

        /// <summary>カメラの位置。</summary>
        public NVector3    CameraPos{ get; set; }

        /// <summary>カメラの注視点。</summary>
        public NVector3    LookAtPos{ get; set; }

        /// <summary>カメラの回転。</summary>
        public NFloat    CameraRotation{ get; set; }

        /// <summary>カメラタイプ。</summary>
        public NInt    CameraProjType{ get; set; }

        /// <summary>アスペクト比。</summary>
        public NFloat    AspectRatio{ get; set; }

        /// <summary>画角(パースペクティブ) / 画像の縦サイズ(オルソ)</summary>
        public NFloat    FOV{ get; set; }

        /// <summary>ニア</summary>
        public NFloat    FrustumNear{ get; set; }

        /// <summary>ファー</summary>
        public NFloat    FrustumFar{ get; set; }


        /// <summary>コンストラクタ</summary>
        public MESSAGEPACKET_Viewer_CameraSettings()
        {
            ModifiedProperties = new NUint(0xFFFFFFFF);
            ModifiedPropertyElements = new NUint(0xFFFFFFFF);

            CameraPos = new NVector3();
            LookAtPos = new NVector3();
            CameraRotation = new NFloat();
            CameraProjType = new NInt();
            AspectRatio = new NFloat();
            FOV = new NFloat();
            FrustumNear = new NFloat();
            FrustumFar = new NFloat();
        }

        /// <summary>コンストラクタ</summary>
        public MESSAGEPACKET_Viewer_CameraSettings(NVector3 _CameraPos, NVector3 _LookAtPos, NFloat _CameraRotation, NInt _CameraProjType, NFloat _AspectRatio, NFloat _FOV, NFloat _FrustumNear, NFloat _FrustumFar)
        {
            ModifiedProperties = new NUint(0xFFFFFFFF);
            ModifiedPropertyElements = new NUint(0xFFFFFFFF);

            CameraPos = _CameraPos;
            LookAtPos = _LookAtPos;
            CameraRotation = _CameraRotation;
            CameraProjType = _CameraProjType;
            AspectRatio = _AspectRatio;
            FOV = _FOV;
            FrustumNear = _FrustumNear;
            FrustumFar = _FrustumFar;
        }
        /// <summary>文字列化</summary>
        public override string ToString()
        {
            string temp = string.Empty;
            temp += "CameraPos:(" + CameraPos.ToString() + ") ";
            temp += "LookAtPos:(" + LookAtPos.ToString() + ") ";
            temp += "CameraRotation:(" + CameraRotation.ToString() + ") ";
            temp += "CameraProjType:(" + CameraProjType.ToString() + ") ";
            temp += "AspectRatio:(" + AspectRatio.ToString() + ") ";
            temp += "FOV:(" + FOV.ToString() + ") ";
            temp += "FrustumNear:(" + FrustumNear.ToString() + ") ";
            temp += "FrustumFar:(" + FrustumFar.ToString() + ") ";
            return temp;
        }

        /// <summary>パケットサイズ（可変サイズに対応）</summary>
        public int PacketSize
        {
            get {
                int temp = (4 + 4);   // + 4 は ModifiedProperties と ModifiedPropertyElements
                temp += CameraPos.Size;
                temp += LookAtPos.Size;
                temp += CameraRotation.Size;
                temp += CameraProjType.Size;
                temp += AspectRatio.Size;
                temp += FOV.Size;
                temp += FrustumNear.Size;
                temp += FrustumFar.Size;
                temp += (4 - (temp & 0x03)) & 0x03;
                return temp;
            }
        }

        /// <summary>後に続く配列の長さ</summary>
        public override int ArrayLength
        {
            get{
                return 0;
            }
        }

        /// <summary>Read</summary>
        public override void Read(Protocol4FReader reader)
        {
             ModifiedProperties.Read(reader);
             ModifiedPropertyElements.Read(reader);
             CameraPos.Read(reader);
             LookAtPos.Read(reader);
             CameraRotation.Read(reader);
             CameraProjType.Read(reader);
             AspectRatio.Read(reader);
             FOV.Read(reader);
             FrustumNear.Read(reader);
             FrustumFar.Read(reader);
        }

        /// <summary>Write</summary>
        public override void Write(Protocol4FWriter writer)
        {
             ModifiedProperties.Write(writer);
             ModifiedPropertyElements.Write(writer);
             CameraPos.Write(writer);
             LookAtPos.Write(writer);
             CameraRotation.Write(writer);
             CameraProjType.Write(writer);
             AspectRatio.Write(writer);
             FOV.Write(writer);
             FrustumNear.Write(writer);
             FrustumFar.Write(writer);
        }
    };

    /// <summary>プレビュー設定のモデルの再生パラメータを送信します</summary>
    public class MESSAGEPACKET_Viewer_SendModelPlaybackParam : BaseMessagePacket
    {
        /// <summary>メッセージタイプ</summary>
        public const MessageType MessagePacketType = MessageType.Viewer_SendModelPlaybackParam;

        /// <summary>構造体にした場合のサイズ</summary>
        public const int StructSize = 20 + 4 + 4;   // + 4 は ModifiedProperties と ModifiedPropertyElements

        /// <summary>後ろに続く配列のサイズ(byte)</summary>
        public virtual int ArraySize{ get{ return 0; } }

        /// <summary>プロパティ</summary>
        [Flags]
        public enum PropertyFlag : uint
        {
            /// <summary>モデルノードＩＤ</summary>
            ModelID = 1<<0,

            /// <summary>再生範囲の開始フレーム</summary>
            AnimStartFrame = 1<<1,

            /// <summary>再生範囲の終了フレーム</summary>
            AnimEndFrame = 1<<2,

            /// <summary>再生速度</summary>
            PlaybackSpeed = 1<<3,

            /// <summary>再生開始フレーム</summary>
            PlaybackStartFrame = 1<<4,

        }

        /// <summary>編集されたプロパティ</summary>
        public NUint    ModifiedProperties{ get; set; }

        /// <summary>編集されたプロパティ要素</summary>
        public NUint    ModifiedPropertyElements{ get; set; }

        /// <summary>モデルノードＩＤ</summary>
        public NInt    ModelID{ get; set; }

        /// <summary>再生範囲の開始フレーム</summary>
        public NInt    AnimStartFrame{ get; set; }

        /// <summary>再生範囲の終了フレーム</summary>
        public NInt    AnimEndFrame{ get; set; }

        /// <summary>再生速度</summary>
        public NFloat    PlaybackSpeed{ get; set; }

        /// <summary>再生開始フレーム</summary>
        public NInt    PlaybackStartFrame{ get; set; }


        /// <summary>コンストラクタ</summary>
        public MESSAGEPACKET_Viewer_SendModelPlaybackParam()
        {
            ModifiedProperties = new NUint(0xFFFFFFFF);
            ModifiedPropertyElements = new NUint(0xFFFFFFFF);

            ModelID = new NInt();
            AnimStartFrame = new NInt();
            AnimEndFrame = new NInt();
            PlaybackSpeed = new NFloat();
            PlaybackStartFrame = new NInt();
        }

        /// <summary>コンストラクタ</summary>
        public MESSAGEPACKET_Viewer_SendModelPlaybackParam(NInt _ModelID, NInt _AnimStartFrame, NInt _AnimEndFrame, NFloat _PlaybackSpeed, NInt _PlaybackStartFrame)
        {
            ModifiedProperties = new NUint(0xFFFFFFFF);
            ModifiedPropertyElements = new NUint(0xFFFFFFFF);

            ModelID = _ModelID;
            AnimStartFrame = _AnimStartFrame;
            AnimEndFrame = _AnimEndFrame;
            PlaybackSpeed = _PlaybackSpeed;
            PlaybackStartFrame = _PlaybackStartFrame;
        }
        /// <summary>文字列化</summary>
        public override string ToString()
        {
            string temp = string.Empty;
            temp += "ModelID:(" + ModelID.ToString() + ") ";
            temp += "AnimStartFrame:(" + AnimStartFrame.ToString() + ") ";
            temp += "AnimEndFrame:(" + AnimEndFrame.ToString() + ") ";
            temp += "PlaybackSpeed:(" + PlaybackSpeed.ToString() + ") ";
            temp += "PlaybackStartFrame:(" + PlaybackStartFrame.ToString() + ") ";
            return temp;
        }

        /// <summary>パケットサイズ（可変サイズに対応）</summary>
        public int PacketSize
        {
            get {
                int temp = (4 + 4);   // + 4 は ModifiedProperties と ModifiedPropertyElements
                temp += ModelID.Size;
                temp += AnimStartFrame.Size;
                temp += AnimEndFrame.Size;
                temp += PlaybackSpeed.Size;
                temp += PlaybackStartFrame.Size;
                temp += (4 - (temp & 0x03)) & 0x03;
                return temp;
            }
        }

        /// <summary>後に続く配列の長さ</summary>
        public override int ArrayLength
        {
            get{
                return 0;
            }
        }

        /// <summary>Read</summary>
        public override void Read(Protocol4FReader reader)
        {
             ModifiedProperties.Read(reader);
             ModifiedPropertyElements.Read(reader);
             ModelID.Read(reader);
             AnimStartFrame.Read(reader);
             AnimEndFrame.Read(reader);
             PlaybackSpeed.Read(reader);
             PlaybackStartFrame.Read(reader);
        }

        /// <summary>Write</summary>
        public override void Write(Protocol4FWriter writer)
        {
             ModifiedProperties.Write(writer);
             ModifiedPropertyElements.Write(writer);
             ModelID.Write(writer);
             AnimStartFrame.Write(writer);
             AnimEndFrame.Write(writer);
             PlaybackSpeed.Write(writer);
             PlaybackStartFrame.Write(writer);
        }
    };

    /// <summary>編集対象となるオブジェクトの指定</summary>
    public class EditSelectedObject : BaseStructPacket
    {
        /// <summary>構造体にした場合のサイズ</summary>
        public const int StructSize = 8;

        /// <summary>IDです。</summary>
        public NUint    Id{ get; set; }

        /// <summary>Indexです。</summary>
        public NUint    Index{ get; set; }


        /// <summary>コンストラクタ</summary>
        public EditSelectedObject()
        {
            Id = new NUint();
            Index = new NUint();
        }

        /// <summary>コンストラクタ</summary>
        public EditSelectedObject(NUint _Id, NUint _Index)
        {
            Id = _Id;
            Index = _Index;
        }
        /// <summary>文字列化</summary>
        public override string ToString()
        {
            string temp = string.Empty;
            temp += "Id:(" + Id.ToString() + ") ";
            temp += "Index:(" + Index.ToString() + ") ";
            return temp;
        }

        /// <summary>Read</summary>
        public override void Read(Protocol4FReader reader)
        {
             Id.Read(reader);
             Index.Read(reader);
        }

        /// <summary>Write</summary>
        public override void Write(Protocol4FWriter writer)
        {
             Id.Write(writer);
             Index.Write(writer);
        }
    };

    /// <summary>メッセージのバイナリ配列要素</summary>
    public class ByteData : BaseStructPacket
    {
        /// <summary>構造体にした場合のサイズ</summary>
        public const int StructSize = 1;

        /// <summary>Byteの配列要素です。</summary>
        public NByte    Data{ get; set; }


        /// <summary>コンストラクタ</summary>
        public ByteData()
        {
            Data = new NByte();
        }

        /// <summary>コンストラクタ</summary>
        public ByteData(NByte _Data)
        {
            Data = _Data;
        }
        /// <summary>文字列化</summary>
        public override string ToString()
        {
            string temp = string.Empty;
            temp += "Data:(" + Data.ToString() + ") ";
            return temp;
        }

        /// <summary>Read</summary>
        public override void Read(Protocol4FReader reader)
        {
             Data.Read(reader);
        }

        /// <summary>Write</summary>
        public override void Write(Protocol4FWriter writer)
        {
             Data.Write(writer);
        }
    };

}
