﻿// ========================================================================
// <copyright file="BoneLinkagePacket.cs" company="Nintendo">
//      Copyright 2009 Nintendo.  All rights reserved.
// </copyright>
//
// These coded instructions, statements, and computer programs contain
// proprietary information of Nintendo of America Inc. and/or Nintendo
// Company Ltd., and are protected by Federal copyright law.  They may
// not be disclosed to third parties or copied or duplicated in any form,
// in whole or in part, without the prior written consent of Nintendo.
// ========================================================================

using System;
using System.IO;
using System.Text;

namespace NW4F.Protocols
{
    #region MESSAGEPACKET_BoneLinkage_ModelNameListOverride

    /// <summary>ビューア側からモデル名リストを送ってもらう</summary>
    public class MESSAGEPACKET_BoneLinkage_ModelNameListOverride :
        MESSAGEPACKET_BoneLinkage_ModelNameList
    {
        /// <summary>Number of loops needed for processing the struct packets.</summary>
        public override int NumProcessLoops
        {
            get { return 2; }
        }

        /// <summary>Create corresponding struct packets.</summary>
        /// <returns>The created struct packet.</returns>
        public override BaseStructPacket CreateStructPacket()
        {
            return new STRUCTPACKET_BoneLinkage_ModelName();
        }

        /// <summary>後に続く配列の長さ</summary>
        public override int ArrayLength
        {
            get { return (int)Count.Value; }
        }
    };

    #endregion

    #region STRUCTPACKET_BoneLinkage_ModelName

    /// <summary>
    /// Model name struct packet for bone linkage
    /// </summary>
    public class STRUCTPACKET_BoneLinkage_ModelName : BaseStructPacket
    {
        private int m_iCurProcessLoop = 0;

        /// <summary>構造体にした場合のサイズ</summary>
        public int StructSize
        {
            get
            {
                return 4 + ModelName.Length + 1;
            }
        }

        /// <summary>Hash IDです。</summary>
        public Uint HashID { get; set; }

        /// <summary>Indexです。</summary>
        public string ModelName { get; set; }


        /// <summary>コンストラクタ</summary>
        public STRUCTPACKET_BoneLinkage_ModelName()
        {
            HashID    = new Uint();
            ModelName = "";
        }

        /// <summary>文字列化</summary>
        public override string ToString()
        {
            string temp = string.Empty;
            temp += "Hash ID:(" + HashID.ToString() + ") ";
            temp += "Model Name:(" + ModelName + ") ";
            return temp;
        }

        /// <summary>Read</summary>
        public override void Read(Protocol4FReader reader)
        {
            if ( m_iCurProcessLoop==0 )
            {
                HashID.Read(reader);
            }
            else if ( m_iCurProcessLoop==1 )
            {
                byte ch = reader.ReadByte();
                while ( ch!=0 )
                {
                    ModelName += (char)ch;
                    ch = reader.ReadByte();
                }
            }
            else
            {
                return;
            }

            ++m_iCurProcessLoop;
        }

        /// <summary>Write</summary>
        public override void Write(Protocol4FWriter writer)
        {
            if ( m_iCurProcessLoop==0 )
            {
                HashID.Write(writer);
            }
            else if ( m_iCurProcessLoop==1 )
            {
                writer.Write(ModelName);
            }
            else
            {
                return;
            }

            ++m_iCurProcessLoop;
        }
    };

    #endregion

    #region MESSAGEPACKET_BoneLinkage_BoneNameListOverride

    /// <summary>ビューア側からボーン名リストを送ってもらう初めはビューアへからパケットを送ると、ボーンリストがもらえる。</summary>
    public class MESSAGEPACKET_BoneLinkage_BoneNameListOverride :
        MESSAGEPACKET_BoneLinkage_BoneNameList
    {
        /// <summary>後に続く配列の長さ</summary>
        public override int ArrayLength
        {
            get { return (int)Count.Value; }
        }

        /// <summary>Create corresponding struct packets.</summary>
        /// <returns>The created struct packet.</returns>
        public override BaseStructPacket CreateStructPacket()
        {
            return new STRUCTPACKET_BoneLinkage_BoneName();
        }
    };

    #endregion

    #region STRUCTPACKET_BoneLinkage_BoneName

    /// <summary>
    /// Model name struct packet for bone linkage
    /// </summary>
    public class STRUCTPACKET_BoneLinkage_BoneName : BaseStructPacket
    {
        /// <summary>構造体にした場合のサイズ</summary>
        public int StructSize
        {
            get
            {
                return BoneName.Length + 1;
            }
        }

        /// <summary>Bone nameです。</summary>
        public string BoneName { get; set; }


        /// <summary>コンストラクタ</summary>
        public STRUCTPACKET_BoneLinkage_BoneName()
        {
            BoneName = "";
        }

        /// <summary>文字列化</summary>
        public override string ToString()
        {
            string temp = string.Empty;
            temp += "Bone Name:(" + BoneName + ") ";
            return temp;
        }

        /// <summary>Read</summary>
        public override void Read(Protocol4FReader reader)
        {
            byte ch = reader.ReadByte();
            while ( ch!=0 )
            {
                BoneName += (char)ch;
                ch = reader.ReadByte();
            }
        }

        /// <summary>Write</summary>
        public override void Write(Protocol4FWriter writer)
        {
            writer.Write(BoneName);
        }
    };

    #endregion
}
