﻿// ========================================================================
// <copyright file="Vector1i.cs" company="Nintendo">
//      Copyright 2011 Nintendo.  All rights reserved.
// </copyright>
//
// These coded instructions, statements, and computer programs contain
// proprietary information of Nintendo of America Inc. and/or Nintendo
// Company Ltd., and are protected by Federal copyright law.  They may
// not be disclosed to third parties or copied or duplicated in any form,
// in whole or in part, without the prior written consent of Nintendo.
// ========================================================================

using System;
using System.Diagnostics;
using System.Xml.Serialization;
using NintendoWare.ToolDevelopmentKit;
using NintendoWare.ToolDevelopmentKit.ComponentModel;

namespace NWCore.DataModel.Major_1.Minor_6.Build_0.Revision_0
{
    /// <summary>
    /// 1 次元のベクトルを表します
    /// </summary>
    public sealed class Vector1i : IVectori, ICloneable, ISettable, IEquatable<Vector1i>
    {
        #region Member Variables

        private int x;

        #endregion

        #region Construtors

        /// <summary>
        /// デフォルトコンストラクタです
        /// </summary>
        public Vector1i()
        {
        }

        /// <summary>
        /// コピーコンストラクタです
        /// </summary>
        /// <param name="source">コピー元のベクトルです</param>
        public Vector1i(Vector1i source)
        {
            this.Set(source);
        }

        /// <summary>
        /// X座標を指定の値に設定するコンストラクタです
        /// </summary>
        /// <param name="x">X座標です</param>
        public Vector1i(int x)
        {
            this.Set(x);
        }

        /// <summary>
        /// X座標を指定の値に設定するコンストラクタです
        /// </summary>
        /// <param name="source">X座標の配列です</param>
        public Vector1i(int[] source)
        {
            this.Set(source);
        }

        #endregion

        #region Properties

        /// <summary>
        /// 次元数を取得します
        /// </summary>
        public int Dimension
        {
            get
            {
                return 1;
            }
        }

        /// <summary>
        /// X 成分を取得または設定します
        /// </summary>
        [XmlElement("x")]
        public int X
        {
            get
            {
                return this.x;
            }

            set
            {
                this.x = value;
            }
        }

        /// <summary>
        /// ベクトルの長さを取得します
        /// </summary>
        /// <returns>ベクトルの長さです</returns>
        public int Length
        {
            get { return this.X; }
        }

        /// <summary>
        /// 長さの2乗を取得します
        /// </summary>
        /// <returns>ベクトルの長さの２乗です</returns>
        public int LengthSquared
        {
            get { return this.X * this.X; }
        }

        /// <summary>
        /// 指定したインデックス位置の成分を取得、または設定します
        /// </summary>
        /// <param name="i">インデックスです</param>
        /// <returns>成分です</returns>
        [XmlIgnore]
        public int this[int i]
        {
            get
            {
                switch (i)
                {
                    case 0:
                        return this.X;
                    default:
                        throw new ArgumentOutOfRangeException();
                }
            }

            set
            {
                switch (i)
                {
                    case 0:
                        this.X = value;
                        break;
                    default:
                        throw new ArgumentOutOfRangeException();
                }
            }
        }

        #endregion

        #region Operator

        /// <summary>
        /// 2つのベクトルを加算します
        /// </summary>
        /// <param name="lhs">演算の左のベクトルです</param>
        /// <param name="rhs">演算の右のベクトルです</param>
        /// <returns>lhsとrhsを加算した結果のベクトルです</returns>
        public static Vector1i operator +(Vector1i lhs, Vector1i rhs)
        {
            Ensure.Argument.NotNull(lhs);
            Ensure.Argument.NotNull(rhs);

            return new Vector1i(lhs.X + rhs.X);
        }

        /// <summary>
        /// 2 つのベクトルを減算します
        /// </summary>
        /// <param name="lhs">演算の左のベクトルです</param>
        /// <param name="rhs">演算の右のベクトルです</param>
        /// <returns>lhsとrhsを減算した結果のベクトルです</returns>
        public static Vector1i operator -(Vector1i lhs, Vector1i rhs)
        {
            Ensure.Argument.NotNull(lhs);
            Ensure.Argument.NotNull(rhs);

            return new Vector1i(lhs.X - rhs.X);
        }

        /// <summary>
        /// ベクトルにスカラーを乗算します
        /// </summary>
        /// <param name="vec">演算の左のベクトルです</param>
        /// <param name="scalar">演算の右のスカラーです</param>
        /// <returns>vecとscalarを乗算した結果のベクトルです</returns>
        public static Vector1i operator *(Vector1i vec, int scalar)
        {
            Ensure.Argument.NotNull(vec);

            return new Vector1i(vec.X * scalar);
        }

        /// <summary>
        /// ベクトルをスカラーで除算します
        /// </summary>
        /// <param name="vec">演算の左のベクトルです</param>
        /// <param name="scalar">演算の右のスカラーです</param>
        /// <returns>vecとscalarを除算した結果のベクトルです</returns>
        public static Vector1i operator /(Vector1i vec, int scalar)
        {
            Ensure.Argument.NotNull(vec);
            Ensure.Operation.DividerNotZero(scalar);

            return new Vector1i(vec.X / scalar);
        }

        #endregion

        #region Utility Functions

        /// <summary>
        /// 2 つのベクトルの内積を計算します
        /// </summary>
        /// <param name="lhs">演算の左のベクトルです</param>
        /// <param name="rhs">演算の右のベクトルです</param>
        /// <returns>lhsとrhsの外積を計算した結果のスカラーです</returns>
        public static int Dot(Vector1i lhs, Vector1i rhs)
        {
            Ensure.Argument.NotNull(lhs);
            Ensure.Argument.NotNull(rhs);

            return lhs.X * rhs.X;
        }

        /// <summary>
        /// X座標を格納した配列を返します
        /// </summary>
        /// <returns>X座標の配列です</returns>
        public int[] ToArray()
        {
            return new int[] { this.X };
        }

        /// <summary>
        /// オブジェクトを設定します
        /// </summary>
        /// <param name="source">設定するオブジェクトです</param>
        public void Set(object source)
        {
            Ensure.Argument.True(source is IVector);
            this.Set(source as IVector);
        }

        /// <summary>
        /// ベクトルを設定します
        /// </summary>
        /// <param name="source">設定するベクトルです同じ次元でなければなりません</param>
        public void Set(IVector source)
        {
            Ensure.Argument.NotNull(source);
            Ensure.Argument.True(this.Dimension == source.Dimension);

            this.Set(source[0]);
        }

        /// <summary>
        /// ベクトルを設定します
        /// </summary>
        /// <param name="source">設定するベクトルです</param>
        public void Set(Vector1i source)
        {
            Ensure.Argument.NotNull(source);

            this.Set(source.X);
        }

        /// <summary>
        /// X座標を指定の値に設定します
        /// </summary>
        /// <param name="x">X座標です</param>
        public void Set(int x)
        {
            this.X = x;
        }

        /// <summary>
        /// X座標を指定の値に設定します
        /// </summary>
        /// <param name="source">X座標の配列です</param>
        public void Set(int[] source)
        {
            Ensure.Argument.NotNull(source);
            Ensure.Argument.True(this.Dimension == source.Length);

            this.Set(source[0]);
        }

        /// <summary>
        /// 成分を0に設定します
        /// </summary>
        public void SetZero()
        {
            this.X = 0;
        }

        /// <summary>
        /// オブジェクトを複製します
        /// </summary>
        /// <returns>複製したオブジェクトです</returns>
        public object Clone()
        {
            return new Vector1i(this);
        }

        /// <summary>
        /// 指定したベクトルとの内積を計算します
        /// </summary>
        /// <param name="vec">掛けるベクトルです</param>
        /// <returns>vecとの内積を計算した結果のスカラーです</returns>
        public int Dot(Vector1i vec)
        {
            return Dot(this, vec);
        }

        /// <summary>
        /// 等値であるかどうか比較します
        /// </summary>
        /// <param name="other">比較対象です</param>
        /// <returns>等値であればtrueを返します</returns>
        public override bool Equals(object other)
        {
            return this.Equals(other as Vector1i);
        }

        /// <summary>
        /// 等値であるかどうか比較します
        /// </summary>
        /// <param name="other">比較対象です</param>
        /// <returns>等値であればtrueを返します</returns>
        public bool Equals(Vector1i other)
        {
            if (other == this)
            {
                return true;
            }

            if (other == null)
            {
                return false;
            }

            return this.X == other.X;
        }

        /// <summary>
        /// ハッシュ値を取得します
        /// </summary>
        /// <returns>ハッシュ値です</returns>
        public override int GetHashCode()
        {
            return
                this.X.GetHashCode();
        }

        /// <summary>
        /// 現在のオブジェクトを表す文字列を返します
        /// </summary>
        /// <returns>現在のオブジェクトを表す文字列です</returns>
        public override string ToString()
        {
            return string.Format("[{0}]", this.X);
        }

        /// <summary>
        /// 長さの2乗を受け取り長さを１に正規化
        /// </summary>
        /// <param name="lengthSquared"></param>
        private void Normalize(int lengthSquared)
        {
            Debug.Assert(lengthSquared == this.LengthSquared, "Unexpected case!");

            if (lengthSquared == 1)
            {
                return;
            }

            int inverseLength = 1 / (int)Math.Sqrt(lengthSquared);
            this.Set(this * inverseLength);
        }

        #endregion
    }
}
