﻿// ========================================================================
// <copyright file="GameConfigData.cs" company="Nintendo">
//      Copyright 2011 Nintendo.  All rights reserved.
// </copyright>
//
// These coded instructions, statements, and computer programs contain
// proprietary information of Nintendo of America Inc. and/or Nintendo
// Company Ltd., and are protected by Federal copyright law.  They may
// not be disclosed to third parties or copied or duplicated in any form,
// in whole or in part, without the prior written consent of Nintendo.
// ========================================================================

using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using System.Diagnostics;
using NintendoWare.ToolDevelopmentKit;
using NWCore.DataModel;

namespace NWCore.DataModel.Major_0.Minor_0.Build_0.Revision_0
{
    /// <summary>
    /// ゲーム設定データ
    /// </summary>
    public sealed class GameConfigData : BaseDataModel<GameConfigData>
    {
        #region Constructors

        /// <summary>
        /// デフォルトコンストラクタ
        /// </summary>
        public GameConfigData()
        {
            Initialize();
        }

        /// <summary>
        /// コピーコンストラクタ
        /// </summary>
        /// <param name="source">コピー元のベクトル</param>
        public GameConfigData(GameConfigData source)
            : this()
        {
            this.Set(source);
        }

        /// <summary>
        /// 初期化
        /// </summary>
        public override void Initialize()
        {
            this.PreviewModelIndex = 0;
            this.BoneIndex = 0;
            this.ModelName = string.Empty;
            this.BoneName = string.Empty;
            this.DispEmitterSet = true;
            this.EmitterSetName = "";
            this.StartFrame = 0;
            this.MultColor = new Vector4(1.0f, 1.0f, 1.0f, 1.0f);
            this.ParticleScale = new Vector2(1.0f, 1.0f);
            this.EmissionParticleScale = new Vector2(1.0f, 1.0f);
            this.MartixSetType = MatrixSetType.Normal;
            this.OffsetScale = new Vector3(1.0f, 1.0f, 1.0f);
            this.EmitterBillboard = false;
            this.OffsetRotateX = 0.0f;
            this.OffsetRotateY = 0.0f;
            this.OffsetRotateZ = 0.0f;
            this.OffsetPosition = new Vector3(0.0f, 0.0f, 0.0f);
            this.EmitterScale = new Vector3(1.0f, 1.0f, 1.0f);
            this.AllDirectionalVelScale = 1.0f;
            this.DirectionalVelScale = 1.0f;
            this.RandomVel = 1.0f;
            this.AdditionalVel = new Vector3(0.0f, 0.0f, 0.0f);
            this.SpecifyDirection = false;
            this.DirectionalVel = new Vector3(0.0f, 1.0f, 0.0f);
        }

        #endregion

        #region Utility Functions

        /// <summary>
        /// ベクトルを設定します。
        /// </summary>
        /// <param name="source">設定するベクトルです。</param>
        public override void Set(GameConfigData source)
        {
            Ensure.Argument.NotNull(source);

            this.PreviewModelIndex = source.PreviewModelIndex;
            this.BoneIndex = source.BoneIndex;
            this.ModelName = source.ModelName;
            this.BoneName = source.BoneName;
            this.DispEmitterSet = source.DispEmitterSet;
            this.EmitterSetName = source.EmitterSetName;
            this.StartFrame = source.StartFrame;
            this.MultColor.Set(source.MultColor);
            this.ParticleScale.Set(source.ParticleScale);
            this.EmissionParticleScale = source.EmissionParticleScale;
            this.MartixSetType = source.MartixSetType;
            this.OffsetScale.Set(source.OffsetScale);
            this.EmitterBillboard = source.EmitterBillboard;
            this.OffsetRotateX = source.OffsetRotateX;
            this.OffsetRotateY = source.OffsetRotateY;
            this.OffsetRotateZ = source.OffsetRotateZ;
            this.OffsetPosition.Set(source.OffsetPosition);
            this.EmitterScale.Set(source.EmitterScale);
            this.AllDirectionalVelScale = source.AllDirectionalVelScale;
            this.DirectionalVelScale = source.DirectionalVelScale;
            this.RandomVel = source.RandomVel;
            this.AdditionalVel.Set(source.AdditionalVel);
            this.SpecifyDirection = source.SpecifyDirection;
            this.DirectionalVel.Set(source.DirectionalVel);
        }


        /// <summary>
        /// オブジェクトを複製します。
        /// </summary>
        /// <returns>複製したオブジェクトです。</returns>
        public override GameConfigData Clone()
        {
            return new GameConfigData(this);
        }

        #endregion

        #region Properties

        /// <summary>
        /// プレビューモデルのインデックス
        /// </summary>
        public int PreviewModelIndex { get; set; }

        /// <summary>
        /// 接続先のモデル
        /// </summary>
        public string ModelName { get; set; }

        /// <summary>
        /// 接続先のボーンのインデックス
        /// </summary>
        public int BoneIndex { get; set; }

        /// <summary>
        /// 接続先のボーン
        /// </summary>
        public string BoneName { get; set; }

        /// <summary>
        /// DEBUG:エミッタセットが有効か？
        /// </summary>
        public bool DispEmitterSet { get; set; }

        /// <summary>
        /// エミッタセットの名前
        /// </summary>
        public string EmitterSetName { get; set; }

        /// <summary>
        /// 呼び出し開始フレーム
        /// </summary>
        public int StartFrame { get; set; }

        /// <summary>
        /// 色（乗算値）
        /// </summary>
        public Vector4 MultColor { get; set; }

        /// <summary>
        /// パーティクルの大きさ
        /// </summary>
        public Vector2 ParticleScale { get; set; }

        /// <summary>
        /// 放出時のみ作用するパーティクルの大きさ
        /// </summary>
        public Vector2 EmissionParticleScale { get; set; }

        /// <summary>
        /// 行列設定タイプ
        /// </summary>
        public MatrixSetType MartixSetType { get; set; }

        /// <summary>
        /// オフセット行列 - スケール
        /// </summary>
        public Vector3 OffsetScale { get; set; }

        /// <summary>
        /// オフセット行列 - エミッタビルボード
        /// </summary>
        public bool EmitterBillboard { get; set; }

        /// <summary>
        /// オフセット行列 - 回転X
        /// </summary>
        public float OffsetRotateX { get; set; }

        /// <summary>
        /// オフセット行列 - 回転Y
        /// </summary>
        public float OffsetRotateY { get; set; }

        /// <summary>
        /// オフセット行列 - 回転Z
        /// </summary>
        public float OffsetRotateZ { get; set; }

        /// <summary>
        /// オフセット行列 - 座標
        /// </summary>
        public Vector3 OffsetPosition { get; set; }

        /// <summary>
        /// エミッタスケール
        /// </summary>
        public Vector3 EmitterScale { get; set; }

        /// <summary>
        /// 初速度 - 全方向速度のスケール値
        /// </summary>
        public float AllDirectionalVelScale { get; set; }

        /// <summary>
        /// 初速度 - 指定方向速度のスケール値
        /// </summary>
        public float DirectionalVelScale { get; set; }

        /// <summary>
        /// 初速度 - ランダムスケール値
        /// </summary>
        public float RandomVel { get; set; }

        /// <summary>
        /// 初速度 - 最終的に出てきた速度に加算する値
        /// </summary>
        public Vector3 AdditionalVel { get; set; }

        /// <summary>
        /// 初速度 - 指定方向の方向を指定するか
        /// </summary>
        public bool SpecifyDirection { get; set; }

        /// <summary>
        /// 初速度 - 指定方向の方向
        /// </summary>
        public Vector3 DirectionalVel { get; set; }

        #endregion
    }
}
