﻿// ========================================================================
// <copyright file="ChildData.cs" company="Nintendo">
//      Copyright 2011 Nintendo.  All rights reserved.
// </copyright>
//
// These coded instructions, statements, and computer programs contain
// proprietary information of Nintendo of America Inc. and/or Nintendo
// Company Ltd., and are protected by Federal copyright law.  They may
// not be disclosed to third parties or copied or duplicated in any form,
// in whole or in part, without the prior written consent of Nintendo.
// ========================================================================

using System;
using System.Diagnostics;
using NintendoWare.ToolDevelopmentKit;

namespace NWCore.DataModel
{
    /// <summary>
    /// チャイルド
    /// </summary>
    public sealed class ChildData : BaseDataModel<ChildData>
    {
        #region Construtors

        /// <summary>
        /// デフォルトコンストラクタ
        /// </summary>
        public ChildData()
        {
            this.Color = new RgbaColor();        // 色
            this.TextureRes = new TextureRes();  // チャイルド用テクスチャ
            this.Scale = new Vector2(1, 1); ;    // スケール値
            this.RandVel = new Vector3();        // ランダム速度

            this.UserShaderParamGroups = new UserShaderParameter[8];
            for ( int i=0;i<this.UserShaderParamGroups.Length;++i )
            {
                this.UserShaderParamGroups[i] = new UserShaderParameter();
                this.UserShaderParamGroups[i].Index = i;
            }
        }

        /// <summary>
        /// コピーコンストラクタ
        /// </summary>
        /// <param name="source">コピー元のベクトル</param>
        public ChildData(ChildData source)
            : this()
        {
            this.Set(source);
        }

        /// <summary>
        /// 初期化
        /// </summary>
        public override void Initialize()
        {
            this.AirRegist = 1.0f;
            this.Alpha = 1.0f;
            this.MeshType = MeshType.Particle;
            this.TextureTrimming = false;
            this.BillboardType = BillboardType.Billboard;
            this.BillboardPrimitiveFileSource = string.Empty;
            this.EmitRate = 1;
            this.EmitStep = 10;
            this.EmitTiming = 60;
            this.DynamicsRandom = 0.0f;
            this.ScaleInheritRate = 1.0f;
            this.VelInheritRate = 0.0f;
            this.VelInheritRate = 1.0f;
            this.UseSoftParticle = false;
            this.SoftParticleProcessingMethod = SoftParticleProcessingMethod.Fragment;
            this.SoftParticleFadeDistance = 0.0f;
            this.SoftParticleVolumeParam = 0.0f;
            this.ShaderType = ShaderType.Particle;
            this.ShaderParam0 = 0.0f;
            this.ShaderParam1 = 0.0f;
            this.UI_ApplyAlphaToRefraction = true;

            for ( int i=0;i<this.UserShaderParamGroups.Length;++i )
                this.UserShaderParamGroups[i].Initialize();

            this.UserShaderCompileDef1 = string.Empty;
            this.UserShaderCompileDef2 = string.Empty;

            this.RandomScale = 0.0f;

            this.InitPosRand = 0.0f;

            float worldScale = s_WorldScaleRange / 100.0f;
            {
                this.FigureVelocity = 0.1f * worldScale;
                this.Scale.Set(1*worldScale, 1*worldScale);
            }

            this.ZBufATestType = ZBufATestType.Normal;
            this.DisplaySide = DisplaySideTypes.Both;

            this.UI_combinerShaderGenFromFile    = true;
            this.UI_combinerShaderProjectPath    = string.Empty;
            this.UI_combinerPrimitiveColorBlend  = CombinerBlendTypes.Mod;
            this.UI_combinerPrimitiveAlphaBlend  = CombinerBlendTypes.Mod;
            this.UI_combinerTextureColorUsage    = CombinerColorUsageTypes.Color;
            this.UI_combinerPrimitiveColorUsage  = CombinerColorUsageTypes.Color;
            this.UI_combinerTextureAlphaUsage    = CombinerAlphaUsageTypes.Color;
            this.UI_combinerPrimitiveAlphaUsage  = CombinerAlphaUsageTypes.Color;
        }

        #endregion

        #region Utility Functions

        /// <summary>
        /// ベクトルを設定します。
        /// </summary>
        /// <param name="source">設定するベクトルです。</param>
        public override void Set(ChildData source)
        {
            Ensure.Argument.NotNull(source);

            this.Color.Set(source.Color);
            this.TextureRes.Set(source.TextureRes);
            this.Scale.Set(source.Scale);
            this.RandVel.Set(source.RandVel);

            this.AirRegist = source.AirRegist;
            this.Alpha = source.Alpha;
            this.AlphaInit = source.AlphaInit;
            this.AlphaTarget = source.AlphaTarget;
            this.MeshType = source.MeshType;
            this.TextureTrimming = source.TextureTrimming;
            this.BillboardType = source.BillboardType;
            this.BillboardPrimitiveFileSource = source.BillboardPrimitiveFileSource;
            this.CombinerType = source.CombinerType;
            this.EmitRate = source.EmitRate;
            this.EmitStep = source.EmitStep;
            this.EmitTiming = source.EmitTiming;
            this.DynamicsRandom = source.DynamicsRandom;
            this.FigureVelocity = source.FigureVelocity;
            this.InitPosRand = source.InitPosRand;
            this.RandomScale = source.RandomScale;
            this.ScaleInheritRate = source.ScaleInheritRate;
            this.VelInheritRate = source.VelInheritRate;
            this.ZBufATestType = source.ZBufATestType;
            this.DisplaySide = source.DisplaySide;
            this.UseSoftParticle = source.UseSoftParticle;
            this.SoftParticleProcessingMethod = source.SoftParticleProcessingMethod;
            this.SoftParticleFadeDistance = source.SoftParticleFadeDistance;
            this.SoftParticleVolumeParam = source.SoftParticleVolumeParam;
            this.ShaderType = source.ShaderType;
            this.ShaderSetting = source.ShaderSetting;
            this.ShaderParam0 = source.ShaderParam0;
            this.ShaderParam1 = source.ShaderParam1;
            this.UI_ApplyAlphaToRefraction = source.UI_ApplyAlphaToRefraction;

            for ( int i=0;i<this.UserShaderParamGroups.Length;++i )
            {
                this.UserShaderParamGroups[i].Set( source.UserShaderParamGroups[i] );
            }

            this.UserShaderCompileDef1 = source.UserShaderCompileDef1;
            this.UserShaderCompileDef2 = source.UserShaderCompileDef2;

            this.UI_combinerShaderGenFromFile    = source.UI_combinerShaderGenFromFile;
            this.UI_combinerShaderProjectPath    = source.UI_combinerShaderProjectPath;
            this.UI_combinerPrimitiveColorBlend  = source.UI_combinerPrimitiveColorBlend;
            this.UI_combinerPrimitiveAlphaBlend  = source.UI_combinerPrimitiveAlphaBlend;
            this.UI_combinerTextureColorUsage    = source.UI_combinerTextureColorUsage;
            this.UI_combinerPrimitiveColorUsage  = source.UI_combinerPrimitiveColorUsage;
            this.UI_combinerTextureAlphaUsage    = source.UI_combinerTextureAlphaUsage;
            this.UI_combinerPrimitiveAlphaUsage  = source.UI_combinerPrimitiveAlphaUsage;
        }

        /// <summary>
        /// オブジェクトを複製します。
        /// </summary>
        /// <returns>複製したオブジェクトです。</returns>
        public override ChildData Clone()
        {
            return new ChildData(this);
        }

        #endregion

        #region Properties

        /// <summary>
        /// 放出量
        /// </summary>
        public int EmitRate { get; set; }

        /// <summary>
        /// タイミング
        /// </summary>
        public int EmitTiming { get; set; }

        /// <summary>
        /// 放出間隔
        /// </summary>
        public int EmitStep { get; set; }

        /// <summary>
        /// 運動量ランダム
        /// </summary>
        public float DynamicsRandom { get; set; }

        /// <summary>
        /// 速度継承率
        /// </summary>
        public float VelInheritRate { get; set; }

        /// <summary>
        /// 形状速度
        /// </summary>
        public float FigureVelocity { get; set; }

        /// <summary>
        /// ランダム速度
        /// </summary>
        public Vector3 RandVel { get; set; }

        /// <summary>
        /// 初期位置のランダム量
        /// </summary>
        public float InitPosRand { get; set; }

        /// <summary>
        /// Mesh type
        /// </summary>
        public MeshType MeshType { get; set; }

        /// <summary>
        /// テクスチャトリミング
        /// </summary>
        public bool TextureTrimming { get; set; }

        /// <summary>
        /// ビルボードタイプ
        /// </summary>
        public BillboardType BillboardType { get; set; }

        /// <summary>
        /// プリミティブファイルパス
        /// </summary>
        public string BillboardPrimitiveFileSource { get; set; }

        /// <summary>
        /// Ｚバッファー、αテストタイプ
        /// </summary>
        public ZBufATestType ZBufATestType { get; set; }

        /// <summary>
        /// カリング
        /// </summary>
        public DisplaySideTypes DisplaySide { get; set; }

        /// <summary>
        /// チャイルド用テクスチャ
        /// </summary>
        public TextureRes TextureRes { get; set; }

        /// <summary>
        /// 色
        /// </summary>
        public RgbaColor Color { get; set; }

        /// <summary>
        /// アルファ
        /// </summary>
        public float Alpha { get; set; }

        /// <summary>
        /// アルファアニメのターゲット
        /// </summary>
        public float AlphaTarget { get; set; }

        /// <summary>
        /// アルファアニメの初期値
        /// </summary>
        public float AlphaInit { get; set; }

        /// <summary>
        /// スケール継承率
        /// </summary>
        public float ScaleInheritRate { get; set; }

        /// <summary>
        /// スケール値
        /// </summary>
        public Vector2 Scale { get; set; }

        /// <summary>
        /// ランダム値
        /// </summary>
        public float RandomScale { get; set; }

        /// <summary>
        /// コンバイナタイプ
        /// </summary>
        public CombinerType CombinerType { get; set; }

        /// <summary>
        /// 空気抵抗
        /// </summary>
        public float AirRegist { get; set; }

        /// <summary>
        /// シェーダータイプ：ソフトパーティクルを利用する
        /// </summary>
        public bool UseSoftParticle { get; set; }

        /// <summary>
        /// ソフトパーティクル・処理方法.
        /// </summary>
        public SoftParticleProcessingMethod SoftParticleProcessingMethod { get; set; }

        /// <summary>
        /// Fade distance for soft particle.
        /// </summary>
        public float SoftParticleFadeDistance { get; set; }

        /// <summary>
        /// Volume parameter for the soft particle
        /// </summary>
        public float SoftParticleVolumeParam { get; set; }

        /// <summary>
        /// シェーダータイプ
        /// </summary>
        public ShaderType ShaderType { get; set; }

        /// <summary>
        /// シェーダセッティング
        /// </summary>
        public uint ShaderSetting { get; set; }

        /// <summary>
        /// シェーダパラメータ
        /// </summary>
        public float ShaderParam0 { get; set; }

        /// <summary>
        /// シェーダパラメータ
        /// </summary>
        public float ShaderParam1 { get; set; }

        /// <summary>
        /// アルファ値を反映する。
        /// </summary>
        public bool UI_ApplyAlphaToRefraction { get; set; }

        /// <summary>
        /// シェーダパラメータ
        /// </summary>
        public Vector2 ShaderParams01
        {
            get
            {
                return new Vector2(ShaderParam0, ShaderParam1);
            }
            set
            {
                ShaderParam0 = value.X;
                ShaderParam1 = value.Y;
            }
        }

        /// <summary>
        /// ユーザーシェーダパラメータ
        /// </summary>
        public UserShaderParameter[] UserShaderParamGroups { get; set; }

        /// <summary>
        /// ユーザーシェーダコンパイル定義１
        /// </summary>
        public string UserShaderCompileDef1 { get; set; }

        /// <summary>
        /// ユーザーシェーダコンパイル定義２
        /// </summary>
        public string UserShaderCompileDef2 { get; set; }

        /// <summary>
        /// コンバイナ：True to generate shader code from the shader project file
        /// instead of the shader code sent from EffectCombinerEditor.
        /// </summary>
        public bool UI_combinerShaderGenFromFile { get; set; }

        /// <summary>
        /// コンバイナ：シェーダプロジェクトパス。
        /// </summary>
        public string UI_combinerShaderProjectPath { get; set; }

        /// <summary>
        /// コンバイナ：プリミティブカラーの上段との合成。
        /// </summary>
        public CombinerBlendTypes UI_combinerPrimitiveColorBlend { get; set; }

        /// <summary>
        /// コンバイナ：プリミティブアルファの上段との合成。
        /// </summary>
        public CombinerBlendTypes UI_combinerPrimitiveAlphaBlend { get; set; }

        /// <summary>
        /// コンバイナ：テクスチャカラーをカラー値か1.0で置き換える。
        /// </summary>
        public CombinerColorUsageTypes UI_combinerTextureColorUsage { get; set; }

        /// <summary>
        /// コンバイナ：プリミティブカラーをカラー値か1.0で置き換える。
        /// </summary>
        public CombinerColorUsageTypes UI_combinerPrimitiveColorUsage { get; set; }

        /// <summary>
        /// コンバイナ：テクスチャアルファをアルファ値か1.0で置き換える。
        /// </summary>
        public CombinerAlphaUsageTypes UI_combinerTextureAlphaUsage { get; set; }

        /// <summary>
        /// コンバイナ：プリミティブアルファをアルファ値か1.0で置き換える。
        /// </summary>
        public CombinerAlphaUsageTypes UI_combinerPrimitiveAlphaUsage { get; set; }

        #endregion
    }
}
