﻿// ========================================================================
// <copyright file="ShellUtility.cs" company="Nintendo">
//      Copyright 2009 Nintendo.  All rights reserved.
// </copyright>
//
// These coded instructions, statements, and computer programs contain
// proprietary information of Nintendo of America Inc. and/or Nintendo
// Company Ltd., and are protected by Federal copyright law.  They may
// not be disclosed to third parties or copied or duplicated in any form,
// in whole or in part, without the prior written consent of Nintendo.
// ========================================================================

using System;
using System.Collections.Generic;
using System.Diagnostics;
using System.Drawing;
using System.Runtime.InteropServices;
using System.IO;
using System.Text;

namespace App.Utility
{
    /// <summary>
    /// シェルユーティリティクラス。
    /// </summary>
    public sealed class ShellUtility
    {
        /// <summary>
        /// エクスプローラーでフォルダを開きます。
        /// </summary>
        public static void OpenPathByExplorer( string path,
                                               bool bSelectFile = true )
        {
            if ( path.Length>0 )
            {
                string commandline;
                if ( bSelectFile==true )
                    commandline = "/e, /select,\"" + path + "\"";
                else
                    commandline = "/e,\"" + path + "\"";

                System.Diagnostics.Process.Start("EXPLORER.EXE", commandline);
            }
        }

        /// <summary>
        /// 相対パスを取得します。
        /// </summary>
        /// <param name="rootPath">元にするルートパス</param>
        /// <param name="fullPath">相対パスを求めるパス</param>
        /// <returns>相対パス</returns>
        public static string GetRelativePath(string rootPath, string fullPath)
        {
            try
            {
                if (String.IsNullOrEmpty(rootPath))
                {
                    return fullPath;
                }
                else
                {
                    Uri uRoot = new Uri(rootPath, UriKind.RelativeOrAbsolute);
                    Uri uPath = new Uri(fullPath, UriKind.RelativeOrAbsolute);
                    uPath = uRoot.MakeRelativeUri(uPath);
                    return uPath.ToString();
                }
            }
            catch
            {
                return fullPath;
            }
        }

        /// <summary>
        /// Helper method for make the given path consistent as an absolute file path.
        /// </summary>
        /// <param name="path">The file path.</param>
        /// <param name="workingFolder">
        /// The working folder. Null to use the application working folder.
        /// </param>
        /// <returns>The absolute path.</returns>
        public static string ToAbsolutePath( string path,
                                             string workingFolder = null )
        {
            if ( string.IsNullOrEmpty(path)==true )
                return string.Empty;

            if ( Path.IsPathRooted(path)==false )
            {
                if ( string.IsNullOrEmpty(workingFolder)==true )
                    return Path.Combine( TheApp.WorkingFolder, path );
                else
                    return Path.Combine( workingFolder, path );
            }

            return string.Copy( path );
        }

        /// <summary>
        /// ドキュメントファイルに関連付けられた実行ファイルのパスを取得。
        /// </summary>
        public static string GetExecutablePath(string documentPath)
        {
            if (File.Exists(documentPath))
            {
                StringBuilder buffer = new StringBuilder(Win32.Constants.MAX_PATH);
                if (Win32.Shell32.FindExecutable(documentPath, null, buffer) > 32)
                {
                    return buffer.ToString();
                }
            }
            return null;
        }

        /// <summary>
        /// ファイルに関連付けられているアイコンを取得。
        /// </summary>
        public static Icon GetFileAssociatedIcon(string filePath, bool smallIcon)
        {
            if (File.Exists(filePath))
            {
                uint flag = (uint)Win32.SHGFI.SHGFI_ICON;
                if (smallIcon)
                {
                    flag |= (uint)Win32.SHGFI.SHGFI_SMALLICON;
                }

                Win32.SHFILEINFO fileInfo = new Win32.SHFILEINFO();
                IntPtr result = Win32.Shell32.SHGetFileInfo(filePath, 0, ref fileInfo, (uint)Marshal.SizeOf(fileInfo), flag);
                if (result != IntPtr.Zero)
                {
                    return Icon.FromHandle(fileInfo.hIcon);
                }
            }
            return null;
        }
    }
}
