﻿// ========================================================================
// <copyright file="DocumentSaver.cs" company="Nintendo">
//      Copyright 2009 Nintendo.  All rights reserved.
// </copyright>
//
// These coded instructions, statements, and computer programs contain
// proprietary information of Nintendo of America Inc. and/or Nintendo
// Company Ltd., and are protected by Federal copyright law.  They may
// not be disclosed to third parties or copied or duplicated in any form,
// in whole or in part, without the prior written consent of Nintendo.
// ========================================================================

using System;
using System.Diagnostics;
using System.Collections.Generic;
using System.IO;
using System.Windows.Forms;
using System.Text.RegularExpressions;
//using App.Controls;
using App.Data;
using App.Utility;
using NWCore.DataModel;
using NWCore.Viewer;

namespace App.IO
{
    #region Class for keeping the document saving information

    public class DocumentSaveInfo
    {
        #region Constructors

        /// <summary>
        /// Constructor for the information for saving documents.
        /// </summary>
        /// <param name="doc">The document to save.</param>
        /// <param name="targetPath">The target file path to save to.</param>
        public DocumentSaveInfo( IDocument doc,
                                 string targetPath )
        {
            this.Document   = doc;
            this.TargetPath = targetPath;
        }


        /// <summary>
        /// Constructor for the information for saving files.
        /// </summary>
        /// <param name="sourcePath">The source file path.</param>
        /// <param name="targetPath">The target file path to save to.</param>
        public DocumentSaveInfo( string sourcePath,
                                 string targetPath )
        {
            this.SourcePath = sourcePath;
            this.TargetPath = targetPath;
        }

        #endregion

        #region Properties

        /// <summary>Get the document to be saved.</summary>
        public IDocument Document
        {
            get { return m_doc; }
            private set
            {
                m_doc = value;

                if ( m_doc!=null )
                    this.SourcePath = m_doc.FilePath;
            }
        }


        /// <summary>Get the original file path.</summary>
        public string SourcePath
        {
            get { return m_srcPath; }
            private set
            {
                if ( string.IsNullOrEmpty(value)==true )
                {
                    m_srcPath     = string.Empty;
                    m_iSrcPathCRC = 0;
                }
                else
                {
                    m_srcPath     = string.Copy( value );
                    m_iSrcPathCRC = TheApp.CRC32Helper.ComputeCRC32Str( m_srcPath.ToLower() );
                }
            }
        }


        /// <summary>Get the CRC32 hash code of the source file path.</summary>
        public uint SourcePathCRC
        {
            get { return m_iSrcPathCRC; }
        }


        /// <summary>Get the target file path.</summary>
        public string TargetPath
        {
            get { return m_trgPath; }
            private set
            {
                if ( string.IsNullOrEmpty( value )==true )
                {
                    m_trgPath     = string.Empty;
                    m_iTrgPathCRC = 0;
                }
                else
                {
                    m_trgPath     = string.Copy( value );
                    m_iTrgPathCRC = TheApp.CRC32Helper.ComputeCRC32Str( m_trgPath.ToLower() );
                }
            }
        }


        /// <summary>Get the CRC32 hash code of the target file path.</summary>
        public uint TargetPathCRC
        {
            get { return m_iTrgPathCRC; }
        }

        #endregion

        #region Member variables

        private IDocument m_doc         = null;
        private string    m_srcPath     = string.Empty;
        private string    m_trgPath     = string.Empty;
        private uint      m_iSrcPathCRC = 0;
        private uint      m_iTrgPathCRC = 0;

        #endregion
    }

    #endregion

    /// <summary>
    /// ドキュメントセーバ。
    /// </summary>
    public class DocumentSaver
    {
        #region Constructor

        public DocumentSaver()
        {
            this.EnableMessageBox = true;
        }

        #endregion

        #region Events

        /// <summary>
        /// ドキュメント保存イベント。
        /// </summary>
        //public event EventHandler DocumentSaved = null;

        /// <summary>
        /// ドキュメント保存ハンドラ。
        /// </summary>
        protected virtual void OnDocumentoSaved(EventArgs e)
        {
            /*
            if ( this.DocumentSaved!=null )
            {
                if ( TheApp.MainFrame.InvokeRequired==true )
                {
                    TheApp.MainFrame.Invoke( new MainFrame.EventInvoker( DocumentSaved ),
                                             this,
                                             e );
                }
                else
                {
                    this.DocumentSaved( this, e );
                }
            }
            */
        }

        #endregion

        #region Properties

        /// <summary>
        /// Get or set the flag indicating if the warning message boxes are enabled.
        /// </summary>
        public bool EnableMessageBox { get; set; }

        #endregion

        #region Save

        /// <summary>
        /// 保存。
        /// </summary>
        /// <param name="document">The document to save.</param>
        /// <param name="filePath">The destination file path.</param>
        /// <returns>True on success.</returns>
        public bool Save( IDocument document,
                          string filePath )
        {
            DocumentPropertyChangedEventArgs changed = new DocumentPropertyChangedEventArgs();

            bool bResult = true;
            if ( document is EffectProjectDocument )
            {
                bResult = SaveWorkspace( document as EffectProjectDocument,
                                         filePath,
                                         changed );
            }
            else if ( document is GameSettingsDocument )
            {
                bResult = SavePrev( document,
                                    filePath,
                                    changed );
            }
            else
            {
                // Check if the textures were assigned to the emitters.
                if ( AreTexturesAssigned( document, MessageBoxButtons.YesNo )==DialogResult.No )
                    return false;

                bResult = Save( document,
                                filePath,
                                changed );
            }

            Document.NotifyPropertyChanged(this, changed);

            return bResult;
        }


        /// <summary>
        /// 保存。
        /// This method accepts a property change event argument
        /// so the user can use this method to operate upon multiple
        /// documents at the same time.
        /// </summary>
        /// <param name="document">The document to save.</param>
        /// <param name="filePath">The destination file path.</param>
        /// <param name="changed">The property change event arguments.</param>
        /// <returns>True on success.</returns>
        public bool Save( IDocument document,
                          string filePath,
                          DocumentPropertyChangedEventArgs changed )
        {
            Debug.Assert( document!=null );

            // 保存場所が決まっていない場合はSaveAsにまわす
            bool bResult = true;
            if ( Path.IsPathRooted( filePath )==false )
            {
                bResult = SaveAs( document, changed );
            }
            else
            {
                EmitterSetDocument esetDoc = document as EmitterSetDocument;
                if ( esetDoc!=null &&
                     this.EnableMessageBox==true )
                {
                    TheApp.ResetUpdateDrawPathDialogFlags();
                    if ( CheckEmitterDrawPathProperty(esetDoc)==false )
                        return false;

                    DialogResult result =
                        CheckEmitterImmortalEndingProperties( esetDoc, MessageBoxButtons.YesNo );
                    if ( result==DialogResult.No )
                        return false;
                }

                string errMsg;
                bResult = SaveInternal( document, filePath, changed, out errMsg );
                if ( bResult==false )
                    ShowErrorSaveFailed( filePath, errMsg );
            }

            return bResult;
        }


        /// <summary>
        /// 保存。
        /// This method accepts a property change event argument
        /// so the user can use this method to operate upon multiple
        /// documents at the same time.
        /// </summary>
        /// <param name="document">The document to save.</param>
        /// <param name="filePath">The destination file path.</param>
        /// <param name="changed">The property change event arguments.</param>
        /// <returns>True on success.</returns>
        public bool SavePrev( IDocument document,
                              string filePath,
                              DocumentPropertyChangedEventArgs changed )
        {
            Debug.Assert( document!=null );

            // 保存場所が決まっていない場合はSaveAsにまわす
            bool bResult = true;
            if ( Path.IsPathRooted( filePath )==false )
            {
                bResult = SavePrevAs( document, changed );
            }
            else
            {
                string errMsg;
                bResult = SaveInternal( document, filePath, changed, out errMsg );
                if ( bResult==false )
                    ShowErrorSaveFailed( filePath, errMsg );
            }

            return bResult;
        }


        /// <summary>
        /// ワークスペースを保存。
        /// This method accepts a property change event argument
        /// so the user can use this method to operate upon multiple
        /// documents at the same time.
        /// </summary>
        /// <param name="project">The project to save.</param>
        /// <param name="filePath">The destination file path.</param>
        /// <param name="changed">The property change event arguments.</param>
        /// <returns>True on success.</returns>
        private bool SaveWorkspace( EffectProjectDocument project,
                                    string filePath,
                                    DocumentPropertyChangedEventArgs changed )
        {
            if ( project==null )
                return false;

            // Check if all the emitter sets have had their file path set.
            foreach ( IEmitterSetDocument doc in project.EmitterSetDocuments )
            {
                // Check emitter set documents ( .eset files )
                if ( System.IO.Path.IsPathRooted( doc.FileLocation )==false )
                {
                    /*
                    ThreadSafeMsgBox.Show( res.Strings.WARNING_MSG_SAVE_WORKFILE_NOT_ALL_FILES_SAVED,
                                           res.Strings.WARNING_CAPTION,
                                           MessageBoxButtons.OK,
                                           MessageBoxIcon.Exclamation );
                    */
                    return false;
                }
            }

            // 保存場所が決まっていない場合はSaveAsにまわす
            bool bResult = true;
            if ( Path.IsPathRooted( filePath )==false )
            {
                bResult = SaveWorkspaceAs( project, changed );
            }
            else
            {
                string errMsg;
                bResult = SaveInternal( project,
                                        filePath,
                                        changed,
                                        out errMsg );
                if ( bResult==false )
                    ShowErrorSaveFailed( filePath, errMsg );
            }

            return bResult;
        }

        #endregion

        #region Save as...

        /// <summary>
        /// ファイル名指定保存。
        /// </summary>
        /// <param name="document">The document to save.</param>
        /// <returns>True on success.</returns>
        public bool SaveAs( IDocument document )
        {
            DocumentPropertyChangedEventArgs changed = new DocumentPropertyChangedEventArgs();

            bool bResult = false;
            if ( document is EffectProjectDocument )
            {
                SaveWorkspaceAs( document as EffectProjectDocument, changed );
            }
            else if ( document is GameSettingsDocument )
            {
                SavePrevAs( document, changed );
            }
            else
            {
                // Check if the textures were assigned to the emitters.
                if ( AreTexturesAssigned( document, MessageBoxButtons.YesNo )==DialogResult.No )
                    return false;

                SaveAs( document, changed );
            }

            Document.NotifyPropertyChanged(this, changed);

            return bResult;
        }


        /// <summary>
        /// ファイル名指定保存。
        /// This method accepts a property change event argument
        /// so the user can use this method to operate upon multiple
        /// documents at the same time.
        /// </summary>
        /// <param name="document">The document to save.</param>
        /// <param name="changed">The property change event arguments.</param>
        /// <returns>True on success.</returns>
        private bool SaveAs( IDocument document,
                             DocumentPropertyChangedEventArgs changed )
        {
            EmitterSetDocument esetDoc = document as EmitterSetDocument;
            if ( esetDoc==null )
                return false;

            if ( this.EnableMessageBox==true )
            {
                TheApp.ResetUpdateDrawPathDialogFlags();
                if ( CheckEmitterDrawPathProperty(esetDoc)==false )
                    return false;

                DialogResult result =
                    CheckEmitterImmortalEndingProperties( esetDoc, MessageBoxButtons.YesNo );
                if ( result==DialogResult.No )
                    return false;
            }

            // Prompt user for the location to save the document.
            SaveFileDialog dialog = new SaveFileDialog();

            dialog.DefaultExt         = document.FileExt;
            dialog.FileName           = document.FileName;
            dialog.Filter             = document.FileFilter;
            dialog.AutoUpgradeEnabled = Config.Data.DocumentIO.AutoUpgradeEnabled;

            dialog.FileOk += new System.ComponentModel.CancelEventHandler( OnSaveFileDialogOK );

            // Initial directory for the dialog.
            if ( Path.IsPathRooted( document.FileLocation )==true )
                dialog.InitialDirectory = document.FileLocation;
            else
                dialog.InitialDirectory =
                    Config.Data.DocumentIO.GetLastAccessedDir( DocumentConstants.Eset );

            // Cancelled?
            /*
            if ( dialog.ShowDialog(TheApp.MainFrame)!=DialogResult.OK )
                return false;
            */

            using ( MCSDisableBlock disableBlock = new MCSDisableBlock() )
            {
                // Remember the directory path.
                Config.Data.DocumentIO.SetLastAccessedDir( DocumentConstants.Eset,
                                                           dialog.FileName );

                string docName = Path.GetFileNameWithoutExtension( dialog.FileName );

                // The document name has to be the same as the file name,
                // rename the document if necessary.
                /*
                if ( docName!=document.Name )
                {
                    using ( MCSEnableBlock enableBlock = new MCSEnableBlock() )
                    {
                        Viewer.Message.Close.Send( NWCore.Protocols.BinaryDataType.Ptcl,
                                                   document.FileTitle );
                    }

                    #region Execute command for changing the document name

                    ICommandSet commandset = new Command.CommandSet();

                    // ファイル名も一緒に変更します。
                    commandset.Add( new Command.DocumentRenameFileTitleCommand( esetDoc.Project,
                                                                                esetDoc,
                                                                                docName ) );

                    // 名前変更コマンド
                    commandset.Add( new Command.DocumentRenameCommand( esetDoc.Project,
                                                                       esetDoc,
                                                                       docName ) );

                    // Execute the command set
                    TheApp.CommandManager.ExecuteIgnoreTargetDocument( commandset );

                    #endregion

                    using ( MCSEnableBlock enableBlock = new MCSEnableBlock() )
                    {
                        Viewer.Message.SendUtility.SendEmitterSetBinary(document as IEmitterSetDocument);
                    }
                }
                */

                // Save the document.
                string errMsg;
                if ( SaveInternal(document,
                                  dialog.FileName,
                                  changed,
                                  out errMsg)==false )
                {
                    ShowErrorSaveFailed( dialog.FileName, errMsg );
                    return false;
                }
            }

            // Save the directory as the default path next time we open the file dialog.
            //Config.Data.DocumentIO.SaveAsPath = Path.GetDirectoryName(dialog.FileName);

            return true;
        }


        /// <summary>
        /// ファイル名指定保存。
        /// This method accepts a property change event argument
        /// so the user can use this method to operate upon multiple
        /// documents at the same time.
        /// </summary>
        /// <param name="document">The document to save.</param>
        /// <param name="changed">The property change event arguments.</param>
        /// <returns>True on success.</returns>
        private bool SavePrevAs( IDocument document,
                                 DocumentPropertyChangedEventArgs changed )
        {
            GameSettingsDocument doc = document as GameSettingsDocument;
            if ( doc==null )
                return false;

            // Prompt user for the location to save the document.
            SaveFileDialog dialog = new SaveFileDialog();

            dialog.DefaultExt         = document.FileExt;
            dialog.FileName           = document.FileName;
            dialog.Filter             = document.FileFilter;
            dialog.AutoUpgradeEnabled = Config.Data.DocumentIO.AutoUpgradeEnabled;

            dialog.FileOk += new System.ComponentModel.CancelEventHandler( OnSaveFileDialogOK );

            // Initial directory for the dialog.
            if ( Path.IsPathRooted( document.FileLocation )==true )
                dialog.InitialDirectory = document.FileLocation;
            else
                dialog.InitialDirectory =
                    Config.Data.DocumentIO.GetLastAccessedDir( DocumentConstants.Prev );

            // Cancelled?
            /*
            if ( dialog.ShowDialog(TheApp.MainFrame)!=DialogResult.OK )
                return false;
            */

            // Remember the directory path.
            Config.Data.DocumentIO.SetLastAccessedDir( DocumentConstants.Prev,
                                                       dialog.FileName );

            string docName = Path.GetFileNameWithoutExtension( dialog.FileName );

            // The document name has to be the same as the file name,
            // rename the document if necessary.
            if ( docName!=document.Name )
            {
                #region Execute command for changing the document name

                /*
                ICommandSet commandset = new Command.CommandSet();

                // 名前変更コマンド
                commandset.Add( new Command.DocumentRenameCommand( doc.EmitterSetDocument.Project,
                                                                   doc,
                                                                   docName ) );

                // Execute the command set
                TheApp.CommandManager.ExecuteIgnoreTargetDocument( commandset );

                doc.SetFileLocation( Path.GetDirectoryName(dialog.FileName) );
                */

                #endregion
            }

            // Save the document.
            string errMsg;
            if ( SaveInternal(document,
                              dialog.FileName,
                              changed,
                              out errMsg)==false )
            {
                ShowErrorSaveFailed( dialog.FileName, errMsg );
                return false;
            }

            return true;
        }


        /// <summary>
        /// ワークスペースをファイル名指定保存。
        /// This method accepts a property change event argument
        /// so the user can use this method to operate upon multiple
        /// documents at the same time.
        /// </summary>
        /// <param name="project">The document to save.</param>
        /// <param name="changed">The property change event arguments.</param>
        /// <returns>True on success.</returns>
        private bool SaveWorkspaceAs( EffectProjectDocument project,
                                      DocumentPropertyChangedEventArgs changed )
        {
            // 上書き確認がファイルダイアログで行われる
            Debug.Assert( project!=null );

            // Check if all the emitter sets have had their file path set.
            foreach ( IEmitterSetDocument doc in project.EmitterSetDocuments )
            {
                // Check emitter set documents ( .eset files )
                if ( System.IO.Path.IsPathRooted( doc.FileLocation )==false )
                {
                    /*
                    ThreadSafeMsgBox.Show( res.Strings.WARNING_MSG_SAVE_WORKFILE_NOT_ALL_FILES_SAVED,
                                           res.Strings.WARNING_CAPTION,
                                           MessageBoxButtons.OK,
                                           MessageBoxIcon.Exclamation );
                    */
                    return false;
                }
            }

            SaveFileDialog dialog = new SaveFileDialog();
            dialog.DefaultExt = project.FileExt;
            dialog.FileName   = project.FileName;
            dialog.Filter     = project.FileFilter;
            dialog.AutoUpgradeEnabled = Config.Data.DocumentIO.AutoUpgradeEnabled;

            dialog.FileOk += new System.ComponentModel.CancelEventHandler( OnSaveFileDialogOK );

            if ( Path.IsPathRooted( project.FileLocation )==true )
                dialog.InitialDirectory = project.FileLocation;
            else
                dialog.InitialDirectory =
                    Config.Data.DocumentIO.GetLastAccessedDir( DocumentConstants.FEnv );

            /*
            if ( dialog.ShowDialog(TheApp.MainFrame)!=DialogResult.OK )
                return false;
            */

            // Remember the directory path.
            Config.Data.DocumentIO.SetLastAccessedDir( DocumentConstants.FEnv,
                                                       dialog.FileName );

            // Save the document.
            string errMsg;
            if ( SaveInternal( project,
                               dialog.FileName,
                               changed,
                               out errMsg )==false )
            {
                ShowErrorSaveFailed( dialog.FileName, errMsg );
                return false;
            }

            return true;
        }

        #endregion

        #region Save all

        /// <summary>
        /// 全て上書き保存。
        /// </summary>
        public bool SaveAll()
        {
            IProjectDocument project = ProjectManager.ActiveProject;
            Debug.Assert( project!=null );

            List<IDocument> documents = GetDocumentSaveList( project, true, Config.Data.Option.SavePreviewOnSaveAll );
            if ( documents.Count<=0 )
                return true;

            List<IDocument> overwriteDocList = new List<IDocument>();
            List<IDocument> pathEmptyDocList = new List<IDocument>();

            foreach ( IDocument doc in documents )
            {
                if ( Path.IsPathRooted( doc.FilePath )==true )
                    overwriteDocList.Add( doc );
                else
                    pathEmptyDocList.Add( doc );
            }

            TheApp.ResetUpdateDrawPathDialogFlags();

            using ( WaitCursor wait = new WaitCursor())
            using ( DocumentSaverStatusIndicator indicator =
                        new DocumentSaverStatusIndicator(this, documents.Count) )
            {
                DocumentPropertyChangedEventArgs changed =
                    new DocumentPropertyChangedEventArgs();

                foreach ( IDocument doc in overwriteDocList )
                {
                    // Check if the all the textures were assigned.
                    DialogResult texDlgResult = AreTexturesAssigned( doc, MessageBoxButtons.YesNoCancel );
                    if ( texDlgResult==DialogResult.No )
                        continue;
                    else if ( texDlgResult==DialogResult.Cancel )
                        return true;

                    EmitterSetDocument esetDoc = doc as EmitterSetDocument;
                    if ( esetDoc!=null &&
                         this.EnableMessageBox==true )
                    {
                        if ( CheckEmitterDrawPathProperty(esetDoc)==false )
                            return false;

                        DialogResult result =
                            CheckEmitterImmortalEndingProperties( esetDoc, MessageBoxButtons.YesNoCancel );
                        if ( result==DialogResult.No )
                            continue;
                        else if ( result==DialogResult.Cancel )
                            return true;
                    }

                    string errMsg;
                    if ( SaveInternal( doc,
                                       doc.FilePath,
                                       changed,
                                       out errMsg )==false )
                    {
                        ShowErrorSaveFailed( doc.FilePath, errMsg );
                    }
                }

                /*
                foreach ( IDocument doc in pathEmptyDocList )
                {
                    DialogResult dialogResult =
                        ThreadSafeMsgBox.Show( TheApp.MainFrame,
                                               doc.Name + res.Strings.DIALOG_MSG_CONFIRM_SAVE_CHANGES,
                                               res.Strings.DIALOG_CAPTION_CONFIRM_SAVE_CHANGES,
                                               MessageBoxButtons.YesNoCancel,
                                               MessageBoxIcon.Information );

                    if ( dialogResult==DialogResult.Yes )
                    {
                        // Check if the all the textures were assigned.
                        DialogResult texDlgResult = AreTexturesAssigned( doc, MessageBoxButtons.YesNoCancel );
                        if ( texDlgResult==DialogResult.No ||
                             texDlgResult==DialogResult.Cancel )
                        {
                            return true;
                        }
                        else
                        {
                            EmitterSetDocument esetDoc = doc as EmitterSetDocument;
                            if ( esetDoc!=null &&
                                 this.EnableMessageBox==true )
                            {
                                if ( CheckEmitterDrawPathProperty(esetDoc)==false )
                                    return false;

                                DialogResult result =
                                    CheckEmitterImmortalEndingProperties( esetDoc, MessageBoxButtons.YesNoCancel );
                                if ( result==DialogResult.No )
                                    continue;
                                else if ( result==DialogResult.Cancel )
                                    return true;
                            }

                            m_bSuppressImmortalWarningDialog = true;

                            // Save the document.
                            if ( doc is GameSettingsDocument )
                            {
                                SavePrevAs( doc, changed );
                            }
                            else if ( doc is EmitterSetDocument )
                            {
                                SaveAs( doc, changed );
                            }

                            m_bSuppressImmortalWarningDialog = false;
                        }
                    }
                    else if ( dialogResult==DialogResult.Cancel )
                    {
                        break;
                    }
                }
                */

                Document.NotifyPropertyChanged( this, changed );
            }

            return true;
        }

        #endregion

        #region Save all to...

        /// <summary>
        /// 全て指定場所に保存。
        /// </summary>
        public bool SaveAllTo( IProjectDocument project )
        {
            if ( project==null )
                return false;

            Debug.Assert( project!=null );

            // Gather the documents for saving.
            List<IDocument> documents = GetDocumentSaveList( project, false );

            return SaveAllTo( documents );
        }


        /// <summary>
        /// 全て指定場所に保存。
        /// </summary>
        public bool SaveAllTo( List<IDocument> documents )
        {
            if ( documents==null ||
                 documents.Count<=0 )
            {
                return false;
            }

            TheApp.ResetUpdateDrawPathDialogFlags();

            List<IDocument> docList = new List<IDocument>( documents.Count );

            // Check if the all the textures were assigned.
            foreach ( IDocument doc in documents )
            {
                DialogResult texDlgResult = AreTexturesAssigned( doc, MessageBoxButtons.YesNoCancel );
                if ( texDlgResult==DialogResult.No )
                    continue;
                else if ( texDlgResult==DialogResult.Cancel )
                    return true;

                EmitterSetDocument esetDoc = doc as EmitterSetDocument;
                if ( esetDoc!=null &&
                     this.EnableMessageBox==true )
                {
                    if ( CheckEmitterDrawPathProperty(esetDoc)==false )
                        return false;

                    DialogResult result =
                        CheckEmitterImmortalEndingProperties( esetDoc, MessageBoxButtons.YesNoCancel );
                    if ( result==DialogResult.No )
                        continue;
                    else if ( result==DialogResult.Cancel )
                        return true;
                }

                docList.Add( doc );
            }

            if ( docList.Count<=0 )
                return true;

            // Create and show the dialog.
            /*
            List<DocumentSaveInfo> saveList =
                TheApp.MainFrame.ShowSaveDirectoryDialog( docList );
            if ( saveList==null || saveList.Count==0 )
                return false;
            */

            bool bFinalResult = true;

            /*
            using ( WaitCursor wait = new WaitCursor() )
            using ( DocumentSaverStatusIndicator indicator =
                        new DocumentSaverStatusIndicator( this, saveList.Count ) )
            {
                bool bCurrResult = true;

                DocumentPropertyChangedEventArgs changed =
                    new DocumentPropertyChangedEventArgs();

                // Save the assets first, otherwise the SaveInternal method would
                // prompt user to copy assets.
                foreach ( DocumentSaveInfo info in saveList )
                {
                    string errMsg = string.Empty;
                    if ( info.Document!=null )
                        continue;

                    #region Copy assets

                    try
                    {
                        // Is the source and target path identical?
                        if ( info.SourcePathCRC==info.TargetPathCRC )
                            continue;

                        // Does the source file exist?
                        if ( File.Exists( info.SourcePath )==false )
                            continue;

                        // Is there already a texture file with the same name?
                        if ( File.Exists( info.TargetPath )==true )
                        {
                            // Another texture file with the same name was found in the search path.
                            // Check if the modify time of these 2 files is different.
                            DateTime srcModTime = File.GetLastWriteTime( info.SourcePath );
                            DateTime dstModTime = File.GetLastWriteTime( info.TargetPath );

                            // The modify time is different, prompt user for overwriting.
                            if ( srcModTime.CompareTo( dstModTime )!=0 )
                            {
                                OverwriteAssetDialog dialog =
                                    new OverwriteAssetDialog( info.SourcePath,
                                                                srcModTime,
                                                                info.TargetPath,
                                                                dstModTime );

                                if ( dialog.ShowDialog()!=DialogResult.Yes )
                                    continue;
                            }
                        }

                        // Does the destination directory exist?
                        string targetDir = Path.GetDirectoryName( info.TargetPath );
                        if ( Directory.Exists( targetDir )==false )
                        {
                            // Create the destination directory.
                            Directory.CreateDirectory( targetDir );
                        }

                        // Copy the file
                        File.Copy( info.SourcePath, info.TargetPath, true );

                        bCurrResult = true;
                    }
                    catch ( Exception exception )
                    {
                        errMsg = exception.Message;

                        #if DEBUG
                            errMsg += "\nCall stack :\n" + exception.StackTrace;
                        #endif

                        bCurrResult = false;
                    }

                    #endregion

                    if ( bCurrResult==false )
                    {
                        ShowErrorSaveFailed( info.TargetPath, errMsg );
                    }
                }

                // Save all the rest of the documents on the list
                foreach ( DocumentSaveInfo info in saveList )
                {
                    string errMsg = string.Empty;
                    if ( info.Document==null )
                        continue;

                    #region Save document

                    bCurrResult = SaveInternal( info.Document,
                                                info.TargetPath,
                                                changed,
                                                out errMsg,
                                                true );

                    #endregion

                    if ( bCurrResult==false )
                    {
                        ShowErrorSaveFailed( info.TargetPath, errMsg );
                    }
                }

                // Trigger event.
                Document.NotifyPropertyChanged( this, changed );
            }
            */

            return bFinalResult;
        }

        #endregion

        #region The internal save method

        /// <summary>
        /// Internal save method. This method is the method that actually
        /// saves the document.
        ///
        /// If bExport is set to true, the saving will not have any affect to
        /// the document, for example, the file path will not be set, the modify
        /// flag will not be removed.
        /// </summary>
        /// <param name="doc">The document to save.</param>
        /// <param name="filePath">The destination file path.</param>
        /// <param name="changed">The property change event arguments.</param>
        /// <param name="errMsg">The error message if any.</param>
        /// <param name="bExport">True to export the document.</param>
        /// <returns>True on success.</returns>
        private bool SaveInternal( IDocument doc,
                                   string filePath,
                                   DocumentPropertyChangedEventArgs changed,
                                   out string errMsg,
                                   bool bExport = false )
        {
            errMsg = string.Empty;

            // Verify the texture location.
            EmitterSetDocument esetDoc = doc as EmitterSetDocument;
            if ( esetDoc!=null && bExport==false )
            {
                VerifyAssetsLocation( esetDoc, filePath );
            }

            // Show information on the status bar.
            UpdateStatusBar(filePath);

            bool bResult = true;

            DateTime start = DateTime.Now;
            try
            {
                if ( bExport==false )
                {
                    CheckRename( doc, filePath );

                    // Update the document.
                    doc.UpdateDocument();
                }

                // Save the document.
                if ( doc.SaveDocument(filePath,
                                      !bExport,
                                      true)==false )
                {
                    MessageLogger.WriteWarning( "{0} : \n\t{1}\n",
                                                res.Strings.IO_ERR_FileDoesNotExport,
                                                filePath );
                    bResult = false;
                }

                if ( bExport==false )
                {
                    // File path changed?
                    if ( doc.FilePath!=filePath )
                        doc.SetFileLocation( DocumentConstants.GetDirectoryName( filePath ) );

                    changed.AddArgs( new DocumentHeaderChangedArgs( doc ) );
                }

                if ( bResult==true )
                {
                    // Trigger the event.
                    OnDocumentoSaved( EventArgs.Empty );

                    // Compute the time spent.
                    TimeSpan time = DateTime.Now - start;

                    // Show log.
                    DebugConsole.WriteLine( "{0} was saved successfully.( time spent {1,4:f3} seconds )",
                                            doc.FileName,
                                            time.TotalSeconds );
                }
            }
            catch ( SystemException ex )
            {
                errMsg = ex.Message;

                #if DEBUG
                    errMsg += "\nCall stack :\n" + ex.StackTrace;
                #endif

                bResult = false;
            }

            // Clear the modify flag
            if ( bExport==false &&
                 bResult==true )
            {
                // Clear the modification flags for the saved documents.
                DocumentManager.ClearDataModifyFlag( doc.DataScrPath );
            }

            var document = doc as Document;
            if (document != null && bExport == false)
                document.NotifyDocumentSaveFinished(!bResult);

            return bResult;
        }

        #endregion

        #region Utility methods

        /// <summary>
        /// 保存対象のファイルリストを取得します。
        /// </summary>
        /// <param name="project">対象とするプロジェクトです。</param>
        /// <param name="bCheckModification">
        /// True to filter out the documents those were not modified.
        /// </param>
        /// <param name="bIncludeGameSettingsDoc">True to include game settings documents.</param>
        /// <returns>保存対象のリスト</returns>
        private List<IDocument> GetDocumentSaveList( IProjectDocument project,
                                                     bool bCheckModification,
                                                     bool bIncludeGameSettingsDoc = false )
        {
            List<IDocument> documents       = new List<IDocument>();
            List<IDocument> effects         = new List<IDocument>();
            List<IDocument> gameSettingsDoc = new List<IDocument>();
            // 各ドキュメントごとに集める
            foreach (IDocument item in project.SubDocuments)
            {
                if ( String.IsNullOrEmpty(item.FileExt)==false )
                {
                    if ( item is IEmitterSetDocument)
                    {
                        // Emitter set document.
                        if ( bCheckModification==false ||
                             item.Modified==true )
                        {
                            effects.Add( item );
                        }

                        // Game settings documents.
                        if ( bIncludeGameSettingsDoc==true )
                        {
                            EmitterSetDocument eset = item as EmitterSetDocument;
                            foreach ( GameSettingsDocument childDoc in eset.GameSettingsDocuments )
                            {
                                if ( bCheckModification==false ||
                                     childDoc.Modified==true )
                                {
                                    gameSettingsDoc.Add( childDoc );
                                }
                            }
                        }
                    }
                }
            }

            if ( effects.Count>0 )
            {
                documents.AddRange(effects.ToArray());
            }

            if ( gameSettingsDoc.Count>0 )
            {
                documents.AddRange( gameSettingsDoc.ToArray() );
            }

            return documents;
        }


        /// <summary>
        /// Update status bar on the bottom of the main frame.
        /// </summary>
        /// <param name="filePath">The file path of the document just being saved.</param>
        protected virtual void UpdateStatusBar(string filePath)
        {
            /*
            if ( TheApp.MainFrame==null )
                return;

            TheApp.MainFrame.StatusMessage =
                StringResource.Get( "IO_SAVING_MSG",
                                    Path.GetFileName(filePath) );
            */
        }


        /// <summary>
        /// Check if the file name or path of the document needs to be changed.
        /// If so, apply the change to the document.
        /// </summary>
        /// <param name="document">The document.</param>
        /// <param name="filePath">The destination file path.</param>
        protected virtual void CheckRename( IDocument document,
                                            string filePath )
        {
            string name = Path.GetFileNameWithoutExtension(filePath);

            // Was the file name changed?
            if ( document.Name==name )
                return;

            // Fire file name changed event.
            DocumentNameChangedArgs nameChangedArgs =
                new DocumentNameChangedArgs( document, document.Name );

            document.FileTitle = name;
            Document.NotifyPropertyChanged(null, nameChangedArgs);
        }


        /// <summary>
        /// セーブ失敗エラーの表示。
        /// </summary>
        /// <param name="filePath">The file path.</param>
        /// <param name="message">The error message to output.</param>
        private void ShowErrorSaveFailed( string filePath,
                                          string message )
        {
            /*
            if ( this.EnableMessageBox==false )
                return;

            string fullMessage = string.Format(res.Strings.IO_SAVE_FAILED_MSG, Path.GetFileName(filePath) + "\n" + message);
            ThreadSafeMsgBox.Show( fullMessage,
                                   res.Strings.ERROR_CAPTION,
                                   System.Windows.Forms.MessageBoxButtons.OK,
                                   System.Windows.Forms.MessageBoxIcon.Error );
            */
        }

        #endregion

        #region Emitter asset related utility methods

        /// <summary>
        /// Check if the textures in the emitter set document are already assigned.
        /// </summary>
        /// <param name="doc">The emitter set document.</param>
        /// <returns>True if the textures are assigned or user choose to continue.</returns>
        private DialogResult AreTexturesAssigned( IDocument doc,
                                                  MessageBoxButtons dlgButtons )
        {
            /*
            EmitterSetDocument esetDoc = doc as EmitterSetDocument;
            if ( esetDoc==null )
                return DialogResult.None;

            if ( this.EnableMessageBox==true &&
                 esetDoc.VerifyTexturesAreSet() == false)
            {
                string msg = string.Format( res.Strings.WARNING_TEXTURE_NOT_FOUND_CANCEL_SAVING,
                                            esetDoc.Name );
                DialogResult result =
                    ThreadSafeMsgBox.Show( TheApp.MainFrame,
                                           msg,
                                           res.Strings.WARNING_CAPTION,
                                           dlgButtons,
                                           MessageBoxIcon.Exclamation );
                return result;
            }
            */

            return DialogResult.None;
        }


        /// <summary>
        /// Verify the assets locations, if they are not in the search paths,
        /// prompt user if overwrite or copy the asset to search path.
        /// </summary>
        /// <param name="doc">The emitter set document.</param>
        /// <param name="esetPath">The location of the emitter set file.</param>
        /// <returns>True on success.</returns>
        private bool VerifyAssetsLocation( EmitterSetDocument doc,
                                           string esetPath )
        {
            return false;

            /*
            if ( this.EnableMessageBox==false )
                return true;

            bool bAllVerified = true;
            bool texturePathProblem = false;
            bool primitivePathProblem = false;
            string srcEsetPath = doc.FileLocation;
            string srcTexFullPath = string.Empty;

            try
            {
                // A list for keeping the source paths of the textures to copy.
                List<EmitterAssetInfo> copyList = new List<EmitterAssetInfo>();

                // Check if the textures are located within the search paths.
                string dstTexFullPath;
                bool   bFound;
                foreach ( EmitterAssetInfo info in doc.Textures )
                {
                    // Does the texture exist?
                    if ( string.IsNullOrEmpty(info.FilePath)==true )
                        continue;

                    if ( File.Exists(info.FilePath)==false )
                    {
                        bFound =
                            DocumentConstants.LocateTextureByFileName( srcEsetPath,
                                                                       Path.GetFileName(info.FilePath),
                                                                       out srcTexFullPath );
                        if ( bFound==false )
                        {
                            texturePathProblem = true;
                            bAllVerified = false;
                            continue;
                        }

                        info.FilePath = string.Copy( srcTexFullPath );
                    }

                    // Find the texture within the search paths.
                    bFound =
                        DocumentConstants.LocateTextureByFileName( esetPath,
                                                                   Path.GetFileName(info.FilePath),
                                                                   out dstTexFullPath );
                    if ( bFound==true &&
                         info.FilePath.ToLower()!=dstTexFullPath.ToLower() )
                    {
                        // Another texture file with the same name was found in the search path.
                        // Check if the modify time of these 2 files is different.
                        DateTime srcModTime = File.GetLastWriteTime( info.FilePath );
                        DateTime dstModTime = File.GetLastWriteTime( dstTexFullPath );

                        // The modify time is different, prompt user for overwriting.
                        if ( srcModTime.CompareTo(dstModTime)!=0 )
                        {
                            OverwriteAssetDialog dialog =
                                new OverwriteAssetDialog( info.FilePath,
                                                            srcModTime,
                                                            dstTexFullPath,
                                                            dstModTime );


                            if ( dialog.ShowDialog()==DialogResult.Yes )
                            {
                                File.Copy( info.FilePath, dstTexFullPath, true );
                            }
                        }
                    }
                    else if ( bFound==false )
                    {
                        // The asset is not in the search path.
                        // Collect the information of these assets and we will
                        // prompt the user for copying these assets to the search
                        // path later on.
                        copyList.Add( info );
                    }
                }

                foreach ( EmitterAssetInfo info in doc.Primitives )
                {
                    // Does the primitive exist?
                    if ( string.IsNullOrEmpty(info.FilePath)==true )
                        continue;

                    if ( File.Exists(info.FilePath)==false )
                    {
                        bFound =
                            DocumentConstants.LocatePrimitiveByFileName( srcEsetPath,
                                                                         Path.GetFileName(info.FilePath),
                                                                         out srcTexFullPath );
                        if ( bFound==false )
                        {
                            primitivePathProblem = true;
                            bAllVerified = false;
                            continue;
                        }

                        info.FilePath = string.Copy( srcTexFullPath );
                    }

                    // Find the texture within the search paths.
                    bFound =
                        DocumentConstants.LocatePrimitiveByFileName( esetPath,
                                                                     Path.GetFileName(info.FilePath),
                                                                     out dstTexFullPath );
                    if ( bFound==true &&
                         info.FilePath.ToLower()!=dstTexFullPath.ToLower() )
                    {
                        // Another texture file with the same name was found in the search path.
                        // Check if the modify time of these 2 files is different.
                        DateTime srcModTime = File.GetLastWriteTime( info.FilePath );
                        DateTime dstModTime = File.GetLastWriteTime( dstTexFullPath );

                        // The modify time is different, prompt user for overwriting.
                        if ( srcModTime.CompareTo(dstModTime)!=0 )
                        {
                            OverwriteAssetDialog dialog =
                                new OverwriteAssetDialog( info.FilePath,
                                                          srcModTime,
                                                          dstTexFullPath,
                                                          dstModTime );

                            if ( dialog.ShowDialog()==DialogResult.Yes )
                            {
                                File.Copy( info.FilePath, dstTexFullPath, true );
                            }
                        }
                    }
                    else if ( bFound==false )
                    {
                        // The asset is not in the search path.
                        // Collect the information of these assets and we will
                        // prompt the user for copying these assets to the search
                        // path later on.
                        copyList.Add( info );
                    }
                }

                if ( copyList.Count>0 )
                {
                    // There are assets that are not in the search path, ask the user if
                    // he wants to copy them to the search path.
                    CopyAssetDialog copyAssetDialog = new CopyAssetDialog();

                    copyAssetDialog.EsetFilePath = esetPath;
                    copyAssetDialog.Assets       = copyList;
                    if ( copyAssetDialog.ShowDialog(TheApp.MainFrame)==DialogResult.OK )
                    {
                        // Copy the textures.
                        string fileName   = string.Empty;
                        string targetPath = string.Empty;
                        foreach ( EmitterAssetInfo info in copyList )
                        {
                            fileName   = Path.GetFileName( info.FilePath );

                            EmitterAssetTypes assetType =
                                info.AssetType & EmitterAssetTypes.AssetTypeMask;
                            if ( assetType==EmitterAssetTypes.TextureAsset )
                                targetPath = copyAssetDialog.TextureCopyDestinationPath;
                            else if ( assetType==EmitterAssetTypes.PrimitiveAsset )
                                targetPath = copyAssetDialog.PrimitiveCopyDestinationPath;
                            else
                                continue;

                            File.Copy( info.FilePath,
                                       Path.Combine(targetPath, fileName),
                                       true );
                        }
                    }
                    else
                    {
                        bAllVerified = false;
                        foreach ( EmitterAssetInfo info in copyList )
                        {
                            string formatStr = string.Empty;
                            switch ( info.AssetType )
                            {
                                case EmitterAssetTypes.EmitterTexture0 :
                                    formatStr = res.Strings.LOG_MSG_TEXTURE_0_NEXT_TIME_CANNOT_BE_FOUND;
                                    texturePathProblem = true;
                                    break;
                                case EmitterAssetTypes.EmitterTexture1 :
                                    formatStr = res.Strings.LOG_MSG_TEXTURE_1_NEXT_TIME_CANNOT_BE_FOUND;
                                    texturePathProblem = true;
                                    break;
                                case EmitterAssetTypes.EmitterTexture2 :
                                    formatStr = res.Strings.LOG_MSG_TEXTURE_2_NEXT_TIME_CANNOT_BE_FOUND;
                                    texturePathProblem = true;
                                    break;
                                case EmitterAssetTypes.ChildTexture0 :
                                    formatStr = res.Strings.LOG_MSG_CHILD_TEXTURE_NEXT_TIME_CANNOT_BE_FOUND;
                                    texturePathProblem = true;
                                    break;
                                case EmitterAssetTypes.EmitterPrimitive :
                                    formatStr = res.Strings.LOG_MSG_MAIN_PRIMITIVE_NEXT_TIME_CANNOT_BE_FOUND;
                                    primitivePathProblem = true;
                                    break;
                                case EmitterAssetTypes.ChildPrimitive :
                                    formatStr = res.Strings.LOG_MSG_CHILD_PRIMITIVE_NEXT_TIME_CANNOT_BE_FOUND;
                                    primitivePathProblem = true;
                                    break;
                                default:
                                    break;
                            }

                            TheApp.Logger.Warn.FormatMessage( formatStr,
                                                              info.EmitterSetDocument.Name,
                                                              info.EmitterDocument.Name,
                                                              info.FilePath );
                        }
                    }
                }
            }
            catch ( Exception ex )
            {
                DebugConsole.WriteLine( "Failed verifying texture location, error message : " +
                                        ex.Message +
                                        "\nCall stack :\n" +
                                        ex.StackTrace );
                bAllVerified = false;
            }

            if ( texturePathProblem )
            {
                DialogResult result =
                    ThreadSafeMsgBox.Show( res.Strings.WARNING_TEXTURE_SAVE_LOC_NOT_REACHABLE,
                                           res.Strings.WARNING_CAPTION,
                                           MessageBoxButtons.OK,
                                           MessageBoxIcon.Exclamation);
            }

            if ( primitivePathProblem )
            {
                DialogResult result =
                    ThreadSafeMsgBox.Show( res.Strings.WARNING_PRIMITIVE_SAVE_LOC_NOT_REACHABLE,
                                           res.Strings.WARNING_CAPTION,
                                           MessageBoxButtons.OK,
                                           MessageBoxIcon.Exclamation);
            }

            return bAllVerified;
            */
        }

        #endregion

        #region Check emitter immortal & ending properties

        /// <summary>
        /// Check the immortal & ending properties of the emitter documents.
        /// Show warning dialog if necessary.
        /// </summary>
        /// <param name="eset">The owner emitter set document.</param>
        /// <param name="dlgButtons">The dialog button options.</param>
        /// <returns>
        /// DialogResult.None to proceed on saving or one of the dialog buttons given.
        /// </returns>
        private DialogResult CheckEmitterImmortalEndingProperties( EmitterSetDocument eset,
                                                                   MessageBoxButtons dlgButtons )
        {
            if ( m_bSuppressImmortalWarningDialog==true )
                return DialogResult.None;

            string emitterNames = string.Empty;
            foreach ( IEmitterDocument emitter in eset.EmitterDocuments )
            {
                if ( emitter==null || emitter.EmitterData==null )
                    continue;

                if ( emitter.EmitterData.AnimEditData.UI_lifeInfinit==(uint)1 &&
                     emitter.EmitterData.UI_fadeOutType==NWCore.DataModel.FadeOutType.StopEmit )
                {
                    emitterNames += (emitterNames.Length>0 ? "\n" : string.Empty) +
                                    eset.Name + ":" + emitter.Name;
                }
            }

            /*
            if ( string.IsNullOrEmpty(emitterNames)==false )
            {
                DialogResult result =
                    ThreadSafeMsgBox.Show( emitterNames + "\n" + res.Strings.WARNING_EMITTER_IMMORTAL_ENDING_PROPERTIES,
                                           res.Strings.WARNING_CAPTION,
                                           dlgButtons,
                                           MessageBoxIcon.Exclamation );
                return result;
            }
            */

            return DialogResult.None;
        }

        #endregion

        #region Check emitter draw path

        /// <summary>
        /// Check if the emitter has valid draw path.
        /// </summary>
        /// <param name="eset">The emitter set document.</param>
        /// <returns>False when cancel button is clicked.</returns>
        private bool CheckEmitterDrawPathProperty( IEmitterSetDocument eset )
        {
            return false;

            /*
            if ( TheApp.UserConfigData==null )
                return true;

            // No draw paths are set in the user config file,
            // and emitter draw path is also empty.
            // Just bail out.
            if ( TheApp.UserConfigData.DrawPaths==null ||
                 TheApp.UserConfigData.DrawPaths.Length<=0 )
            {
                return true;
            }

            foreach ( IEmitterDocument emitter in eset.EmitterDocuments )
            {
                if ( TheApp.UserConfigData.FindDrawPathByText(emitter.EmitterData.UI_drawPath)!=null )
                    continue;

                if ( TheApp.ShouldSkipUpdateDrawPathDialog()==true )
                {
                    if ( TheApp.ShouldUpdateDrawPath()==true )
                        emitter.EmitterData.UI_drawPath = TheApp.UserConfigData.DefaultDrawPath;
                    continue;
                }

                bool bResetDrawPath = false;

                UpdateInvalidDrawPathDialog dialog =
                    new UpdateInvalidDrawPathDialog( eset.Name + " : " + emitter.Name );

                dialog.ShowCancelButton = true;

                DialogResult result = dialog.ShowDialog( TheApp.MainFrame );

                if ( result==DialogResult.Cancel )
                    return false;
                else if ( result==DialogResult.Yes )
                    bResetDrawPath = true;
                else if ( result==DialogResult.No )
                    bResetDrawPath = false;
                else
                    continue;

                if ( dialog.IsApplyToAll==true )
                    TheApp.SetUpdateDrawPathDialogAction( bResetDrawPath );

                if ( bResetDrawPath==true )
                    emitter.EmitterData.UI_drawPath = TheApp.UserConfigData.DefaultDrawPath;
            }

            return true;
            */
        }

        #endregion

        #region Event handler for save file dialog

        /// <summary>
        /// Handle FileOK event for the save file dialogs.
        /// </summary>
        /// <param name="sender">The sender of the event.</param>
        /// <param name="e">The event arguments.</param>
        private void OnSaveFileDialogOK( object sender,
                                         System.ComponentModel.CancelEventArgs e )
        {
            SaveFileDialog dialog = sender as SaveFileDialog;
            if ( dialog==null )
                return;

            string filePath = dialog.FileName;

            Regex validateFileName = new Regex( res.Strings.REGEXP_DOCUMENT_NAME_VALIDATION );

            if ( validateFileName.IsMatch(Path.GetFileNameWithoutExtension(filePath))==false )
            {
                /*
                ThreadSafeMsgBox.Show( TheApp.MainFrame,
                                       res.Strings.DEFAULT_INVALID_NAME_ERROR_MSG,
                                       res.Strings.MSGBOX_CAPTION_NAME_INPUT_DIALOG_INVALID_NAME,
                                       System.Windows.Forms.MessageBoxButtons.OK,
                                       System.Windows.Forms.MessageBoxIcon.Exclamation );
                */
                e.Cancel = true;
            }

            // Filename length check
            if (Path.GetFileNameWithoutExtension(filePath).Length > DocumentIO.s_MaxFileNameLength)
            {
                DocumentIO.ErrorFileNameLengthSave(filePath);
                e.Cancel = true;
            }
        }

        #endregion

        #region Member variables

        private bool m_bSuppressImmortalWarningDialog = false;

        #endregion
    }

    #region Class for showing save progress on the bottom of main frame

    /// <summary>
    /// ドキュメントセーバステータスインジケータクラス。
    /// </summary>
    public sealed class DocumentSaverStatusIndicator : StatusIndicator
    {
        /// <summary>
        /// コンストラクタ。
        /// </summary>
        public DocumentSaverStatusIndicator( DocumentSaver saver,
                                             int stepCount ) :
            base(stepCount)
        {
            m_saver = saver;
            /*
            if ( m_saver!=null )
                m_saver.DocumentSaved += new EventHandler( OnDocumentSaved );
            */
        }


        /// <summary>
        /// 破棄。
        /// </summary>
        public override void Dispose()
        {
            /*
            if ( m_saver!=null )
                m_saver.DocumentSaved -= new EventHandler( OnDocumentSaved );
            */

            base.Dispose();
        }


        /// <summary>
        /// Handle DocumentSaved event sent from the saver.
        /// </summary>
        /// <param name="sender">The sender of the event.</param>
        /// <param name="e">The event arguments.</param>
        private void OnDocumentSaved(object sender, EventArgs e)
        {
            AdvanceProgress();
        }


        private DocumentSaver m_saver = null;
    }

    #endregion
}
