﻿// ========================================================================
// <copyright file="UserConfigData.cs" company="Nintendo">
//      Copyright 2009 Nintendo.  All rights reserved.
// </copyright>
//
// These coded instructions, statements, and computer programs contain
// proprietary information of Nintendo of America Inc. and/or Nintendo
// Company Ltd., and are protected by Federal copyright law.  They may
// not be disclosed to third parties or copied or duplicated in any form,
// in whole or in part, without the prior written consent of Nintendo.
// ========================================================================

using System;
using System.Collections.Generic;
using System.Xml.Serialization;

namespace App
{
    /// <summary>
    /// Serializer class for the user config data.
    /// </summary>
    [Serializable]
    public class UserConfigData
    {
        #region Serializer class for the draw path data

        /// <summary>
        /// Serializer class for the draw path data.
        /// </summary>
        [Serializable]
        public class DrawPathData
        {
            /// <summary>
            /// Default constructor.
            /// </summary>
            public DrawPathData()
            {
                this.ID   = 0;
                this.Text = string.Empty;

                this.ShaderCompileDef1 = string.Empty;
                this.ShaderCompileDef2 = string.Empty;
            }


            /// <summary>The ID of the draw path data.</summary>
            [XmlAttribute("id")]
            public int ID { get; set; }


            /// <summary>The text(name) of the draw path data.</summary>
            [XmlAttribute("text")]
            public string Text { get; set; }


            /// <summary>The shader compile definition.</summary>
            [XmlAttribute( "shaderCompileDef1" )]
            public string ShaderCompileDef1 { get; set; }


            /// <summary>The shader compile definition.</summary>
            [XmlAttribute( "shaderCompileDef2" )]
            public string ShaderCompileDef2 { get; set; }
        }

        #endregion

        #region Serializer class for user data labels

        /// <summary>
        /// Serializer class for user data labels
        /// </summary>
        [Serializable]
        public class UserDataLabels
        {
            /// <summary>
            /// Default constructor.
            /// </summary>
            public UserDataLabels()
            {
                this.BitLabel         = null;
                this.IntNumberLabel   = null;
                this.FloatNumberLabel = null;
            }


            /// <summary>The labels for the bit data fields.</summary>
            [XmlArrayItem( "string" )]
            public string[] BitLabel { get; set; }


            /// <summary>The labels for the integer data fields.</summary>
            [XmlArrayItem( "string" )]
            public string[] IntNumberLabel { get; set; }


            /// <summary>The labels for the float data fields.</summary>
            [XmlArrayItem("string")]
            public string[] FloatNumberLabel { get; set; }
        }

        #endregion

        #region Constructor

        /// <summary>
        /// Default constructor.
        /// </summary>
        public UserConfigData()
        {
            this.DrawPaths                = null;
            this.DefaultDrawPath          = string.Empty;
            this.DefaultDrawPathID        = -1;
            this.DefaultShaderCompileDef1 = string.Empty;
            this.DefaultShaderCompileDef2 = string.Empty;
            this.EmitterUserData          = null;
            this.EmitterSetUserData       = null;
        }

        #endregion

        #region Properties

        /// <summary>The draw paths.</summary>
        [XmlArrayItem("path")]
        public DrawPathData[] DrawPaths { get; set; }


        /// <summary>The default draw path.</summary>
        public string DefaultDrawPath { get; set; }


        /// <summary>The ID of the default draw path.</summary>
        public int DefaultDrawPathID { get; set; }


        /// <summary>The default user shader compile definition 1.</summary>
        public string DefaultShaderCompileDef1 { get; set; }


        /// <summary>The default user shader compile definition 2.</summary>
        public string DefaultShaderCompileDef2 { get; set; }


        /// <summary>The user data labels for the emitters.</summary>
        public UserDataLabels EmitterUserData { get; set; }


        /// <summary>The user data labels for the emitters.</summary>
        public UserDataLabels EmitterSetUserData { get; set; }

        #endregion

        #region Utility methods

        /// <summary>
        /// Find the draw path data with its text.
        /// </summary>
        /// <param name="text">The text of the draw path to find.</param>
        /// <returns>The draw path data or null if no matches are found.</returns>
        public DrawPathData FindDrawPathByText( string text )
        {
            if ( this.DrawPaths==null || this.DrawPaths.Length<=0 )
                return null;

            foreach ( DrawPathData data in this.DrawPaths )
            {
                if ( data==null ||
                     string.IsNullOrEmpty(data.Text)==true ||
                     data.Text.Length!=text.Length )
                {
                    continue;
                }

                if ( data.Text==text )
                    return data;
            }

            return null;
        }

        #endregion
    }
}
