﻿// ========================================================================
// <copyright file="IconManager.cs" company="Nintendo">
//      Copyright 2011 Nintendo.  All rights reserved.
// </copyright>
//
// These coded instructions, statements, and computer programs contain
// proprietary information of Nintendo of America Inc. and/or Nintendo
// Company Ltd., and are protected by Federal copyright law.  They may
// not be disclosed to third parties or copied or duplicated in any form,
// in whole or in part, without the prior written consent of Nintendo.
// ========================================================================

using System;
using System.Threading;
using System.IO;
using System.Drawing;
using System.Drawing.Imaging;
using System.Collections.Generic;
using System.Runtime.InteropServices;

//using nw.g3d.nw4f_3dif;

using NWCore.DataModel;

using App.Win32;
using App.IO;

namespace App.Data
{
    // Do typedef.
    using IconManagerTaskData = Tuple<IconManagerUserTickets,
                                      string,
                                      IconManager.AsyncThumbnailLoadedEvent>;

    /// <summary>
    /// Enum for the users of the icon manager.
    ///
    /// When requesting icon from the icon manager, one must provide
    /// the corresponding ticket so that icon manager knows which
    /// tasks need to be canceled whenever needed.
    ///
    /// Eq. when switched to another folder on EffectBrowser, the
    /// task requesting for the icons of the previous folder should
    /// be canceled, so the icon manager won't keep loading the
    /// icons that's not needed anymore.
    /// </summary>
    public enum IconManagerUserTickets
    {
        EffectBrowser,
        OpenAssetDialog,
        SaveDirectoryDialog
    }


    /// <summary>
    /// Cache dictionary class
    /// </summary>
    /// <typeparam name="TKey"></typeparam>
    /// <typeparam name="TValue"></typeparam>
    public class CacheDictionary<TKey, TValue>
    {
        /// <summary>
        /// Constructor
        /// </summary>
        public CacheDictionary(int size, int cacheRemovalCount = 1)
        {
            m_dictionary = new Dictionary<TKey, LinkedListNode<Tuple<TKey, TValue>>>(size);
            m_linkedList = new LinkedList<Tuple<TKey, TValue>>();
            m_size = size;
            m_cacheRemovalCount = cacheRemovalCount;
        }

        /// <summary>
        /// Add
        /// </summary>
        /// <param name="key"></param>
        /// <param name="?"></param>
        public void Add(TKey key, TValue value)
        {
            if (m_dictionary.ContainsKey(key))
            {
                return;
            }

            if (m_dictionary.Count >= m_size)
            {
                RemoveLast(m_cacheRemovalCount);
            }

            var tuple = Tuple.Create(key, value);
            m_linkedList.AddFirst(tuple);
            m_dictionary.Add(key, m_linkedList.First);
        }

        /// <summary>
        /// Remove last item(s)
        /// </summary>
        /// <param name="count"></param>
        public void RemoveLast(int count = 1)
        {
            for (int i = 0; i < count && m_dictionary.Count > 0; ++i)
            {
                LinkedListNode<Tuple<TKey, TValue>> node = m_linkedList.Last;
                m_dictionary.Remove(node.Value.Item1);
                m_linkedList.RemoveLast();
            }
        }

        /// <summary>
        /// Remove item by key
        /// </summary>
        /// <param name="key"></param>
        /// <returns></returns>
        public bool Remove(TKey key)
        {
            if (!m_dictionary.ContainsKey(key))
            {
                return false;
            }

            var node = m_dictionary[key];
            m_linkedList.Remove(node);

            return m_dictionary.Remove(key);
        }

        /// <summary>
        /// Contains key
        /// </summary>
        /// <param name="key9"></param>
        /// <returns></returns>
        public bool ContainsKey(TKey key)
        {
            return m_dictionary.ContainsKey(key);
        }

        /// <summary>
        /// Clear
        /// </summary>
        public void Clear()
        {
            m_dictionary.Clear();
            m_linkedList.Clear();
        }

        /// <summary>
        /// Indexer
        /// </summary>
        /// <param name="key"></param>
        /// <returns></returns>
        public TValue this[TKey key]
        {
            get
            {
                var node = m_dictionary[key];
                m_linkedList.Remove(node);
                m_linkedList.AddFirst(node);
                TValue value = node.Value.Item2;

                return value;
            }
        }

        Dictionary<TKey, LinkedListNode<Tuple<TKey, TValue>>> m_dictionary = null;
        LinkedList<Tuple<TKey, TValue>> m_linkedList = null;
        int m_size;
        int m_cacheRemovalCount;

    }

    /// <summary>
    /// Class for the icon manager.
    /// </summary>
    public class IconManager
    {
        /// <summary>
        /// The callback delegation called when icon manager finishes
        /// loading the requested icon/thumbnail.
        /// </summary>
        /// <param name="path">The path of the file to load.</param>
        /// <param name="result">True on success.</param>
        public delegate void AsyncThumbnailLoadedEvent( string path,
                                                        bool result );

        /// <summary>
        /// Class for storing the cache data for the loaded icons.
        /// </summary>
        public class IconCache
        {
            /// <summary>
            /// Constructor
            /// </summary>
            /// <param name="image"></param>
            public IconCache(Image image)
            {
                m_Image = image;
                m_ModTime = new DateTime(0);
                m_CheckModTime = false;
                m_ListIndex = -1;
            }

            /// <summary>
            /// Constructor
            /// </summary>
            /// <param name="image"></param>
            /// <param name="modTime"></param>
            public IconCache(Image image, DateTime modTime)
            {
                m_Image = image;
                m_ModTime = modTime;
                m_CheckModTime = true;
                m_ListIndex = -1;
            }

            /// <summary>
            ///
            /// </summary>
            /// <param name="image"></param>
            /// <param name="gammaCorrected"></param>
            /// <param name="modTime"></param>
            public IconCache(Image image, Image gammaCorrected, DateTime modTime)
            {
                m_Image = image;
                m_GammaCorrected = gammaCorrected;
                m_ModTime = modTime;
                m_CheckModTime = true;
                m_ListIndex = -1;
            }

            /// <summary>
            /// Image
            /// </summary>
            public Image Image
            {
                get
                {
                    if ( TheApp.IsGammaCorrectionEnabled==true && m_GammaCorrected != null)
                        return m_GammaCorrected;
                    else
                        return m_Image;
                }
            }

            /// <summary>
            /// Modification time
            /// </summary>
            public DateTime ModTime { get { return m_ModTime; } }

            /// <summary>
            /// Check modification time flag
            /// </summary>
            public bool CheckModTime { get { return m_CheckModTime; } }

            /// <summary>
            /// List index
            /// </summary>
            public int ListIndex { get { return m_ListIndex; } set { m_ListIndex = value; } }


            Image    m_Image = null;
            Image    m_GammaCorrected = null;
            DateTime m_ModTime;
            bool     m_CheckModTime;
            int      m_ListIndex;
        }

        /// <summary>
        /// Constructor
        /// </summary>
        IconManager(){}

        /// <summary>
        /// Destructor
        /// </summary>
        ~IconManager(){}

        /// <summary>
        /// Get thumbnail by path
        /// </summary>
        /// <param name="path"></param>
        /// <returns></returns>
        private static IconCache GetThumbnail(string path)
        {
            IconCache image = null;

            Monitor.Enter(m_ThumbnailCacheDictionary);
            if (m_ThumbnailCacheDictionary.ContainsKey(path))
            {
                image = m_ThumbnailCacheDictionary[path];
            }
            Monitor.Exit(m_ThumbnailCacheDictionary);

            return image;
        }

        /// <summary>
        /// Add thumbnail
        /// </summary>
        /// <param name="path"></param>
        /// <param name="image"></param>
        private static void AddThumbnail(string path, IconCache image)
        {
            Monitor.Enter(m_ThumbnailCacheDictionary);
            m_ThumbnailCacheDictionary.Add(path, image);
            Monitor.Exit(m_ThumbnailCacheDictionary);
        }

        /// <summary>
        /// Remove icon
        /// </summary>
        /// <param name="path"></param>
        /// <param name="image"></param>
        private static void RemoveThumbnail(string path)
        {
            Monitor.Enter(m_ThumbnailCacheDictionary);
            m_ThumbnailCacheDictionary.Remove(path);
            Monitor.Exit(m_ThumbnailCacheDictionary);
        }

        /// <summary>
        ///
        /// </summary>
        /// <param name="path"></param>
        /// <returns></returns>
        private static bool ThumbnailDictionaryContainsKey(string path)
        {
            bool bContains;

            Monitor.Enter(m_ThumbnailCacheDictionary);
            bContains = m_ThumbnailCacheDictionary.ContainsKey(path);
            Monitor.Exit(m_ThumbnailCacheDictionary);

            return bContains;
        }

        /// <summary>
        /// Get icon
        /// </summary>
        /// <param name="path"></param>
        /// <returns></returns>
        private static IconCache GetIcon(string path)
        {
            IconCache image = null;

            Monitor.Enter(m_IconDictionary);
            if (m_IconDictionary.ContainsKey(path))
            {
                image = m_IconDictionary[path];
            }
            Monitor.Exit(m_IconDictionary);

            return image;
        }

        /// <summary>
        /// Add icon
        /// </summary>
        /// <param name="path"></param>
        /// <param name="image"></param>
        private static void AddIcon(string path, IconCache image)
        {
            Monitor.Enter(m_IconDictionary);
            m_IconDictionary.Add(path, image);
            Monitor.Exit(m_IconDictionary);
        }

        /// <summary>
        /// Remove icon
        /// </summary>
        /// <param name="path"></param>
        /// <param name="image"></param>
        private static void RemoveIcon(string path)
        {
            Monitor.Enter(m_IconDictionary);
            m_IconDictionary.Remove(path);
            Monitor.Exit(m_IconDictionary);
        }

        /// <summary>
        ///
        /// </summary>
        /// <param name="path"></param>
        /// <returns></returns>
        private static bool IconDictionaryContainsKey(string path)
        {
            bool bContains;

            Monitor.Enter(m_IconDictionary);
            bContains = m_IconDictionary.ContainsKey(path);
            Monitor.Exit(m_IconDictionary);

            return bContains;
        }


        /// <summary>
        ///
        /// </summary>
        /// <param name="fileExt"></param>
        /// <param name="icon"></param>
        /// <returns></returns>
        public static bool SetFileAssociatedIcon(string fileExt, Image icon)
        {
            if (icon == null)
                return false;

            string extL = fileExt.ToLower();

            if (IconDictionaryContainsKey(fileExt))
            {
                RemoveIcon(fileExt);
            }

            AddIcon(extL, new IconCache(icon));

            return true;
        }

        /// <summary>
        /// Get icon from a path.
        /// If the thumbnail is not exist then return null and asynchronously
        /// load/make the thumbnail.
        /// </summary>
        /// <param name="ticket">The ticket that indicating who is requesting the icon.</param>
        /// <param name="info">The file info or directory info.</param>
        /// <param name="callback">Thumbnail loaded event(called in another thread)</param>
        /// <returns>Thumbnail image</returns>
        public static Image GetIcon( IconManagerUserTickets ticket,
                                     FileSystemInfo info,
                                     AsyncThumbnailLoadedEvent callback )
        {
            return GetIcon( ticket, info.FullName, callback );
        }

        /// <summary>
        /// Get icon from a path.
        /// If the thumbnail is not exist then return null and asynchronously
        /// load/make the thumbnail.
        /// </summary>
        /// <param name="ticket">The ticket that indicating who is requesting the icon.</param>
        /// <param name="path">Filepath</param>
        /// <param name="callback">Thumbnail loaded event(called in another thread)</param>
        /// <returns>Thumbnail image</returns>
        public static Image GetIcon( IconManagerUserTickets ticket,
                                     string path,
                                     AsyncThumbnailLoadedEvent callback )
        {
            string ext = Path.GetExtension(path).ToLower();
            bool bTexture = ext.Equals(DocumentConstants.DotFtxa) ||
                            ext.Equals(DocumentConstants.DotFtxb) ||
                            ext.Equals(DocumentConstants.DotTga);

            if ( bTexture==true )
            {
                // Textures
                IconCache cache   = GetThumbnail( path );
                DateTime  modTime = File.GetLastWriteTimeUtc( path );
                if ( cache!=null )
                {
                    if ( cache.CheckModTime==false ||
                         cache.ModTime.Equals(modTime)==true )
                    {
                        if ( cache.Image==null )
                        {
                            IconCache ftxbCache = GetIcon(DocumentConstants.DotFtxb);
                            return ( (ftxbCache!=null) ? ftxbCache.Image : null );
                        }
                        else
                        {
                            return cache.Image;
                        }
                    }
                    else
                    {
                        RemoveThumbnail( path );
                    }
                }

                AddTask( ticket, path, callback );

                Monitor.Enter( m_ManualResetEvent );
                m_ManualResetEvent.Set();
                Monitor.Exit( m_ManualResetEvent );

                if ( m_Thread.IsAlive==false )
                    m_Thread.Start();

                {
                    IconCache ftxbCache = GetIcon(DocumentConstants.DotFtxb);
                    return ( (ftxbCache!=null) ? ftxbCache.Image : null );
                }
            }
            else if ( Directory.Exists(path)==true )
            { // Directories

                if (m_FolderIcon == null)
                {
                    IntPtr hIcon = GetDirectoryIconHandle(path);

                    if (hIcon == IntPtr.Zero)
                        return m_DefaultIcon;

                    var image = GetSystemIcon(hIcon);
                    if (image == null)
                        return m_DefaultIcon;

                    m_FolderIcon = image;
                }

                return m_FolderIcon;
            }
            else
            { // Other files
                IconCache image = null;

                // Find by ext
                if ( (image = GetIcon(ext))!= null )
                    return image.Image;

                IntPtr hIcon = GetFileIconHandle(path);
                string key = hIcon.ToInt64().ToString();

                // Find by icon handle
                if ((image = GetIcon(key)) != null)
                    return image.Image;

                image = new IconCache(GetSystemIcon(hIcon));
                if (image == null)
                {
                    image = new IconCache(m_DefaultIcon);
                }

                AddIcon(key, image);

                return image.Image;
            }
        }

        /// <summary>
        /// Get handle of file icon
        /// </summary>
        /// <param name="path"></param>
        /// <returns></returns>
        private static IntPtr GetFileIconHandle(string path)
        {
            uint flags = (uint)SHGFI.SHGFI_ICON | (uint)SHGFI.SHGFI_USEFILEATTRIBUTES;
            flags |= (uint)SHGFI.SHGFI_SMALLICON;

            Win32.SHFILEINFO shfi = new Win32.SHFILEINFO();

            // Get file info, 0x10 = FILE_ATTRIBUTE_NORMAL
            Shell32.SHGetFileInfo(path,
                                  0x80,
                                  ref shfi,
                                  (uint)Marshal.SizeOf(shfi),
                                  flags);

            return shfi.hIcon;
        }

        /// <summary>
        /// Get handle of directory icon
        /// </summary>
        /// <param name="path"></param>
        /// <returns></returns>
        private static IntPtr GetDirectoryIconHandle(string path)
        {
            uint flags = (uint)SHGFI.SHGFI_ICON | (uint)SHGFI.SHGFI_USEFILEATTRIBUTES;
            flags |= (uint)SHGFI.SHGFI_SMALLICON;

            Win32.SHFILEINFO shfi = new Win32.SHFILEINFO();

            // Get file info, 0x10 = FILE_ATTRIBUTE_NORMAL
            Shell32.SHGetFileInfo(path,
                                  0x10,
                                  ref shfi,
                                  (uint)Marshal.SizeOf(shfi),
                                  flags);

            return shfi.hIcon;
        }

        /// <summary>
        /// Get system icon
        /// </summary>
        /// <param name="ext"></param>
        /// <returns></returns>
        private static Image GetSystemIcon(IntPtr hIcon)
        {
            if ( hIcon==IntPtr.Zero )
                return null;

            Icon icon = Icon.FromHandle(hIcon);
            if (icon == null)
            {
                icon = SystemIcons.WinLogo;
            }

            return icon.ToBitmap();
        }

        /// <summary>
        /// Finalize thread
        /// </summary>
        public static void FinalizeThread()
        {
            m_bThreadEnd = true;
            m_ManualResetEvent.Set();
            if (m_Thread.IsAlive)
            {
                m_Thread.Join();
            }
        }

        /// <summary>
        /// Cancel the queued tasks of the specified user.
        /// </summary>
        /// <param name="ticket">The ticket indicating the user whose tasks to be canceled.</param>
        public static void CancelTasks( IconManagerUserTickets ticket )
        {
            Monitor.Enter( m_QueryQueue );

            // Clean up the tasks of the user from the task queue first.
            for ( int i=m_QueryQueue.Count-1;i>=0;--i )
            {
                IconManagerTaskData data = m_QueryQueue[i];
                if ( data!=null && data.Item1==ticket )
                    m_QueryQueue.RemoveAt( i );
            }

            Monitor.Exit( m_QueryQueue );

            Monitor.Enter( m_CancelTaskUsers );

            // Add the user ticket to our task canceling user list.
            m_CancelTaskUsers.Add( ticket );

            Monitor.Exit( m_CancelTaskUsers );

            m_CancellationNotification.Set();
        }

        /// <summary>
        /// Queue a task to load texture and create thumbnail.
        /// </summary>
        /// <param name="ticket">The ticket that indicating who is requesting the task.</param>
        /// <param name="path">The file path of the texture to load.</param>
        /// <param name="callback">The callback function when the task is finished.</param>
        private static void AddTask( IconManagerUserTickets ticket,
                                     string path,
                                     AsyncThumbnailLoadedEvent callback )
        {
            Monitor.Enter(m_QueryQueue);
            m_QueryQueue.Add( new IconManagerTaskData(ticket, path, callback) );
            Monitor.Exit(m_QueryQueue);
        }

        /// <summary>
        /// Sequentially process the queue.
        /// </summary>
        private static void ProcessQueue()
        {
            m_ManualResetEvent.Reset();
            while ( true )
            {
                // Copy the task queue.
                Monitor.Enter( m_QueryQueue );

                // Copy out the tasks so that the queue can still be
                // accessed while processing the tasks.
                IconManagerTaskData[] queueList = m_QueryQueue.ToArray();
                m_QueryQueue.Clear();

                Monitor.Exit(m_QueryQueue);

                // enclose the code that make a cold sequence from m_CancelTaskUsers
                // into a lambda expression for recall later in the method
                Func<IconManagerUserTickets[]> makeColdCancelTaskUsers = () =>
                {
                    IconManagerUserTickets[] tmpCancelTaskUsers = null;

                    // Copy the user tickets whose tasks to be canceled.
                    lock (m_CancelTaskUsers)
                    {
                        tmpCancelTaskUsers = m_CancelTaskUsers.ToArray();
                        m_CancelTaskUsers.Clear();
                    }

                    return tmpCancelTaskUsers;
                };

                IconManagerUserTickets[] cancelTaskUsers = makeColdCancelTaskUsers();

                // Execute the tasks.
                foreach ( IconManagerTaskData task in queueList )
                {
                    if (m_CancellationNotification.IsSet)
                    {
                        // update the cold sequence
                        cancelTaskUsers = makeColdCancelTaskUsers();
                        m_CancellationNotification.Reset();
                    }

                    // Check if we should cancel the task.
                    bool bCancel = false;
                    if (cancelTaskUsers != null)
                    {
                        foreach (IconManagerUserTickets ticket in cancelTaskUsers)
                        {
                            if ( task.Item1==ticket )
                            {
                                bCancel = true;
                                break;
                            }
                        }
                    }

                    if ( bCancel==true )
                        continue;

                    // Execute the task.
                    LoadThumbnail( task.Item2, task.Item3 );
                }

                // Clean up.
                queueList = null;

                // Is the queue empty?
                bool queueIsEmpty = false;
                Monitor.Enter(m_QueryQueue);
                queueIsEmpty = ( m_QueryQueue.Count<=0 );
                Monitor.Exit(m_QueryQueue);

                // Exit loop if flag is raised
                if ( m_bThreadEnd==true )
                    break;

                // Wait if queue is empty
                if ( queueIsEmpty==true )
                {
                    m_ManualResetEvent.Reset();
                    m_ManualResetEvent.Wait();
                }

                // Exit loop if flag is raised
                if ( m_bThreadEnd==true )
                    break;
            }
        }

        /// <summary>
        /// Load and cache thumbnail
        /// </summary>
        /// <param name="path"></param>
        /// <param name="callback"></param>
        private static void LoadThumbnail( string path,
                                           AsyncThumbnailLoadedEvent callback )
        {
            // Check dictionary again
            IconCache image  = GetThumbnail(path);

            // If the image is already loaded by former queue
            if (image!=null)
            {
                if (callback != null)
                {
                    callback(path, true);
                }

                return;
            }

            /*
            TextureInfo texInfo;
            Bitmap[] colorImgs;
            Bitmap[] alphaImgs;
            Enum pixelFormat;
            texture_info_comp_selValue[] compSelector;

            if ( TheApp.TextureManager.LoadTextureDirect(path,
                                                         out texInfo,
                                                         out colorImgs,
                                                         out alphaImgs,
                                                         out pixelFormat,
                                                         out compSelector)==LoadTextureResults.Success )
            {
                const int iconSize = 16;

                Bitmap colorBitMap    = new Bitmap( iconSize*2,
                                                    iconSize,
                                                    PixelFormat.Format24bppRgb );
                using (Graphics colorGraphics = Graphics.FromImage(colorBitMap))
                {
                    colorGraphics.DrawImage(colorImgs[0],
                                             new Rectangle(0, 0, iconSize, iconSize));

                    colorGraphics.DrawImage(alphaImgs[0],
                                             new Rectangle(iconSize, 0, iconSize, iconSize),
                                             0, 0, alphaImgs[0].Width, alphaImgs[0].Height,
                                             GraphicsUnit.Pixel);
                }
                // Swizzle linear flag.
                bool[] linearSwizzled = new bool[4];
#if false
                for (int i = 0; i < 4; ++i)
                {
                    switch (m_compSelector[i])
                    {
                        case texture_info_comp_selValue.r:
                            linearSwizzled[i] = m_texData.Linear[0];
                            break;
                        case texture_info_comp_selValue.g:
                            linearSwizzled[i] = m_texData.Linear[1];
                            break;
                        case texture_info_comp_selValue.b:
                            linearSwizzled[i] = m_texData.Linear[2];
                            break;
                        case texture_info_comp_selValue.a:
                            linearSwizzled[i] = m_texData.Linear[3];
                            break;
                        default:
                            linearSwizzled[i] = false;
                            break;
                    }
                }
#else
                // EffectMaker上ではリニアフラグに依存せず
                // リニアモードならガンマ補正をかけるので全部true
                linearSwizzled[0] = true;
                linearSwizzled[1] = true;
                linearSwizzled[2] = true;
                linearSwizzled[3] = false;
#endif


                // 読み込み時にガンマ補正無/有の両方の画像を作成します
                Bitmap gammaCorrected;
                if ( TheApp.TextureManager.IsGammaCorrectedImage(texInfo.NativeDataFormat)==true )
                {
                    // SRGBの場合はオリジナルイメージがガンマ補正済みとみなす
                    gammaCorrected = new Bitmap( colorBitMap.Width,
                                                 colorBitMap.Height,
                                                 System.Drawing.Imaging.PixelFormat.Format24bppRgb );
                    App.Utility.GraphicsUtility.CopyBitmap( colorBitMap, gammaCorrected );

                    NintendoWare.ColorPicker.ColorUtil.ApplyInvImageGamma( colorBitMap,
                                                                           true );
                }
                else
                {
                    gammaCorrected = new Bitmap( colorBitMap.Width,
                                                 colorBitMap.Height,
                                                 System.Drawing.Imaging.PixelFormat.Format24bppRgb );
                    App.Utility.GraphicsUtility.CopyBitmap( colorBitMap, gammaCorrected );

                    NintendoWare.ColorPicker.ColorUtil.ApplyImageGamma( gammaCorrected,
                                                                        true,
                                                                        linearSwizzled[0],
                                                                        linearSwizzled[1],
                                                                        linearSwizzled[2],
                                                                        linearSwizzled[3] );
                }

                image = new IconCache(colorBitMap, gammaCorrected, File.GetLastWriteTimeUtc(path));

                AddThumbnail(path, image);

                if ( callback!=null )
                    callback(path, true);
            }
            else
            */
            {
                AddThumbnail( path, new IconCache(null, null, File.GetLastWriteTimeUtc(path)) );

                if ( callback!=null )
                    callback(path, false);
            }
        }

        private const string ICON_LOADING_IMAGE_KEY = "ICON_LOADING_IMAGE_KEY";
        private const int    MaxCacheCount = 256;

        static private Dictionary<string, IconCache>      m_IconDictionary           = new Dictionary<string, IconCache>(MaxCacheCount);
        static private CacheDictionary<string, IconCache> m_ThumbnailCacheDictionary = new CacheDictionary<string, IconCache>(MaxCacheCount, 1);
        static private List<IconManagerTaskData>          m_QueryQueue               = new List<IconManagerTaskData>();
        static private List<IconManagerUserTickets>       m_CancelTaskUsers          = new List<IconManagerUserTickets>();

        static private ManualResetEventSlim m_CancellationNotification = new ManualResetEventSlim();
        static private Thread               m_Thread = new Thread(new ThreadStart(ProcessQueue));
        static private ManualResetEventSlim m_ManualResetEvent = new ManualResetEventSlim();
        static private bool                 m_bThreadEnd       = false;
        static private Image                m_DefaultIcon      = SystemIcons.WinLogo.ToBitmap();
        static private Image                m_FolderIcon;
    }
}
