﻿// ========================================================================
// <copyright file="AssociationManager.cs" company="Nintendo">
//      Copyright 2009 Nintendo.  All rights reserved.
// </copyright>
//
// These coded instructions, statements, and computer programs contain
// proprietary information of Nintendo of America Inc. and/or Nintendo
// Company Ltd., and are protected by Federal copyright law.  They may
// not be disclosed to third parties or copied or duplicated in any form,
// in whole or in part, without the prior written consent of Nintendo.
// ========================================================================

using System;
using System.Collections.Generic;
using System.Text;
using Microsoft.Win32;
using System.ComponentModel;
using System.Diagnostics;
using System.Runtime.InteropServices;
using System.Windows.Forms;
using System.Collections;
using System.Globalization;
using System.ComponentModel.Design.Serialization;
using System.IO;
//using App.Controls;

namespace App.Data
{
    /// <summary>
    /// Provides more streamlined interface for associating a single or multiple
    /// extensions with a single program.
    /// </summary>
    public class AssociationManager
    {
        #region Internal Class - App

        /// <summary>
        /// Application to open the file with
        /// </summary>
        public class App
        {
            /// <summary>
            /// constructor
            /// </summary>
            public App()
            {
                this.appPath = null;
            }

            /// <summary>
            /// constructor
            /// </summary>
            /// <param name="appPath">path of executable</param>
            public App(string appPath)
            {
                // add validation logic here
                this.appPath = appPath;
            }

            private string appPath;

            /// <summary>
            /// Name of executable
            /// </summary>
            public string AppPath
            {
                get { return appPath; }
                set { appPath = value; }
            }
        }
        #endregion

        #region Internal Class - AppCollection

        /// <summary>
        /// A collection of applications available to open the file with
        /// </summary>
        [Serializable]
        public class AppCollection : CollectionBase
        {
            /// <summary>
            /// Get the app by index
            /// </summary>
            /// <param name="index">index of the app</param>
            /// <returns>App object</returns>
            public App this[int index]
            {
                get { return (App)List[index]; }
                set { List[index] = value; }
            }

            /// <summary>
            /// Add app to the list
            /// </summary>
            /// <param name="item">App to be added</param>
            /// <returns>index of the app</returns>
            public int Add(App item)
            {
                return List.Add(item);
            }

            /// <summary>
            /// Remove app from the list
            /// </summary>
            /// <param name="item">App to be removed</param>
            public void Remove(App item)
            {
                List.Remove(item);
            }
        }
        #endregion

        #region Data Structure

        /// <summary>
        /// Data structure
        /// </summary>
        public struct COPYDATASTRUCT
        {
            /// <summary>
            /// Pointer to the data
            /// </summary>
            public IntPtr dwData;

            /// <summary>
            /// Integer of data
            /// </summary>
            public int cbData;

            /// <summary>
            /// String data
            /// </summary>
            [MarshalAs(UnmanagedType.LPStr)]
            public string lpData;
        }

        #endregion

        #region Construtor

        /// <summary>
        /// Constructor of AssociationManager
        /// </summary>
        public AssociationManager()
        {
            AppName = "EffectMaker.exe";
            Description = "Nintendoware EffectMaker project file";
            EmbbedIcon = true;
            Ext = ".cepj";
            Handler = "Nintendoware";
            IconName = "AnimationCurve.ico";
            IconPos = 0;
            OpenText = "Open";
            HKCR = "HKEY_CLASSES_ROOT\\";

            openWith.Add(new App(Application.ExecutablePath));
        }

        #endregion

        #region Methods

        /// <summary>
        /// Register file types into registries
        /// </summary>
        public void RegisterFileType()
        {
            Registry.SetValue(HKCR + Ext, "", Handler);
            Registry.SetValue(HKCR + Handler, "", Description);
            Registry.ClassesRoot.CreateSubKey("DefaultIcon");

            if (EmbbedIcon)
            {
                Registry.SetValue(HKCR + Handler + "\\DefaultIcon", "", "\"" +
                                  Application.StartupPath + "\\" + AppName + "\"," + IconPos + "");
            }
            else
            {
                Registry.SetValue(HKCR + Handler + "\\DefaultIcon", "", "\"" +
                                  Application.StartupPath + "\\" + IconName + "\"");
            }

            Registry.SetValue(HKCR + Handler + "\\shell", "", "");
            Registry.SetValue(HKCR + Handler + "\\shell\\open", "", OpenText);
            Registry.SetValue(HKCR + Handler + "\\shell\\open\\command", "", "\"" + Application.StartupPath + "\\" + AppName + @""" ""%1""");

            for (int i = 0; i < openWith.Count; i++)
            {
                string app = openWith[i].AppPath;
                string originalApp = app;
                app = app.Substring(app.LastIndexOf("\\") + 1);
                app = app.Substring(0, app.LastIndexOf('.'));

                string runHandler = "CustomFileExtension." + app + ".Run.Handler";

                Registry.ClassesRoot.CreateSubKey(Ext + "\\OpenWithProgids");
                Registry.SetValue(HKCR + Ext + "\\OpenWithProgids", runHandler, "");

                // openWith handler
                Registry.SetValue(HKCR + runHandler, "", "");

                Registry.SetValue(HKCR + runHandler + "\\shell", "", "");
                Registry.SetValue(HKCR + runHandler + "\\shell\\open", "", "");
                Registry.SetValue(HKCR + runHandler + "\\shell\\open\\command", "", "\"" + originalApp + @""" ""%1""");
            }

            SHChangeNotify(HChangeNotifyEventID.SHCNE_ASSOCCHANGED,
                           HChangeNotifyFlags.SHCNF_IDLIST,
                           IntPtr.Zero,
                           IntPtr.Zero);
        }

        /// <summary>
        /// Remove file types from the registries
        /// </summary>
        public void RemoveFileType()
        {
            RegistryKey regExt = Registry.ClassesRoot.OpenSubKey(Ext);
            RegistryKey regHandler = Registry.ClassesRoot.OpenSubKey(Handler);

            if (regExt != null && regHandler != null)
            {
                RegistryKey rk = Registry.ClassesRoot.OpenSubKey(Ext + "\\OpenWithProgids");

                string[] names = null;
                if (rk != null)
                {
                    names = rk.GetValueNames();

                    for (int i = 0; i < names.Length; i++)
                    {
                        Registry.ClassesRoot.DeleteSubKeyTree(names[i]);
                    }
                }

                Registry.ClassesRoot.DeleteSubKeyTree(Ext);
                Registry.ClassesRoot.DeleteSubKeyTree(Handler);

                SHChangeNotify(HChangeNotifyEventID.SHCNE_ASSOCCHANGED,
                               HChangeNotifyFlags.SHCNF_IDLIST,
                               IntPtr.Zero,
                               IntPtr.Zero);
            }
        }

        /// <summary>
        /// Notify the shell to update registry changes
        /// </summary>
        /// <param name="wEventId">event id</param>
        /// <param name="uFlags">notify flag</param>
        /// <param name="dwItem1">item pointer</param>
        /// <param name="dwItem2">item pointer</param>
        [DllImport("shell32.dll")]
        static extern void SHChangeNotify(HChangeNotifyEventID wEventId,
                                          HChangeNotifyFlags uFlags,
                                          IntPtr dwItem1,
                                          IntPtr dwItem2);

        /// <summary>
        /// Sends the file path to the application instance
        /// </summary>
        /// <param name="args">arguments passed from main method</param>
        /// <returns>return true if there's more then one instance active</returns>
        public bool SendFilePath(string[] args)
        {
            Process proc = Process.GetCurrentProcess();
            Process[] processes = Process.GetProcessesByName(proc.ProcessName);

            if (processes.Length == 0)
                return false;

            string message = "";

            for (int i = 0; i < args.Length; i++)
                message += args[i] + ";";

            byte[] data = Encoding.Default.GetBytes(message);
            int dataLen = data.Length;

            COPYDATASTRUCT cds;
            cds.dwData = (IntPtr)100;
            cds.lpData = message;
            cds.cbData = dataLen + 1;

            // iterate through all the application processes
            if (processes.Length > 1)
            {
                foreach (Process p in processes)
                {
                    if (p.Id != proc.Id)
                    {
                        SendMessage(p.MainWindowHandle,
                                    WM_COPYDATA,
                                    0,
                                    ref cds);
                        return true;
                    }
                }
            }

            return false;
        }

        /// <summary>
        /// Sends the file path to the application instance
        /// </summary>
        /// <param name="processName">送信先のプロセス名</param>
        /// <param name="messageNum">メッセージ番号</param>
        /// <param name="messageData">メッセージデータ</param>
        /// <returns>return true if there's more then one instance active</returns>
        public bool SendFilePath(string processName, uint messageNum, string messageData)
        {
            Process[] processes = Process.GetProcessesByName(processName);
            if (processes.Length == 0)
            {
                return false;
            }

            byte[] data = Encoding.Default.GetBytes(messageData);
            int dataLen = data.Length;

            COPYDATASTRUCT cds = new COPYDATASTRUCT();
            cds.dwData = (IntPtr)messageNum;
            cds.lpData = messageData;
            cds.cbData = dataLen + 1;

            // iterate through all the application processes
            if (processes.Length > 0)
            {
                foreach (Process process in processes)
                {
                    SendMessage(process.MainWindowHandle, WM_COPYDATA, 0, ref cds);
                }
                return true;
            }
            return false;
        }

        /// <summary>
        /// Send message to the app
        /// </summary>
        /// <param name="hwnd">pointer to the handler</param>
        /// <param name="Msg">message to be sent</param>
        /// <param name="wParam">parameters to be sent along with</param>
        /// <param name="lParam">copy of data structure</param>
        /// <returns></returns>
        [DllImport("user32.dll", CharSet = CharSet.Auto, SetLastError = true)]
        private static extern IntPtr SendMessage(IntPtr hwnd,
                                                 uint Msg,
                                                 int wParam,
                                                 ref COPYDATASTRUCT lParam);

        #endregion

        #region enum HChangeNotifyEventID
        /// <summary>
        /// Describes the event that has occurred.
        /// Typically, only one event is specified at a time.
        /// If more than one event is specified, the values contained
        /// in the <i>dwItem1</i> and <i>dwItem2</i>
        /// parameters must be the same, respectively, for all specified events.
        /// This parameter can be one or more of the following values.
        /// </summary>
        /// <remarks>
        /// <para><b>Windows NT/2000/XP:</b> <i>dwItem2</i> contains the index
        /// in the system image list that has changed.
        /// <i>dwItem1</i> is not used and should be <see langword="null"/>.</para>
        /// <para><b>Windows 95/98:</b> <i>dwItem1</i> contains the index
        /// in the system image list that has changed.
        /// <i>dwItem2</i> is not used and should be <see langword="null"/>.</para>
        /// </remarks>
        [Flags]
        enum HChangeNotifyEventID
        {
            /// <summary>
            /// All events have occurred.
            /// </summary>
            SHCNE_ALLEVENTS = 0x7FFFFFFF,

            /// <summary>
            /// A file type association has changed. <see cref="HChangeNotifyFlags.SHCNF_IDLIST"/>
            /// must be specified in the <i>uFlags</i> parameter.
            /// <i>dwItem1</i> and <i>dwItem2</i> are not used and must be <see langword="null"/>.
            /// </summary>
            SHCNE_ASSOCCHANGED = 0x08000000,

            /// <summary>
            /// The attributes of an item or folder have changed.
            /// <see cref="HChangeNotifyFlags.SHCNF_IDLIST"/> or
            /// <i>dwItem1</i> contains the item or folder that has changed.
            /// <i>dwItem2</i> is not used and should be <see langword="null"/>.
            /// </summary>
            SHCNE_ATTRIBUTES = 0x00000800,

            /// <summary>
            /// A nonfolder item has been created.
            /// <see cref="HChangeNotifyFlags.SHCNF_IDLIST"/> or
            /// <i>dwItem1</i> contains the item that was created.
            /// <i>dwItem2</i> is not used and should be <see langword="null"/>.
            /// </summary>
            SHCNE_CREATE = 0x00000002,

            /// <summary>
            /// A nonfolder item has been deleted.
            /// <see cref="HChangeNotifyFlags.SHCNF_IDLIST"/> or
            /// <i>dwItem1</i> contains the item that was deleted.
            /// <i>dwItem2</i> is not used and should be <see langword="null"/>.
            /// </summary>
            SHCNE_DELETE = 0x00000004,

            /// <summary>
            /// A drive has been added.
            /// <see cref="HChangeNotifyFlags.SHCNF_IDLIST"/> or
            /// <i>dwItem1</i> contains the root of the drive that was added.
            /// <i>dwItem2</i> is not used and should be <see langword="null"/>.
            /// </summary>
            SHCNE_DRIVEADD = 0x00000100,

            /// <summary>
            /// A drive has been added and the Shell should create a new window for the drive.
            /// <see cref="HChangeNotifyFlags.SHCNF_IDLIST"/> or
            /// <i>dwItem1</i> contains the root of the drive that was added.
            /// <i>dwItem2</i> is not used and should be <see langword="null"/>.
            /// </summary>
            SHCNE_DRIVEADDGUI = 0x00010000,

            /// <summary>
            /// A drive has been removed. <see cref="HChangeNotifyFlags.SHCNF_IDLIST"/> or
            /// <i>dwItem1</i> contains the root of the drive that was removed.
            /// <i>dwItem2</i> is not used and should be <see langword="null"/>.
            /// </summary>
            SHCNE_DRIVEREMOVED = 0x00000080,

            /// <summary>
            /// Not currently used.
            /// </summary>
            SHCNE_EXTENDED_EVENT = 0x04000000,

            /// <summary>
            /// The amount of free space on a drive has changed.
            /// <see cref="HChangeNotifyFlags.SHCNF_IDLIST"/> or
            /// <i>dwItem1</i> contains the root of the drive on which the free space changed.
            /// <i>dwItem2</i> is not used and should be <see langword="null"/>.
            /// </summary>
            SHCNE_FREESPACE = 0x00040000,

            /// <summary>
            /// Storage media has been inserted into a drive.
            /// <see cref="HChangeNotifyFlags.SHCNF_IDLIST"/> or
            /// <i>dwItem1</i> contains the root of the drive that contains the new media.
            /// <i>dwItem2</i> is not used and should be <see langword="null"/>.
            /// </summary>
            SHCNE_MEDIAINSERTED = 0x00000020,

            /// <summary>
            /// Storage media has been removed from a drive.
            /// <see cref="HChangeNotifyFlags.SHCNF_IDLIST"/> or
            /// <i>dwItem1</i> contains the root of the drive from which the media was removed.
            /// <i>dwItem2</i> is not used and should be <see langword="null"/>.
            /// </summary>
            SHCNE_MEDIAREMOVED = 0x00000040,

            /// <summary>
            /// A folder has been created. <see cref="HChangeNotifyFlags.SHCNF_IDLIST"/>
            /// <i>dwItem1</i> contains the folder that was created.
            /// <i>dwItem2</i> is not used and should be <see langword="null"/>.
            /// </summary>
            SHCNE_MKDIR = 0x00000008,

            /// <summary>
            /// A folder on the local computer is being shared via the network.
            /// <see cref="HChangeNotifyFlags.SHCNF_IDLIST"/> or
            /// <i>dwItem1</i> contains the folder that is being shared.
            /// <i>dwItem2</i> is not used and should be <see langword="null"/>.
            /// </summary>
            SHCNE_NETSHARE = 0x00000200,

            /// <summary>
            /// A folder on the local computer is no longer being shared via the network.
            /// <see cref="HChangeNotifyFlags.SHCNF_IDLIST"/> or
            /// <i>dwItem1</i> contains the folder that is no longer being shared.
            /// <i>dwItem2</i> is not used and should be <see langword="null"/>.
            /// </summary>
            SHCNE_NETUNSHARE = 0x00000400,

            /// <summary>
            /// The name of a folder has changed.
            /// <see cref="HChangeNotifyFlags.SHCNF_IDLIST"/> or
            /// <i>dwItem1</i> contains the previous pointer to an item identifier list (PIDL) or name of the folder.
            /// <i>dwItem2</i> contains the new PIDL or name of the folder.
            /// </summary>
            SHCNE_RENAMEFOLDER = 0x00020000,

            /// <summary>
            /// The name of a nonfolder item has changed.
            /// <see cref="HChangeNotifyFlags.SHCNF_IDLIST"/> or
            /// <i>dwItem1</i> contains the previous PIDL or name of the item.
            /// <i>dwItem2</i> contains the new PIDL or name of the item.
            /// </summary>
            SHCNE_RENAMEITEM = 0x00000001,

            /// <summary>
            /// A folder has been removed.
            /// <see cref="HChangeNotifyFlags.SHCNF_IDLIST"/> or
            /// <i>dwItem1</i> contains the folder that was removed.
            /// <i>dwItem2</i> is not used and should be <see langword="null"/>.
            /// </summary>
            SHCNE_RMDIR = 0x00000010,

            /// <summary>
            /// The computer has disconnected from a server.
            /// <see cref="HChangeNotifyFlags.SHCNF_IDLIST"/> or
            /// <i>dwItem1</i> contains the server from which the computer was disconnected.
            /// <i>dwItem2</i> is not used and should be <see langword="null"/>.
            /// </summary>
            SHCNE_SERVERDISCONNECT = 0x00004000,

            /// <summary>
            /// The contents of an existing folder have changed,
            /// but the folder still exists and has not been renamed.
            /// <see cref="HChangeNotifyFlags.SHCNF_IDLIST"/> or
            /// <i>dwItem1</i> contains the folder that has changed.
            /// <i>dwItem2</i> is not used and should be <see langword="null"/>.
            /// If a folder has been created, deleted, or renamed, use SHCNE_MKDIR, SHCNE_RMDIR, or
            /// SHCNE_RENAMEFOLDER, respectively, instead.
            /// </summary>
            SHCNE_UPDATEDIR = 0x00001000,

            /// <summary>
            /// An image in the system image list has changed.
            /// <see cref="HChangeNotifyFlags.SHCNF_DWORD"/> must be specified in <i>uFlags</i>.
            /// </summary>
            SHCNE_UPDATEIMAGE = 0x00008000,
        }

        #endregion

        #region enum HChangeNotifyFlags

        /// <summary>
        /// Flags that indicate the meaning of the <i>dwItem1</i> and <i>dwItem2</i> parameters.
        /// The uFlags parameter must be one of the following values.
        /// </summary>
        [Flags]
        public enum HChangeNotifyFlags
        {
            /// <summary>
            /// The <i>dwItem1</i> and <i>dwItem2</i> parameters are DWORD values.
            /// </summary>
            SHCNF_DWORD = 0x0003,
            /// <summary>
            /// <i>dwItem1</i> and <i>dwItem2</i> are the addresses of ITEMIDLIST structures that
            /// represent the item(s) affected by the change.
            /// Each ITEMIDLIST must be relative to the desktop folder.
            /// </summary>
            SHCNF_IDLIST = 0x0000,
            /// <summary>
            /// <i>dwItem1</i> and <i>dwItem2</i> are the addresses of null-terminated strings of
            /// maximum length MAX_PATH that contain the full path names
            /// of the items affected by the change.
            /// </summary>
            SHCNF_PATHA = 0x0001,
            /// <summary>
            /// <i>dwItem1</i> and <i>dwItem2</i> are the addresses of null-terminated strings of
            /// maximum length MAX_PATH that contain the full path names
            /// of the items affected by the change.
            /// </summary>
            SHCNF_PATHW = 0x0005,
            /// <summary>
            /// <i>dwItem1</i> and <i>dwItem2</i> are the addresses of null-terminated strings that
            /// represent the friendly names of the printer(s) affected by the change.
            /// </summary>
            SHCNF_PRINTERA = 0x0002,
            /// <summary>
            /// <i>dwItem1</i> and <i>dwItem2</i> are the addresses of null-terminated strings that
            /// represent the friendly names of the printer(s) affected by the change.
            /// </summary>
            SHCNF_PRINTERW = 0x0006,
            /// <summary>
            /// The function should not return until the notification
            /// has been delivered to all affected components.
            /// As this flag modifies other data-type flags, it cannot by used by itself.
            /// </summary>
            SHCNF_FLUSH = 0x1000,
            /// <summary>
            /// The function should begin delivering notifications to all affected components
            /// but should return as soon as the notification process has begun.
            /// As this flag modifies other data-type flags, it cannot by used by itself.
            /// </summary>
            SHCNF_FLUSHNOWAIT = 0x2000
        }

        #endregion

        #region Member Variables

        private bool m_bEmbeddedIcon = false;
        private string m_strExt;
        private string m_strHandler;
        private string m_strDesc;
        private string m_strIconName;
        private string m_strOpenText;
        private string m_strAppName;
        private string m_strHKCR;
        private int m_iIconPos = 0;

        private const int WM_COPYDATA = 0x4A;
        AppCollection openWith = new AppCollection();

        #endregion

        #region Properties

        /// <summary>
        /// Whether to embed an icon or not
        /// </summary>
        public bool EmbbedIcon
        {
            get { return m_bEmbeddedIcon; }
            set {m_bEmbeddedIcon = value; }
        }

        /// <summary>
        /// Extension of file to be associated with
        /// </summary>
        public string Ext
        {
            get { return m_strExt; }
            set { m_strExt = value; }
        }

        /// <summary>
        /// Handler of the files
        /// </summary>
        public string Handler
        {
            get { return m_strHandler; }
            set { m_strHandler = value; }
        }

        /// <summary>
        /// Description to be set in the registry
        /// </summary>
        public string Description
        {
            get { return m_strDesc; }
            set { m_strDesc = value; }
        }

        /// <summary>
        /// Position of the icon in the resource
        /// </summary>
        public int IconPos
        {
            get { return m_iIconPos; }
            set { m_iIconPos = value; }
        }

        /// <summary>
        /// Name of the icon in the resource
        /// </summary>
        public string IconName
        {
            get { return m_strIconName; }
            set { m_strIconName = value; }
        }

        /// <summary>
        /// Open text of the file
        /// </summary>
        public string OpenText
        {
            get { return m_strOpenText; }
            set { m_strOpenText = value; }
        }

        /// <summary>
        /// Name of the application to open the file with
        /// </summary>
        public string AppName
        {
            get { return m_strAppName; }
            set { m_strAppName = value; }
        }

        /// <summary>
        /// Registry path; Don't usually need to change this.
        /// </summary>
        public string HKCR
        {
            get { return m_strHKCR; }
            set { m_strHKCR = value; }
        }

        /// <summary>
        /// Get data as string array separated by semi column
        /// </summary>
        /// <param name="data">string input</param>
        /// <returns>array of strings; token parsed from input string</returns>
        public string[] GetData(string data)
        {
            data = data.TrimEnd(';');
            string[] result = data.Split(';');

            return result;
        }

        #endregion
    }
}


