﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using System.Threading.Tasks;
using System.Xml.Linq;
using EffectMaker.DataModel.DataModels;
using EffectMaker.Foundation.Command;
using EffectMaker.Foundation.Input;
using EffectMaker.Foundation.Interfaces;
using EffectMaker.UILogic.Commands;

namespace EffectMaker.UILogic.ViewModels
{
    /// <summary>
    /// ユーザーページのビューモデルです.
    /// </summary>
    public class UserPageViewModel : PropertyPageViewModel<UserPageData>, IModificationFlagOwner
    {
        /// <summary>
        /// The constructor.
        /// </summary>
        /// <param name="parent">The parent view model.</param>
        /// <param name="dataModel">The data model to encapsulate.</param>
        /// <param name="addViewModelForContents">True to automatically add an user group view model for the contents data.</param>
        public UserPageViewModel(
            HierarchyViewModel parent,
            UserPageData dataModel,
            bool addViewModelForContents = true)
            : base(parent, dataModel)
        {
            if (addViewModelForContents == true)
            {
                this.Contents = new UserGroupViewModel(this, dataModel.ContentsData);
                this.AddChildViewModel(this.Contents);
            }

            // Always create the modification flag view model IN THE END of the constructor
            // to prevent any initialization triggers the modification events.
            this.ModificationFlagViewModel = new ModificationFlagViewModel(this);
        }

        /// <summary>
        /// ページ内コンテンツ.
        /// </summary>
        public UserGroupViewModel Contents { get; set; }

        /// <summary>
        /// Get the view model that holds the modification flags of
        /// this view model's properties.
        /// </summary>
        public ModificationFlagViewModel ModificationFlagViewModel { get; private set; }

        /// <summary>
        /// 型名をユーザーデータ固有の型名に置き換えてエクスポートします。
        /// </summary>
        /// <returns>XML化したデータ</returns>
        public override string ExportValuesAsXml()
        {
            Type type = this.GetUserDataType();
            if (type == null)
            {
                return null;
            }

            var root = new XElement(
                "root",
                new XAttribute("type", type.FullName));

            try
            {
                this.ExportValues(new XmlElementContainer(root));
            }
            catch
            {
                // TODO: report the error
                return null;
            }

            return root.ToString();
        }

        /// <summary>
        /// Set values based on a XML content.
        /// </summary>
        /// <param name="xmlContent">The XML content to set values from.</param>
        /// <returns>Returns true if import succeeded, false otherwise.</returns>
        public override bool ImportValuesFromXml(string xmlContent)
        {
            XElement root = null;

            try
            {
                root = XElement.Parse(xmlContent);
            }
            catch
            {
                // TODO: report the error
                return false;
            }

            Type type = this.GetUserDataType();
            if (type == null)
            {
                return false;
            }

            if ((string)root.Attribute("type") != type.FullName)
            {
                // TODO: report the error ?
                return false;
            }

            try
            {
                this.ImportValues(root.Elements());
            }
            catch
            {
                // TODO: report the error
                return false;
            }

            return true;
        }

        /// <summary>
        /// 同じユーザーページかどうかまでチェックした上で、ペーストの可否を判定します。
        /// </summary>
        /// <param name="xmlContent">The XML content to set values from.</param>
        /// <returns>Returns true if can import, false otherwise.</returns>
        public override bool CanImportValuesFromXml(string xmlContent)
        {
            XElement root = null;

            try
            {
                root = XElement.Parse(xmlContent);
            }
            catch
            {
                return false;
            }

            Type type = this.GetUserDataType();
            if (type == null)
            {
                return false;
            }

            if ((string)root.Attribute("type") != type.FullName)
            {
                return false;
            }

            return true;
        }

        /// <summary>
        /// ユーザーデータの型を取得します。
        /// </summary>
        /// <returns>ユーザーデータの型情報。</returns>
        public Type GetUserDataType()
        {
            return this.DataModel.ContentsData.GetType();
        }

        /// <summary>
        /// Called to reset the page data to its default values.
        /// </summary>
        /// <param name="parameter">The executable parameters.</param>
        protected override void OnResetDefaultValues(object parameter)
        {
            CommandManager.Execute(new ResetDefaultValueCommand(this.Contents));
        }
    }
}
