﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using System;
using System.Collections.Generic;
using System.ComponentModel;
using System.Dynamic;
using System.Linq;

using EffectMaker.Foundation.Extensions;
using EffectMaker.Foundation.Log;

namespace EffectMaker.UILogic.ViewModels
{
    /// <summary>
    /// 初期値と変更フラグを取得する関数のペア
    /// </summary>
    public class ModificationFuncs
    {
        /// <summary>
        /// 初期値取得関数
        /// </summary>
        public Func<bool> IsDefault { get; set; }

        /// <summary>
        /// 変更取得関数
        /// </summary>
        public Func<bool> IsModified { get; set; }
    }

    /// <summary>
    /// 変更フラグのビューモデル.
    /// 複数のプロパティの変更をまとめて管理するクラス.
    /// </summary>
    public class MultipleModificationFlagsViewModel : ModificationFlagViewModel
    {
        /// <summary>
        /// プロパティの親子関係を管理するコンテナ
        /// </summary>
        private readonly Dictionary<string, string[]> propertyDictionary =
            new Dictionary<string, string[]>();

        /// <summary>
        /// 変更、デフォル状態についてフラグを返すViewModelのコンテナ
        /// </summary>
        private readonly Dictionary<string, IModificationPropertyOwner[]> modificationDictionary =
            new Dictionary<string, IModificationPropertyOwner[]>();

        /// <summary>
        /// Constructor.
        /// </summary>
        /// <param name="parent">親のビューモデル.</param>
        public MultipleModificationFlagsViewModel(ViewModelBase parent) :
            base(parent)
        {
        }

        /// <summary>
        /// 親子プロパティを設定する
        /// </summary>
        /// <param name="parentProperty">親プロパティ名</param>
        /// <param name="childProperties">子プロパティ名のリスト</param>
        public void SetPropertyDictionary(
            string parentProperty,
            IEnumerable<string> childProperties)
        {
            this.propertyDictionary.Add(parentProperty, childProperties.ToArray());
        }

        /// <summary>
        /// プロパティに、変更とデフォル状態についてフラグを返すViewModelの配列を割り当てる。
        /// </summary>
        /// <param name="property">プロパティ名</param>
        /// <param name="viewModels">プロパティに関連づけるViewModelの配列</param>
        public void SetModificationDictionary(string property, IEnumerable<IModificationPropertyOwner> viewModels)
        {
            this.modificationDictionary.Add(property, viewModels.ToArray());
        }

        /// <summary>
        /// 変更フラグをクリアします。
        /// </summary>
        public override void ClearModificationFlags()
        {
            base.ClearModificationFlags();

            // Issue property change event for all the compound properties
            // so the UI can update the modification flags accordingly.
            this.propertyDictionary.ForEach(p => this.OnPropertyChanged(p.Key));
        }

        /// <summary>
        /// Try to get the requested property value from the data model.
        /// </summary>
        /// <param name="binder">The get member binder.</param>
        /// <param name="result">The property value.</param>
        /// <returns>True on success.</returns>
        public override bool TryGetMember(
            GetMemberBinder binder,
            out object result)
        {
            if (this.Parent == null)
            {
                result = ModificationData.Empty;
                return false;
            }

            // 親プロパティに登録されている場合は、子プロパティのModificationDataからresultを合成する.
            string[] properties = null;
            if (this.propertyDictionary.TryGetValue(binder.Name, out properties))
            {
                bool isDefaultValue = true;
                bool isModified = false;

                ModificationData data = null;
                foreach (string property in properties)
                {
                    if (this.ModifyFlags.TryGetValue(property, out data) == false)
                    {
                        // If the modification data is not found, means the property has
                        // not been modified, so we only need to check if the property is
                        // default value.
                        isDefaultValue &= this.Parent.IsPropertyDefaultValue(property);
                    }
                    else
                    {
                        isDefaultValue &= data.IsDefaultValue;
                        isModified |= data.IsModified;
                    }
                }

                IModificationPropertyOwner[] viewModels = null;
                if (this.modificationDictionary.TryGetValue(binder.Name, out viewModels))
                {
                    foreach (var viewModel in viewModels)
                    {
                        isDefaultValue &= viewModel.IsDefault;
                        isModified |= viewModel.IsModified;
                    }
                }

                result = new ModificationData()
                {
                    IsDefaultValue = isDefaultValue,
                    IsModified = isModified,
                    OriginalValue = null,
                    DefaultValue = null,
                    CurrentValue = null,
                };
            }
            else
            {
                // This is not a compound property, just get from the base class.
                return base.TryGetMember(binder, out result);
            }

            return true;
        }

        /// <summary>
        /// Handle PropertyChanged event issues when the properties of the parent is changed.
        /// </summary>
        /// <param name="sender">The sender of the event.</param>
        /// <param name="e">The event arguments.</param>
        protected override void OnParentPropertyChanged(
            object sender,
            PropertyChangedEventArgs e)
        {
            if (ModificationFlagViewModel.IgnoreParentPropertyChangedEvents == true)
            {
                return;
            }

            if (IgnorePropertyNamesStatic.Contains(e.PropertyName))
            {
                return;
            }

            if (this.IgnorePropertyNames.Contains(e.PropertyName))
            {
                return;
            }

            base.OnParentPropertyChanged(sender, e);

            // Find the compound property names those contain the modified property.
            IEnumerable<string> compoundPropertyNames =
                from pair in this.propertyDictionary
                where pair.Value.Any(p => p == e.PropertyName)
                select pair.Key;

            // Issue property change event for the compound properties who contains
            // the modified property.
            compoundPropertyNames.ForEach(name => this.OnPropertyChanged(name));
        }
    }
}
