﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using System;
using System.Collections.Generic;
using System.Diagnostics;
using System.IO;
using System.Linq;
using System.Xml;
using EffectMaker.BusinessLogic.IO;
using EffectMaker.BusinessLogic.Options;
using EffectMaker.BusinessLogic.UserData;
using EffectMaker.Foundation.Log;

namespace EffectMaker.UILogic.ViewModels.IO
{
    /// <summary>
    /// FE1世代のesetをFE2で読めるようにするためのアップデータです。
    /// コードが汚れるのを防ぐため、FE1アップデートに関する処理はできる限りこのクラスに閉じ込めます。
    /// </summary>
    public class AncientEsetUpdater
    {
        /// <summary>
        /// 唯一のインスタンスです。
        /// </summary>
        public static readonly AncientEsetUpdater Instance = new AncientEsetUpdater();

        /// <summary>
        /// esetの世代を表す列挙体です。
        /// </summary>
        public enum EsetType
        {
            /// <summary>
            /// FE1
            /// </summary>
            FE1,

            /// <summary>
            /// CE2
            /// </summary>
            CE2,
        }

        /// <summary>
        /// FE1Updater.exeのファイルパスです。
        /// </summary>
        private readonly string fe1UpdaterPath;

        /// <summary>
        /// CE2Updater.exeのファイルパスです。
        /// </summary>
        private readonly string ce2UpdaterPath;

        /// <summary>
        /// コンバートデータの出力フォルダパスです。
        /// コンバートリストファイル、コンバートデータの保存に使用します。
        /// </summary>
        private readonly string workingFolderPath;

        /// <summary>
        /// アップデートファイルリストのxmlファイルパスです。
        /// </summary>
        private readonly string updateListPath;

        /// <summary>
        /// アップデート対象のesetファイルリストです。
        /// </summary>
        private readonly Dictionary<string, bool> esetList;

        /// <summary>
        /// コンストラクタです。
        /// </summary>
        public AncientEsetUpdater()
        {
            // esetファイルリストを初期化
            this.esetList = new Dictionary<string, bool>();

            // アップデータのexeファイルパスを取得
            this.fe1UpdaterPath = Path.Combine(IOConstants.ExecutableFolderPath, @"FE1Updater\FE1Updater.exe");
            this.ce2UpdaterPath = Path.Combine(IOConstants.ExecutableFolderPath, @"CE2Updater\CE2Updater.exe");

            // コンバートデータの出力フォルダパスを取得
            this.workingFolderPath = Path.Combine(IOConstants.TemporaryFolderPath, "EsetUpdater");

            // コンバートリストファイルのパスを取得
            this.updateListPath = Path.Combine(this.workingFolderPath, "UpdateList.xml");
        }

        /// <summary>
        /// ファイルオープン前にアップデート処理を行います。
        /// </summary>
        /// <param name="targetList">ファイルリスト</param>
        /// <param name="type">どの世代のesetをアップデートするか</param>
        public void PreOpenUpdate(List<string> targetList, EsetType type)
        {
            // リセット処理
            {
                // ファイルリストをクリア
                this.esetList.Clear();

                // 作業フォルダを、内容を含め削除
                if (Directory.Exists(this.workingFolderPath))
                {
                    // 作業フォルダ内のサブディレクトリを削除
                    foreach (string directory in Directory.GetDirectories(this.workingFolderPath))
                    {
                        Directory.Delete(directory, true);
                    }

                    // 作業フォルダ内の全ファイルを削除
                    foreach (string file in Directory.GetFiles(this.workingFolderPath))
                    {
                        File.Delete(file);
                    }
                }
                else
                {
                    // 作業フォルダを作成
                    Directory.CreateDirectory(this.workingFolderPath);
                }
            }

            Func<string, bool> checkFunc = null;
            Action updateFunc = null;

            // アップデート処理を設定
            switch (type)
            {
                case EsetType.FE1:
                    checkFunc = Instance.IsFe1Eset;
                    updateFunc = Instance.RunFe1Update;
                    break;

                case EsetType.CE2:
                    checkFunc = Instance.IsCe2Eset;
                    updateFunc = Instance.RunCe2Update;
                    break;
            }

            // esetファイルを列挙
            foreach (string targetPath in targetList)
            {
                if (checkFunc(targetPath))
                {
                    this.esetList.Add(targetPath, true);
                }
            }

            // 該当するファイルがないとき処理を終了する
            if (this.esetList.Count == 0)
            {
                return;
            }

            if (OptionStore.RuntimeOptions.IsCommandLineMode)
            {
                // コマンドライン実行時は同期で実行
                updateFunc();
            }
            else
            {
                // プログレスダイアログを表示してアップデートを実行
                WorkspaceRootViewModel.Instance.Dialogs.OnShowAncientEsetUpdaterDialogExecutable.Execute(
                    new object[]
                {
                    Properties.Resources.AncientEsetUpdaterDialogMessage,
                    Properties.Resources.AncientEsetUpdaterDialogCaption,
                    updateFunc
                });
            }
        }

        /// <summary>
        /// FE2世代のesetファイルパスを取得します。
        /// </summary>
        /// <param name="filePath">FE1またはFE2世代のesetファイルパス</param>
        /// <returns>FE2世代のesetファイルパスを返します。</returns>
        public string GetFe2FilePath(string filePath)
        {
            // リストからパスを検索
            bool found;
            this.esetList.TryGetValue(filePath, out found);

            if (found)
            {
                // リストにパスがあれば、アップデートしてできたファイルのパスを返す
                string fileName = Path.GetFileName(filePath);
                string fe2Path = Path.Combine(this.workingFolderPath, fileName);

                // アップデートに失敗していたときはnullを返す
                if (File.Exists(fe2Path) == false)
                {
                    return null;
                }

                return fe2Path;
            }
            else
            {
                // リストにパスがないとき、そのままfilePathを返す
                return filePath;
            }
        }

        /// <summary>
        /// FE1世代のesetファイルかどうか取得します。
        /// </summary>
        /// <param name="filePath">ファイルパス</param>
        /// <returns>FE1世代のesetファイルのときtrueを返します。</returns>
        public bool IsFe1Eset(string filePath)
        {
            bool isFe1Eset = false;
            bool foundElement = false;

            // esetファイルを開く
            using (var fs = new FileStream(filePath, FileMode.Open, FileAccess.Read))
            {
                // XMLパーサーを作成
                var xmlReaderSettings = new XmlReaderSettings { IgnoreComments = true, IgnoreWhitespace = true };
                var xmlReader = XmlReader.Create(fs, xmlReaderSettings);

                while (xmlReader.Read())
                {
                    if (xmlReader.IsEmptyElement)
                    {
                        continue;
                    }

                    // 要素名を取得
                    string elementName = xmlReader.Name;

                    if (elementName == "EmitterSet")
                    {
                        // "EmitterSet"要素があるesetはFE1世代のデータ
                        isFe1Eset = true;
                        foundElement = true;
                        break;
                    }
                    else if (elementName == "EmitterSetData")
                    {
                        // "EmitterSetData"要素があるesetはFE2世代のデータ
                        isFe1Eset = false;
                        foundElement = true;
                        break;
                    }
                }

                xmlReader.Close();
            }

            // 判別失敗
            if (foundElement == false)
            {
                Logger.Log(LogLevels.Warning, "AncientEsetLoader.IsFe1Eset : Feature element not found.");
            }

            return isFe1Eset;
        }

        /// <summary>
        /// CE2世代のesetファイルかどうか取得します。
        /// </summary>
        /// <param name="filePath">ファイルパス</param>
        /// <returns>CE2世代のesetファイルのときtrueを返します。</returns>
        public bool IsCe2Eset(string filePath)
        {
            bool isCe2Eset = false;
            bool foundElement = false;

            // esetファイルを読み込み
            XmlDocument doc = new XmlDocument();
            doc.Load(filePath);

            XmlElement root = doc.DocumentElement;
            if (root.Name == "EmitterSet")
            {
                foreach (XmlElement elem in root)
                {
                    if (elem.Name == "Version")
                    {
                        int? resCompare = CompareVersion(elem.InnerText, "1.7.0.0");

                        if (resCompare != null && resCompare < 0)
                        {
                            // "1.7.0.0"より古いesetはCE2世代のデータとみなす
                            isCe2Eset = true;
                        }

                        foundElement = true;
                        break;
                    }
                }
            }
            else if (root.Name == "EmitterSetData")
            {
                // "EmitterSetData"要素があるesetはFE2世代のデータ
                foundElement = true;
            }

            // 判別失敗
            if (foundElement == false)
            {
                Logger.Log(LogLevels.Warning, "AncientEsetLoader.IsCe2Eset : Feature element not found.");
            }

            return isCe2Eset;
        }

        /// <summary>
        /// バージョン文字列を比較します。
        /// </summary>
        /// <returns>
        /// A > B のとき 0 より大きい数、A == B のとき 0、A < B のとき 0 より小さい数を返します。
        /// 文字列が不正のとき null を返します。
        /// </returns>
        private int? CompareVersion(string strA, string strB)
        {
            Version versionA, versionB;
            bool resParse;

            resParse = Version.TryParse(strA, out versionA);
            if (resParse == false) return null;

            resParse = Version.TryParse(strB, out versionB);
            if (resParse == false) return null;

            return versionA.CompareTo(versionB);
        }

        /// <summary>
        /// アップデータに渡すコンバートリストを作成します。
        /// </summary>
        /// <returns>処理が正常に完了したときtrueを返します。</returns>
        private bool CreateConvertList()
        {
            // コンバートリストファイルを開く
            using (var fs = new FileStream(this.updateListPath, FileMode.Create))
            {
                // XMLライターを作成
                var xmlWriterSettings = new XmlWriterSettings { Indent = true };
                var xmlWriter = XmlWriter.Create(fs, xmlWriterSettings);

                // XMLを書き出す
                // <SEADPtclConverterArg>
                //   <name>FE1Update</name>
                //   <EmitterSetList>
                //     <EmitterSet>
                //       <file>"FilePath1"</file>
                //       <file>"FilePath2"</file>
                //     </EmitterSet>
                //   </EmitterSetList>
                // </SEADPtclConverterArg>
                xmlWriter.WriteStartElement("SEADPtclConverterArg");
                    xmlWriter.WriteStartElement("name");
                        xmlWriter.WriteString("FE1Update");
                    xmlWriter.WriteEndElement();
                    xmlWriter.WriteStartElement("EmitterSetList");
                        xmlWriter.WriteStartElement("EmitterSet");

                            foreach (string filePath in this.esetList.Keys)
                            {
                                xmlWriter.WriteStartElement("file");
                                    xmlWriter.WriteString(filePath);
                                xmlWriter.WriteEndElement();
                            }

                        xmlWriter.WriteEndElement();
                    xmlWriter.WriteEndElement();
                xmlWriter.WriteEndElement();

                xmlWriter.Close();
            }

            return true;
        }

        /// <summary>
        /// プログレスダイアログ表示中に行うアップデート処理です。
        /// </summary>
        private void RunFe1Update()
        {
            // アップデータに渡すコンバートリストを作成
            bool created = this.CreateConvertList();
            if (created == false)
            {
                return;
            }

            // アップデータを実行
            {
                // 標準の引数設定
                string args = string.Format(
                    "-ps -u2 \"{0}\" \"{1}\"",
                    this.updateListPath,
                    this.workingFolderPath);

                // ユーザデータ指定引数
                string userDataArgs = String.Empty;

                var csDefFilePath = OptionStore.ProjectConfig.CustomShaderPath;
                if (!string.IsNullOrEmpty(csDefFilePath))
                {
                    var def = UserDataHelper.Instance.LoadCustomShaderUserData(csDefFilePath, false);
                    if (def.UsdDefinitions.Any())
                    {
                        userDataArgs = string.Format("-us \"{0}\"", csDefFilePath);
                    }
                }

                var caDefFilePath = OptionStore.ProjectConfig.CustomActionPath;
                if (!string.IsNullOrEmpty(caDefFilePath))
                {
                    var def = UserDataHelper.Instance.LoadCustomActionUserData(caDefFilePath, false);
                    if (def.UddDefinitions.Any())
                    {
                        if (!string.IsNullOrEmpty(userDataArgs))
                        {
                            userDataArgs += " ";
                        }

                        userDataArgs += string.Format("-ua \"{0}\"", Path.GetDirectoryName(caDefFilePath));
                    }
                }

                if (!string.IsNullOrEmpty(userDataArgs))
                {
                    args = userDataArgs + " " + args;
                }

                // アップデータの起動設定
                var startInfo = new ProcessStartInfo
                {
                    FileName = this.fe1UpdaterPath,
                    Arguments = args,
                    CreateNoWindow = true,
                    UseShellExecute = false
                };

                // アップデータを実行
                Process process = Process.Start(startInfo);
                process.WaitForExit();
            }
        }

        /// <summary>
        /// プログレスダイアログ表示中に行うアップデート処理です。
        /// </summary>
        private void RunCe2Update()
        {
            // アップデータに渡すコンバートリストを作成
            bool created = this.CreateConvertList();
            if (created == false)
            {
                return;
            }

            // アップデータを実行
            {
                // CE2→FE1
                {
                    // 標準の引数設定
                    string args = string.Format(
                        "-u \"{0}\"",
                        this.updateListPath);

                    // アップデータの起動設定
                    var startInfo = new ProcessStartInfo
                    {
                        FileName = this.ce2UpdaterPath,
                        Arguments = args,
                        CreateNoWindow = true,
                        UseShellExecute = false
                    };

                    // アップデータを実行
                    Process process = Process.Start(startInfo);
                    process.WaitForExit();
                }

                // FE1→FE2
                {
                    List<string> targetList = new List<string>();
                    foreach (string target in this.esetList.Keys)
                    {
                        targetList.Add(target);
                    }

                    this.PreOpenUpdate(targetList, EsetType.FE1);
                }
            }
        }
    }
}
