﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using System;
using System.Collections.Generic;
using System.Threading.Tasks;
using EffectMaker.BusinessLogic.Options;
using EffectMaker.DataModel.AnimationTable;
using EffectMaker.DataModel.DataModels;
using EffectMaker.DataModel.Specific.DataModels;
using EffectMaker.DataModelLogic.Utilities;
using EffectMaker.Foundation.Attributes;
using EffectMaker.Foundation.Command;
using EffectMaker.Foundation.Dynamic;
using EffectMaker.Foundation.EventArguments;
using EffectMaker.Foundation.Extensions;
using EffectMaker.Foundation.Input;
using EffectMaker.Foundation.Interfaces;
using EffectMaker.Foundation.Utility;
using EffectMaker.UILogic.Attributes;
using EffectMaker.UILogic.Commands;

namespace EffectMaker.UILogic.ViewModels
{
    /// <summary>
    /// Class for the view model of the SpinBasicParamViewModel.
    /// </summary>
    public class SpinBasicParamViewModel : PropertyGroupViewModel<SpinParamData>, IModificationFlagOwner
    {
        /// <summary>
        /// VelocityEnableOtherKeySnap プロパティのバッキングフィールドです。
        /// </summary>
        private bool velocityEnableOtherKeySnap;

        /// <summary>
        /// VelocitySnapTime プロパティのバッキングフィールドです。
        /// </summary>
        private int velocitySnapTime;

        /// <summary>
        /// VelocitySnapValue プロパティのバッキングフィールドです。
        /// </summary>
        private int velocitySnapValue;

        /// <summary>
        /// 回転速度アニメーションのスナップ設定名です。
        /// </summary>
        private string velocitySnapSettingName;

        /// <summary>
        /// OuterEnableOtherKeySnap プロパティのバッキングフィールドです。
        /// </summary>
        private bool outerEnableOtherKeySnap;

        /// <summary>
        /// OuterSnapTime プロパティのバッキングフィールドです。
        /// </summary>
        private int outerSnapTime;

        /// <summary>
        /// OuterSnapValue プロパティのバッキングフィールドです。
        /// </summary>
        private int outerSnapValue;

        /// <summary>
        /// 拡散速度アニメーションのスナップ設定名です。
        /// </summary>
        private string outerSnapSettingName;

        /// <summary>
        /// コンストラクタです。
        /// </summary>
        /// <param name="parent">親ビューモデル</param>
        /// <param name="dataModel">The data model to encapsulate.</param>
        public SpinBasicParamViewModel(HierarchyViewModel parent, SpinParamData dataModel)
            : base(parent, dataModel)
        {
            this.FieldSpinAxisTypeItems = new []
            {
                new KeyValuePair<string, object>(Properties.Resources.FieldSpinAxisTypeX, 0),
                new KeyValuePair<string, object>(Properties.Resources.FieldSpinAxisTypeY, 1),
                new KeyValuePair<string, object>(Properties.Resources.FieldSpinAxisTypeZ, 2),
            };

            this.OnGetParticleLifeExecutable = new AnonymousExecutable((o) =>
            {
                var param = o as float[];
                var emitterVm = this.FindNearestParentOfType<EmitterViewModel>();
                param[0] = emitterVm.EmitterParticleViewModel.EmitterParticleLifeViewModel.GetDataModelValue<int>("Life");
            });

            this.OnVelocityResetExecutable = new AnonymousExecutable(this.OnVelocityReset);
            this.OnOuterResetExecutable = new AnonymousExecutable(this.OnOuterReset);

            this.InitVelocitySettings();
            this.InitOuterSettings();

            this.SetDefaultVelocitySnapSetting("FielsSpinBasicVelocity");
            this.SetDefaultOuterSnapSetting("FieldSpinBasicOuter");

            var table = this.GetDataModelValue(() => dataModel.VelocityAnimationTable);
            if (table.Count == 0)
            {
                this.InitVelocityData();
            }

            table = this.GetDataModelValue(() => dataModel.OuterAnimationTable);
            if (table.Count == 0)
            {
                this.InitOuterData();
            }

            // デフォルト値を更新
            this.UpdatePropertyDescriptors();

            var spinVelocityCurveProperty = EnumerableUtility.Enumerate(
                    ViewModelBase.NameOf(() => dataModel.EnableVelocityAnimation),
                    ViewModelBase.NameOf(() => dataModel.VelocityAnimationTable),
                    ViewModelBase.NameOf(() => dataModel.VelocityLoopMode),
                    ViewModelBase.NameOf(() => dataModel.VelocityLoopEnd),
                    ViewModelBase.NameOf(() => dataModel.VelocityRandomize),
                    ViewModelBase.NameOf(() => dataModel.SpinVelocity));

            var outerCurveProperty = EnumerableUtility.Enumerate(
                    ViewModelBase.NameOf(() => dataModel.EnableOuterAnimation),
                    ViewModelBase.NameOf(() => dataModel.OuterAnimationTable),
                    ViewModelBase.NameOf(() => dataModel.OuterLoopMode),
                    ViewModelBase.NameOf(() => dataModel.OuterLoopEnd),
                    ViewModelBase.NameOf(() => dataModel.OuterRandomize),
                    ViewModelBase.NameOf(() => dataModel.SpinOuter));

            // Always create the modification flag view model IN THE END of the constructor
            // to prevent any initialization triggers the modification events.
            var mmfvm = new MultipleModificationFlagsViewModel(this);
            mmfvm.SetPropertyDictionary("SpinVelocityCurve", spinVelocityCurveProperty);
            mmfvm.SetPropertyDictionary("OuterCurve", outerCurveProperty);
            this.ModificationFlagViewModel = mmfvm;

            // データ変更フラグをクリア。
            this.ModificationFlagViewModel.ClearModificationFlags();
            this.ModificationFlagViewModel.ClearChildModificationFlags();
        }

        /// <summary>軸タイプの項目を取得します.</summary>
        public IEnumerable<KeyValuePair<string, object>> FieldSpinAxisTypeItems { get; private set; }

        /// <summary>
        /// 回転速度エディタXチャンネルチェックボックス
        /// </summary>
        public bool VelocityEnableX { get; set; }

        /// <summary>
        /// 回転速度エディタ他キースナップ
        /// </summary>
        public bool VelocityEnableOtherKeySnap
        {
            get
            {
                return this.velocityEnableOtherKeySnap;
            }

            set
            {
                this.velocityEnableOtherKeySnap = value;
                this.UpdateVelocitySnapSettingOption();
            }
        }

        /// <summary>
        /// 回転速度エディタ時間スナップ
        /// </summary>
        public int VelocitySnapTime
        {
            get
            {
                return this.velocitySnapTime;
            }

            set
            {
                this.velocitySnapTime = value;
                this.UpdateVelocitySnapSettingOption();
            }
        }

        /// <summary>
        /// 回転速度エディタ値スナップ
        /// </summary>
        public int VelocitySnapValue
        {
            get
            {
                return this.velocitySnapValue;
            }

            set
            {
                this.velocitySnapValue = value;
                this.UpdateVelocitySnapSettingOption();
            }
        }

        /// <summary>
        /// 回転速度0固定モード
        /// </summary>
        public int VelocityZeroPinMode { get; set; }

        /// <summary>
        /// 回転速度テーブル(キー数変更時リロード対応)
        /// </summary>
        [UseDataModelOriginalValue]
        public AnimationTableData VelocityAnimationTable
        {
            get
            {
                return this.DataModel.VelocityAnimationTable;
            }

            set
            {
                int count = this.VelocityAnimationTable.Count;
                var binder = new EffectMakerSetMemberBinder("VelocityAnimationTable", false, false);
                MessageBlockerWithSendBinaryOnce.ExecuteOnDemandReload(
                    () => this.TrySetMember(binder, value),
                    value.Count != count,
                    this.DataModel);
            }
        }

        /// <summary>
        /// 拡散速度エディタXチャンネルチェックボックス
        /// </summary>
        public bool OuterEnableX { get; set; }

        /// <summary>
        /// 拡散速度エディタ他キースナップ
        /// </summary>
        public bool OuterEnableOtherKeySnap
        {
            get
            {
                return this.outerEnableOtherKeySnap;
            }

            set
            {
                this.outerEnableOtherKeySnap = value;
                this.UpdateOuterSnapSettingOption();
            }
        }

        /// <summary>
        /// 拡散速度エディタ時間スナップ
        /// </summary>
        public int OuterSnapTime
        {
            get
            {
                return this.outerSnapTime;
            }

            set
            {
                this.outerSnapTime = value;
                this.UpdateOuterSnapSettingOption();
            }
        }

        /// <summary>
        /// 拡散速度エディタ値スナップ
        /// </summary>
        public int OuterSnapValue
        {
            get
            {
                return this.outerSnapValue;
            }

            set
            {
                this.outerSnapValue = value;
                this.UpdateOuterSnapSettingOption();
            }
        }

        /// <summary>
        /// 拡散速度0固定モード
        /// </summary>
        public int OuterZeroPinMode { get; set; }

        /// <summary>
        /// 拡散速度テーブル(キー数変更時リロード対応)
        /// </summary>
        [UseDataModelOriginalValue]
        public AnimationTableData OuterAnimationTable
        {
            get
            {
                return this.DataModel.OuterAnimationTable;
            }

            set
            {
                int count = this.OuterAnimationTable.Count;
                var binder = new EffectMakerSetMemberBinder("OuterAnimationTable", false, false);
                MessageBlockerWithSendBinaryOnce.ExecuteOnDemandReload(
                    () => this.TrySetMember(binder, value),
                    value.Count != count,
                    this.DataModel);
            }
        }

        /// <summary>
        /// Get the view model that holds the modification flags of
        /// this view model's properties.
        /// </summary>
        public ModificationFlagViewModel ModificationFlagViewModel { get; private set; }

        /// <summary>
        /// パーティクルの寿命を取得するExecutableを取得します。
        /// </summary>
        public IExecutable OnGetParticleLifeExecutable { get; private set; }

        /// <summary>
        /// Get an executable to run when reset.
        /// </summary>
        public IExecutable OnVelocityResetExecutable { get; private set; }

        /// <summary>
        /// Get an executable to run when reset.
        /// </summary>
        public IExecutable OnOuterResetExecutable { get; private set; }

        /// <summary>
        /// 回転速度エディタの初期化
        /// </summary>
        private void InitVelocitySettings()
        {
            this.VelocityEnableX = true;
            this.VelocityEnableOtherKeySnap = true;
            this.VelocitySnapTime = 0;
            this.VelocitySnapValue = 2;
            this.VelocityZeroPinMode = -2;
        }

        /// <summary>
        /// 拡散速度エディタの初期化
        /// </summary>
        private void InitOuterSettings()
        {
            this.OuterEnableX = true;
            this.OuterEnableOtherKeySnap = true;
            this.OuterSnapTime = 0;
            this.OuterSnapValue = 2;
            this.OuterZeroPinMode = -2;
        }

        /// <summary>
        /// 回転速度のスナップ設定のデフォルト値を設定します。
        /// </summary>
        /// <param name="settingName">設定名</param>
        /// <param name="isEnabled">スナップを有効にするかどうか</param>
        /// <param name="snapTime">時間スナップ</param>
        /// <param name="snapValue">値スナップ</param>
        public void SetDefaultVelocitySnapSetting(string settingName, bool enableSnap = true, int snapTime = 0, int snapValue = 2)
        {
            // スナップのデフォルト値を適用
            this.VelocityEnableOtherKeySnap = enableSnap;
            this.VelocitySnapTime = snapTime;
            this.VelocitySnapValue = snapValue;

            // オプションに保存されたデフォルト値を取得
            CurveEditorSnapSetting setting;
            OptionStore.RootOptions.Interface.CurveEditorSnapSettings.TryGetValue(settingName, out setting);

            // オプションに保存されたデフォルト値があればそれを適用する
            if (setting != null)
            {
                this.VelocityEnableOtherKeySnap = setting.EnableSnap;
                this.VelocitySnapTime = setting.SnapValueX;
                this.VelocitySnapValue = setting.SnapValueY;
            }

            // スナップ設定名に有効な値をセットすると、スナップ値の変更がオプションに反映されるようになるため
            // 全てのスナップ値の変更が終わってからスナップ設定名をセットする
            this.velocitySnapSettingName = settingName;
        }

        /// <summary>
        /// 拡散速度のスナップ設定のデフォルト値を設定します。
        /// </summary>
        /// <param name="settingName">設定名</param>
        /// <param name="isEnabled">スナップを有効にするかどうか</param>
        /// <param name="snapTime">時間スナップ</param>
        /// <param name="snapValue">値スナップ</param>
        public void SetDefaultOuterSnapSetting(string settingName, bool enableSnap = true, int snapTime = 0, int snapValue = 2)
        {
            // スナップのデフォルト値を適用
            this.OuterEnableOtherKeySnap = enableSnap;
            this.OuterSnapTime = snapTime;
            this.OuterSnapValue = snapValue;

            // オプションに保存されたデフォルト値を取得
            CurveEditorSnapSetting setting;
            OptionStore.RootOptions.Interface.CurveEditorSnapSettings.TryGetValue(settingName, out setting);

            // オプションに保存されたデフォルト値があればそれを適用する
            if (setting != null)
            {
                this.OuterEnableOtherKeySnap = setting.EnableSnap;
                this.OuterSnapTime = setting.SnapValueX;
                this.OuterSnapValue = setting.SnapValueY;
            }

            // スナップ設定名に有効な値をセットすると、スナップ値の変更がオプションに反映されるようになるため
            // 全てのスナップ値の変更が終わってからスナップ設定名をセットする
            this.outerSnapSettingName = settingName;
        }

        /// <summary>
        /// 回転速度の現在のスナップ設定をオプションに反映させます。
        /// </summary>
        private void UpdateVelocitySnapSettingOption()
        {
            if (string.IsNullOrEmpty(this.velocitySnapSettingName) == false)
            {
                CurveEditorSnapSetting setting = new CurveEditorSnapSetting
                {
                    EnableSnap = this.VelocityEnableOtherKeySnap,
                    SnapValueX = this.VelocitySnapTime,
                    SnapValueY = this.VelocitySnapValue
                };

                OptionStore.RootOptions.Interface.CurveEditorSnapSettings[this.velocitySnapSettingName] = setting;
            }
        }

        /// <summary>
        /// 拡散速度の現在のスナップ設定をオプションに反映させます。
        /// </summary>
        private void UpdateOuterSnapSettingOption()
        {
            if (string.IsNullOrEmpty(this.outerSnapSettingName) == false)
            {
                CurveEditorSnapSetting setting = new CurveEditorSnapSetting
                {
                    EnableSnap = this.OuterEnableOtherKeySnap,
                    SnapValueX = this.OuterSnapTime,
                    SnapValueY = this.OuterSnapValue
                };

                OptionStore.RootOptions.Interface.CurveEditorSnapSettings[this.outerSnapSettingName] = setting;
            }
        }

        /// <summary>
        /// 回転速度のリセット処理
        /// </summary>
        private void OnVelocityReset()
        {
            if (ExportableViewModel.IsPasting)
            {
                return;
            }

            using (new MessageBlockerWithSendBinaryOnce(this.DataModel))
            using (new CommandCombiner())
            {
                this.TrySetMember(new EffectMakerSetMemberBinder("VelocityLoopMode", false, false), 0);
                this.TrySetMember(new EffectMakerSetMemberBinder("VelocityRandomize", false, false), 0);
                this.SetDataModelValue(100, () => this.DataModel.VelocityLoopEnd);
                this.InitVelocityData();
            }

            this.InitVelocitySettings();
        }

        /// <summary>
        /// 拡散速度のリセット処理
        /// </summary>
        private void OnOuterReset()
        {
            if (ExportableViewModel.IsPasting)
            {
                return;
            }

            using (new MessageBlockerWithSendBinaryOnce(this.DataModel))
            using (new CommandCombiner())
            {
                this.TrySetMember(new EffectMakerSetMemberBinder("OuterLoopMode", false, false), 0);
                this.TrySetMember(new EffectMakerSetMemberBinder("OuterRandomize", false, false), 0);
                this.SetDataModelValue(100, () => this.DataModel.OuterLoopEnd);
                this.InitOuterData();
            }

            this.InitOuterSettings();
        }

        /// <summary>
        /// 回転速度の値を初期化
        /// </summary>
        private void InitVelocityData()
        {
            var table = new AnimationTableData(0, 0.0f, 0.0f, 0.0f, 0.0f);
            this.SetDataModelValue(table, () => this.VelocityAnimationTable);
        }

        /// <summary>
        /// 拡散速度の値を初期化
        /// </summary>
        private void InitOuterData()
        {
            var table = new AnimationTableData(0, 0.0f, 0.0f, 0.0f, 0.0f);
            this.SetDataModelValue(table, () => this.OuterAnimationTable);
        }
    }
}
