﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using System;
using System.Collections.ObjectModel;
using EffectMaker.DataModel.DataModels;
using EffectMaker.DataModel.Specific.DataModels;
using EffectMaker.DataModelLogic.Utilities;
using EffectMaker.Foundation.Attributes;
using EffectMaker.UILogic.Attributes;
using EffectMaker.UILogic.Commands;

namespace EffectMaker.UILogic.ViewModels
{
    /// <summary>
    /// Class for the view model of the FieldData.
    /// </summary>
    [Tag("WorkspaceNodeViewModelKind")]
    public abstract class FieldViewModel : WorkspaceNodeViewModelBase
    {
        /// <summary>
        /// The constructor.
        /// </summary>
        /// <param name="parent">The parent view model.</param>
        /// <param name="dataModel">The data model to encapsulate.</param>
        public FieldViewModel(HierarchyViewModel parent, FieldDataBase dataModel)
            : base(parent, dataModel)
        {
        }

        /// <summary>
        /// 新ランダムフィールドが作成可能か調べる.
        /// </summary>
        public bool CanCreateNewRandomField
        {
            get { return WorkspaceRootViewModel.Instance.CanHaveRandomField(this.Parent); }
        }

        /// <summary>
        /// ランダムフィールドが作成可能か調べる.
        /// </summary>
        public bool CanCreateNewRandomFe1Field
        {
            get { return WorkspaceRootViewModel.Instance.CanHaveRandomFe1Field(this.Parent); }
        }

        /// <summary>
        /// 磁力フィールドが作成可能か調べる.
        /// </summary>
        public bool CanCreateNewMagnetField
        {
            get { return WorkspaceRootViewModel.Instance.CanHaveMagnetField(this.Parent); }
        }

        /// <summary>
        /// スピンフィールドが作成可能か調べる.
        /// </summary>
        public bool CanCreateNewSpinField
        {
            get { return WorkspaceRootViewModel.Instance.CanHaveSpinField(this.Parent); }
        }

        /// <summary>
        /// 収束フィールドが作成可能か調べる.
        /// </summary>
        public bool CanCreateNewConvergeField
        {
            get { return WorkspaceRootViewModel.Instance.CanHaveConvergeField(this.Parent); }
        }

        /// <summary>
        /// "位置に加算"フィールドが作成可能か調べる.
        /// </summary>
        public bool CanCreateNewAddLocationField
        {
            get { return WorkspaceRootViewModel.Instance.CanHaveAddLocationField(this.Parent); }
        }

        /// <summary>
        /// シンプルコリジョンフィールドが作成可能か調べる.
        /// </summary>
        public bool CanCreateNewCollisionField
        {
            get { return WorkspaceRootViewModel.Instance.CanHaveCollisionField(this.Parent); }
        }

        /// <summary>
        /// カールノイズフィールドが作成可能か調べる.
        /// </summary>
        public bool CanCreateNewCurlNoiseField
        {
            get { return WorkspaceRootViewModel.Instance.CanHaveCurlNoiseField(this.Parent); }
        }

        /// <summary>
        /// カスタムフィールドが作成可能か調べる.
        /// </summary>
        public bool CanCreateNewCustomField
        {
            get { return WorkspaceRootViewModel.Instance.CanHaveCustomField(this.Parent); }
        }

        /// <summary>
        /// ノードデータをバイナリにコンバートするかどうか取得または設定します。
        /// </summary>
        [UseDataModelOriginalValue]
        public bool EnableConvert
        {
            get
            {
                return this.GetDataModelValue(() => this.EnableConvert);
            }

            set
            {
                using (new MessageBlockerWithSendBinaryOnce(this.Proxy.DataModel))
                {
                    this.SetDataModelValue(value, () => this.EnableConvert);

                    // エミッタのGPUモードを更新
                    HierarchyViewModel.GetParent<EmitterViewModel>(this).EmitterBasicViewModel.EmitterBasicBasicViewModel.RaiseCanUsingGpu();
                }
            }
        }
    }

    /// <summary>
    /// ジェネリック版のFieldViewModelです。
    /// </summary>
    /// <typeparam name="T">データモデルの型</typeparam>
    [Tag("WorkspaceNodeViewModelKind")]
    public abstract class FieldViewModel<T> : FieldViewModel where T : FieldDataBase
    {
        /// <summary>
        /// コンストラクタです。
        /// </summary>
        /// <param name="parent">親ビューモデル</param>
        /// <param name="dataModel">関連付けるデータモデル</param>
        protected FieldViewModel(HierarchyViewModel parent, T dataModel)
            : base(parent, dataModel)
        {
        }

        /// <summary>
        /// 関連付けられたデータモデルを取得します。
        /// </summary>
        public T DataModel
        {
            get { return this.Proxy.DataModel; }
        }
    }
}
