﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using System;
using System.Collections.Generic;
using System.IO;
using System.Linq;
using System.Reflection;
using System.Text;
using System.Threading.Tasks;
using System.Xml;
using System.Xml.Linq;
using System.Xml.Serialization;
using EffectMaker.DataModel.DataModels;
using EffectMaker.DataModel.Specific.DataModels;
using EffectMaker.DataModelLogic.DataModelProxies;
using EffectMaker.Foundation.Core;
using EffectMaker.Foundation.Dynamic;
using EffectMaker.Foundation.Extensions;
using EffectMaker.Foundation.Interfaces;
using EffectMaker.UILogic.TypeSerializers;

namespace EffectMaker.UILogic.ViewModels
{
    /// <summary>
    /// Extended ViewModelBase class with import/export capabilities.
    /// </summary>
    public class ExportableViewModel : ViewModelBase
    {
        /// <summary>
        /// Gets the array of known xml type serializers.
        /// </summary>
        private static IXmlTypeSerializer[] serializers = new IXmlTypeSerializer[]
        {
            new LanguagePrimitiveXmlTypeSerializer(),
            new XmlSerializableXmlTypeSerializer(),
            new DataModelXmlTypeSerializer(),
        };

        /// <summary>
        /// コピペの対象にしないプロパティ名リスト
        /// </summary>
        private List<string> ignorePropertyNames = new List<string>();

        /// <summary>
        /// Initializes the ExportableViewModel instance.
        /// </summary>
        /// <param name="dataModel">Data model to represent.</param>
        public ExportableViewModel(DataModelBase dataModel)
            : base(dataModel)
        {
        }

        /// <summary>
        /// ペースト、及びそれに類する処理(値リセット・プレビュー置き換え読み込み)で、
        /// 望ましくない処理をブロックするためのフラグです。
        /// </summary>
        public static bool IsPasting { get; set; }

        /// <summary>
        /// コピペの対象にしないプロパティ名リストを追加します.
        /// </summary>
        /// <param name="names">プロパティ名リスト</param>
        public void AddIgnoreCopyProperties(string[] names)
        {
            foreach (string name in names)
            {
                this.ignorePropertyNames.Add(name);
            }
        }

        /// <summary>
        /// Export values to a XML string.
        /// </summary>
        /// <returns>Returns a XML string containing the serialized values.</returns>
        public virtual string ExportValuesAsXml()
        {
            Type type = this.GetType();
            XElement root;

            var namedVm = this as INamedViewModel;
            if (namedVm != null)
            {
                root = new XElement(
                    "root",
                    new XAttribute("type", type.FullName),
                    new XAttribute("group", namedVm.BindingName));
            }
            else
            {
                root = new XElement(
                    "root",
                    new XAttribute("type", type.FullName));
            }

            try
            {
                this.ExportValues(new XmlElementContainer(root));
            }
            catch
            {
                // TODO: report the error
                return null;
            }

            return root.ToString();
        }

        /// <summary>
        /// Check if this view model can import values from the given XML contents.
        /// </summary>
        /// <param name="xmlContent">The XML content to set values from.</param>
        /// <returns>Returns true if can import, false otherwise.</returns>
        public virtual bool CanImportValuesFromXml(string xmlContent)
        {
            XElement root = null;

            try
            {
                root = XElement.Parse(xmlContent);
            }
            catch
            {
                return false;
            }

            Type type = this.GetType();

            if ((string)root.Attribute("type") != type.FullName)
            {
                return false;
            }

            var namedVm = this as INamedViewModel;
            if (namedVm != null)
            {
                if ((string) root.Attribute("group") != namedVm.BindingName)
                {
                    return false;
                }
            }

            return true;
        }

        /// <summary>
        /// Set values based on a XML content.
        /// </summary>
        /// <param name="xmlContent">The XML content to set values from.</param>
        /// <returns>Returns true if import succeeded, false otherwise.</returns>
        public virtual bool ImportValuesFromXml(string xmlContent)
        {
            XElement root = null;

            try
            {
                root = XElement.Parse(xmlContent);
            }
            catch
            {
                // TODO: report the error
                return false;
            }

            Type type = this.GetType();

            if ((string)root.Attribute("type") != type.FullName)
            {
                // TODO: report the error ?
                return false;
            }

            try
            {
                this.ImportValues(root.Elements());
            }
            catch
            {
                // TODO: report the error
                return false;
            }

            return true;
        }

        /// <summary>
        /// When overridden, exports the values to XML.
        /// </summary>
        /// <param name="container">Container to export values to.</param>
        protected virtual void ExportValues(IPushOnlyContainer<XElement> container)
        {
            IEnumerable<PropertyInfo> dataModelProperties =
                (this.Proxy as DataModelProxy).GetDynamicPropertyInfos();

            var dataModelRoot = new XElement("data");

            foreach (PropertyInfo propertyInfo in dataModelProperties)
            {
                // get the serializer for the property type
                var serializer = this.GetXmlTypeSerializer(propertyInfo);

                // skip if there is there is no serialized (that means the type is unknown)
                if (serializer == null)
                {
                    continue;
                }

                XElement element = serializer.Serialize(
                    propertyInfo.Name,
                    propertyInfo.GetValue((this.Proxy as DataModelProxy).DataModel));

                dataModelRoot.Add(element);
            }

            container.Add(dataModelRoot);

            Type type = this.GetType();

            // retrieve properties of the view model
            PropertyInfo[] viewModelProperties = type
                .GetProperties(BindingFlags.Public | BindingFlags.Instance);

            var viewModelRoot = new XElement("vm");

            foreach (var propertyInfo in viewModelProperties)
            {
                if (this.ignorePropertyNames.Contains(propertyInfo.Name))
                {
                    continue;
                }

                // get the serializer for the property type
                var serializer = this.GetXmlTypeSerializer(propertyInfo);

                // skip if there is there is no serialized (that means the type is unknown)
                if (serializer == null)
                {
                    continue;
                }

                XElement element = serializer.Serialize(
                    propertyInfo.Name,
                    propertyInfo.GetValue(this));

                viewModelRoot.Add(element);
            }

            container.Add(viewModelRoot);
        }

        /// <summary>
        /// When overridden, imports the values from XML.
        /// </summary>
        /// <param name="elements">Sequence of XML elements to import from.</param>
        protected virtual void ImportValues(IEnumerable<XElement> elements)
        {
            if (elements == null)
            {
                throw new ArgumentNullException("elements");
            }

            XElement dataRoot = elements.FirstOrDefault(x => x.Name == "data");
            if (dataRoot == null)
            {
                throw new FormatException("data");
            }

            string name = null;
            object value = null;

            // for each element in the XML data
            foreach (XElement element in dataRoot.Elements())
            {
                // for each deserializer
                foreach (IXmlTypeSerializer serializer in serializers)
                {
                    // look for a working deserializer
                    if (serializer.Deserialize(element, out name, out value) == false)
                    {
                        // this deserializer is not matching, try with the next one
                        continue;
                    }

                    //// from here, we found a matching deserializer
                    //// and retrieved data name and value

                    // try to find a property that match the name
                    PropertyInfo propertyInfo;
                    if ((this.Proxy as DataModelProxy).GetDynamicPropertyInfo(name, out propertyInfo) == false)
                    {
                        break;
                    }

                    if (propertyInfo.Name == "Guid" &&
                        propertyInfo.PropertyType == typeof(Guid))
                    {
                        break;
                    }

                    // make sure we can set the value
                    if (propertyInfo.CanPubliclyWrite() == false)
                    {
                        break;
                    }

                    try
                    {
                        object original;
                        this.TryGetMember(new EffectMakerGetMemberBinder(name), out original);

                        var dm = original as DataModelBase;
                        if (dm != null)
                        {
                            dm.SetWithoutGuid(value);
                        }
                        else
                        {
                            // set the value through dynamic support
                            this.TrySetMember(new EffectMakerSetMemberBinder(name), value);
                        }
                    }
                    catch (ArgumentException)
                    {
                        // type mismatch may happen
                        continue;
                    }

                    break;
                }
            }

            XElement viewModelRoot = elements.FirstOrDefault(x => x.Name == "vm");
            if (viewModelRoot == null)
            {
                throw new FormatException("vm");
            }

            Type viewModelType = this.GetType();

            // for each element in the XML data
            foreach (XElement element in viewModelRoot.Elements())
            {
                // for each deserializer
                foreach (IXmlTypeSerializer serializer in serializers)
                {
                    // look for a working deserializer
                    if (serializer.Deserialize(element, out name, out value) == false)
                    {
                        // this deserializer is not matching, try with the next one
                        continue;
                    }

                    //// from here, we found a matching deserializer
                    //// and retrieved data name and value

                    // try to find a property that match the name
                    PropertyInfo propertyInfo = viewModelType.GetProperty(
                        name,
                        BindingFlags.Public | BindingFlags.Instance);
                    if (propertyInfo == null)
                    {
                        break;
                    }

                    // make sure we can set the value
                    if (propertyInfo.CanPubliclyWrite() == false)
                    {
                        break;
                    }

                    try
                    {
                        // set the value
                        propertyInfo.SetValue(this, value);
                    }
                    catch (ArgumentException)
                    {
                        // type mismatch may happen
                        continue;
                    }

                    break;
                }
            }
        }

        /// <summary>
        /// Gets the IXmlTypeSerializer instance matching the given property.
        /// </summary>
        /// <param name="propertyInfo">The PropertyInfo to retrieve the IXmlTypeSerializer.</param>
        /// <returns>Returns an IXmlTypeSerializer, or null otherwise.</returns>
        private IXmlTypeSerializer GetXmlTypeSerializer(PropertyInfo propertyInfo)
        {
            //// skip the properties than cannot be publicly read and written
            //// because they cannot be used
            if (propertyInfo.CanPubliclyRead() == false ||
                propertyInfo.CanPubliclyWrite() == false)
            {
                return null;
            }

            // skip properties with the XmlIgnore attribute
            XmlIgnoreAttribute attr =
                propertyInfo.GetCustomAttribute<XmlIgnoreAttribute>(false);
            if (attr != null)
            {
                return null;
            }

            // find a serializer for the property type
            return serializers
                .FirstOrDefault(s => s.IsSupportedType(propertyInfo.PropertyType));
        }

        /// <summary>
        /// Private container to export XML serialized values.
        /// </summary>
        protected class XmlElementContainer : IPushOnlyContainer<XElement>
        {
            /// <summary>
            /// Stores the root XElement.
            /// </summary>
            private XElement root;

            /// <summary>
            /// Initializes the XmlElementContainer instance.
            /// </summary>
            /// <param name="root">The root XElement.</param>
            public XmlElementContainer(XElement root)
            {
                if (root == null)
                {
                    throw new ArgumentNullException("root");
                }

                this.root = root;
            }

            /// <summary>
            /// Adds a value to the child collection of the root element.
            /// </summary>
            /// <param name="value">The value to add to the container.</param>
            public void Add(XElement value)
            {
                this.root.Add(value);
            }
        }
    }
}
