﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using System;
using System.Collections.Generic;
using EffectMaker.DataModel.DataModels;
using EffectMaker.DataModel.Specific.DataModels;
using EffectMaker.Foundation.Attributes;
using EffectMaker.Foundation.Command;
using EffectMaker.Foundation.EventArguments;
using EffectMaker.Foundation.Input;
using EffectMaker.Foundation.Interfaces;
using EffectMaker.UILogic.Attributes;

namespace EffectMaker.UILogic.ViewModels
{
    /// <summary>
    /// Class for the view model of the EmitterTexturePatternAnimationViewModel.
    /// </summary>
    public class EmitterTexturePatternAnimationViewModel : PropertyGroupViewModel<EmitterTexturePatternAnimationData>, IModificationFlagOwner
    {
        /// <summary>
        /// 変更の対象に含めないプロパティ名のリストです.
        /// </summary>
        private readonly string[] ignorePropertyNames = new string[]
        {
            "PatternTableVisibility",
            "PatternAnimTypeInCommentVisibility",
            "DurationCountPerFrameVisibility",
            "PatternTableCountVisibility",
            "RandomPatternTableCountVisibility",
            "EnableSimpleTableRandomVisibility",
            "EnableTableInputVisibility",
            "SurfaceSubGroupVisibility"
        };

        /// <summary>
        /// The constructor.
        /// </summary>
        /// <param name="parent">The parent view model.</param>
        /// <param name="dataModel">The data model to encapsulate.</param>
        public EmitterTexturePatternAnimationViewModel(
            HierarchyViewModel parent, EmitterTexturePatternAnimationData dataModel)
            : base(parent, dataModel)
        {
            this.AvailablePatternAnimTypes = new []
            {
                new KeyValuePair<string, object>(Properties.Resources.EmitterTextureAnimationPlayTypeNone, 0),
                new KeyValuePair<string, object>(Properties.Resources.EmitterTextureAnimationPlayTypeLifeFit, 1),
                new KeyValuePair<string, object>(Properties.Resources.EmitterTextureAnimationPlayTypeClamp, 2),
                new KeyValuePair<string, object>(Properties.Resources.EmitterTextureAnimationPlayTypeLoop, 3),
                new KeyValuePair<string, object>(Properties.Resources.EmitterTextureAnimationPlayTypeRandom, 4),
                new KeyValuePair<string, object>(Properties.Resources.EmitterTextureAnimationPlayTypeSurface, 5),
            };

            // Always create the modification flag view model IN THE END of the constructor
            // to prevent any initialization triggers the modification events.
            this.ModificationFlagViewModel = new ModificationFlagViewModel(this);
            this.ModificationFlagViewModel.AddIgnoreProperties(this.ignorePropertyNames);
        }

        /// <summary>
        /// 再生方法の項目を取得します.
        /// </summary>
        public IEnumerable<KeyValuePair<string, object>> AvailablePatternAnimTypes { get; private set; }

        /// <summary>
        /// エミッタのタイプを設定または取得します。
        /// </summary>
        [UseDataModelOriginalValue]
        public int PatternAnimType
        {
            get
            {
                return this.GetDataModelValue(() => this.PatternAnimType);
            }

            set
            {
                this.SetDataModelValue(value, () => this.PatternAnimType);

                // プロパティ変更通知.
                this.OnPropertyChanged(() => this.PatternTableVisibility);
                this.OnPropertyChanged(() => this.DurationCountPerFrameVisibility);
                this.OnPropertyChanged(() => this.PatternTableCountVisibility);
                this.OnPropertyChanged(() => this.RandomPatternTableCountVisibility);
                this.OnPropertyChanged(() => this.EnableSimpleTableRandomVisibility);
                this.OnPropertyChanged(() => this.EnableTableInputVisibility);
                this.OnPropertyChanged(() => this.PatternAnimTypeName);
                this.OnPropertyChanged(() => this.PatternAnimTypeInCommentVisibility);
                this.OnPropertyChanged(() => this.SurfaceSubGroupVisibility);
            }
        }

        /// <summary>
        /// 再生方法名
        /// </summary>
        public string PatternAnimTypeName
        {
            get
            {
                string str = string.Empty;
                foreach (KeyValuePair<string, object> patternAnimType in this.AvailablePatternAnimTypes)
                {
                    if (this.PatternAnimType == (int)patternAnimType.Value)
                    {
                        str = patternAnimType.Key;
                    }
                }

                return str;
            }
        }

        /// <summary>
        /// ランダム設定時の使用パターンテーブル数です。
        /// </summary>
        public int RandomPatternTableCountForViewModel
        {
            get
            {
                return this.GetDataModelValue<int>(() => this.DataModel.RandomPatternTableCount);
            }

            set
            {
                this.SetDataModelValue<int>(value, () => this.DataModel.RandomPatternTableCount);
            }
        }

        /// <summary>
        /// パターンテーブルの表示設定を取得します。
        /// </summary>
        public bool PatternTableVisibility
        {
            get
            {
                // 再生方法がライフフィット,クランプ,ループ,ランダムのときtrue.
                int animType = this.DataModel.PatternAnimType;

                return animType == 1 || animType == 2 || animType == 3 || animType == 4;
            }

            set
            {
            }
        }

        /// <summary>
        /// 1コマの継続フレーム数の表示設定を取得します。
        /// </summary>
        public bool DurationCountPerFrameVisibility
        {
            get
            {
                // 再生方法がクランプ,ループのときtrue.
                int animType = this.DataModel.PatternAnimType;

                return animType == 2 || animType == 3;
            }

            set
            {
            }
        }

        /// <summary>
        /// 使用パターンテーブル数の表示設定を取得します。
        /// </summary>
        public bool PatternTableCountVisibility
        {
            get
            {
                // 再生方法がライフフィット,クランプ,ループのときtrue.
                int animType = DataModel.PatternAnimType;

                return animType == 1 || animType == 2 || animType == 3;
            }

            set
            {
            }
        }

        /// <summary>
        /// テーブル数の表示設定を取得します。
        /// </summary>
        public bool RandomPatternTableCountVisibility
        {
            get
            {
                // 再生方法がランダムのときtrue.
                int animType = this.DataModel.PatternAnimType;

                return animType == 4;
            }

            set
            {
            }
        }

        /// <summary>
        /// 開始テーブルランダムの表示設定を取得します。
        /// </summary>
        public bool EnableSimpleTableRandomVisibility
        {
            get
            {
                // 再生方法がループのときtrue.
                int animType = this.DataModel.PatternAnimType;

                return animType == 3;
            }

            set
            {
            }
        }

        /// <summary>テーブル入力の表示設定を取得します。</summary>
        public bool EnableTableInputVisibility
        {
            get
            {
                // 再生方法がライフフィット,クランプ,ループのときtrue.
                int animType = this.DataModel.PatternAnimType;

                return animType == 1 || animType == 2 || animType == 3;
            }

            set
            {
            }
        }

        /// <summary>
        /// ライフフィット時のコメントの表示設定を取得します。
        /// </summary>
        public bool PatternAnimTypeInCommentVisibility
        {
            get
            {
                // 再生方法がライフフィットのときtrue
                int animType = this.DataModel.PatternAnimType;

                return animType == 1;
            }

            set
            {
            }
        }

        /// <summary>
        /// サーフェイスサブグループの表示設定を取得します。
        /// </summary>
        public bool SurfaceSubGroupVisibility
        {
            get
            {
                // 再生方法がサーフェイスのときtrue
                int animType = this.DataModel.PatternAnimType;

                return animType == 5;
            }

            set
            {
            }
        }

        /// <summary>
        /// Get the view model that holds the modification flags of
        /// this view model's properties.
        /// </summary>
        public ModificationFlagViewModel ModificationFlagViewModel { get; private set; }
    }
}
