﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using System;
using EffectMaker.DataModel.AnimationTable;
using EffectMaker.DataModel.Specific.DataModels;
using EffectMaker.Foundation.Dynamic;
using EffectMaker.Foundation.Input;
using EffectMaker.Foundation.Interfaces;
using EffectMaker.Foundation.Primitives;
using EffectMaker.Foundation.Utility;
using EffectMaker.UILogic.Commands;

namespace EffectMaker.UILogic.ViewModels
{
    /// <summary>
    /// Class for the view model of the EmitterScaleAnimationTableViewModel.
    /// </summary>
    public class EmitterScaleAnimationCurveViewModel : CurveEditorOwnerViewModel<EmitterScaleAnimationCurveData>, IModificationFlagOwner
    {
        /// <summary>
        /// 変更の対象に含めないプロパティ名のリストです.
        /// </summary>
        private readonly string[] ignoreModifyPropertyNames = new string[]
        {
            "AdjustTrigger"
        };

        /// <summary>
        /// ビューポート調整のトリガー値.
        /// </summary>
        private bool adjustTrigger = false;

        /// <summary>
        /// The constructor.
        /// </summary>
        /// <param name="parent">The parent view model.</param>
        /// <param name="dataModel">The data model to encapsulate.</param>
        public EmitterScaleAnimationCurveViewModel(
            HierarchyViewModel parent, EmitterScaleAnimationCurveData dataModel)
            : base(parent, dataModel)
        {
            this.DefaultValue = new Vector4f(1.0f, 1.0f, 1.0f, 0.0f);

            if (dataModel.AnimationTable.Count == 0)
            {
                this.InitData();
            }

            this.InitSettings();
            this.SetDefaultSnapSetting("EmitterScaleParticle");

            this.AdaptPresetExecutable = new AnonymousExecutable(this.AdaptPreset);

            var mmfvm = new MultipleModificationFlagsViewModel(this);
            var scaleAnimeParams = EnumerableUtility.Enumerate(
                ViewModelBase.NameOf(() => dataModel.AnimationTable),
                ViewModelBase.NameOf(() => dataModel.LoopMode),
                ViewModelBase.NameOf(() => dataModel.LoopEnd),
                ViewModelBase.NameOf(() => dataModel.Randomize));
            mmfvm.SetPropertyDictionary("ScaleAnimeParams", scaleAnimeParams);
            mmfvm.AddIgnoreProperties(this.ignoreModifyPropertyNames);

            // Always create the modification flag view model IN THE END of the constructor
            // to prevent any initialization triggers the modification events.
            this.ModificationFlagViewModel = mmfvm;
        }

        /// <summary>
        /// カーブにプリセットを適用する処理を取得します。
        /// </summary>
        public IExecutable AdaptPresetExecutable { get; private set; }

        /// <summary>
        /// Get the view model that holds the modification flags of
        /// this view model's properties.
        /// </summary>
        public ModificationFlagViewModel ModificationFlagViewModel { get; private set; }

        /// <summary>
        /// ビューポートの調整処理をトリガーします。
        /// </summary>
        public bool AdjustTrigger
        {
            get
            {
                return this.adjustTrigger;
            }

            set
            {
                this.adjustTrigger = value;
                if (!value)
                {
                    return;
                }

                // View側に通知
                this.OnPropertyChanged();

                // 通知したら元の値に戻す
                this.adjustTrigger = false;
            }
        }

        /// <summary>
        /// プリセットを適用します。
        /// </summary>
        /// <param name="parameter">プリセットの適用パターンを表す文字列(A,B,C)</param>
        private void AdaptPreset(object parameter)
        {
            var args = parameter as string;
            if (string.IsNullOrEmpty(args))
            {
                return;
            }

            var binder = new EffectMakerSetMemberBinder("AnimationTable", false, true);
            var table = new AnimationTableData();

            if (args == "A")
            {
                table.AddKeyFrame(0, 1.0f, 1.0f, 1.0f, 0.0f, true);
                table.AddKeyFrame(100, 0.0f, 0.0f, 0.0f, 0.0f, false);
            }
            else if (args == "B")
            {
                table.AddKeyFrame(0, 0.0f, 0.0f, 0.0f, 0.0f, true);
                table.AddKeyFrame(50, 1.0f, 1.0f, 1.0f, 0.0f, false);
                table.AddKeyFrame(100, 0.0f, 0.0f, 0.0f, 0.0f, false);
            }
            else if (args == "C")
            {
                table.AddKeyFrame(0, 0.0f, 0.0f, 0.0f, 0.0f, true);
                table.AddKeyFrame(25, 1.0f, 1.0f, 1.0f, 0.0f, false);
                table.AddKeyFrame(75, 1.0f, 1.0f, 1.0f, 0.0f, false);
                table.AddKeyFrame(100, 0.0f, 0.0f, 0.0f, 0.0f, false);
            }
            else if (args == "D")
            {
                table.AddKeyFrame(0, 1.0f, 1.0f, 1.0f, 0.0f, true);
                table.AddKeyFrame(50, 1.0f, 1.0f, 1.0f, 0.0f, false);
                table.AddKeyFrame(100, 0.0f, 0.0f, 0.0f, 0.0f, false);
            }
            else if (args == "E")
            {
                table.AddKeyFrame(0, 1.0f, 1.0f, 1.0f, 0.0f, true);
                table.AddKeyFrame(25, 0.5f, 0.5f, 0.5f, 0.0f, false);
                table.AddKeyFrame(75, 0.5f, 0.5f, 0.5f, 0.0f, false);
                table.AddKeyFrame(100, 0.0f, 0.0f, 0.0f, 0.0f, false);
            }
            else if (args == "F")
            {
                table.AddKeyFrame(0, 0.0f, 0.0f, 0.0f, 0.0f, true);
                table.AddKeyFrame(100, 1.0f, 1.0f, 1.0f, 0.0f, false);
            }
            else
            {
                return;
            }

            using (new MessageBlockerWithSendBinaryOnce(this.DataModel))
            {
                if (this.TrySetMember(binder, table))
                {
                    this.AdjustTrigger = true;
                }
            }
        }
    }
}
