﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using System;
using System.Collections.Generic;
using System.IO;
using System.Linq;
using System.Xml.Linq;

using EffectMaker.BusinessLogic;
using EffectMaker.BusinessLogic.IO;
using EffectMaker.BusinessLogic.UserData;

using EffectMaker.DataModel.DataModels;
using EffectMaker.DataModel.Manager;
using EffectMaker.DataModel.Specific.DataModels;

using EffectMaker.DataModelLogic.DataModelProxies;
using EffectMaker.DataModelLogic.Utilities;

using EffectMaker.Foundation.Attributes;
using EffectMaker.Foundation.Command;
using EffectMaker.Foundation.Dynamic;
using EffectMaker.Foundation.EventArguments;
using EffectMaker.Foundation.Input;
using EffectMaker.Foundation.Interfaces;
using EffectMaker.Foundation.Log;

using EffectMaker.UILogic.Properties;

namespace EffectMaker.UILogic.ViewModels
{
    /// <summary>
    /// エミッタカスタムシェーダビューモデル
    /// </summary>
    public class EmitterCustomShaderGroupViewModel : PropertyGroupViewModel<EmitterCustomShaderData>, IModificationFlagOwner
    {
        /// <summary>
        /// 変更の対象に含めないプロパティ名のリストです.
        /// </summary>
        private readonly string[] ignorePropertyNames = new string[]
        {
            "AvailableSettingViewModels",
            "SelectedSettingViewModel",
        };

        /// <summary>
        /// The constructor.
        /// </summary>
        /// <param name="parent">The parent view model.</param>
        /// <param name="dataModel">The data model to encapsulate.</param>
        public EmitterCustomShaderGroupViewModel(
            HierarchyViewModel parent,
            EmitterCustomShaderData dataModel)
            : base(parent, dataModel)
        {
            // Generate view model for the custom shader user data models.
            this.UpdateUserData(true);

            UserDataManager.UserDataUpdated += this.OnUserDataUpdated;

            // Always create the modification flag view model IN THE END of the constructor
            // to prevent any initialization triggers the modification events.
            this.ModificationFlagViewModel = new ModificationFlagViewModel(this);
            this.ModificationFlagViewModel.AddIgnoreProperties(this.ignorePropertyNames);
        }

        /// <summary>
        /// Get the available custom shader setting view models.
        /// </summary>
        public IEnumerable<object> AvailableSettingViewModels
        {
            get
            {
                return from ch in this.Children
                       where ch is EmitterCustomShaderSettingDataViewModel
                       select (EmitterCustomShaderSettingDataViewModel)ch;
            }
        }

        /// <summary>
        /// Get the selected custom shader setting data view model.
        /// </summary>
        public object SelectedSettingViewModel
        {
            get
            {
                if (this.DataModel == null)
                {
                    return null;
                }

                int index = this.DataModel.SelectedSettingIndex;
                if (index < 0 || index >= this.DataModel.Settings.Count)
                {
                    return null;
                }

                var dataModel = this.DataModel.Settings[index];
                if (dataModel == null)
                {
                    return null;
                }

                foreach (IHierarchyObject child in this.Children)
                {
                    var vm = child as EmitterCustomShaderSettingDataViewModel;
                    if (vm != null && vm.DataModel == dataModel)
                    {
                        return vm;
                    }
                }

                return null;
            }
        }

        /// <summary>
        /// Get or set the selected custom setting index.
        /// </summary>
        public int SelectedSettingIndex
        {
            get
            {
                return this.GetDataModelValue(() => this.SelectedSettingIndex);
            }

            set
            {
                this.SetDataModelValue(value, () => this.SelectedSettingIndex);
                this.OnPropertyChanged(() => this.SelectedSettingViewModel);
            }
        }

        /// <summary>
        /// Get the view model that holds the modification flags of
        /// this view model's properties.
        /// </summary>
        public ModificationFlagViewModel ModificationFlagViewModel
        {
            get;
            private set;
        }

        /// <summary>
        /// Dispose this object.
        /// </summary>
        public override void Dispose()
        {
            UserDataManager.UserDataUpdated -= this.OnUserDataUpdated;

            base.Dispose();
        }

        /// <summary>
        /// ViewModelコレクションの変更処理をトリガーする
        /// </summary>
        public void RaiseViewModelsChanged()
        {
            this.OnPropertyChanged(() => this.AvailableSettingViewModels);
        }

        /// <summary>
        /// 現在インデックスしているViewModelの変更処理をトリガーする
        /// </summary>
        public void RaiseSelectedIndexChanged()
        {
            this.OnPropertyChanged(() => this.SelectedSettingIndex);
            this.OnPropertyChanged(() => this.SelectedSettingViewModel);
        }

        /// <summary>
        /// Check if this view model can import values from the given XML contents.
        /// </summary>
        /// <param name="xmlContent">The XML content to set values from.</param>
        /// <returns>Returns true if can import, false otherwise.</returns>
        public override bool CanImportValuesFromXml(string xmlContent)
        {
            XElement root = null;

            try
            {
                root = XElement.Parse(xmlContent);
            }
            catch
            {
                return false;
            }

            Type type = this.GetType();

            if ((string)root.Attribute("type") != type.FullName)
            {
                return false;
            }

            try
            {
                var data = root.Element("data");
                if (data != null)
                {
                    var lp = data.Element("lp");
                    if (lp != null)
                    {
                        if (lp.Attribute("name").Value == "SelectedSettingIndex")
                        {
                            var index = lp.Attribute("value");
                            if (index.Value != this.SelectedSettingIndex.ToString())
                            {
                                return false;
                            }
                        }
                    }
                }
            }
            catch
            {
                return false;
            }

            return true;
        }

        /// <summary>
        /// Create a data model proxy.
        /// This method is called in the constructor.
        /// If you need a specific type of data model proxy,
        /// override this method and return the desired data model proxy.
        /// </summary>
        /// <param name="dataModel">The data model.</param>
        /// <returns>The created data model proxy.</returns>
        protected override DataModelProxy CreateDataModelProxy(DataModelBase dataModel)
        {
            return new EmitterCustomShaderDataProxy(dataModel);
        }

        /// <summary>
        /// Handle UserDataUpdated event from the user data manager.
        /// </summary>
        /// <param name="sender">The sender of the event.</param>
        /// <param name="e">The event arguments.</param>
        private void OnUserDataUpdated(
            object sender,
            UserDataUpdatedEventArgs e)
        {
            if (e.IsUserDataTypeUpdated(UpdatedUserDataTypes.CustomShader) == false)
            {
                // The updated user data is not for custom shader, bail out.
                return;
            }

            this.UpdateUserData(false);
        }

        /// <summary>
        /// Helper method for updating user data.
        /// </summary>
        /// <param name="onlyUpdateViewModels">
        /// True to update only the view models for to the existing user data models.
        /// Otherwise, remove all the user data models and update them.
        /// </param>
        private void UpdateUserData(bool onlyUpdateViewModels)
        {
            // First remove all the user page view models.
            var viewModels = this.Children.Where(
                ch => ch is EmitterCustomShaderSettingDataViewModel).ToArray();
            foreach (var vm in viewModels)
            {
                this.Children.Remove(vm);
            }

            if (onlyUpdateViewModels == false)
            {
                // Update user data models.
                (this.Proxy as EmitterCustomShaderDataProxy).UpdateUserData();
            }

            // Create user page view models with the new user data models.
            int index = 0;
            foreach (EmitterCustomShaderSettingData dm in this.DataModel.Settings)
            {
                var vm = new EmitterCustomShaderSettingDataViewModel(this, dm, index);
                this.Children.Add(vm);

                // Advance the index.
                ++index;
            }

            // Deselect custom action.
            if (this.SelectedSettingIndex >= index)
            {
                this.SelectedSettingIndex = -1;
            }

            // Clear modification flags.
            if (this.ModificationFlagViewModel != null)
            {
                this.ModificationFlagViewModel.ClearModificationFlags();
                this.ModificationFlagViewModel.ClearChildModificationFlags();
            }
        }
    }
}
