﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using System;
using System.Collections.Generic;
using System.IO;
using EffectMaker.BusinessLogic.EffectCombinerEditor;
using EffectMaker.BusinessLogic.IO;
using EffectMaker.BusinessLogic.Options;
using EffectMaker.DataModel.AnimationTable;
using EffectMaker.DataModel.DataModels;
using EffectMaker.DataModel.Specific.DataModels;
using EffectMaker.DataModelLogic.Utilities;
using EffectMaker.Foundation.Attributes;
using EffectMaker.Foundation.Editting;
using EffectMaker.Foundation.Extensions;
using EffectMaker.Foundation.Input;
using EffectMaker.Foundation.Interfaces;
using EffectMaker.Foundation.Primitives;
using EffectMaker.UILogic.Attributes;

namespace EffectMaker.UILogic.ViewModels
{
    /// <summary>
    /// Class for the view model of the EmitterCombinerEditorData.
    /// </summary>
    public class EmitterCombinerEditorViewModel : PropertyGroupViewModel<EmitterCombinerEditorData>, IModificationFlagOwner
    {
        /// <summary>
        /// 変更の対象に含めないプロパティ名のリストです.
        /// </summary>
        private readonly string[] ignorePropertyNames = new string[]
        {
            "IsEffectCombinerEnabled",
            "EmitterCombinerEditorAnimationCurveViewModel0",
            "EmitterCombinerEditorAnimationCurveData0",
        };

        /// <summary>
        /// CombinerEditorProjectPath設定時に、バイナリを転送するかどうか
        /// </summary>
        private bool isSendingBinary = true;

        /// <summary>
        /// 親のビューモデル
        /// </summary>
        private EmitterCombinerViewModel parentViewModel = null;

        /// <summary>
        /// The constructor.
        /// </summary>
        /// <param name="parent">The parent view model.</param>
        /// <param name="dataModel">The data model to encapsulate.</param>
        public EmitterCombinerEditorViewModel(
            HierarchyViewModel parent,
            EmitterCombinerEditorData dataModel)
            : base(parent, dataModel)
        {
            this.parentViewModel = (EmitterCombinerViewModel)this.Parent;

            this.OnReloadCombinerProjectExecutable = new AnonymousExecutable(this.OnReloadCombinerProject);

            var combiner = EffectCombinerCommunicationManager.CommunicationBridge;
            if (combiner != null)
            {
                combiner.ShaderChanged += this.OnShaderChanged;
            }

            this.EmitterCombinerEditorAnimationCurveViewModel0 = new EmitterCombinerEditorAnimationCurveViewModel(
                this, dataModel.EmitterCombinerEditorAnimationCurveData0);
            this.Children.Add(this.EmitterCombinerEditorAnimationCurveViewModel0);

            this.RaisePropertyChangedForGroupVisibility();

            // Always create the modification flag view model IN THE END of the constructor
            // to prevent any initialization triggers the modification events.
            this.ModificationFlagViewModel = new ModificationFlagViewModel(this);
            this.ModificationFlagViewModel.AddIgnoreProperties(this.ignorePropertyNames);
        }

        /// <summary>
        /// EmitterCombinerEditorAnimationCurveViewModel.
        /// </summary>
        public EmitterCombinerEditorAnimationCurveViewModel EmitterCombinerEditorAnimationCurveViewModel0
        {
            get; private set;
        }

        /// <summary>
        /// Get the view model that holds the modification flags of
        /// this view model's properties.
        /// </summary>
        public ModificationFlagViewModel ModificationFlagViewModel { get; private set; }

        /// <summary>
        /// Get the executable for reloading combiner project.
        /// </summary>
        public IExecutable OnReloadCombinerProjectExecutable { get; private set; }

        /// <summary>
        /// Get the flag indicating whether the effect combiner feature is enabled.
        /// </summary>
        public bool IsEffectCombinerEnabled
        {
            get { return OptionStore.ProjectConfig.IsEftCombinerEditorEnabled; }
        }

        /// <summary>
        /// Get or set the shader combiner editor project path.
        /// </summary>
        public string CombinerEditorProjectPath
        {
            get
            {
                return this.GetDataModelValue(() => this.CombinerEditorProjectPath);
            }

            set
            {
                // コンバイナシェーダのxmlファイルであるかチェックする.
                if (this.CheckCombinerShader(value) == false)
                {
                    bool existing = File.Exists(value);
                    var emitterViewModel = this.FindNearestParentOfType<EmitterViewModel>();
                    emitterViewModel.UpdateAssetStatus(!existing, existing);
                    return;
                }

                if (this.GetDataModelValue(() => this.CombinerEditorProjectPath) != value)
                {
                    this.SetDataModelValue(value, () => this.CombinerEditorProjectPath);
                    if (!string.IsNullOrEmpty(value))
                    {
                        var combiner = EffectCombinerCommunicationManager.CommunicationBridge;
                        if (combiner != null)
                        {
                            combiner.DiscardCache(value);
                        }
                    }

                    this.SendEmitterSetBinary();
                }

                // プロパティ変更通知.
                this.RaisePropertyChangedForGroupVisibility();
            }
        }

        /// <summary>
        /// オブジェクトが破棄される時にシェーダ変更イベントハンドラを削除します。
        /// </summary>
        public override void Dispose()
        {
            var combiner = EffectCombinerCommunicationManager.CommunicationBridge;
            if (combiner != null)
            {
                combiner.ShaderChanged -= this.OnShaderChanged;
            }

            base.Dispose();
        }

        /// <summary>
        /// グループ表示状態の更新を反映
        /// </summary>
        public override void FirePropertyChanges()
        {
            this.RaisePropertyChangedForGroupVisibility();

            base.FirePropertyChanges();
        }

        /// <summary>
        /// CombinerEditorProjectPathを更新する。
        /// 更新時に、エミッタセットのバイナリをビューアに転送するか指定できる
        /// </summary>
        /// <param name="filePath">コンバイナシェーダのパス</param>
        /// <param name="sendBinary">エミッタセットのバイナリをビューアに転送するか</param>
        public void SetCombinerEditorProjectPath(string filePath, bool sendBinary)
        {
            this.isSendingBinary = sendBinary;

            // コンバイナシェーダのパスを更新
            this.CombinerEditorProjectPath = filePath;

            // 常にエミッタセットのバイナリを転送するようにしておく
            this.isSendingBinary = true;
        }

        /// <summary>
        /// オプション変更でコンバイナエディタ関連のUI表示状態を更新する
        /// </summary>
        /// <param name="sender">使用しない</param>
        /// <param name="e">使用しない</param>
        protected override void OnOptionChanged(object sender, EventArgs e)
        {
            base.OnOptionChanged(sender, e);
            this.OnPropertyChanged(() => this.IsEffectCombinerEnabled);
            this.RaisePropertyChangedForGroupVisibility();
        }

        /// <summary>
        /// コンバイナシェーダが有効か否かに応じてグループの表示状態を切り替えます
        /// </summary>
        private void RaisePropertyChangedForGroupVisibility()
        {
            bool enableCombinerShader = !string.IsNullOrEmpty(this.CombinerEditorProjectPath);
            this.parentViewModel.EnableParticleShaderGroup = !enableCombinerShader;

            // カスタムシェーダで使いたいという要望があったため、暫定措置として常に true
//            this.EmitterCombinerEditorAnimationCurveViewModel0.GroupVisibility = enableCombinerShader;
            this.EmitterCombinerEditorAnimationCurveViewModel0.GroupVisibility = true;
        }

        /// <summary>
        /// Handle reloading combiner project.
        /// </summary>
        /// <param name="parameter">Custom parameter</param>
        private void OnReloadCombinerProject(object parameter)
        {
            // キャッシュを破棄してファイルから確実にリロードされるようにする
            if (!string.IsNullOrEmpty(this.CombinerEditorProjectPath))
            {
                var combiner = EffectCombinerCommunicationManager.CommunicationBridge;
                if (combiner != null)
                {
                    combiner.DiscardCache(this.CombinerEditorProjectPath);
                }
            }

            // エミッタのリロード
            this.SendEmitterSetBinary();
        }

        /// <summary>
        /// Helper method to send emitter set binary to the viewer.
        /// </summary>
        private void SendEmitterSetBinary()
        {
            if (!this.isSendingBinary)
            {
                return;
            }

            using (new ForceRefreshBinary())
            {
                var dataModel = this.DataModel.Parent;
                ViewerMessageHelper.FindPropertyParentDataAndSend(dataModel);
            }
        }

        /// <summary>
        /// シェーダが送信されてきたらリロードします。
        /// </summary>
        /// <param name="sender">nullが飛んできます。</param>
        /// <param name="e">受信したシェーダの情報が入ります。</param>
        private void OnShaderChanged(object sender, EffectCombinerShaderChangedEventArgs e)
        {
            if (e.ProjectPath == this.CombinerEditorProjectPath)
            {
                this.SendEmitterSetBinary();
            }
        }

        /// <summary>
        /// 指定されたファイルが、コンバイナシェーダであるか確かめます
        /// </summary>
        /// <param name="filePath">コンバイナシェーダのファイルパス</param>
        /// <returns>正常なコンバイナシェーダだったら、true</returns>
        private bool CheckCombinerShader(string filePath)
        {
            // 空っぽだったら、trueを返してしまう.
            if (string.IsNullOrEmpty(filePath) == true)
            {
                return true;
            }

            // ルートノードが、"project"になっているか
            if (!IOConstants.CheckRootElement(filePath, "project"))
            {
                return false;
            }

            return true;
        }
    }
}
