﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using System.Threading.Tasks;
using EffectMaker.Foundation.Attributes;
using EffectMaker.Foundation.Collections.Generic;
using EffectMaker.Foundation.Command;
using EffectMaker.Foundation.Interfaces;
using EffectMaker.UILogic.ViewModels;

namespace EffectMaker.UILogic.Commands
{
    /// <summary>
    /// SwapNodesCommand
    /// </summary>
    [Alias("SwapNodes")]
    public class SwapNodesCommand : CommandBase
    {
        /// <summary>
        /// Stores the command stack target.
        /// </summary>
        private object commandStackTarget;

        /// <summary>
        /// Stores the parent node in which a child is swapped.
        /// </summary>
        private WorkspaceNodeViewModelBase parentNode;

        /// <summary>
        /// Store the child node that is being swapped.
        /// </summary>
        private WorkspaceNodeViewModelBase sourceChildNode;

        /// <summary>
        /// Stores the index where the child is before swapping.
        /// </summary>
        private int sourceIndex;

        /// <summary>
        /// Stores the index there the child is after swapping.
        /// </summary>
        private int targetIndex;

        /// <summary>
        /// コンストラクタです。
        /// </summary>
        public SwapNodesCommand()
        {
        }

        /// <summary>
        /// コンストラクタです。
        /// </summary>
        /// <param name="parentNode">親ノード</param>
        /// <param name="sourceIndex">交換元のインデックス</param>
        /// <param name="targetIndex">交換先のインデックス</param>
        public SwapNodesCommand(WorkspaceNodeViewModelBase parentNode, int sourceIndex, int targetIndex)
        {
            object[] args = new object[] { parentNode, sourceIndex, targetIndex };

            this.Initialize(args);
        }

        /// <summary>
        /// Initializes the SwapNodesCommand instance.
        /// </summary>
        /// <param name="args">Command arguments.</param>
        /// <returns>Returns true on success, false on error.</returns>
        public override bool Initialize(object[] args)
        {
            if (args == null || args.Length != 3)
            {
                return false;
            }

            this.parentNode = args[0] as WorkspaceNodeViewModelBase;
            if (this.parentNode == null)
            {
                return false;
            }

            if ((args[1] is int) == false || (args[2] is int) == false)
            {
                return false;
            }

            this.sourceIndex = (int)args[1];
            this.targetIndex = (int)args[2];

            this.sourceChildNode =
                this.parentNode.Children[this.sourceIndex] as WorkspaceNodeViewModelBase;

            if (this.sourceChildNode == null)
            {
                return false;
            }

            this.commandStackTarget =
                CommandBase.FindCommandStackTargetFromHierarchyObject(this.parentNode);

            this.IsInitialized = true;
            return true;
        }

        /// <summary>
        /// Get the command's target.
        /// </summary>
        /// <returns>Return the command's target.</returns>
        public override object GetTarget()
        {
            return this.commandStackTarget;
        }

        /// <summary>
        /// Performs the node swapping.
        /// </summary>
        /// <returns>Returns true on success, false on error.</returns>
        public override bool Do()
        {
            try
            {
                using (new SuppressNodeTypeOrdering(this.parentNode))
                {
                    // Move the data model.
                    this.parentNode.Proxy.MoveChildDataModel(
                        this.sourceChildNode.Proxy.DataModel,
                        this.targetIndex - this.sourceIndex);

                    if (this.sourceChildNode is PreviewViewModel)
                    {
                        ModificationFlagViewModel.IgnoreParentPropertyChangedEvents = true;
                    }

                    // Move the data model in the child list.
                    this.parentNode.Children.RemoveAt(this.sourceIndex);
                    this.parentNode.Children.Insert(this.targetIndex, this.sourceChildNode);

                    if (this.sourceChildNode is PreviewViewModel)
                    {
                        ModificationFlagViewModel.IgnoreParentPropertyChangedEvents = false;
                    }
                }

                // force reset selection
                this.sourceChildNode.IsSelected = false;
                this.sourceChildNode.IsSelected = true;

                return true;
            }
            catch
            {
                return false;
            }
        }

        /// <summary>
        /// Performs the back swpping.
        /// </summary>
        /// <returns>Returns true on success, false on error.</returns>
        public override bool Undo()
        {
            try
            {
                using (new SuppressNodeTypeOrdering(this.parentNode))
                {
                    // Move the data model in its own list.
                    this.parentNode.Proxy.MoveChildDataModel(
                        this.sourceChildNode.Proxy.DataModel,
                        this.sourceIndex - this.targetIndex);

                    if (this.sourceChildNode is PreviewViewModel)
                    {
                        ModificationFlagViewModel.IgnoreParentPropertyChangedEvents = true;
                    }

                    // Move the data model in the child list.
                    this.parentNode.Children.RemoveAt(this.targetIndex);
                    this.parentNode.Children.Insert(this.sourceIndex, this.sourceChildNode);

                    if (this.sourceChildNode is PreviewViewModel)
                    {
                        ModificationFlagViewModel.IgnoreParentPropertyChangedEvents = false;
                    }
                }

                // force reset selection
                this.sourceChildNode.IsSelected = false;
                this.sourceChildNode.IsSelected = true;

                return true;
            }
            catch
            {
                return false;
            }
        }
    }
}
