﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using System.Threading.Tasks;
using EffectMaker.DataModelLogic.Utilities;
using EffectMaker.Foundation.Attributes;
using EffectMaker.Foundation.Command;
using EffectMaker.Foundation.Dynamic;
using EffectMaker.Foundation.Extensions;
using EffectMaker.Foundation.Interfaces;
using EffectMaker.Foundation.Log;
using EffectMaker.UILogic.ViewModels;

namespace EffectMaker.UILogic.Commands
{
    /// <summary>
    /// PasteGroupCommand
    /// </summary>
    [Alias("PasteGroup")]
    public class PasteGroupCommand : CommandBase
    {
        /// <summary>
        /// Stores the PropertyGroupViewModel instance.
        /// </summary>
        private PropertyGroupViewModel viewModel;

        /// <summary>
        /// Stores the command stack target.
        /// </summary>
        private object commandStackTarget;

        /// <summary>
        /// Stores the old values of the group.
        /// </summary>
        private string oldValues;

        /// <summary>
        /// Stores the new values of the group.
        /// </summary>
        private string newValues;

        /// <summary>
        /// 複数選択時の編集操作をアクションリストで保持します。
        /// </summary>
        private List<Action> multiUndoActions, multiRedoActions;

        /// <summary>
        /// コンストラクタ
        /// </summary>
        public PasteGroupCommand()
        {
            this.multiUndoActions = new List<Action>();
            this.multiRedoActions = new List<Action>();
        }

        /// <summary>
        /// コンストラクタ
        /// </summary>
        /// <param name="viewModel">ビューモデル</param>
        /// <param name="oldValues">元の値</param>
        /// <param name="newValues">新しい値</param>
        public PasteGroupCommand(PropertyGroupViewModel viewModel, string oldValues, string newValues)
            : this()
        {
            object[] args = new object[] { viewModel, oldValues, newValues };

            this.Initialize(args);
        }

        /// <summary>
        /// Initializes the PasteGroupCommand instance.
        /// </summary>
        /// <param name="args">Arguments of the command.</param>
        /// <returns>Returns true on sucess, false otherwise.</returns>
        public override bool Initialize(object[] args)
        {
            if (args == null || args.Length != 3)
            {
                return false;
            }

            this.viewModel = args[0] as PropertyGroupViewModel;
            if (this.viewModel == null)
            {
                return false;
            }

            if ((args[1] is string) == false || (args[2] is string) == false)
            {
                return false;
            }

            this.oldValues = (string)args[1];
            this.newValues = (string)args[2];

            if (string.IsNullOrWhiteSpace(this.oldValues) ||
                string.IsNullOrWhiteSpace(this.newValues))
            {
                return false;
            }

            this.commandStackTarget =
                CommandBase.FindCommandStackTargetFromHierarchyObject(this.viewModel);

            this.IsInitialized = true;
            return true;
        }

        /// <summary>
        /// Gets the command's target.
        /// </summary>
        /// <returns>Returns the command's target.</returns>
        public override object GetTarget()
        {
            return this.commandStackTarget;
        }

        /// <summary>
        /// Sets the new values to the group.
        /// </summary>
        /// <returns>Returns true on success, false otherwise.</returns>
        public override bool Do()
        {
            // 複数選択対象の同じプロパティに対して同じ値をセットする
            this.multiUndoActions.Clear();
            this.multiRedoActions.Clear();
            foreach (var node in WorkspaceRootViewModel.Instance.MultiSelectedNodes)
            {
                var sameTarget = HierarchyObjectExtensions.FindChildrenRecursive(
                    this.viewModel,
                    node,
                    MultiNodeEditUtil.IsSameTarget) as HierarchyViewModel;
                if (sameTarget != null)
                {
                    string currentValues = sameTarget.ExportValuesAsXml();
                    Action undoAction = () => PasteProc(sameTarget, currentValues);
                    this.multiUndoActions.Add(undoAction);

                    Action redoAction = () => PasteProc(sameTarget, this.newValues);
                    this.multiRedoActions.Add(redoAction);
                    redoAction();
                }
            }

            bool result = PasteProc(this.viewModel, this.newValues);
            if (result)
            {
                // メニューの状態を更新
                WorkspaceRootViewModel.Instance.UpdateUIStates();

                // Send the whole binary again.
                using (new ForceRefreshBinary())
                {
                    ViewerMessageHelper.FindPropertyParentDataAndSend(this.viewModel.Proxy.DataModel);
                }
            }

            return result;
        }

        /// <summary>
        /// Reset the old values of the group.
        /// </summary>
        /// <returns>Returns true on success, false otherwise.</returns>
        public override bool Undo()
        {
            // アクションリストに保存された同時編集をUndo
            foreach (var editAction in this.multiUndoActions)
            {
                try
                {
                    editAction();
                }
                catch (Exception e)
                {
                    // エミッタセットを削除された場合には対応しきれないのでログだけ吐いておく
                    Logger.Log(LogLevels.Error, "Multi Edit Undo Error:" + e.Message);
                }
            }

            bool result = PasteProc(this.viewModel, this.oldValues);
            if (result)
            {
                // メニューの状態を更新
                WorkspaceRootViewModel.Instance.UpdateUIStates();

                // Send the whole binary again.
                using (new ForceRefreshBinary())
                {
                    ViewerMessageHelper.FindPropertyParentDataAndSend(this.viewModel.Proxy.DataModel);
                }
            }

            return result;
        }

        /// <summary>
        /// Reset the new values of the group.
        /// </summary>
        /// <returns>Returns true on success, false otherwise.</returns>
        public override bool Redo()
        {
            // アクションリストに保存された同時編集をUndo
            foreach (var editAction in this.multiRedoActions)
            {
                try
                {
                    editAction();
                }
                catch (Exception e)
                {
                    // エミッタセットを削除された場合には対応しきれないのでログだけ吐いておく
                    Logger.Log(LogLevels.Error, "Multi Edit Redo Error:" + e.Message);
                }
            }

            bool result = PasteProc(this.viewModel, this.newValues);
            if (result)
            {
                // メニューの状態を更新
                WorkspaceRootViewModel.Instance.UpdateUIStates();

                // Send the whole binary again.
                using (new ForceRefreshBinary())
                {
                    ViewerMessageHelper.FindPropertyParentDataAndSend(this.viewModel.Proxy.DataModel);
                }
            }

            return result;
        }

        /// <summary>
        /// ペースト処理のコアです。
        /// </summary>
        /// <param name="target">ペースト対象のグループビューモデル</param>
        /// <param name="values">値</param>
        /// <returns>成功したらtrue,失敗したらfalse.</returns>
        private static bool PasteProc(HierarchyViewModel target, string values)
        {
            try
            {
                ExportableViewModel.IsPasting = true;

                target.ImportValuesFromXml(values);

                // 全てのプロパティへPropertyChanged更新イベントを発行.
                target.FirePropertyChanges();

                ExportableViewModel.IsPasting = false;

                // カーブエディタウィンドウの更新
                WorkspaceRootViewModel.Instance.UpdateCurveEditorWindow(target);

                return true;
            }
            catch
            {
                ExportableViewModel.IsPasting = false;
                return false;
            }
        }
    }
}
