﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using System;
using System.IO;
using EffectMaker.BusinessLogic.DataModelOperation;
using EffectMaker.BusinessLogic.Options;
using EffectMaker.BusinessLogic.UserData;
using EffectMaker.DataModel.DataModels;
using EffectMaker.DataModel.Specific.DataModels;
using EffectMaker.DataModelLogic.DataModelProxies;
using EffectMaker.Foundation.Attributes;
using EffectMaker.Foundation.Command;
using EffectMaker.Foundation.Interfaces;
using EffectMaker.Foundation.Log;
using EffectMaker.UILogic.Commands;
using EffectMaker.UILogic.Properties;
using EffectMaker.UILogic.ViewModels;

namespace EffectMaker.BusinessLogic.Commands
{
    /// <summary>
    /// Command to create reserved shader node to an emitter.
    /// </summary>
    [Alias("CreateReservedShader")]
    public class CreateReservedShaderCommand : CommandBase
    {
        /// <summary>The emitter view model to create reserved shader for.</summary>
        private EmitterViewModel parentViewModel = null;

        /// <summary>The created reserved shader node view model.</summary>
        private ReservedShaderNodeViewModel reservedShaderViewModel = null;

        /// <summary>The command stack target.</summary>
        private IHierarchyObject commandStackTarget = null;

        /// <summary>The user data info of the reserved shader to create.</summary>
        private UserDataInfo reservedShaderUserDataInfo = null;

        /// <summary>以前選択していたエミッタの処理タイプ</summary>
        private int previousUsingCpu = -1;

        /// <summary>
        /// Constructor.
        /// </summary>
        public CreateReservedShaderCommand()
        {
        }

        /// <summary>
        /// コンストラクタです.
        /// </summary>
        /// <param name="parentViewModel">親ビューモデル</param>
        /// <param name="userDataInfo">シェーダユーザーデータ情報</param>
        /// <param name="insertIndex">挿入するインデックス</param>
        public CreateReservedShaderCommand(HierarchyViewModel parentViewModel, UserDataInfo userDataInfo)
        {
            object[] args = new object[] { parentViewModel, userDataInfo };

            this.Initialize(args);
        }

        /// <summary>
        /// Initialize the command and process the arguments.
        /// </summary>
        /// <param name="args">
        /// The arguments for the command.
        /// [0] : The emitter view model to remove the node from.
        /// [1] : The user data info for the reserved shader.
        /// </param>
        /// <returns>True on success.</returns>
        public override bool Initialize(object[] args)
        {
            if (args == null || args.Length != 2)
            {
                return false;
            }

            // Check if there are any reserved shader definitions loaded.
            if (ReservedShaderUserDataManager.Definition == null)
            {
                Logger.Log("LogView", LogLevels.Warning, Resources.WarningAbortCreateReservedShader);
                return false;
            }

            // Retrieve and save the emitter view model.
            if ((args[0] is EmitterViewModel) == false)
            {
                return false;
            }

            this.parentViewModel = (EmitterViewModel)args[0];

            // Retrieve and save the reserved shader user data info.
            if ((args[1] is UserDataInfo) == false)
            {
                return false;
            }

            this.reservedShaderUserDataInfo = (UserDataInfo)args[1];

            // Find the target command stack by looping through the parents of the emitter
            // and find the first parent who has CommandStackOwnerAttribute defined.
            var cmdStackTarget = this.parentViewModel as IHierarchyObject;
            this.commandStackTarget = CommandBase.FindCommandStackTargetFromHierarchyObject(cmdStackTarget) as IHierarchyObject;

            this.IsInitialized = true;
            return true;
        }

        /// <summary>
        /// Get the target command stack.
        /// </summary>
        /// <returns>The target command stack.</returns>
        public override object GetTarget()
        {
            return this.commandStackTarget;
        }

        /// <summary>
        /// Execute the command.
        /// </summary>
        /// <returns>True on success.</returns>
        public override bool Do()
        {
            // using構文内でエミッタセットの転送を抑制し、ディスポーザーでエミッタセットの転送を行います。
            using (new MessageBlockerWithSendBinaryOnce(this.parentViewModel.DataModel))
            {
                // Create the reserved shader node view model.
                this.reservedShaderViewModel =
                    this.parentViewModel.CreateNewReservedShader(this.reservedShaderUserDataInfo);
                if (this.reservedShaderViewModel == null)
                {
                    return false;
                }

                // Add the reserved shader to the emitter.
                if (this.parentViewModel.AddReservedShader(this.reservedShaderViewModel) == false)
                {
                    return false;
                }

                // Select the created reserved shader node.
                this.reservedShaderViewModel.IsSelected = true;

                // Update UI states.
                this.previousUsingCpu =
                    this.parentViewModel.EmitterBasicViewModel.EmitterBasicBasicViewModel.RaiseCanUsingGpu();
                WorkspaceRootViewModel.Instance.UpdateUIStates();
            }

            return true;
        }

        /// <summary>
        /// Undo the command.
        /// </summary>
        /// <returns>True on success.</returns>
        public override bool Undo()
        {
            // using構文内でエミッタセットの転送を抑制し、ディスポーザーでエミッタセットの転送を行います。
            using (new MessageBlockerWithSendBinaryOnce(this.parentViewModel.DataModel))
            {
                // If the reserved shader node is currently selected, select the parent emitter first.
                // (to prevent the reserved shader page from staying on the property panel)
                // (this is necessary before deleting the reserved shader node)
                this.parentViewModel.IsSelected = true;

                // Remove the reserved shader node.
                if (this.parentViewModel.RemoveReservedShader(this.reservedShaderViewModel) == false)
                {
                    return false;
                }

                // Update UI states.
                this.parentViewModel.EmitterBasicViewModel.EmitterBasicBasicViewModel.RaiseCanUsingGpu(
                    this.previousUsingCpu);
                WorkspaceRootViewModel.Instance.UpdateUIStates();
            }

            return true;
        }

        /// <summary>
        /// Redo the command.
        /// </summary>
        /// <returns>True on success.</returns>
        public override bool Redo()
        {
            // using構文内でエミッタセットの転送を抑制し、ディスポーザーでエミッタセットの転送を行います。
            using (new MessageBlockerWithSendBinaryOnce(this.parentViewModel.DataModel))
            {
                // Add the created reserved shader node back to the emitter.
                if (this.parentViewModel.AddReservedShader(this.reservedShaderViewModel) == false)
                {
                    return false;
                }

                // Update UI states.
                this.previousUsingCpu =
                    this.parentViewModel.EmitterBasicViewModel.EmitterBasicBasicViewModel.RaiseCanUsingGpu();

                OptionStore.TriggerOptionChangedEvent(null);
                WorkspaceRootViewModel.Instance.UpdateUIStates();
            }

            return true;
        }
    }
}
