﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using System;
using System.Collections.Generic;
using System.IO;
using System.Linq;
using System.Text;
using System.Text.RegularExpressions;
using System.Xml;
using EffectMaker.BusinessLogic.BinaryHeaders;
using EffectMaker.BusinessLogic.BinaryHeaders.Helpers;
using EffectMaker.BusinessLogic.GfxToolsUtility;
using EffectMaker.BusinessLogic.IO;
using EffectMaker.BusinessLogic.Options;
using EffectMaker.BusinessLogic.ProjectConfig;
using EffectMaker.BusinessLogic.SpecDefinitions;
using EffectMaker.DataModel.Specific.DataModels;
using EffectMaker.DataModelLogic.Utilities;
using EffectMaker.Foundation.Debugging.Profiling;
using EffectMaker.Foundation.Extensions;
using EffectMaker.Foundation.Log;
using EffectMaker.Foundation.Utility;
using EffectMaker.ShaderCodeGeneratorGeneric;

using CompileShaderCompleteCallback = System.Action<System.Collections.Generic.List<long>, System.Action<bool>, EffectMaker.Foundation.Debugging.Profiling.ProfileTimer, EffectMaker.Foundation.Debugging.Profiling.ProfileTimer, bool, bool, byte[], long, int, long, int, System.Action<System.IO.Stream>>;

namespace EffectMaker.SpecGeneric.Shader
{
    /// <summary>
    /// Genericスペック向けのシェーダコンバートを管理するクラスです。
    /// Vfxベースのスペック向けの処理を集約します。
    /// </summary>
    public partial class ShaderManager
    {
        /// <summary>
        /// バリエーションファイルを生成するときに使用する、ShaderKeyから抽出するマクロ名
        /// </summary>
        private static List<string> _macroNames = null;

        /// <summary>
        /// ComputeShaderのバリエーションファイルを生成するときに使用するコンスタント変数名とそのデリゲート
        /// </summary>
        private static List<Tuple<string, Func<EmitterData, bool>>> _computeShaderVariationConstantNameSets = null;

        /// <summary>ShaderConverter.exeで利用するファイルパス</summary>
        private static readonly string ShaderIntermediateDirectoryPath = ShaderConverterExecuter.Instance.IntermediateDirectoryPath;
        private static readonly string VertexShaderPath = Path.Combine(ShaderIntermediateDirectoryPath, "shader.vsh");
        private static readonly string PixelShaderPath = Path.Combine(ShaderIntermediateDirectoryPath, "shader.psh");
        private static readonly string IncludeDirectoryPath = Path.Combine(IOConstants.ExecutableFolderPath, @"Converter\shader");
        private static readonly string VariationPath = Path.Combine(ShaderIntermediateDirectoryPath, "shaderVariation.xml");
        private static readonly string OutputPath = Path.Combine(ShaderIntermediateDirectoryPath, "output.grsf");

        /// <summary>ShaderConverter.exeがコンピュートシェーダのシェーダバイナリを生成するときに使用するファイルパス</summary>
        private static readonly string ComputeShaderPath = Path.Combine(ShaderIntermediateDirectoryPath, "computeShader.vsh");
        private static readonly string ComputeShaderVariationPath = Path.Combine(ShaderIntermediateDirectoryPath, "computeShaderVariation.xml");
        private static readonly string ComputeShaderOutputPath = Path.Combine(ShaderIntermediateDirectoryPath, "computeShaderOutput.grsf");

        /// <summary>ShaderConverter.exeによるソースコードのdumpなどに使用されるファイルパス</summary>
        private static readonly string ShaderSourceDumpDirectoryPath = ShaderConverterExecuter.Instance.IntermediateDirectoryPath;

        /// <summary>
        /// バーテックスシェーダのダンプ結果出力パスを取得します。
        /// </summary>
        private static string VertexShaderDumpPath
        {
            get
            {
                bool isBin = SpecManager.CurrentSpec.ShaderConversionOption.CodeType.StartsWith("Binary");
                return Path.Combine(
                    ShaderSourceDumpDirectoryPath,
                    string.Format("0_vs_{0}.glsl", isBin ? "bin" : "src"));
            }
        }

        /// <summary>
        /// ピクセルシェーダのダンプ結果出力パスを取得します。
        /// </summary>
        private static string PixelShaderDumpPath
        {
            get
            {
                bool isBin = SpecManager.CurrentSpec.ShaderConversionOption.CodeType.StartsWith("Binary");
                return Path.Combine(
                    ShaderSourceDumpDirectoryPath,
                    string.Format("0_ps_{0}.glsl", isBin ? "bin" : "src"));
            }
        }


       /// <summary>
        /// コンピュートシェーダのダンプ結果出力パスを取得します。
        /// </summary>
        private static string ComputeShaderDumpPath
        {
            get
            {
                bool isBin = SpecManager.CurrentSpec.ShaderConversionOption.CodeType.StartsWith("Binary");
                return Path.Combine(
                    ShaderSourceDumpDirectoryPath,
                    string.Format("0_cs_{0}.glsl", isBin ? "bin" : "src"));
            }
        }

        /// <summary>
        /// バリエーションファイルを生成するときに使用する、ShaderKeyから抽出するマクロ名を取得します。
        /// </summary>
        private static List<string> MacroNames
        {
            get
            {
                if (_macroNames != null)
                {
                    return _macroNames;
                }

                _macroNames = new List<string>()
                {
                    "VFX_USE_INSTANS_DRAW",
                    "VFX_USE_REGISTER_DRAW",
                    "_USE_NORMAL_ATTR",
                    "_USE_TANGENT_ATTR",
                    "_USE_COLOR0_ATTR",
                    "_USE_TEXCOORD_ATTR",
                    "_CALC_TYPE_CPU",
                    "_CALC_TYPE_GPU_TIME",
                    "_CALC_TYPE_GPU_SO",
                    "_PARTICLE_TYPE_BILLBOARD",
                    "_PARTICLE_TYPE_COMPLEX_BILLBOARD",
                    "_PARTICLE_TYPE_Y_BILLBOARD",
                    "_PARTICLE_TYPE_Y_BILLBOARD_PARALLEL",
                    "_PARTICLE_TYPE_WITH_SCALE_Z",
                    "_PARTICLE_TYPE_WITH_SCALE_Z_DIRECTIONAL",
                    "_PARTICLE_TYPE_POLYGON_XY",
                    "_PARTICLE_TYPE_POLYGON_XZ",
                    "_PARTICLE_TYPE_VEL_LOOK",
                    "_PARTICLE_TYPE_VEL_LOOK_POLYGON",
                    "_EMITTER_FOLLOW_TYPE_ALL",
                    "_EMITTER_FOLLOW_TYPE_NONE",
                    "_EMITTER_FOLLOW_TYPE_POS",
                    "_FLUCTUATION_SCALE_ENABLE",
                    "_FLUCTUATION_ALPHA_ENABLE",
                    "_FLUCTUATION_SCALE_Y_SEPARATE",
                    "_COLOR_0_FIXED",
                    "_COLOR_0_RANDOM",
                    "_COLOR_1_FIXED",
                    "_COLOR_1_RANDOM",
                    "_ALPHA_0_FIXED",
                    "_ALPHA_1_FIXED",
                    "_TEX_0_SHIFT_ANIM_ROTATE",
                    "_TEX_0_SHIFT_ANIM",
                    "_TEX_1_SHIFT_ANIM_ROTATE",
                    "_TEX_1_SHIFT_ANIM",
                    "_TEX_2_SHIFT_ANIM_ROTATE",
                    "_TEX_2_SHIFT_ANIM",
                    "_TEX_0_PATTERN_ANIM",
                    "_TEX_1_PATTERN_ANIM",
                    "_TEX_2_PATTERN_ANIM",
                    "_TEX_0_SPHERE_MAP",
                    "_TEX_1_SPHERE_MAP",
                    "_TEX_2_SPHERE_MAP",
                    "_USE_ROTATE",
                    "_ROTATE_YZX",
                    "_ROTATE_XYZ",
                    "_ROTATE_ZXY",
                    "_NEAR_DIST_ALPHA",
                    "_FAR_DIST_ALPHA",
                    "_MASKING_CHECK",
                    "_VERTEX_SOFT",
                    "_VERTEX_FRESNEL_ALPHA",
                    "_PARTICLE",
                    "_PRIMITIVE",
                    "_PARTICLE_SCALE_LIMIT_NEAR",
                    "_PARTICLE_SCALE_LIMIT_FAR",
                    "_LINK_FOVY_TO_SCALE_LIMIT",
                    "_AVOID_Z_FIGHTING",
                    "_SHADER_ANIM",
                    "_COLOR0_INHERIT",
                    "_COLOR1_INHERIT",
                    "_ALPHA0_INHERIT",
                    "_ALPHA1_INHERIT",
                    "_CONNECT_PTCL_SCALE_TO_Z_OFFSET",
                    "_VELOCITY_SCALE_Y",
                    "_DEPTH_OFFSET",
                    "_CAMERA_OFFSET",
                    "_CAMERA_OFFSET_FIXED_SIZE",
                    "_WORLD_GRAVITY",
                    "_FIELD_RANDOM",
                    "_TEX0_KEEP_TEXTURE_SIZE_U",
                    "_TEX1_KEEP_TEXTURE_SIZE_U",
                    "_TEX2_KEEP_TEXTURE_SIZE_U",
                    "_TEX0_KEEP_TEXTURE_SIZE_V",
                    "_TEX1_KEEP_TEXTURE_SIZE_V",
                    "_TEX2_KEEP_TEXTURE_SIZE_V",
                    "_STRIPE_BILLBOARD",
                    "_STRIPE_EMITTER_MATRIX",
                    "_STRIPE_EMITTER_UPDOWN",
                    "_STRIPE_FOLLOW_EMITTER_MATRIX",
                    "_STRIPE_RIBBON",
                    "_SHADER_TYPE_NORMAL",
                    "_SHADER_TYPE_REFRACT",
                    "_SHADER_TYPE_DISTORTION",
                    "_COLOR_PROCESS_COLOR",
                    "_COLOR_PROCESS_TEXTURE",
                    "_COLOR_PROCESS_TEXTURE_INTERPOLATE",
                    "_COLOR_PROCESS_TEXTURE_ADD",
                    "_ALPHA_PROCESS_MOD",
                    "_ALPHA_PROCESS_SUB",
                    "_ALPHA_PROCESS_MOD_A0_A1",
                    "_ALPHA_PROCESS_SUB_A0_MOD_A1",
                    "_ALPHA_PROCESS_DIST_FILED",
                    "_TEX0_COLOR_INPUT_ONE",
                    "_TEX1_COLOR_INPUT_ONE",
                    "_TEX2_COLOR_INPUT_ONE",
                    "_TEX0_ALPHA_INPUT_ONE",
                    "_TEX1_ALPHA_INPUT_ONE",
                    "_TEX2_ALPHA_INPUT_ONE",
                    "_TEX0_ALPHA_INPUT_RED",
                    "_TEX1_ALPHA_INPUT_RED",
                    "_TEX2_ALPHA_INPUT_RED",
                    "_TEXTURE0_ENABLE",
                    "_TEXTURE1_ENABLE",
                    "_TEXTURE2_ENABLE",
                    "_TEXTURE1_COLOR_BLEND_MOD",
                    "_TEXTURE1_COLOR_BLEND_ADD",
                    "_TEXTURE1_COLOR_BLEND_SUB",
                    "_TEXTURE2_COLOR_BLEND_MOD",
                    "_TEXTURE2_COLOR_BLEND_ADD",
                    "_TEXTURE2_COLOR_BLEND_SUB",
                    "_TEXTURE1_ALPHA_BLEND_MOD",
                    "_TEXTURE1_ALPHA_BLEND_ADD",
                    "_TEXTURE1_ALPHA_BLEND_SUB",
                    "_TEXTURE2_ALPHA_BLEND_MOD",
                    "_TEXTURE2_ALPHA_BLEND_ADD",
                    "_TEXTURE2_ALPHA_BLEND_SUB",
                    "_PRIMITIVE_COLOR_BLEND_MOD",
                    "_PRIMITIVE_COLOR_BLEND_ADD",
                    "_PRIMITIVE_COLOR_BLEND_SUB",
                    "_PRIMITIVE_ALPHA_BLEND_MOD",
                    "_PRIMITIVE_ALPHA_BLEND_ADD",
                    "_PRIMITIVE_ALPHA_BLEND_SUB",
                    "_PRIMITIVE_COLOR_INPUT_ONE",
                    "_PRIMITIVE_COLOR_INPUT_SRC",
                    "_PRIMITIVE_ALPHA_INPUT_A",
                    "_PRIMITIVE_ALPHA_INPUT_RED",
                    "_PRIMITIVE_ALPHA_INPUT_ONE",
                    "_FRAGMENT_SOFT",
                    "_DECAL",
                    "_FRAGMENT_FRESNEL_ALPHA",
                    "_REFRACT_APPLY_ALPHA",
                    "_ALPHA_COMPARE_NEVER",
                    "_ALPHA_COMPARE_LESS",
                    "_ALPHA_COMPARE_EQUAL",
                    "_ALPHA_COMPARE_LEQUAL",
                    "_ALPHA_COMPARE_GREATER",
                    "_ALPHA_COMPARE_NOTEQUAL",
                    "_ALPHA_COMPARE_GEQUAL",
                    "_ALPHA_COMPARE_ALWAYS",
                    "_TEX0_COLOR_INPUT_SQUARE",
                    "_TEX1_COLOR_INPUT_SQUARE",
                    "_TEX2_COLOR_INPUT_SQUARE",
                    "_REFRACT_BY_CAMERA_DISTANCE",
                    "_TEX0_COLOR_INPUT_ONE_MINUS_RGB",
                    "_TEX1_COLOR_INPUT_ONE_MINUS_RGB",
                    "_TEX2_COLOR_INPUT_ONE_MINUS_RGB",
                    "_TEX0_COLOR_INPUT_ALPHA",
                    "_TEX1_COLOR_INPUT_ALPHA",
                    "_TEX2_COLOR_INPUT_ALPHA",
                    "_TEX0_COLOR_INPUT_ONE_MINUS_ALPHA",
                    "_TEX1_COLOR_INPUT_ONE_MINUS_ALPHA",
                    "_TEX2_COLOR_INPUT_ONE_MINUS_ALPHA",
                    "_TEX0_ALPHA_INPUT_ONE_MINUS_RED",
                    "_TEX1_ALPHA_INPUT_ONE_MINUS_RED",
                    "_TEX2_ALPHA_INPUT_ONE_MINUS_RED",
                    "_TEX0_ALPHA_INPUT_ONE_MINUS_ALPHA",
                    "_TEX1_ALPHA_INPUT_ONE_MINUS_ALPHA",
                    "_TEX2_ALPHA_INPUT_ONE_MINUS_ALPHA",
                    "_PRIMITIVE_COLOR_INPUT_ONE_MINUS_RGB",
                    "_PRIMITIVE_COLOR_INPUT_ALPHA",
                    "_PRIMITIVE_COLOR_INPUT_ONE_MINUS_ALPHA",
                    "_PRIMITIVE_ALPHA_INPUT_ONE_MINUS_RED",
                    "_PRIMITIVE_ALPHA_INPUT_ONE_MINUS_ALPHA",
                    "_TEX0_MIPMAP_LEVEL_LIMIT",
                    "_TEX1_MIPMAP_LEVEL_LIMIT",
                    "_TEX2_MIPMAP_LEVEL_LIMIT",
                    "_TEX_0_SURFACE_ANIM",
                    "_TEX_1_SURFACE_ANIM",
                    "_TEX_2_SURFACE_ANIM",
                    "_TEX_0_SURFACE_ANIM_CROSSFADE",
                    "_TEX_1_SURFACE_ANIM_CROSSFADE",
                    "_TEX_2_SURFACE_ANIM_CROSSFADE",
                    "_TEXTURE0_USE_UV1",
                    "_TEXTURE1_USE_UV1",
                    "_TEXTURE2_USE_UV1",
                    "DRAW_PATH_NONE",
                    "CUSTOM_SHADER_NONE",
                    "USR_SETTING_NONE",
                    "CUSTOM_SHADER_FLAG_NONE",
                    "CUSTOM_SHADER_SWITCH_FLAG_NONE",
                    "CUSTOM_FIELD_ENABLED",
                    "CUSTOM_FIELD_NONE",
                };

                for (int i = 0; i < 8; ++i)
                {
                    _macroNames.Add(string.Format("_SCALE_ANIM_{0}_KEY", i + 1));
                }

                for (int i = 0; i < 8; ++i)
                {
                    _macroNames.Add(string.Format("_COLOR_0_ANIM_{0}_KEY", i + 1));
                }

                for (int i = 0; i < 8; ++i)
                {
                    _macroNames.Add(string.Format("_COLOR_1_ANIM_{0}_KEY", i + 1));
                }

                for (int i = 0; i < 8; ++i)
                {
                    _macroNames.Add(string.Format("_ALPHA_0_ANIM_{0}_KEY", i + 1));
                }

                for (int i = 0; i < 8; ++i)
                {
                    _macroNames.Add(string.Format("_ALPHA_1_ANIM_{0}_KEY", i + 1));
                }

                for (int i = 0; i < 64; ++i)
                {
                    _macroNames.Add(string.Format("DRAW_PATH_{0}", i));
                }

                for (int i = 0; i < 8; ++i)
                {
                    _macroNames.Add(string.Format("RESERVED_SHADER_{0}", i));
                }

                for (int i = 0; i < 8; ++i)
                {
                    _macroNames.Add(string.Format("CUSTOM_SHADER_INDEX_{0}", i + 1));
                    _macroNames.Add(string.Format("USR_SETTING_{0}", i + 1));
                }

                for (int i = 0; i < 64; ++i)
                {
                    _macroNames.Add(string.Format("CUSTOM_SHADER_FLAG_{0}", i));
                    _macroNames.Add(string.Format("USR_FLAG_{0}", i));
                }

                for (int i = 0; i < 64; ++i)
                {
                    _macroNames.Add(string.Format("CUSTOM_SHADER_SWITCH_FLAG_{0}", i));
                    _macroNames.Add(string.Format("USR_SWITCH_FLAG_{0}", i));
                }

                for (int i = 0; i < 32; ++i)
                {
                    _macroNames.Add(string.Format("CUSTOM_FIELD_FLAG_{0}", i));
                }

                return _macroNames;
            }
        }

        /// <summary>
        /// パーティクルのエミッタ追従タイプ。
        /// </summary>
        private enum ParticleFollowType
        {
            ParticleFollowType_EmitterFull = 0,        //!< 完全追従
            ParticleFollowType_None = 1,               //!< 追従しない
            ParticleFollowType_EmitterPosition = 2,    //!< 位置だけ追従
            ParticleFollowType_MaxFollowType = 3,      //!< エミッタ追従タイプ総数
        };


        /// <summary>
        /// コンピュートシェーダのバリエーションファイルを生成するときに使用します。
        /// EmitterDataを元に、どのビットフラグを立てるか決定します。
        /// </summary>
        private static List<Tuple<string, Func<EmitterData, bool>>> ComputeShaderVariationConstantNameSets
        {
            get
            {
                if (_computeShaderVariationConstantNameSets != null)
                {
                    return _computeShaderVariationConstantNameSets;
                }

                _computeShaderVariationConstantNameSets = new List<Tuple<string, Func<EmitterData, bool>>>()
                {
                    // 重力世界方法
                    new Tuple<string, Func<EmitterData, bool>>("bitFlagWorldGrabityEnabled", emitterData => emitterData.EmitterEmissionData.EmitterEmissionGravityData.EnableWorldCoordinate ),

                    // フィールド
                    new Tuple<string, Func<EmitterData, bool>>("bitFlagFieldRandom"        , emitterData => emitterData.ActiveFieldList.Any( field => field is RandomData ) ),
                    new Tuple<string, Func<EmitterData, bool>>("bitFlagFieldRandomFe1"     , emitterData => emitterData.ActiveFieldList.Any( field => field is RandomFe1Data) ),
                    new Tuple<string, Func<EmitterData, bool>>("bitFlagFieldPosadd"        , emitterData => emitterData.ActiveFieldList.Any( field => field is AddLocationData) ),
                    new Tuple<string, Func<EmitterData, bool>>("bitFlagFieldMagnet"        , emitterData => emitterData.ActiveFieldList.Any( field => field is MagnetData) ),
                    new Tuple<string, Func<EmitterData, bool>>("bitFlagFieldConvergence"   , emitterData => emitterData.ActiveFieldList.Any( field => field is ConvergeData) ),
                    new Tuple<string, Func<EmitterData, bool>>("bitFlagFieldSpin"          , emitterData => emitterData.ActiveFieldList.Any( field => field is SpinData) ),
                    new Tuple<string, Func<EmitterData, bool>>("bitFlagFieldCollision"     , emitterData => emitterData.ActiveFieldList.Any( field => field is CollisionData) ),
                    new Tuple<string, Func<EmitterData, bool>>("bitFlagFieldCurlnoise"     , emitterData => emitterData.ActiveFieldList.Any( field => field is CurlNoiseData) ),

                    // エミッタ追従
                    new Tuple<string, Func<EmitterData, bool>>("bitFlagEmitterFollowAll"   , emitterData => emitterData.EmitterBasicSettingData.EmitterBasicBasicData.ParticleFollowType == (int)ParticleFollowType.ParticleFollowType_EmitterFull ),
                    new Tuple<string, Func<EmitterData, bool>>("bitFlagEmitterFollowPos"   , emitterData => emitterData.EmitterBasicSettingData.EmitterBasicBasicData.ParticleFollowType == (int)ParticleFollowType.ParticleFollowType_EmitterPosition ),
                    new Tuple<string, Func<EmitterData, bool>>("bitFlagEmitterFollowNone"  , emitterData => emitterData.EmitterBasicSettingData.EmitterBasicBasicData.ParticleFollowType == (int)ParticleFollowType.ParticleFollowType_None )
                };

                return _computeShaderVariationConstantNameSets;
            }
        }

        /// <summary>
        /// ShaderConverter.exeで使用するバリエーションファイルを生成します。
        /// ShaderKeyを元に、生成します。
        /// </summary>
        /// <param name="filePath">バリエーションファイルのファイルパス</param>
        /// <param name="shaderKeyList">元になるShaderKey</param>
        /// <returns></returns>
        private static bool MakeShaderConverterXmlFile(string filePath, IEnumerable<VariationElement> shaderKeyList)
        {
            // シェーダキーリストとコンバイナシェーダのインデックスのリスト
            var definedMacrosList = new List<Tuple<List<string>, int>>();

            // "#define"のあとの文字列を引っこ抜く正規表現
            var regex = new Regex("^#define[ \t]+(?<name>[a-zA-Z0-9_]+).*$", RegexOptions.IgnoreCase);

            foreach (var shaderKey in shaderKeyList)
            {
                string[] shaderKeyArray = shaderKey.ShaderKey.Split(Environment.NewLine.ToCharArray());
                var definedMacros = shaderKeyArray.Select(str => regex.Match(str).Groups["name"].Value).ToList();
                definedMacrosList.Add(new Tuple<List<string>, int>(definedMacros, shaderKey.CombinerShaderIndex));
            }

            // shaderKeyListにはユーザー定義文字列が入ってくるので、バリエーションに含めるために全文字列をマクロ一覧に追加しておく
            var drawPaths = OptionStore.ProjectConfig.DrawPaths;
            var userDefineStrings = new List<string>();
            userDefineStrings.AddRange(drawPaths.Select(p => p.ShaderCompileDef1).Where(s => !string.IsNullOrEmpty(s)));
            userDefineStrings.AddRange(drawPaths.Select(p => p.ShaderCompileDef2).Where(s => !string.IsNullOrEmpty(s)));
            foreach (var defineString in userDefineStrings)
            {
                // 重複はさせないから毎回通しても大丈夫
                if (!MacroNames.Contains(defineString))
                {
                    MacroNames.Add(defineString);
                }
            }

            var document = new XmlDocument();
            var declaration = document.CreateXmlDeclaration("1.0", "utf-8", null);  // XML宣言
            var gfxShaderVariation = document.CreateElement("GfxShaderVariation");  // ルート要素
            gfxShaderVariation.SetAttribute("version", "0.0.0");
            document.AppendChild(declaration);
            document.AppendChild(gfxShaderVariation);

            var shaderVariationDefinition = document.CreateElement("ShaderVariationDefinition");
            {
                Action<XmlElement, string> makeVariation = (variationElement, attributeName) =>
                {
                    var variationConstantBuffer = document.CreateElement("VariationConstantBuffer");
                    variationConstantBuffer.SetAttribute("name", attributeName);
                    variationElement.AppendChild(variationConstantBuffer);

                    var preprocessorDefinitionDefinitionArray =
                        document.CreateElement("PreprocessorDefinitionDefinitionArray");

                    // バリエーション種別の個数は「マクロ数+1」となる（コンバイナインデックスのぶん）
                    preprocessorDefinitionDefinitionArray.SetAttribute("length", (MacroNames.Count + 1).ToString());
                    for (int i = 0; i < MacroNames.Count; i++)
                    {
                        var element = document.CreateElement("PreprocessorDefinitionDefinition");
                        element.SetAttribute("index", i.ToString());
                        element.SetAttribute("name", MacroNames[i] + "_CONVERTER");
                        preprocessorDefinitionDefinitionArray.AppendChild(element);
                    }

                    // コンバイナインデックスをバリエーションとして追加
                    {
                        var element = document.CreateElement("PreprocessorDefinitionDefinition");
                        element.SetAttribute("index", MacroNames.Count.ToString());
                        element.SetAttribute("name", "COMBINER_SHADER_INDEX");
                        preprocessorDefinitionDefinitionArray.AppendChild(element);
                    }

                    variationElement.AppendChild(preprocessorDefinitionDefinitionArray);
                };

                var vertexShaderVariationDefinition = document.CreateElement("VertexShaderVariationDefinition");
                makeVariation(vertexShaderVariationDefinition, "VariationConstantsVs");
                shaderVariationDefinition.AppendChild(vertexShaderVariationDefinition);

                var pixelShaderVariationDefinition = document.CreateElement("PixelShaderVariationDefinition");
                makeVariation(pixelShaderVariationDefinition, "VariationConstantsPs");
                shaderVariationDefinition.AppendChild(pixelShaderVariationDefinition);
            }

            gfxShaderVariation.AppendChild(shaderVariationDefinition);

            var shaderVariationValueArray = document.CreateElement("ShaderVariationValueArray");
            shaderVariationValueArray.SetAttribute("length", definedMacrosList.Count.ToString());
            {
                for (int emitterIndex = 0; emitterIndex < definedMacrosList.Count; ++emitterIndex)
                {
                    var definedMacros = definedMacrosList[emitterIndex];

                    var shaderVariationValue = document.CreateElement("ShaderVariationValue");
                    shaderVariationValue.SetAttribute("index", emitterIndex.ToString());
                    {
                        Action<XmlElement> addMacroValues = (elem) =>
                        {
                            var preprocessorDefinitionValueArray = document.CreateElement("PreprocessorDefinitionValueArray");

                            // バリエーション種別の個数は「マクロ数+1」となる（コンバイナインデックスのぶん）
                            preprocessorDefinitionValueArray.SetAttribute("length", (MacroNames.Count + 1).ToString());
                            {
                                for (int i = 0; i < MacroNames.Count; i++)
                                {
                                    var preprocessorDefinitionValue = document.CreateElement("PreprocessorDefinitionValue");
                                    int index = definedMacros.Item1.IndexOf(MacroNames[i]);
                                    var innerText = index == -1 ? "( 0 )" : "( 1 )";
                                    innerText += "\r\n#if " + MacroNames[i] + "_CONVERTER\r\n#define " + MacroNames[i] + "\r\n#endif";
                                    preprocessorDefinitionValue.InnerText = innerText;
                                    preprocessorDefinitionValue.SetAttribute("index", i.ToString());
                                    preprocessorDefinitionValueArray.AppendChild(preprocessorDefinitionValue);
                                }

                                // コンバイナインデックスの値を追加(-1は不使用)
                                {
                                    var preprocessorDefinitionValue = document.CreateElement("PreprocessorDefinitionValue");
                                    preprocessorDefinitionValue.InnerText = definedMacros.Item2.ToString();
                                    preprocessorDefinitionValue.SetAttribute("index", MacroNames.Count.ToString());
                                    preprocessorDefinitionValueArray.AppendChild(preprocessorDefinitionValue);
                                }
                            }

                            elem.AppendChild(preprocessorDefinitionValueArray);
                        };

                        var vertexShaderVariationValue = document.CreateElement("VertexShaderVariationValue");
                        addMacroValues(vertexShaderVariationValue);
                        shaderVariationValue.AppendChild(vertexShaderVariationValue);

                        var pixelShaderVariationValue = document.CreateElement("PixelShaderVariationValue");
                        addMacroValues(pixelShaderVariationValue);
                        shaderVariationValue.AppendChild(pixelShaderVariationValue);
                    }

                    shaderVariationValueArray.AppendChild(shaderVariationValue);
                }
            }

            gfxShaderVariation.AppendChild(shaderVariationValueArray);

            document.Save(filePath);

            return true;
        }

        // コンピュートシェーダのバリエーションを、コンスタント変数で出力したい場合はtrue
        // マクロで出力したい場合はfalse
        private static bool forceToUseConstantVariationForComputeShader = false;

        /// <summary>
        /// ShaderConverter.exeで使用するコンピュートシェーダのバリエーションファイルを生成します。
        /// WriteBinaryDataContextを元に、生成します。
        /// </summary>
        /// <param name="filePath">バリエーションファイルのファイルパス</param>
        /// <param name="emitterList">元になるEmitterDataのリスト</param>
        /// <returns></returns>
        private static bool MakeComputeShaderConverterXmlFile(string filePath, IEnumerable<EmitterData> emitterList)
        {

            var document = new XmlDocument();
            var declaration = document.CreateXmlDeclaration("1.0", "utf-8", null);  // XML宣言
            var gfxShaderVariation = document.CreateElement("GfxShaderVariation");  // ルート要素
            gfxShaderVariation.SetAttribute("version", "0.0.0");
            document.AppendChild(declaration);
            document.AppendChild(gfxShaderVariation);

            var shaderVariationDefinition = document.CreateElement("ShaderVariationDefinition");
            {
                Action<XmlElement, string> makeVariation = (variationElement, attributeName) =>
                {
                    var variationConstantBuffer = document.CreateElement("VariationConstantBuffer");
                    variationConstantBuffer.SetAttribute("name", attributeName);
                    variationElement.AppendChild(variationConstantBuffer);

                    if (forceToUseConstantVariationForComputeShader == true)
                    {
                        if (emitterList.Any() == true)
                        {
                            var variationConstantDefinitionArray = document.CreateElement("VariationConstantDefinitionArray");
                            variationConstantDefinitionArray.SetAttribute("length", ComputeShaderVariationConstantNameSets.Count.ToString());
                            for (int i = 0; i < ComputeShaderVariationConstantNameSets.Count; i++)
                            {
                                var element = document.CreateElement("VariationConstantDefinition");
                                element.SetAttribute("index", i.ToString());
                                element.SetAttribute("name", ComputeShaderVariationConstantNameSets[i].Item1);
                                element.SetAttribute("type", "bool");
                                variationConstantDefinitionArray.AppendChild(element);
                            }
                            variationElement.AppendChild(variationConstantDefinitionArray);
                        }
                    }

                    var preprocessorDefinitionDefinitionArray = document.CreateElement("PreprocessorDefinitionDefinitionArray");
                    int preprocessorDefinitionCounts = 1;
                    if (forceToUseConstantVariationForComputeShader == false)
                    {
                        preprocessorDefinitionCounts += ComputeShaderVariationConstantNameSets.Count;
                    }
                    preprocessorDefinitionDefinitionArray.SetAttribute("length", preprocessorDefinitionCounts.ToString());
                    {
                        // COMPUTE_SHADERフラグを手動で立てる
                        var preprocessorDefinitionDefinition = document.CreateElement("PreprocessorDefinitionDefinition");
                        preprocessorDefinitionDefinition.SetAttribute("index", "0");
                        preprocessorDefinitionDefinition.SetAttribute("name", "COMPUTE_SHADER");
                        preprocessorDefinitionDefinitionArray.AppendChild(preprocessorDefinitionDefinition);

                        if (forceToUseConstantVariationForComputeShader == false)
                        {
                            for (int i = 0; i < ComputeShaderVariationConstantNameSets.Count; i++)
                            {
                                var element = document.CreateElement("PreprocessorDefinitionDefinition");
                                element.SetAttribute("index", (i + 1).ToString());
                                element.SetAttribute("name", ComputeShaderVariationConstantNameSets[i].Item1);
                                preprocessorDefinitionDefinitionArray.AppendChild(element);
                            }
                        }
                    }
                    variationElement.AppendChild(preprocessorDefinitionDefinitionArray);
                };

                var computeShaderVariationDefinition = document.CreateElement("ComputeShaderVariationDefinition");
                makeVariation(computeShaderVariationDefinition, "VariationConstantsCs");
                shaderVariationDefinition.AppendChild(computeShaderVariationDefinition);
            }

            gfxShaderVariation.AppendChild(shaderVariationDefinition);

            var shaderVariationValueArray = document.CreateElement("ShaderVariationValueArray");

            if (emitterList.Any() == true)
            {
                shaderVariationValueArray.SetAttribute("length", emitterList.Count().ToString());
                {
                    int emitterIndex = 0;
                    foreach (var emitterData in emitterList)
                    {
                        var shaderVariationValue = document.CreateElement("ShaderVariationValue");
                        shaderVariationValue.SetAttribute("index", emitterIndex.ToString());
                        {
                            Action<XmlElement> addMacroValues = (elem) =>
                            {
                                var variationConstantValueArray = document.CreateElement("VariationConstantValueArray");

                                // コンスタント変数のバリエーション
                                if (forceToUseConstantVariationForComputeShader == true)
                                {
                                    variationConstantValueArray.SetAttribute("length", ComputeShaderVariationConstantNameSets.Count.ToString());
                                    {
                                        for (int i = 0; i < ComputeShaderVariationConstantNameSets.Count; i++)
                                        {
                                            var variationConstantValueValue = document.CreateElement("VariationConstantValue");
                                            variationConstantValueValue.InnerText = Convert.ToInt32(ComputeShaderVariationConstantNameSets[i].Item2(emitterData)).ToString();
                                            variationConstantValueValue.SetAttribute("index", i.ToString());
                                            variationConstantValueArray.AppendChild(variationConstantValueValue);
                                        }
                                    }
                                    elem.AppendChild(variationConstantValueArray);
                                }

                                var preprocessorDefinitionValueArray = document.CreateElement("PreprocessorDefinitionValueArray");
                                int preprocessorDefinitionCount = 1;
                                if (forceToUseConstantVariationForComputeShader == false)
                                {
                                    preprocessorDefinitionCount += ComputeShaderVariationConstantNameSets.Count;
                                }
                                preprocessorDefinitionValueArray.SetAttribute("length", preprocessorDefinitionCount.ToString());
                                {
                                    // COMPUTE_SHADERの値
                                    var computeShaderPreprocessorDefinitionValue = document.CreateElement("PreprocessorDefinitionValue");
                                    computeShaderPreprocessorDefinitionValue.SetAttribute("index", "0");
                                    computeShaderPreprocessorDefinitionValue.InnerText = "( 1 )";
                                    preprocessorDefinitionValueArray.AppendChild(computeShaderPreprocessorDefinitionValue);

                                    if (forceToUseConstantVariationForComputeShader == false)
                                    {
                                        for (int i = 0; i < ComputeShaderVariationConstantNameSets.Count; i++)
                                        {
                                            var preprocessorDefinitionValue =
                                                document.CreateElement("PreprocessorDefinitionValue");
                                            var innerText = ComputeShaderVariationConstantNameSets[i].Item2(emitterData) ? "( true )" : "( false )";
                                            preprocessorDefinitionValue.InnerText = innerText;
                                            preprocessorDefinitionValue.SetAttribute("index", (i + 1).ToString());
                                            preprocessorDefinitionValueArray.AppendChild(preprocessorDefinitionValue);
                                        }
                                    }
                                }
                                elem.AppendChild(preprocessorDefinitionValueArray);
                            };

                            var computeShaderVariationValue = document.CreateElement("ComputeShaderVariationValue");
                            addMacroValues(computeShaderVariationValue);
                            shaderVariationValue.AppendChild(computeShaderVariationValue);
                        }

                        shaderVariationValueArray.AppendChild(shaderVariationValue);

                        emitterIndex++;
                    }
                }

            }
            else
            {
                shaderVariationValueArray.SetAttribute("length", "1");
                {
                    var shaderVariationValue = document.CreateElement("ShaderVariationValue");
                    shaderVariationValue.SetAttribute("index", "0");
                    {
                        var computeShaderVariationValue = document.CreateElement("ComputeShaderVariationValue");
                        {
                            // COMPUTE_SHADERフラグを手動で立てる
                            var preprocessorDefinitionValueArray = document.CreateElement("PreprocessorDefinitionValueArray");
                            int preprocessorDefinitionCount = 1;
                            if (forceToUseConstantVariationForComputeShader == false)
                            {
                                preprocessorDefinitionCount += ComputeShaderVariationConstantNameSets.Count;
                            }
                            preprocessorDefinitionValueArray.SetAttribute("length", preprocessorDefinitionCount.ToString());
                            {
                                var computeShaderPreprocessorDefinitionValue = document.CreateElement("PreprocessorDefinitionValue");
                                computeShaderPreprocessorDefinitionValue.SetAttribute("index", "0");
                                computeShaderPreprocessorDefinitionValue.InnerText = "( 1 )";
                                preprocessorDefinitionValueArray.AppendChild(computeShaderPreprocessorDefinitionValue);

                                if (forceToUseConstantVariationForComputeShader == false)
                                {
                                    for (int i = 0; i < ComputeShaderVariationConstantNameSets.Count; i++)
                                    {
                                        // ここでは、バイナリサイズ縮小のため、バイナリサイズが小さくなるマクロの組み合わせを指定している。
                                        // ノンバリシェーダであるため、本来は全てのマクロをtrueにしておくべき。
                                        // 当分の間ノンバリコンピュートシェーダは利用されないので問題ない。
                                        var preprocessorDefinitionValue = document.CreateElement("PreprocessorDefinitionValue");
                                        var innerText = "( false )";
                                        // 「エミッタSRT完全追従」＆「ランダムフィールド」な、エミッタはバイナリサイズが小さい
                                        if (ComputeShaderVariationConstantNameSets[i].Item1 == "bitFlagFieldRandomFe1" ||
                                            ComputeShaderVariationConstantNameSets[i].Item1 == "bitFlagEmitterFollowAll")
                                        {
                                            innerText = "( true )";
                                        }
                                        preprocessorDefinitionValue.InnerText = innerText;
                                        preprocessorDefinitionValue.SetAttribute("index", (i + 1).ToString());
                                        preprocessorDefinitionValueArray.AppendChild(preprocessorDefinitionValue);
                                    }
                                }
                            }
                            computeShaderVariationValue.AppendChild(preprocessorDefinitionValueArray);
                        }
                        shaderVariationValue.AppendChild(computeShaderVariationValue);
                    }
                    shaderVariationValueArray.AppendChild(shaderVariationValue);
                }
            }

            gfxShaderVariation.AppendChild(shaderVariationValueArray);

            document.Save(filePath);

            return true;
        }

        /// <summary>
        /// シェーダキーのマクロと、必要なシェーダのソースコードを生成する.
        /// </summary>
        /// <param name="conversionInputData">シェーダキーの生成に使用されるエミッタの情報</param>
        /// <param name="userDefineIndex">ユーザー定義文字列のインデックス</param>
        /// <param name="shaderKey">ShaderKeyのマクロ</param>
        /// <param name="vertexShader">バーテックスシェーダのソースコード</param>
        /// <param name="fragmentShader">フラグメントシェーダのソースコード</param>
        /// <param name="computeShader">コンピュートシェーダのソースコード</param>
        /// <returns></returns>
        private static bool GenerateShaderKeyAndCodes(ShaderConversionInputData conversionInputData,
                                                      uint userDefineIndex,
                                                      out string shaderKey,
                                                      out string vertexShader,
                                                      out string fragmentShader,
                                                      out string computeShader)
        {
            // ShaderKeyが取得できるかテスト
            using (var converter = new ShaderCodeGeneratorGeneric.ShaderCodeGeneratorGeneric(ShaderBinaryHelper.Handle))
            {
                converter.SetShaderCodes(CopyShaderCodeForShaderConversion(true));
                conversionInputData.UserDefineIndex = userDefineIndex;
                if (!converter.GenerateVfxShaderKeyAndCodes(conversionInputData,
                                                            out shaderKey,
                                                            out vertexShader,
                                                            out fragmentShader,
                                                            out computeShader))
                {
                    return false;
                }
            }

            return true;
        }

        /// <summary>
        /// ShaderConverter.exeを用いて、シェーダバイナリを生成します.
        /// </summary>
        /// <param name="shaderKeyList">シェーダバリエーションで使用するシェーダキーのマクロのリスト</param>
        /// <param name="vertexShader">バーテックスシェーダのソースコード</param>
        /// <param name="fragmentSHader">フラグメントシェーダのソースコード</param>
        /// <param name="binaryOutputFilePath">シェーダバイナリの出力ファイルパス</param>
        /// <param name="standardOutput">標準出力</param>
        /// <param name="additionalOption">ShaderConverter.exeに追加で付けるオプション(--dump-directoryなど)</param>
        /// <returns></returns>
        private static bool GenerateShaderBinary(IEnumerable<VariationElement> shaderKeyList,
                                                 string vertexShader,
                                                 string fragmentSHader,
                                                 string binaryOutputFilePath,
                                                 out string standardOutput,
                                                 string additionalOption = null)
        {
            var vertexShaderFile = new FileStream(VertexShaderPath, FileMode.Create);
            byte[] vertexShaderBuffer =
                System.Text.Encoding.GetEncoding("UTF-8").GetBytes(vertexShader);
            vertexShaderFile.Write(vertexShaderBuffer, 0, vertexShaderBuffer.Length);
            vertexShaderFile.Close();

            var pixelShaderFile = new FileStream(PixelShaderPath, FileMode.Create);
            byte[] pixelShaderBuffer =
                System.Text.Encoding.GetEncoding("UTF-8").GetBytes(fragmentSHader);
            pixelShaderFile.Write(pixelShaderBuffer, 0, pixelShaderBuffer.Length);
            pixelShaderFile.Close();

            // バリエーションファイルを生成する
            if (MakeShaderConverterXmlFile(VariationPath, shaderKeyList) == false)
            {
                standardOutput = string.Empty;
                return false;
            }

            // ShaderConveterに渡すオプションの設定
            string args = string.Format(SpecManager.CurrentSpec.ShaderConversionOption.ConverterOption +
                                        " --code-type=" + SpecManager.CurrentSpec.ShaderConversionOption.CodeType +
                                        " -o\"" + binaryOutputFilePath +
                                        "\" --vertex-shader=\"" + VertexShaderPath +
                                        "\" --pixel-shader=\"" + PixelShaderPath +
                                        "\" --include-directory=\"" + GetStringDirectories() +
                                        "\" --variation=\"" + VariationPath +
                                        (string.IsNullOrEmpty(OptionStore.RuntimeOptions.ShaderCacheDirectory)
                                            ? string.Empty : "\" --shader-cache-directory=\""
                                            + OptionStore.RuntimeOptions.ShaderCacheDirectory) +
                                        (OptionStore.ProjectConfig.EnableShaderPreprocessOption
                                            ? "\" --preprocess" : "\"") +
                                        " --print-progress --code-page 65001" +
                                        GetStringDefinitions());

            if (OptionStore.RuntimeOptions.IsCommandLineMode && OptionStore.RuntimeOptions.IsJobsNumberAssigned)
            {
                args += " -j=\"" + OptionStore.RuntimeOptions.JobsNumber + "\"";
            }

            if (additionalOption != null)
            {
                args += additionalOption;
            }

            var resExecute = ShaderConverterExecuter.Instance.Execute(args);

            standardOutput = resExecute.GetAllLog();

            return resExecute.ResultCode == 0;
        }

        /// <summary>
        /// ShaderConverter.exeを用いて、コンピュートシェーダのシェーダバイナリを生成します.
        /// </summary>
        /// <param name="emitterList">コンバートするエミッタのリスト</param>
        /// <param name="computeShader">コンピュートシェーダのソースコード</param>
        /// <param name="binaryOutputFilePath">シェーダバイナリの出力ファイルパス</param>
        /// <param name="standardOutput">標準出力</param>
        /// <param name="additionalOption">ShaderConverter.exeに追加で付けるオプション(--dump-directoryなど)</param>
        /// <returns></returns>
        private static bool GenerateComputeShaderBinary(IEnumerable<EmitterData> emitterList,
                                                        string computeShader,
                                                        string binaryOutputFilePath,
                                                        out string standardOutput,
                                                        string additionalOption = null)
        {
            // コンピュートシェーダを一時ファイルに保存しておく
            var computeShaderFile = new FileStream(ComputeShaderPath, FileMode.Create);
            byte[] computeShaderBuffer =
                System.Text.Encoding.GetEncoding("UTF-8").GetBytes(computeShader);
            computeShaderFile.Write(computeShaderBuffer, 0, computeShaderBuffer.Length);
            computeShaderFile.Close();

            // バリエーションファイルを生成する
            if (MakeComputeShaderConverterXmlFile(ComputeShaderVariationPath, emitterList) == false)
            {
                standardOutput = string.Empty;
                return false;
            }

            // ShaderConveterに渡すオプションの設定
            string args = string.Format(SpecManager.CurrentSpec.ShaderConversionOption.ConverterOption +
                                        " --code-type=" + SpecManager.CurrentSpec.ShaderConversionOption.CodeType +
                                        " -o\"" + binaryOutputFilePath +
                                        "\" --compute-shader=\"" + ComputeShaderPath +
                                        "\" --include-directory=\"" + GetStringDirectories() +
                                        "\" --variation=\"" + ComputeShaderVariationPath +
                                        (string.IsNullOrEmpty(OptionStore.RuntimeOptions.ShaderCacheDirectory)
                                            ? string.Empty : "\" --shader-cache-directory=\""
                                            + OptionStore.RuntimeOptions.ShaderCacheDirectory) +
                                        (OptionStore.ProjectConfig.EnableShaderPreprocessOption
                                            ? "\" --preprocess" : "\"") +
                                        " --print-progress --code-page 65001" +
                                        GetStringDefinitions());

            if (OptionStore.RuntimeOptions.IsCommandLineMode && OptionStore.RuntimeOptions.IsJobsNumberAssigned)
            {
                args += " -j=\"" + OptionStore.RuntimeOptions.JobsNumber + "\"";
            }

            if (additionalOption != null)
            {
                args += (" " + additionalOption);
            }

            var resExecute = ShaderConverterExecuter.Instance.Execute(args);

            standardOutput = resExecute.GetAllLog();

            return resExecute.ResultCode == 0;
        }

        /// <summary>
        /// Gfxのコンバータによるシェーダソースを生成します。
        /// </summary>
        /// <param name="emitter">入力データ</param>
        /// <param name="userDefineIndex">ユーザー定義文字列のインデックス</param>
        /// <param name="isSourceCode">ソースコードを取得するときtrue、アセンブリを取得するときfalse</param>
        /// <param name="vertexShader">頂点シェーダの出力</param>
        /// <param name="fragmentShader">ピクセルシェーダの出力</param>
        /// <param name="computeShader">コンピュートシェーダの出力</param>
        /// <param name="latencyInfo">シェーダの負荷情報の出力</param>
        /// <param name="conversionInputData">emitterをShaderConversionInputDataクラスにコンバートしたもの</param>
        /// <returns>成功すればtrueを返します。</returns>
        private static bool GenerateGfxShaderCore(
            EmitterData emitter,
            uint userDefineIndex,
            bool isSourceCode,
            out string vertexShader,
            out string fragmentShader,
            out string computeShader,
            out string latencyInfo,
            out ShaderConversionInputData conversionInputData)
        {
            latencyInfo = string.Empty;

            // ShaderKeyのリスト
            var shaderKeyList = new List<VariationElement>();

            // ShaderConverter.exeに渡すシェーダファイル
            string vertexShaderForShaderConverter;
            string fragmentShaderForShaderConverter;
            string computeShaderForShaderConverter;
            string shaderKey;

            // ShaderKey生成のために、ShaderConversionInputDataを用意する
            var conversionInputDataList = PrepareEmitterForShaderConversion(emitter);
            if (conversionInputDataList == null || conversionInputDataList.Count <= 0)
            {
                vertexShader = null;
                fragmentShader = null;
                computeShader = null;
                latencyInfo = null;
                conversionInputData = null;
                return false;
            }
            conversionInputData = conversionInputDataList[0];

            // ShaderKeyが取得できるかテスト
            if (GenerateShaderKeyAndCodes(conversionInputData,
                userDefineIndex,
                out shaderKey,
                out vertexShaderForShaderConverter,
                out fragmentShaderForShaderConverter,
                out computeShaderForShaderConverter) == false)
            {
                vertexShader = string.Empty;
                fragmentShader = string.Empty;
                computeShader = string.Empty;
                return false;
            }

            string combinderShader = conversionInputData.OverrideShaderCode;
            if (!string.IsNullOrEmpty(combinderShader))
            {
                // コンバイナシェーダを付加（コード生成モードでは1エミッタしか処理対象にならないのでインデックスは0で固定）
                var builder = new StringBuilder(fragmentShaderForShaderConverter);
                builder.AppendLine(string.Format("#if COMBINER_SHADER_INDEX == {0}", 0));
                builder.AppendLine(combinderShader);
                builder.AppendLine("#endif");
                fragmentShaderForShaderConverter = builder.ToString();
            }

            // コード生成モードでは1エミッタしか処理対象にならないので、コンバイナシェーダなしなら-1、ありなら0の2択
            shaderKeyList.Add(new VariationElement(null, shaderKey, !string.IsNullOrEmpty(combinderShader) ? 0 : -1));

            // ShaderConverter.exeでコンバートしないとシェーダのソースコードが出力されないので、ShaderConverter.exeを実行する
            string additionalOption = " --dump-directory=\"" + ShaderSourceDumpDirectoryPath + "\"";
            string standardOutput;
            if (GenerateShaderBinary(
                shaderKeyList,
                vertexShaderForShaderConverter,
                fragmentShaderForShaderConverter,
                OutputPath,
                out standardOutput,
                additionalOption) == false)
            {
                string errorLogFilePath = Path.Combine(ShaderConverterExecuter.Instance.ErrorLogDirecotoryPath, "vertexAndPixelShaders.log");
                GfxToolExecutionUtility.SaveAndShowErrorMessage(errorLogFilePath, standardOutput);

                vertexShader = standardOutput;
                fragmentShader = standardOutput;
                computeShader = standardOutput;
                latencyInfo = string.Empty;
                return false;
            }

            if (isSourceCode)
            {
                ShaderConverterDumpInfo vertexShaderDumpInfo = new ShaderConverterDumpInfo();

                vertexShaderDumpInfo.Open(VertexShaderDumpPath);
                vertexShader = vertexShaderDumpInfo.GetShaderSourceCode();

                ShaderConverterDumpInfo fragmentShaderDumpInfo = new ShaderConverterDumpInfo();

                fragmentShaderDumpInfo.Open(PixelShaderDumpPath);
                fragmentShader = fragmentShaderDumpInfo.GetShaderSourceCode();
            }
            else
            {
                ShaderConverterDumpInfo vertexShaderDumpInfo = new ShaderConverterDumpInfo();

                vertexShaderDumpInfo.Open(VertexShaderDumpPath);
                vertexShader = vertexShaderDumpInfo.GetAssemblyInfo();

                ShaderConverterDumpInfo fragmentShaderDumpInfo = new ShaderConverterDumpInfo();

                fragmentShaderDumpInfo.Open(PixelShaderDumpPath);
                fragmentShader = fragmentShaderDumpInfo.GetAssemblyInfo();
            }

            // ShaderStaticsからおおよその負荷を抽出する

            string vLatency = "--";
            var vLines = vertexShader.Split(new[] { "\n" }, StringSplitOptions.None);
            foreach (var vLine in vLines)
            {
                if (vLine.Contains("Latency"))
                {
                    vLatency = vLine.Substring(9, vLine.Length - 9);
                    break;
                }
            }

            string pLatency = "--";
            var pLines = fragmentShader.Split(new[] { "\n" }, StringSplitOptions.None);
            foreach (var pLine in pLines)
            {
                if (pLine.Contains("Latency"))
                {
                    pLatency = pLine.Substring(9, pLine.Length - 9);
                    break;
                }
            }

            latencyInfo = string.Format("Latency: VS({0}) / PS ({1})", vLatency, pLatency);

            var emitterList = new List<EmitterData>();
            emitterList.Add(emitter);
            // コンピュートシェーダを使ってシェーダバイナリを生成
            if (GenerateComputeShaderBinary(emitterList, computeShaderForShaderConverter, ComputeShaderOutputPath, out standardOutput, additionalOption) == false)
            {
                string errorLogFilePath = Path.Combine(ShaderConverterExecuter.Instance.ErrorLogDirecotoryPath, "computeShader.log");
                GfxToolExecutionUtility.SaveAndShowErrorMessage(errorLogFilePath, standardOutput);

                computeShader = standardOutput;
                return false;
            }

            if (isSourceCode)
            {
                ShaderConverterDumpInfo computeShaderDumpInfo = new ShaderConverterDumpInfo();

                computeShaderDumpInfo.Open(ComputeShaderDumpPath);
                computeShader = computeShaderDumpInfo.GetShaderSourceCode();
            }
            else
            {
                ShaderConverterDumpInfo computeShaderDumpInfo = new ShaderConverterDumpInfo();

                computeShaderDumpInfo.Open(ComputeShaderDumpPath);
                computeShader = computeShaderDumpInfo.GetAssemblyInfo();
            }

            return true;
        }

        /// <summary>
        /// Write shader binary data.
        /// </summary>
        /// <param name="stream">バイナリーストリーム</param>
        /// <param name="completeCallback">The callback to execute when finished.</param>
        /// <param name="emitterContexts">エミッタのバイナリーコンテキスト</param>
        /// <param name="shaderBinaryOutputFilePath">通常シェーダバイナリを別ファイルで出力する場合のファイルパス</param>
        /// <param name="computeShaderBinaryOutputFilePath">コンピュートシェーダバイナリを別ファイルで出力する場合のファイルパス</param>
        /// <param name="userDefineIndex">ユーザー定義文字列のインデックス</param>
        /// <param name="handleShaderCompileComplete"></param>
        /// <returns>True on success.</returns>
        private static bool WriteGfxShaderResourcesCore(Stream stream,
                                                        string shaderBinaryOutputFilePath,
                                                        string computeShaderBinaryOutputFilePath,
                                                        Action<bool> completeCallback,
                                                        IEnumerable<WriteBinaryDataContext> emitterContexts,
                                                        uint userDefineIndex,
                                                        CompileShaderCompleteCallback handleShaderCompileComplete)
        {
            // アライメント計算ユーティリティの定義
            int shaderAlignment = SpecManager.CurrentSpec.ShaderConversionOption.BinaryAlignment;
            Func<int, int, int> calcPadding = (pos, align) =>
                align <= 0 || pos % align == 0 ? 0 : align - (pos % align);
            Action<Stream, int> writePadding = (memst, size) =>
            {
                if (size > 0)
                {
                    var alignmentBytes = Enumerable.Repeat((byte)0, size).ToArray();
                    memst.Write(alignmentBytes, 0, size);
                }
            };

            using (var profiler = new ProfileTimer("ShaderConvertForGfx"))
            {
                if (OptionStore.RuntimeOptions.IsCommandLineMode)
                {
                    Logger.Log("Console", LogLevels.Information, "Shader convert is starting...");
                }

                // ShaderKeyのリスト(エミッタセット・エミッタ名, シェーダーキー)
                var shaderKeyList = new List<VariationElement>();

                // Emitterのバイナリ上でのオフセット
                var emitterOffsets = new List<long>();

                // 各Emitterのシェーダインデックス
                var emitterShaderIndexes = new List<ShaderConversionInputData>();

                // ShaderConverter.exeに渡すシェーダファイル
                string vertexShaderForShaderConverter = string.Empty;
                string fragmentShaderForShaderConverter = string.Empty;
                string computeShaderForShaderConverter = string.Empty;

                // コンバート対象のデータに含まれるコンバイナシェーダをここに集約する
                var combinerShaderCodeList = new List<string>();

                uint shaderIndex = 0;
                uint computeShaderIndex = 0;
                var emitterListForComputeShader = new List<EmitterData>();

                // シェーダキーの追加を行うクロージャ
                // 戻り値として、シェーダインデックスを返します。
                Func<ShaderConversionInputData, EmitterData, uint, int, Tuple<uint, bool>> generateShaderKey =
                    (conversionInputData, emitterData, userDefineShaderIndex, combinerShaderIndex) =>
                {
                    if ( userDefineShaderIndex == 0 ||
                        (userDefineShaderIndex == 1 && emitterData.EmitterBasicSettingData.EmitterBasicRenderData.DrawPathShaderDef1.IsNullOrEmpty() == false) ||
                        (userDefineShaderIndex == 2 && emitterData.EmitterBasicSettingData.EmitterBasicRenderData.DrawPathShaderDef2.IsNullOrEmpty() == false) )
                    {
                        string shaderKey;
                        if (GenerateShaderKeyAndCodes(conversionInputData,
                                                      userDefineShaderIndex,
                                                      out shaderKey,
                                                      out vertexShaderForShaderConverter,
                                                      out fragmentShaderForShaderConverter,
                                                      out computeShaderForShaderConverter) == false)
                        {
                            if (OptionStore.RuntimeOptions.IsCommandLineMode)
                            {
                                string name = emitterData != null ? emitterData.Name : "invalid data";
                                Logger.Log("Console", LogLevels.Error, "Error on generating shader key at {0}", name);
                            }

                            handleShaderCompileComplete(
                                emitterOffsets,
                                completeCallback,
                                profiler,
                                null,
                                false, // isSuccessful
                                false, // shaderCompileFailed
                                null,  // shaderBinary
                                0,     // shaderBinarySize
                                0,     // shaderCount
                                0,     // computeShaderBinarySize
                                0,     // computeShaderCount
                                null); // writeShaderIndicesAction

                            return new Tuple<uint, bool>(uint.MaxValue, false);
                        }

                        shaderKeyList.Add(new VariationElement(emitterData, shaderKey, combinerShaderIndex));

                        return new Tuple<uint, bool>(shaderIndex++, true);
                    }
                    else
                    {
                        return new Tuple<uint, bool>(uint.MaxValue, true);
                    }
                };

                foreach (var emitterContext in emitterContexts)
                {
                    var emitterData = emitterContext.BinaryStruct.DataModel as EmitterData;
                    ShaderConversionInputData emitterShaderIndex = new ShaderConversionInputData();

                    List<ShaderConversionInputData> conversionInputDataList = null;
                    try
                    {
                        conversionInputDataList = PrepareEmitterForShaderConversion(emitterData);
                    }
                    catch (Exception)
                    {
                        // 例外が発生したのでシェーダコンバートが続行不可。
                        // (例): BinaryFieldInstances.cs の structInstances が foreach 中に別の非同期処理から書き換えられたケース
                        // MEMO: これが発生すること自体正しくはないので、同時アクセスがそもそも起きないような根本対処が理想的。
                        //       アプリが止まる失敗ではなく、ユーザー側でも対処が難しいので、警告類は表示しない。

                        // 後処理コールバックが失敗扱いで呼ぶことで、メッセージが詰まるのを防いでおく。
                        handleShaderCompileComplete(
                            emitterOffsets,
                            completeCallback,
                            profiler,
                            null,
                            false, // isSuccessful
                            false, // shaderCompileFailed
                            null,  // shaderBinary
                            0,     // shaderBinarySize
                            0,     // shaderCount
                            0,     // computeShaderBinarySize
                            0,     // computeShaderCount
                            null); // writeShaderIndicesAction

                        return false;
                    }

                    if (conversionInputDataList == null || conversionInputDataList.Count <= 0)
                    {
                        if (OptionStore.RuntimeOptions.IsCommandLineMode)
                        {
                            string name = emitterData != null ? emitterData.Name : "invalid data";
                            Logger.Log("Console", LogLevels.Error, "Error on input data at {0}", name);
                        }

                        handleShaderCompileComplete(
                            emitterOffsets,
                            completeCallback,
                            profiler,
                            null,
                            false, // isSuccessful
                            false, // shaderCompileFailed
                            null,  // shaderBinary
                            0,     // shaderBinarySize
                            0,     // shaderCount
                            0,     // computeShaderBinarySize
                            0,     // computeShaderCount
                            null); // writeShaderIndicesAction

                        return false;
                    }

                    int combinerShaderIndex = -1;
                    var combinerShaderCode = conversionInputDataList[0].OverrideShaderCode;
                    if (!string.IsNullOrEmpty(combinerShaderCode))
                    {
                        if (!combinerShaderCodeList.Contains(combinerShaderCode))
                        {
                            // 初見のデータは追加した上で末尾のインデックスを得る
                            combinerShaderCodeList.Add(combinerShaderCode);
                            combinerShaderIndex = combinerShaderCodeList.Count - 1;
                        }
                        else
                        {
                            // 既出のデータは該当するインデックスを得るだけ
                            combinerShaderIndex = combinerShaderCodeList.IndexOf(combinerShaderCode);
                        }
                    }

                    // ShaderKeyのマクロと、シェーダのソースコードを生成する.
                    Tuple<uint, bool> result = generateShaderKey(conversionInputDataList[0], emitterData,
                                                                 0, combinerShaderIndex);
                    if (result.Item2 == false)
                    {
                        return false;
                    }
                    emitterShaderIndex.VertexShaderIndex = result.Item1;

                    if (emitterData.EmitterBasicSettingData.EmitterBasicBasicData.ProcessType == 2)
                    {
                        emitterShaderIndex.PixelShaderIndex = computeShaderIndex++;
                        emitterListForComputeShader.Add(emitterData);
                    }
                    else
                    {
                        emitterShaderIndex.PixelShaderIndex = uint.MaxValue;
                    }

                    // 描画パスの付属シェーダ(ShaderCompileDef1)
                    result = generateShaderKey(conversionInputDataList[0], emitterData,
                                               1, combinerShaderIndex);
                    if (result.Item2 == false)
                    {
                        return false;
                    }
                    emitterShaderIndex.UserVertexShaderIndex1 = result.Item1;

                    // 描画パスの付属シェーダ(ShaderCompileDef2)
                    result = generateShaderKey(conversionInputDataList[0], emitterData,
                                               2, combinerShaderIndex);
                    if (result.Item2 == false)
                    {
                        return false;
                    }
                    emitterShaderIndex.UserVertexShaderIndex2 = result.Item1;

                    // 各エミッタ(ResEmitter)のvertexShaderIndexを上書きするために、
                    // 各エミッタのバイナリヘッダのオフセットをここで保存しておく。
                    var emitter = emitterContext.BinaryStruct.DataModel;
                    var helper = BinaryHeaderHelperSelector.GetHelper(emitter);
                    var offset = helper.GetOffset(emitter, (uint)emitterContext.WritePosition);
                    emitterOffsets.Add(emitterContext.WritePosition + offset + conversionInputDataList[0].ShaderIndexOffset);
                    emitterShaderIndexes.Add(emitterShaderIndex);
                }

                // コンバイナシェーダコードを付加する
                if (combinerShaderCodeList.Any())
                {
                    var builder = new StringBuilder(fragmentShaderForShaderConverter);
                    int i = 0;
                    foreach (var code in combinerShaderCodeList)
                    {
                        builder.AppendLine(string.Format("#if COMBINER_SHADER_INDEX == {0}", i));
                        builder.AppendLine(code);
                        builder.AppendLine("#endif");
                        ++i;
                    }

                    fragmentShaderForShaderConverter = builder.ToString();
                }

                string standardOutput;
                string binaryOutputFilePath;
                byte[] grsnData;
                byte[] grscData;

                if (string.IsNullOrEmpty(shaderBinaryOutputFilePath))
                {
                    binaryOutputFilePath = OutputPath;
                }
                else
                {
                    binaryOutputFilePath = shaderBinaryOutputFilePath;
                }

                // バーテックスシェーダとピクセルシェーダを使ってシェーダバイナリを生成
                if (GenerateShaderBinary(
                    shaderKeyList,
                    vertexShaderForShaderConverter,
                    fragmentShaderForShaderConverter,
                    binaryOutputFilePath,
                    out standardOutput) == true)
                {
                    // ShaderConverter.exeがコンバートしたバイナリファイルを読み込む.
                    grsnData = File.ReadAllBytes(binaryOutputFilePath);
                }
                else
                {
                    string errorLogFilePath = Path.Combine(ShaderConverterExecuter.Instance.ErrorLogDirecotoryPath, "vertexAndPixelShaders.log");
                    GfxToolExecutionUtility.SaveAndShowErrorMessage(errorLogFilePath, standardOutput);

                    OutputErrorLog(errorLogFilePath, standardOutput, shaderKeyList);

                    handleShaderCompileComplete(
                        emitterOffsets,
                        completeCallback,
                        profiler,
                        null,
                        false, // isSuccessful
                        true,  // shaderCompileFailed
                        null,  // shaderBinary
                        0,     // shaderBinarySize
                        0,     // shaderCount
                        0,     // computeShaderBinarySize
                        0,     // computeShaderCount
                        null); // writeShaderIndicesAction
                    return false;
                }

                // コンピュートシェーダのコンバート時間を測定する
                ProfileTimer computeShaderProfile = new ProfileTimer("Compute shader");

                if (string.IsNullOrEmpty(computeShaderBinaryOutputFilePath))
                {
                    binaryOutputFilePath = ComputeShaderOutputPath;
                }
                else
                {
                    binaryOutputFilePath = computeShaderBinaryOutputFilePath;
                }

                // コンピュートシェーダを使ってシェーダバイナリを生成
                if (GenerateComputeShaderBinary(emitterListForComputeShader,
                                                computeShaderForShaderConverter,
                                                binaryOutputFilePath,
                                                out standardOutput) == true)
                {
                    // ShaderConverter.exeがコンバートしたコンピュートシェーダのバイナリファイルを読み込む.
                    grscData = File.ReadAllBytes(binaryOutputFilePath);
                }
                else
                {
                    string errorLogFilePath = Path.Combine(ShaderConverterExecuter.Instance.ErrorLogDirecotoryPath, "computeShader.log");
                    GfxToolExecutionUtility.SaveAndShowErrorMessage(errorLogFilePath, standardOutput);

                    OutputErrorLog(errorLogFilePath, standardOutput, shaderKeyList);

                    handleShaderCompileComplete(
                        emitterOffsets,
                        completeCallback,
                        profiler,
                        null,
                        false, // isSuccessful
                        true,  // shaderCompileFailed
                        null,  // shaderBinary
                        0,     // shaderBinarySize
                        0,     // shaderCount
                        0,     // computeShaderBinarySize
                        0,     // computeShaderCount
                        null); // writeShaderIndicesAction
                    return false;
                }

                computeShaderProfile.Stop(false);

                // 各ResShaderのvertexShaderIndexを、ShaderConverterのシェーダバイナリのindexに置き換える
                // 実際は、emitterContexts(emitterOffsets)の順番になっているだけ.
                int emitterIndex = 0;
                foreach (long emitterOffset in emitterOffsets)
                {
                    stream.Seek(emitterOffset, SeekOrigin.Begin);

                    // 頂点もフラグメントもバリエーションも全て解決済みなので、頂点シェーダのインデックスのみ設定する
                    // ※ 2016/07/04 ピクセルシェーダのインデックスは、compute shaderのインデックスとして仮に利用中
                    BinaryConversionUtility.ForResource.WriteStream(stream, emitterShaderIndexes[emitterIndex].VertexShaderIndex);
                    BinaryConversionUtility.ForResource.WriteStream(stream, emitterShaderIndexes[emitterIndex].PixelShaderIndex);
                    BinaryConversionUtility.ForResource.WriteStream(stream, emitterShaderIndexes[emitterIndex].UserVertexShaderIndex1);
                    BinaryConversionUtility.ForResource.WriteStream(stream, emitterShaderIndexes[emitterIndex].UserVertexShaderIndex2);

                    ++emitterIndex;
                }

                // 終端にシーク
                stream.Seek(0, SeekOrigin.End);

                // バイナリに書き込むか、別ファイルとしてシェーダバイナリを出力するか
                if (string.IsNullOrEmpty(shaderBinaryOutputFilePath) &&
                    string.IsNullOrEmpty(computeShaderBinaryOutputFilePath))
                {
                    // GRSN(通常のシェーダバイナリ)のヘッダをメインストリームに書き込む
                    var grsn = new BinaryStructHeader
                    {
                        Tag = "GRSN",
                        BinarySize = (uint)grsnData.Length,
                    };
                    grsn.Child = grsn.Offset;
                    uint grsnPos = (uint)stream.Position;
                    BinaryStructHeader.Empty.Write(stream);

                    // GRSC(コンピュートシェーダ)のヘッダを書き込む
                    var grsc = new BinaryStructHeader
                    {
                        Tag = "GRSC",
                        BinarySize = (uint)grscData.Length
                    };
                    uint grscPos = (uint)stream.Position;
                    BinaryStructHeader.Empty.Write(stream);

                    // ヘッダを2つ書き込んだ状態でシェーダバイナリ用のパディングを入れる
                    writePadding(stream, calcPadding((int)stream.Position, shaderAlignment));

                    // grsnのオフセットを求めて書き込む
                    grsn.Offset = (uint)stream.Position - grsnPos;
                    stream.Seek(grsnPos, SeekOrigin.Begin);
                    grsn.Write(stream);
                    stream.Seek(0, SeekOrigin.End);

                    // シェーダバイナリ用のサブストリームを作る。
                    var memoryStream = new MemoryStream();

                    // GRSNのシェーダバイナリを書き込む
                    memoryStream.Write(grsnData, 0, grsnData.Length);

                    // GRSNとGRSCの間のパディングを入れる
                    writePadding(memoryStream, calcPadding((int)memoryStream.Position, shaderAlignment));

                    // grscのオフセットを求めて書き込む
                    grsc.Offset = (uint)stream.Position + (uint)memoryStream.Position - grscPos;
                    stream.Seek(grscPos, SeekOrigin.Begin);
                    grsc.Write(stream);
                    stream.Seek(0, SeekOrigin.End);

                    // GRSCのコンピュートシェーダのバイナリを書き込む
                    memoryStream.Write(grscData, 0, grscData.Length);

                    handleShaderCompileComplete(
                        emitterOffsets,
                        completeCallback,
                        profiler,
                        computeShaderProfile,
                        true,
                        false, // shaderCompileFailed
                        memoryStream.ToArray(), // GetBuffer()はCapacity分書き込んでしまう
                        (long)grsnData.Length,  // shaderBinarySize
                        shaderKeyList.Count,
                        (long)grscData.Length,
                        emitterListForComputeShader.Count,
                        null);
                }
                else
                {
                    // 別ファイルとしてシェーダバイナリを出力する場合は、
                    // 本バイナリには何も書き込まない

                    byte[] data = new byte[0];

                    handleShaderCompileComplete(
                        emitterOffsets,
                        completeCallback,
                        profiler,
                        computeShaderProfile,
                        true,
                        false, // shaderCompileFailed
                        data.ToArray(), // GetBuffer()はCapacity分書き込んでしまう
                        0,  // shaderBinarySize
                        shaderKeyList.Count,
                        0,
                        emitterListForComputeShader.Count,
                        null);
                }
            }

            if (OptionStore.RuntimeOptions.IsCommandLineMode)
            {
                Logger.Log("Console", LogLevels.Information, "Shader convert is successfully finished.");
            }

            return true;
        }

        /// <summary>
        /// エラーログをコンソールまたはログビューに出力します。
        /// </summary>
        /// <param name="logPath">出力済みのログファイルパス</param>
        /// <param name="standardOutput">取得したログ</param>
        /// <param name="keyList">バリエーションとエミッタ名のテーブル</param>
        private static void OutputErrorLog(string logPath, string standardOutput, List<VariationElement> keyList)
        {
            string logTarget = OptionStore.RuntimeOptions.IsCommandLineMode ? "Console" : "LogView";
            Logger.Log(logTarget, LogLevels.Error, "Shader convert error:");

            string[] logLines = standardOutput.Split(new[] { Environment.NewLine }, StringSplitOptions.None);

            if (logPath.Contains("computeShader"))
            {
                Logger.Log(logTarget, LogLevels.Error, "  at compute shader");
            }
            else
            {
                bool outputName = false;
                string errorVariation = logLines.FirstOrDefault(l => l.Contains("[Variation: "));
                if (!string.IsNullOrEmpty(errorVariation))
                {
                    int errorIndex;
                    if (int.TryParse(errorVariation.Substring(12, errorVariation.Length - 13), out errorIndex))
                    {
                        if (0 <= errorIndex && errorIndex < keyList.Count)
                        {
                            Logger.Log(logTarget, LogLevels.Error, "  at EmitterSetName: {0}", keyList[errorIndex].EmitterSetName);
                            Logger.Log(logTarget, LogLevels.Error, "        EmitterName: {0}", keyList[errorIndex].EmitterName);
                            outputName = true;
                        }
                    }
                }

                if (!outputName)
                {
                    Logger.Log(logTarget, LogLevels.Error, "  at unknown variation");
                }
            }

            if (OptionStore.RuntimeOptions.IsCommandLineMode)
            {
                bool outputEnable = false;
                foreach (var logLine in logLines)
                {
                    if (logLine.Contains("----Error Log----"))
                    {
                        outputEnable = true;
                    }

                    if (outputEnable)
                    {
                        Logger.Log(logTarget, LogLevels.Error, "{0}", logLine);
                    }
                }

                if (outputEnable)
                {
                    Logger.Log(logTarget, LogLevels.Error, "----End of Log----");
                }
            }

            Logger.Log(logTarget, LogLevels.Error, "Error detail log file path:");
            Logger.Log(logTarget, LogLevels.Error, logPath);
        }

        /// <summary>
        /// シェーダインクルードディレクトリ文字列を作成し、取得します。
        /// </summary>
        /// <returns></returns>
        private static string GetStringDirectories()
        {
            string dirString = IncludeDirectoryPath;
            var dirs = OptionStore.ProjectConfig.AdditionalShaderIncludeDirectories;
            if (dirs.Any())
            {
                var dirBuilder = new StringBuilder();
                foreach (var dir in dirs)
                {
                    dirBuilder.Append(
                        "\" --include-directory=\"" +
                        PathUtility.ToAbsolutePath(dir, Path.GetDirectoryName(OptionStore.ProjectConfig.ConfigFilePath)));
                }

                dirString += dirBuilder.ToString();
            }

            return dirString;
        }

        /// <summary>
        /// 追加のシェーダ定義文字列を生成します。
        /// </summary>
        /// <returns>追加のシェーダ定義文字列を返します。</returns>
        private static string GetStringDefinitions()
        {
            List<string> definitions = new List<string>(OptionStore.ProjectConfig.AdditionalShaderPreprocessorDefinitions);

            // プロジェクト設定に応じて必要なシェーダ定義文字列を追加する
            {
                // テクスチャの原点を指定するためのシェーダ定義文字列を追加する
                if (definitions.Contains("_TEXTURE_ORIGIN_LOWER_LEFT") == false &&
                    definitions.Contains("_TEXTURE_ORIGIN_UPPER_LEFT") == false)
                {
                    string strTextureOriginMode = OptionStore.ProjectConfig.TextureOriginMode == TextureOriginMode.LowerLeft
                        ? "_TEXTURE_ORIGIN_LOWER_LEFT"
                        : "_TEXTURE_ORIGIN_UPPER_LEFT";

                    definitions.Add(strTextureOriginMode);
                }

                // デプスの範囲を指定するためのシェーダ定義文字列を追加する
                if (definitions.Contains("_DEPTH_MODE_NEAR_IS_MINUS_W") == false &&
                    definitions.Contains("_DEPTH_MODE_NEAR_IS_ZERO") == false)
                {
                    string strDepthMode = OptionStore.ProjectConfig.DepthMode == DepthMode.NearIsMinusW
                        ? "_DEPTH_MODE_NEAR_IS_MINUS_W"
                        : "_DEPTH_MODE_NEAR_IS_ZERO";

                    definitions.Add(strDepthMode);
                }
            }

            StringBuilder strDefinitions = new StringBuilder();

            // シェーダ定義文字列を生成
            foreach (var definition in definitions)
            {
                strDefinitions.Append(" --preprocessor-definition=" + definition);
            }

            return strDefinitions.ToString();
        }

        /// <summary>
        /// バリエーションとエミッタセット・エミッタ名を対応付けるための要素です。
        /// </summary>
        private class VariationElement
        {
            /// <summary>
            /// コンストラクタ
            /// </summary>
            /// <param name="emitter">エミッタデータ</param>
            /// <param name="shaderKey">シェーダキー</param>
            /// <param name="combinerShaderIndex">コンバイナシェーダのインデックス(-1で不使用)</param>
            public VariationElement(EmitterData emitter, string shaderKey, int combinerShaderIndex)
            {
                if (emitter != null)
                {
                    this.EmitterSetName = emitter.FindParentOfType<EmitterSetData>().Name;
                    this.EmitterName = emitter.Name;
                }

                this.ShaderKey = shaderKey;
                this.CombinerShaderIndex = combinerShaderIndex;
            }

            /// <summary>
            /// エミッタセット名を取得します。
            /// </summary>
            public string EmitterSetName { get; private set; }

            /// <summary>
            /// エミッタ名を取得します。
            /// </summary>
            public string EmitterName { get; private set; }

            /// <summary>
            /// シェーダキーを取得します。
            /// </summary>
            public string ShaderKey { get; private set; }

            /// <summary>
            /// コンバイナシェーダのインデックスを取得します。
            /// </summary>
            public int CombinerShaderIndex { get; private set; }
        }
    }
}
