﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using System.Threading.Tasks;
using EffectMaker.BusinessLogic.DataModelOperation;
using EffectMaker.BusinessLogic.UserData;
using EffectMaker.DataModel;
using EffectMaker.DataModel.DataModels;
using EffectMaker.DataModel.Manager;
using EffectMaker.DataModel.Specific.DataModels;
using EffectMaker.DataModelLogic.Events;
using EffectMaker.Foundation.Extensions;
using EffectMaker.Foundation.Log;

namespace EffectMaker.DataModelLogic.DataModelProxies
{
    /// <summary>
    /// Data model proxy for emitter data.
    /// </summary>
    public class EmitterDataProxy : DataModelProxy, IEmitterBuilder
    {
        /// <summary>
        /// レンダーステートのプロパティとカスタムプロパティを対応付ける辞書
        /// </summary>
        private static readonly Dictionary<string, List<string>> RenderStateDictionary = new Dictionary<string, List<string>>();

        /// <summary>
        /// Constructor.
        /// </summary>
        /// <param name="dataModel">The data model for the proxy.</param>
        public EmitterDataProxy(DataModelBase dataModel) :
            base(dataModel)
        {
            RenderStateDictionary.Clear();
            RenderStateDictionary["ZbufferAlphaTestType"] = new List<string>
            {
////                "IsBlendEnable",
////                "IsDepthTest",
////                "DepthFunc",
////                "IsDepthMask",
                "IsAlphaTest",  // これ一つでリロードトリガーになる
////                "AlphaFunc",
////                "AlphaThreshold",
            };
            RenderStateDictionary["EnableZBufferTest"] = new List<string>
            {
                "IsDepthTest",
                "DepthFunc",
                "IsDepthMask",
            };
            RenderStateDictionary["ZBufferTestPassType"] = new List<string>
            {
                "DepthFunc",
            };
            RenderStateDictionary["EnableZBufferWrite"] = new List<string>
            {
                "IsDepthMask",
            };
            RenderStateDictionary["EnableAlphaTest"] = new List<string>
            {
                "IsAlphaTest",  // これ一つでリロードトリガーになる
////                "AlphaFunc",
////                "AlphaThreshold",
            };
            RenderStateDictionary["AlphaTestPassType"] = new List<string>
            {
                "AlphaFunc",
            };
            RenderStateDictionary["AlphaTestRefValue"] = new List<string>
            {
                "AlphaThreshold",
            };
            RenderStateDictionary["EnableTransparency"] = new List<string>
            {
                "IsBlendEnable",
            };

            DataModelProxy.PropertyModified += this.OnDataModelPropertyModified;

            this.UpdateUserData();
        }

        /// <summary>
        /// Disposes the instance.
        /// </summary>
        public override void Dispose()
        {
            DataModelProxy.PropertyModified -= this.OnDataModelPropertyModified;
            base.Dispose();
        }

        /// <summary>
        /// Update user data models.
        /// </summary>
        public void UpdateUserData()
        {
            var emitterData = this.DataModel as EmitterData;

            // First clear all the user pages.
            var userPages = emitterData.UserDataList.ToArray();
            emitterData.UserDataList.Clear();
            userPages.ForEach(p => emitterData.Children.Remove(p));

            // Create user data models.
            IEnumerable<UserDataBase> userDataModels =
                UserDataManager.CreateUserDataForOwner(typeof(EmitterData));

            // Create one user page data for each of the user data model.
            var createdUserPages = userDataModels.Select(
                dm => new UserPageData() { ContentsData = dm });

            // Add these created user pages.
            foreach (UserPageData pageData in createdUserPages)
            {
                emitterData.UserDataList.Add(pageData);
                emitterData.AddChild(pageData);
            }

            // Fire event.
            this.TriggerPropertyModifiedEvent("UserDataList");
        }

        /// <summary>
        /// Add the given data model to the children data models.
        /// </summary>
        /// <param name="dataModel">The data model.</param>
        public override void AddChildDataModel(DataModelBase dataModel)
        {
            if (dataModel is EmitterData)
            {
                this.AddEmitter((EmitterData)dataModel);
            }
            else if (dataModel is FieldDataBase)
            {
                this.AddField((FieldDataBase)dataModel);
            }
            else
            {
                throw new ArgumentException();
            }
        }

        /// <summary>
        /// Add the given data model to the children data models.
        /// </summary>
        /// <param name="index">The index of the data model.</param>
        /// <param name="dataModel">The data model.</param>
        public override void InsertChildDataModel(int index, DataModelBase dataModel)
        {
            if (dataModel is EmitterData)
            {
                this.InsertEmitter(index, (EmitterData)dataModel);
            }
            else if (dataModel is FieldDataBase)
            {
                this.InsertField(index, (FieldDataBase)dataModel);
            }
            else
            {
                throw new ArgumentException();
            }
        }

        /// <summary>
        /// Remove the specified data model from the children data models.
        /// </summary>
        /// <param name="dataModel">The dat model to be removed.</param>
        public override void RemoveChildDataModel(DataModelBase dataModel)
        {
            if (dataModel is EmitterData)
            {
                this.RemoveEmitter((EmitterData)dataModel);
            }
            else if (dataModel is FieldDataBase)
            {
                this.RemoveField((FieldDataBase)dataModel);
            }
            else
            {
                throw new ArgumentException();
            }
        }

        /// <summary>
        /// Move the specified data model to the specified index.
        /// </summary>
        /// <param name="dataModel">The data model to be moved.</param>
        /// <param name="distance">
        /// The distance to move the child.
        /// E.g. 2 means move down 2 items, -3 means move up 3 items.
        /// </param>
        public override void MoveChildDataModel(DataModelBase dataModel, int distance)
        {
            if (dataModel is EmitterData)
            {
                this.MoveEmitter((EmitterData)dataModel, distance);
            }
            else if (dataModel is FieldDataBase)
            {
                this.MoveField((FieldDataBase)dataModel, distance);
            }
            else
            {
                throw new ArgumentException();
            }
        }

        /// <summary>
        /// Create a new emitter as a child of the emitter set.
        /// </summary>
        /// <param name="name">The name of the emitter.</param>
        /// <returns>The created emitter.</returns>
        public EmitterData CreateEmitter(string name)
        {
            var ownerData = this.DataModel as EmitterData;
            if (ownerData == null)
            {
                return null;
            }

            // Create the emitter.
            EmitterData emitterData = DataModelCreator.CreateEmitterDataModel();
            emitterData.Name = name;

            return emitterData;
        }

        /// <summary>
        /// Add the given emitter data to the emitter set.
        /// </summary>
        /// <param name="emitterData">The emitter data.</param>
        public void AddEmitter(EmitterData emitterData)
        {
            var ownerData = this.DataModel as EmitterData;
            if (ownerData == null)
            {
                return;
            }

            // Check if the emitter has already been added.
            if (ownerData.EmitterList.IndexOf(emitterData) >= 0)
            {
                return;
            }

            // Add the emitter to the emitter set.
            ownerData.EmitterList.Add(emitterData);
            ownerData.AddChild(emitterData);

            // Fire event.
            this.TriggerPropertyModifiedEvent("EmitterList");
        }

        /// <summary>
        /// Add the given emitter data to the emitter set.
        /// </summary>
        /// <param name="index">The index of the emitter.</param>
        /// <param name="emitterData">The emitter data.</param>
        public void InsertEmitter(int index, EmitterData emitterData)
        {
            var ownerData = this.DataModel as EmitterData;
            if (ownerData == null)
            {
                return;
            }

            // Check if the emitter has already been added.
            if (ownerData.EmitterList.IndexOf(emitterData) >= 0)
            {
                return;
            }

            // Add the emitter to the emitter set.
            ownerData.EmitterList.Insert(index, emitterData);
            ownerData.InsertChild(index, emitterData);

            // Fire event.
            this.TriggerPropertyModifiedEvent("EmitterList");
        }

        /// <summary>
        /// Remove the specified emitter from the emitter set.
        /// </summary>
        /// <param name="emitterData">The emitter to be removed.</param>
        public void RemoveEmitter(EmitterData emitterData)
        {
            var ownerData = this.DataModel as EmitterData;
            if (ownerData == null)
            {
                return;
            }

            // Remove the emitter from the emitter set.
            ownerData.EmitterList.Remove(emitterData);
            ownerData.RemoveChild(emitterData);

            // Fire event.
            this.TriggerPropertyModifiedEvent("EmitterList");
        }

        /// <summary>
        /// Move the specified child to the specified index.
        /// </summary>
        /// <param name="child">The child to be moved.</param>
        /// <param name="distance">
        /// The distance to move the child.
        /// E.g. 2 means move down 2 items, -3 means move up 3 items.
        /// </param>
        public void MoveEmitter(EmitterData child, int distance)
        {
            var ownerData = this.DataModel as EmitterData;
            if (ownerData == null)
            {
                return;
            }

            // Find the original index.
            int originalIndex = ownerData.EmitterList.IndexOf(child);
            int newIndex = originalIndex + distance;

            // Remove the emitter from the emitter set.
            ownerData.EmitterList.Remove(child);

            // Insert the emitter back to the specified index.
            ownerData.EmitterList.Insert(newIndex, child);

            // Fire event.
            this.TriggerPropertyModifiedEvent("EmitterList");
        }

        /// <summary>
        /// Create a new field as a child of the emitter.
        /// </summary>
        /// <param name="fieldType">The type of the field.</param>
        /// <returns>The created field.</returns>
        public FieldDataBase CreateField(FieldTypes fieldType)
        {
            var emitterData = this.DataModel as EmitterData;
            if (emitterData == null)
            {
                return null;
            }

            // Create the field.
            FieldDataBase fieldData = DataModelCreator.CreateFieldDataModel(fieldType);

            return fieldData;
        }

        /// <summary>
        /// Add the given field data to the emitter.
        /// </summary>
        /// <param name="fieldData">The field data.</param>
        public void AddField(FieldDataBase fieldData)
        {
            var emitterData = this.DataModel as EmitterData;
            if (emitterData == null)
            {
                return;
            }

            // Check if the field has already been added.
            if (emitterData.FieldList.IndexOf(fieldData) >= 0)
            {
                return;
            }

            // Add the field to the emitter.
            emitterData.FieldList.Add(fieldData);
            emitterData.AddChild(fieldData);

            // Fire event.
            this.TriggerPropertyModifiedEvent("FieldList");
            this.TriggerPropertyModifiedEvent("ActiveFieldList");
        }

        /// <summary>
        /// Add the given field data to the emitter.
        /// </summary>
        /// <param name="index">The index of the field.</param>
        /// <param name="fieldData">The field data.</param>
        public void InsertField(int index, FieldDataBase fieldData)
        {
            var emitterData = this.DataModel as EmitterData;
            if (emitterData == null)
            {
                return;
            }

            // Check if the field has already been added.
            if (emitterData.FieldList.IndexOf(fieldData) >= 0)
            {
                return;
            }

            // Add the field to the field.
            emitterData.FieldList.Insert(index, fieldData);
            emitterData.InsertChild(index, fieldData);

            // Fire event.
            this.TriggerPropertyModifiedEvent("FieldList");
            this.TriggerPropertyModifiedEvent("ActiveFieldList");
        }

        /// <summary>
        /// Remove the specified field from the emitter.
        /// </summary>
        /// <param name="fieldData">The field to be removed.</param>
        public void RemoveField(FieldDataBase fieldData)
        {
            var emitterData = this.DataModel as EmitterData;
            if (emitterData == null)
            {
                return;
            }

            // Remove the field from the emitter.
            emitterData.FieldList.Remove(fieldData);
            emitterData.RemoveChild(fieldData);

            // Fire event.
            this.TriggerPropertyModifiedEvent("FieldList");
            this.TriggerPropertyModifiedEvent("ActiveFieldList");
        }

        /// <summary>
        /// Move the specified child to the specified index.
        /// </summary>
        /// <param name="child">The child to be moved.</param>
        /// <param name="distance">
        /// The distance to move the child.
        /// E.g. 2 means move down 2 items, -3 means move up 3 items.
        /// </param>
        public void MoveField(FieldDataBase child, int distance)
        {
            var ownerData = this.DataModel as EmitterData;
            if (ownerData == null)
            {
                return;
            }

            // Find the original index.
            int originalIndex = ownerData.FieldList.IndexOf(child);
            int newIndex = originalIndex + distance;

            // Remove the field data from the owner.
            ownerData.FieldList.Remove(child);

            // Insert the field data back to the specified index.
            ownerData.FieldList.Insert(newIndex, child);

            // Fire event.
            this.TriggerPropertyModifiedEvent("FieldList");
            this.TriggerPropertyModifiedEvent("ActiveFieldList");
        }

        /// <summary>
        /// Create a new reserved shader.
        /// </summary>
        /// <param name="reservedShaderUserDataInfo">The user data info for the reserved shader to create.</param>
        /// <returns>The created reserved shader node data model.</returns>
        public ReservedShaderNodeData CreateReservedShader(UserDataInfo reservedShaderUserDataInfo)
        {
            var emitterData = this.DataModel as EmitterData;
            if (emitterData == null)
            {
                return null;
            }

            // Create the reserved shader node data.
            var nodeData = new ReservedShaderNodeData();

            // Create the user data with the provided user data info and add it to the user page data.
            nodeData.PageData.ContentsData =
                UserDataManager.CreateUserDataFromInfo(reservedShaderUserDataInfo);

            return nodeData;
        }

        /// <summary>
        /// Add the given reserved shader data to the emitter.
        /// </summary>
        /// <param name="reservedShaderData">The reserved shader node data model to add.</param>
        public void AddReservedShader(ReservedShaderNodeData reservedShaderData)
        {
            var emitterData = this.DataModel as EmitterData;
            if (emitterData == null)
            {
                return;
            }

            // Add the reserved shader node data to the emitter.
            emitterData.ReservedShader = reservedShaderData;
            emitterData.AddChild(reservedShaderData);

            // Fire event.
            this.TriggerPropertyModifiedEvent("ReservedShader");
            this.TriggerPropertyModifiedEvent("ActiveReservedShader");
        }

        /// <summary>
        /// Remove the specified reserved shader from the emitter.
        /// </summary>
        /// <param name="reservedShaderData">The reserved shader to be removed.</param>
        public void RemoveReservedShader(ReservedShaderNodeData reservedShaderData)
        {
            var emitterData = this.DataModel as EmitterData;
            if (emitterData == null)
            {
                return;
            }

            // Remove the stripe from the emitter.
            emitterData.ReservedShader = null;
            emitterData.RemoveChild(reservedShaderData);

            // Fire event.
            this.TriggerPropertyModifiedEvent("ReservedShader");
            this.TriggerPropertyModifiedEvent("ActiveReservedShader");
        }

        /// <summary>
        /// カスタムアクションデータモデルの作成
        /// </summary>
        /// <returns>カスタムアクションデータモデル.</returns>
        public CustomActionData CreateCustomAction()
        {
            var emitterData = this.DataModel as EmitterData;
            if (emitterData == null)
            {
                return null;
            }

            return new CustomActionData();
        }

        /// <summary>
        /// Add the given custom action data to the emitter.
        /// </summary>
        /// <param name="customActionData">The custom action data.</param>
        public void AddCustomAction(CustomActionData customActionData)
        {
            var emitterData = this.DataModel as EmitterData;
            if (emitterData == null)
            {
                return;
            }

            // Add the field to the emitter.
            emitterData.CustomActionData = customActionData;
            emitterData.AddChild(customActionData);

            // Fire event.
            this.TriggerPropertyModifiedEvent("CustomActionData");
            this.TriggerPropertyModifiedEvent("ActiveCustomActionIndex");
        }

        /// <summary>
        /// Remove the specified custom action from the emitter.
        /// </summary>
        /// <param name="customActionData">The custom action to be removed.</param>
        public void RemoveCustomAction(CustomActionData customActionData)
        {
            var emitterData = this.DataModel as EmitterData;
            if (emitterData == null)
            {
                return;
            }

            // Remove the stripe from the emitter.
            emitterData.CustomActionData = null;
            emitterData.RemoveChild(customActionData);

            // Fire event.
            this.TriggerPropertyModifiedEvent("CustomActionData");
            this.TriggerPropertyModifiedEvent("ActiveCustomActionIndex");
        }

        /// <summary>
        /// エミッタ拡張パラメータモデルの作成
        /// </summary>
        /// <returns>エミッタ拡張パラメータモデル.</returns>
        public EmitterExtParams CreateEmitterExtParams()
        {
            var emitterData = this.DataModel as EmitterData;
            if (emitterData == null)
            {
                return null;
            }

            return new EmitterExtParams();
        }

        /// <summary>
        /// Add the given Emitter Extension Parameters to the emitter.
        /// </summary>
        /// <param name="emitterExtParams">The Emitter Extension Parameters.</param>
        public void AddEmitterExtParams(EmitterExtParams emitterExtParams)
        {
            var emitterData = this.DataModel as EmitterData;
            if (emitterData == null)
            {
                return;
            }

            // Add the field to the emitter.
            emitterData.EmitterExtParams = emitterExtParams;
            emitterData.AddChild(emitterExtParams);

            // Fire event.
            this.TriggerPropertyModifiedEvent("EmitterExtParams");
            this.TriggerPropertyModifiedEvent("SelectedEmitterExtParams");
        }

        /// <summary>
        /// Remove the specified Emitter Extension Parameters from the emitter.
        /// </summary>
        /// <param name="emitterExtParams">The Emitter Extension Parameters to be removed.</param>
        public void RemoveEmitterExtParams(EmitterExtParams emitterExtParams)
        {
            var emitterData = this.DataModel as EmitterData;
            if (emitterData == null)
            {
                return;
            }

            // Remove the stripe from the emitter.
            emitterData.EmitterExtParams = null;
            emitterData.RemoveChild(emitterExtParams);

            // Fire event.
            this.TriggerPropertyModifiedEvent("EmitterExtParams");
            this.TriggerPropertyModifiedEvent("SelectedEmitterExtParams");
        }

        /// <summary>
        /// Create a new stripe as a child of the emitter.
        /// </summary>
        /// <param name="stripeType">The type of the stripe.</param>
        /// <returns>The created stripe.</returns>
        public StripeData CreateStripe(Type stripeType)
        {
            var emitterData = this.DataModel as EmitterData;
            if (emitterData == null)
            {
                return null;
            }

            if (stripeType.Equals(typeof(StripeComplexData)))
            {
                // Create the strip data.
                return new StripeComplexData();
            }
            else if (stripeType.Equals(typeof(StripeHistoryData)))
            {
                // Add the strip data to the emitter.
                return new StripeHistoryData();
            }
            else if (stripeType.Equals(typeof(StripeSuperData)))
            {
                // Add the strip data to the emitter.
                return new StripeSuperData();
            }

            return null;
        }

        /// <summary>
        /// Add the given stripe data to the emitter.
        /// </summary>
        /// <param name="stripeData">The stripe data.</param>
        public void AddStripe(StripeData stripeData)
        {
            var emitterData = this.DataModel as EmitterData;
            if (emitterData == null)
            {
                return;
            }

            // Add the field to the emitter.
            emitterData.StripeData = stripeData;
            emitterData.AddChild(stripeData);

            // Fire event.
            this.TriggerPropertyModifiedEvent("StripeData");
        }

        /// <summary>
        /// Remove the specified stripe from the emitter.
        /// </summary>
        /// <param name="stripeData">The stripe to be removed.</param>
        public void RemoveStripe(StripeData stripeData)
        {
            var emitterData = this.DataModel as EmitterData;
            if (emitterData == null)
            {
                return;
            }

            // Remove the stripe from the emitter.
            emitterData.StripeData = null;
            emitterData.RemoveChild(stripeData);

            // Fire event.
            this.TriggerPropertyModifiedEvent("StripeData");
        }

        /// <summary>
        /// Handle PropertyModified event from data model proxy.
        /// </summary>
        /// <param name="sender">The sender of the event.</param>
        /// <param name="e">The event arguments.</param>
        private void OnDataModelPropertyModified(
            object sender,
            DataModelPropertyModifiedEventArgs e)
        {
            var myDataModel = this.DataModel as EmitterData;
            if (myDataModel == null)
            {
                return;
            }

            if (e.DataModel == myDataModel.EmitterBasicSettingData.EmitterBasicRenderData)
            {
                if (RenderStateDictionary.ContainsKey(e.PropertyName))
                {
                    foreach (string customProperty in RenderStateDictionary[e.PropertyName])
                    {
                        this.TriggerPropertyModifiedEvent(
                            customProperty,
                            myDataModel.EmitterBasicSettingData.EmitterBasicRenderData);
                    }
                }
            }
            else if (e.DataModel == myDataModel.CustomActionData)
            {
                if (e.PropertyName == "SelectedSettingIndex")
                {
                    this.TriggerPropertyModifiedEvent("ActiveCustomActionIndex", myDataModel);
                }
            }
        }
    }
}
