﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using System;
using System.Collections.Generic;

using EffectMaker.DataModelLogic.BinaryConverters;
using EffectMaker.Foundation.Log;

namespace EffectMaker.DataModelLogic.BinaryData
{
    /// <summary>
    /// Class for the definition of the converter to use in a binary field.
    /// </summary>
    public class ConverterDefinition
    {
        /// <summary>The parameters for the converter.</summary>
        private KeyValuePair<string, object>[] parameters;

        /// <summary>
        /// Constructor.
        /// </summary>
        /// <param name="converterType">The type of the converter.</param>
        /// <param name="parameters">The parameters for the converter.</param>
        public ConverterDefinition(
            Type converterType,
            params KeyValuePair<string, object>[] parameters)
        {
            this.ConverterType = converterType;
            this.parameters = parameters;
        }

        /// <summary>
        /// Get the type of the converter.
        /// </summary>
        public Type ConverterType { get; private set; }

        /// <summary>
        /// Create the converter with the parameters set to it.
        /// </summary>
        /// <returns>The created converter.</returns>
        public ConverterBase CreateConverter()
        {
            // Create the converter.
            var converter =
                Activator.CreateInstance(this.ConverterType) as ConverterBase;

            // Set parameters to the converter.
            foreach (KeyValuePair<string, object> param in this.parameters)
            {
                if (converter.SetParameter(param.Key, param.Value) == false)
                {
                    Logger.Log(
                        "ConverterDefinition.CreateConverter : The parameter list contains an invalid parameter '{0}' = {1}",
                        param.Key,
                        param.Value == null ? "null" : param.Value.ToString());
                }
            }

            return converter;
        }
    }
}
