﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using System;
using EffectMaker.DataModel.Attributes;
using EffectMaker.DataModel.DataModels;
using EffectMaker.Foundation.Core;
using EffectMaker.Foundation.Extensions;
using EffectMaker.Foundation.Interfaces;
using EffectMaker.Foundation.Primitives;
using EffectMaker.Foundation.Utility;

namespace EffectMaker.DataModelLogic.BinaryConverters
{
    /// <summary>
    /// Class for the default field converter that passes value directly to output.
    /// </summary>
    [EffectMaker.Foundation.Attributes.Default]
    public class ValueScalingConverter : ConverterBase
    {
        /// <summary>Constant parameter name for the selected channel index.</summary>
        private const string ScaleParamName = "Scale";

        /// <summary>The converter parameter holder.</summary>
        private static ConverterParamHolder paramHolder = new ConverterParamHolder();

        /// <summary>
        /// Static constructor.
        /// </summary>
        static ValueScalingConverter()
        {
            // Register parameters.
            paramHolder.RegisterParameter(
                ScaleParamName,
                "The scale.",
                ValidateScaleParamter);
        }

        /// <summary>
        /// Constructor.
        /// </summary>
        public ValueScalingConverter() :
            base(paramHolder)
        {
        }

        /// <summary>
        /// Get the maximum number of input values.
        /// </summary>
        public override int MaxInputValueCount
        {
            get { return 1; }
        }

        /// <summary>
        /// Get the description of the converter.
        /// </summary>
        public override string Description
        {
            get { return Properties.Resources.ValueScalingConverterDesc; }
        }

        /// <summary>
        /// Get the output value description.
        /// </summary>
        public override string OutputValueDescription
        {
            get { return Properties.Resources.ValueScalingConverterOutputValueDesc; }
        }

        /// <summary>
        /// Get the description of the input value.
        /// </summary>
        /// <param name="index">The index to the input value.</param>
        /// <returns>The description.</returns>
        public override string GetInputValueDescription(int index)
        {
            if (index == 0)
            {
                return Properties.Resources.ValueScalingConverterInputValue0Desc;
            }
            else
            {
                return Properties.Resources.WarningInvalidConverterInputValueIndex;
            }
        }

        /// <summary>
        /// Convert data model field values.
        /// </summary>
        /// <returns>True on success.</returns>
        public override bool Convert()
        {
            object param;
            if (this.GetParameter(ScaleParamName, out param) != true)
            {
                return false;
            }

            object convertedValue = param;
            float validScale;
            if (TypeConversionUtility.TryConvert(param.GetType(), typeof(float), ref convertedValue))
            {
                validScale = (float)convertedValue;
            }
            else
            {
                return false;
            }

            object inputValue = this.GetInputValue(0);

            if (inputValue is IPrimitive)
            {
                var outputValue = inputValue.CloneIfCloneable();

                var primI = outputValue as Primitivei;
                var primU = outputValue as Primitiveu;
                var primF = outputValue as Primitivef;
                var primD = outputValue as Primitived;

                if (primI != null)
                {
                    for (int i = 0; i < primI.Count; ++i)
                    {
                        primI[i] = (int)(primI[i] * validScale);
                    }

                    this.SetOutputValue(primI);
                }
                else if (primU != null)
                {
                    for (int i = 0; i < primU.Count; ++i)
                    {
                        primU[i] = (uint)(primU[i] * validScale);
                    }

                    this.SetOutputValue(primU);
                }
                else if (primF != null)
                {
                    for (int i = 0; i < primF.Count; ++i)
                    {
                        primF[i] *= validScale;
                    }

                    this.SetOutputValue(primF);
                }
                else if (primD != null)
                {
                    for (int i = 0; i < primD.Count; ++i)
                    {
                        primD[i] *= validScale;
                    }

                    this.SetOutputValue(primD);
                }
                else
                {
                    return false;
                }
            }
            else if (inputValue is int)
            {
                this.SetOutputValue((int)((int)inputValue * validScale));
            }
            else if (inputValue is uint)
            {
                this.SetOutputValue((uint)((uint)inputValue * validScale));
            }
            else if (inputValue is long)
            {
                this.SetOutputValue((long)((long)inputValue * validScale));
            }
            else if (inputValue is ulong)
            {
                this.SetOutputValue((ulong)((ulong)inputValue * validScale));
            }
            else if (inputValue is float)
            {
                this.SetOutputValue((float)inputValue * validScale);
            }
            else if (inputValue is double)
            {
                this.SetOutputValue((double)inputValue * validScale);
            }
            else
            {
                return false;
            }

            return true;
        }

        /// <summary>
        /// Validate channel index parameter.
        /// </summary>
        /// <param name="value">The parameter value.</param>
        /// <returns>True if the value is valid.</returns>
        private static bool ValidateScaleParamter(object value)
        {
            bool result = false;
            if (value is string)
            {
                string target = (string)value;
                float tmpFloat;
                result |= float.TryParse(target, out tmpFloat);
            }
            else if (value != null)
            {
                object convertedValue = value;
                result |= TypeConversionUtility.TryConvert(
                    value.GetType(),
                    typeof(float),
                    ref convertedValue);
            }

            return result;
        }
    }
}
